/**
 * \brief RefString object module for CoolRain.
 */
#if !defined(COOLRAIN__REFSTRING__H__)
#define COOLRAIN__REFSTRING__H__

#include <string.h>
#include <stdlib.h>
#include <glib/gtypes.h>
#include <glib/gmacros.h>

#include <CoolRain/memory.h>


G_BEGIN_DECLS;

/**
 *
 */
typedef struct {
	char const	*begin;		/*!< Pointer to start string. */
	char const	*end;		/*!< Pointer to end of string, or null if null terminated. */
} coolrain_refstring_t;



/**
 * Fix end pointer for null terminated string.
 * \param str		Refstring object
 */
inline static void coolrain_refstring_fix(coolrain_refstring_t *str)
{
	if (str->end == NULL)
		str->end = str->begin + strlen(str->begin);
}




/**
 * Length of refstring
 * \param str		Refstring object
 * \return Length of string.
 */
inline static size_t coolrain_refstring_length(coolrain_refstring_t const *str)
{
	return (str->end == NULL)? strlen(str->begin) : (size_t)(str->end - str->begin);
}

/**
 * Compare refstring.
 * \param l		Left hand string.
 * \param rbegin	Pointer to begiging of right hand string
 * \param rend		Pointer to end of right hand, or null if null terminated
 * \return TURE if l and (rbegin, rend) is equal, otherwise FALSE.
 */
static inline gboolean coolrain_refstring_equal_(coolrain_refstring_t const* l, char const *rbegin, char const *rend)
{
	size_t rlen = (rend == NULL)? strlen(rbegin) : (size_t)(rend - rbegin);
	return (coolrain_refstring_length(l) != rlen)? FALSE : (memcmp(l->begin, rbegin, rlen) == 0);
}

/**
 * Compare refstring (alternate)
 * \param l		Left hand string
 * \param r		Right hand string.
 * \return TRUE if l and r is equal, otherwise FALSE.
 */
static inline gboolean coolrain_refstring_equal(coolrain_refstring_t const *l, coolrain_refstring_t const *r)
{
	return coolrain_refstring_equal_(l, r->begin, r->end);
}



/**
 * Calcurate hash value of refstring
 * \param str		Refstring object
 * \return Hash value.
 */
static inline guint coolrain_refstring_hash(coolrain_refstring_t const *str)
{
	guint r = 0;
	char const *p = str->begin;

	if (str->end == NULL)	{ while (*p) { r <<= 2; r += *(p++); } }
	else			{ while (p != str->end) { r <<= 2; r += *(p++); } }

	return r;
}


static inline int coolrain_refstring_toint(coolrain_refstring_t const *str)
{
	char buf[64];
	size_t len = MIN(63, coolrain_refstring_length(str));
	memcpy(buf, str->begin, len);
	buf[len] = '\0';

	return strtol(buf, (char**)NULL, 10);
}


static inline char *coolrain_refstring_dup(coolrain_refstring_t const *str)
{
	if (str != NULL) {
		size_t len = coolrain_refstring_length(str);
		char *r = (char*)coolrain_malloc(len + 1);
		memcpy(r, str->begin, len);
		r[len] = '\0';
		return r;
	}
	return NULL;
}

G_END_DECLS;

#endif

