/*
 * The MIT License

   BLDConograph (Bravais lattice determination module in Conograph)

Copyright (c) <2012> <Ryoko Oishi-Tomiyasu, KEK>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
 *
 */
#ifndef SYMMAT3_H_
#define SYMMAT3_H_

#include <complex>
#include "VecDat3.hh"

using namespace std;

// Class of a symmmetric matrix (a_ij) determined by m_mat as below.
//          m_mat[0] m_mat[3] m_mat[4]
// (a_ij) = m_mat[3] m_mat[1] m_mat[5]	for each 0<=i<=2, 0<=j<=2
//          m_mat[4] m_mat[5] m_mat[2]
template <class T>
class SymMat3
{
	template <class S>
	friend inline SymMat3<S> real(const SymMat3< complex<S> >& rhs);
	template <class S>
	friend inline SymMat3< complex<S> > conj(const SymMat3< complex<S> >& rhs);
	template <class S>
	friend inline VecDat3<S> symmetric_transform(const SymMat3<S>&, const VecDat3<S>&);
	template <class S>
	friend inline SymMat3<S> product_transpose(const VecDat3<S>&, const VecDat3<S>&);
	template <class S>
	friend inline SymMat3<S> product_self_transpose(const VecDat3<S>&);

protected:
	enum{ ISIZE = 3, NUM_ELEMENT=6 };
	T m_mat[NUM_ELEMENT];
	
	inline T& operator[](const int&);
	inline const T& operator[](const int&) const;

public:
	SymMat3();
	SymMat3(const T&);
	SymMat3(const SymMat3<T>&); // copy constructor
	virtual ~SymMat3();
	SymMat3<T>& operator=(const T&);
	SymMat3<T>& operator=(const SymMat3<T>&);
	SymMat3<T>& operator+=(const SymMat3<T>&);
	SymMat3<T>& operator-=(const SymMat3<T>&);
	SymMat3<T>& operator*=(const T&);
	inline SymMat3<T> operator+(const SymMat3<T>&) const;
	inline SymMat3<T> operator-(const SymMat3<T>&) const;
	inline SymMat3<T> operator*(const T&) const;
	inline T& operator()(const int&, const int&);
	inline const T& operator()(const int&, const int&) const;
	static const SymMat3<T> DiagMatrix(const T&, const T&, const T&);
};

template <class T>
SymMat3<T>::SymMat3()
{
	for(int k=0; k<NUM_ELEMENT; k++) m_mat[k]=0;
}

template <class T>
SymMat3<T>::SymMat3(const T& a)
{
	for(int k=0; k<NUM_ELEMENT; k++) m_mat[k]=a;
}

template <class T>
SymMat3<T>::SymMat3(const SymMat3 &rhs)
{
	for(int k=0; k<NUM_ELEMENT; k++) m_mat[k]=rhs.m_mat[k];
}

template <class T>
SymMat3<T>::~SymMat3()
{
}

template <class T>
const SymMat3<T> SymMat3<T>::DiagMatrix(const T& a0, const T& a1, const T& a2)
{
	SymMat3<T> ans;
	ans.m_mat[0]=a0;
	ans.m_mat[1]=a1;
	ans.m_mat[2]=a2;
	return ans;
}

template <class T>
SymMat3<T>& SymMat3<T>::operator=(const T& a)
{
	for(int k=0; k<NUM_ELEMENT; k++) m_mat[k]=a;
	return *this;
}

template <class T>
SymMat3<T>& SymMat3<T>::operator=(const SymMat3<T>& rhs)
{
	if(this != &rhs)
		for(int k=0; k<NUM_ELEMENT; k++) m_mat[k]=rhs.m_mat[k];
	return *this;
}

template <class T>
SymMat3<T>& SymMat3<T>::operator+=(const SymMat3<T>& rhs)
{
	for(int k=0; k<NUM_ELEMENT; k++) m_mat[k]+=rhs.m_mat[k];
	return *this;
}

template <class T>
SymMat3<T>& SymMat3<T>::operator-=(const SymMat3<T>& rhs)
{
	for(int k=0; k<NUM_ELEMENT; k++) m_mat[k]-=rhs.m_mat[k];
	return *this;
}

template <class T>
SymMat3<T>& SymMat3<T>::operator*=(const T& a)
{
	for(int k=0; k<NUM_ELEMENT; k++) m_mat[k]*=a;
	return *this;
}

template <class T>
inline SymMat3<T> SymMat3<T>::operator+(const SymMat3& rhs) const
{
	SymMat3<T> ans(*this);
	for(int k=0; k<NUM_ELEMENT; k++) ans.m_mat[k]+=rhs.m_mat[k];
	return ans;
}

template <class T>
inline SymMat3<T> SymMat3<T>::operator-(const SymMat3<T>& rhs) const
{
	SymMat3<T> ans(*this);
	for(int k=0; k<NUM_ELEMENT; k++) ans.m_mat[k]-=rhs.m_mat[k];
	return ans;
}

template <class T>
inline SymMat3<T> SymMat3<T>::operator*(const T& a) const
{
	SymMat3<T> ans(*this);
	for(int k=0; k<NUM_ELEMENT; k++) ans.m_mat[k]*=a;
	return ans;
}

template <class T>
inline T& SymMat3<T>::operator[](const int& j)
{
	return m_mat[j];
}

template <class T>
inline const T& SymMat3<T>::operator[](const int& j) const
{
	return m_mat[j];
}

template <class T>
inline T& SymMat3<T>::operator()(const int& j, const int& k)
{
	if(k==j) return m_mat[j];
	else return m_mat[2+k+j];
}

template <class T>
inline const T& SymMat3<T>::operator()(const int& j, const int& k) const
{
	if(k==j) return m_mat[j];
	else return m_mat[2+k+j];
}

template <class T>
inline SymMat3<T> real(const SymMat3< complex<T> >& rhs)
{
	SymMat3<T> ans;
	for(int k=0; k<6; k++) ans.m_mat[k]=real( rhs.m_mat[k] );
	return ans;
}

template <class T>
inline SymMat3< complex<T> > conj(const SymMat3< complex<T> >& rhs)
{
	SymMat3< complex<T> > ans;
	for(int k=0; k<6; k++) ans.m_mat[k]=conj( rhs.m_mat[k] );
	return ans;
}

template <class T>
inline VecDat3<T> symmetric_transform(const SymMat3<T>& lhs, const VecDat3<T>& rhs)
{
	VecDat3<T> ans;
	ans[0]=lhs.m_mat[0]*rhs[0]+lhs.m_mat[3]*rhs[1]+lhs.m_mat[4]*rhs[2];
	ans[1]=lhs.m_mat[3]*rhs[0]+lhs.m_mat[1]*rhs[1]+lhs.m_mat[5]*rhs[2];
	ans[2]=lhs.m_mat[4]*rhs[0]+lhs.m_mat[5]*rhs[1]+lhs.m_mat[2]*rhs[2];
	return ans;
}

template <class T>
inline SymMat3<T> product_transpose(const VecDat3<T>& lhs, const VecDat3<T>& rhs)
{
	SymMat3<T> ans;
	ans.m_mat[0]=lhs[0]*rhs[0];
	ans.m_mat[1]=lhs[1]*rhs[1];
	ans.m_mat[2]=lhs[2]*rhs[2];
	ans.m_mat[3]=lhs[0]*rhs[1];
	ans.m_mat[4]=lhs[0]*rhs[2];
	ans.m_mat[5]=lhs[1]*rhs[2];
	return ans;
}

template <class T>
inline SymMat3<T> product_self_transpose(const VecDat3<T>& vec)
{
	return product_transpose(vec, vec);
}


#endif /*SYMMAT3_H_*/
