/*
 * The MIT License

   BLDConograph (Bravais lattice determination module in Conograph)

Copyright (c) <2012> <Ryoko Oishi-Tomiyasu, KEK>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
 *
 */
#include "../utility_lattice_reduction/put_Minkowski_reduced_lattice.hh"
#include "LatticeFigureOfMeritToCheckSymmetry.hh"

const string LatticeFigureOfMeritToCheckSymmetry::CS_LABEL[NUM_LS] =
		{	"01", "02", "03", "04", "05", "06", "07", "08", "09", "10", "11", "12", "13", "14" };

LatticeFigureOfMeritToCheckSymmetry::LatticeFigureOfMeritToCheckSymmetry(const Double& rhs)
	: m_label(-1), m_latfom(rhs),
	  m_S_red( SymMat43_Double( SymMat<Double>(3), NRMat<Int4>(4,3) ) )
{
	m_distance = 0.0;
}


LatticeFigureOfMeritToCheckSymmetry::LatticeFigureOfMeritToCheckSymmetry(const BravaisType& brat,
		const SymMat43_Double& S)
	: m_label(-1),
	  m_S_red( SymMat43_Double( SymMat<Double>(3), NRMat<Int4>(4,3) ) )
{
	m_distance = 0.0;
	setLatticeConstants43(brat, S);
}


void LatticeFigureOfMeritToCheckSymmetry::setDistance(const SymMat<Double>& S_super0)
{
	assert( !m_lattice_equiv[(ArrayIndex)Triclinic].empty() );

	const NRMat<Int4> transMat = mprod( m_lattice_equiv[(ArrayIndex)Triclinic][0].transMat,
											put_transform_matrix_row4to3( m_S_red.second ) );
	const SymMat<Double> S_super_equiv = transform_sym_matrix(transMat, m_S_red.first);

	Double ans = 0.0;
	for(Int4 i=0; i<3; i++)
	{
		const Double diff = S_super0(i,i) - S_super_equiv(i,i);
		ans += diff * diff;

		for(Int4 j=0; j<i; j++)
		{
			const Double diff = S_super0(i,j) - S_super_equiv(i,j);
			ans += diff * diff * 2.0;
		}
	}
	m_distance = sqrt(ans);
}


#ifdef DEBUG
static bool checkInitialLatticeParameters(
		const BravaisType& brat,
		const SymMat43_Double& S_red)
{
	const SymMat<Double> dbl_S_red( S_red.first );

	if( brat.enumPointGroup() == Ci && brat.enumBravaisLattice() == Prim )
	{
		assert( dbl_S_red(2,2)*0.9999 < dbl_S_red(1,1) && dbl_S_red(1,1)*0.9999 < dbl_S_red(0,0)
				&& fabs( dbl_S_red(0,1) ) * 1.9999 < dbl_S_red(1,1)
				&& fabs( dbl_S_red(0,2) ) * 1.9999 < dbl_S_red(2,2)
				&& fabs( dbl_S_red(1,2) ) * 1.9999 < dbl_S_red(2,2) );
	}
	else if( brat.enumPointGroup() == C2h_Y && brat.enumBravaisLattice() == Prim )
	{
		assert( 0.0 <= dbl_S_red(0,2) && dbl_S_red(2,2)*0.9999 < dbl_S_red(0,0)
				&& fabs( dbl_S_red(0,2) ) * 1.9999 < dbl_S_red(2,2) && fabs( dbl_S_red(0,2) ) * 1.9999 < dbl_S_red(0,0) );
	}
	else if( brat.enumPointGroup() == C2h_Z && brat.enumBravaisLattice() == Prim )
	{
		assert( 0.0 <= dbl_S_red(0,1) && dbl_S_red(1,1)*0.9999 < dbl_S_red(0,0)
				&& fabs( dbl_S_red(0,1) ) * 1.9999 < dbl_S_red(0,0) && fabs( dbl_S_red(0,1) ) * 1.9999 < dbl_S_red(1,1) );
	}
	else if( brat.enumPointGroup() == C2h_X && brat.enumBravaisLattice() == Prim )
	{
		assert( 0.0 <= dbl_S_red(1,2) && dbl_S_red(2,2)*0.9999 < dbl_S_red(1,1)
				&& fabs( dbl_S_red(1,2) ) * 1.9999 < dbl_S_red(1,1) && fabs( dbl_S_red(1,2) ) * 1.9999 < dbl_S_red(2,2) );
	}
	else if( brat.enumPointGroup() == C2h_Y && brat.enumBravaisLattice() == BaseZ )
	{
		assert( 0.0 <= dbl_S_red(0,2) && fabs( dbl_S_red(0,2) ) * 1.9999 < dbl_S_red(2,2) && fabs( dbl_S_red(0,2) ) * 0.9999 < dbl_S_red(0,0) );
	}
	else if( brat.enumPointGroup() == C2h_Z && brat.enumBravaisLattice() == BaseX )
	{
		assert( 0.0 <= dbl_S_red(0,1) && fabs( dbl_S_red(0,1) ) * 1.9999 < dbl_S_red(0,0) && fabs( dbl_S_red(0,1) ) * 0.9999 < dbl_S_red(1,1) );
	}
	else if( brat.enumPointGroup() == C2h_X && brat.enumBravaisLattice() == BaseY )
	{
		assert( 0.0 <= dbl_S_red(1,2) && fabs( dbl_S_red(1,2) ) * 1.9999 < dbl_S_red(1,1) && fabs( dbl_S_red(1,2) ) * 0.9999  < dbl_S_red(2,2) );
	}
	else if( brat.enumPointGroup() == D2h
			&& brat.enumBravaisLattice() != BaseX
			&& brat.enumBravaisLattice() != BaseY
			&& brat.enumBravaisLattice() != BaseZ )
	{
		assert( dbl_S_red(2,2)*0.9999 < dbl_S_red(1,1) && dbl_S_red(1,1)*0.9999 < dbl_S_red(0,0) );
	}

	const SymMat<Double> S_super = transform_sym_matrix(S_red.second, S_red.first);
	assert( S_super(0,1) <= 0.0
			&& S_super(0,2) <= 0.0
			&& S_super(0,3) <= 0.0
			&& S_super(1,2) <= 0.0
			&& S_super(1,3) <= 0.0
			&& S_super(2,3) <= 0.0 );

	return true;
}
#endif


void LatticeFigureOfMeritToCheckSymmetry::setLatticeConstants43(const BravaisType& brat, const SymMat43_Double& S)
{
	m_S_red = S;
	assert( checkInitialLatticeParameters(brat, m_S_red) );

	m_latfom.setLatticeConstants43(brat, S);
	for(Int4 i=0; i<NUM_LS; i++) m_lattice_equiv[i].clear();
}




bool LatticeFigureOfMeritToCheckSymmetry::checkIfLatticeIsMonoclinic(const ePointGroup& epg_new,
		const Double& resol,
		map< SymMat<Double>, NRMat<Int4> >& ans) const
{
	ans.clear();

	SymMat<Double> ans0 = m_S_red.first;
	cal_average_crystal_system(C2h_X, ans0);

	SymMat<Double> S_red(3);
	NRMat<Int4> trans_mat2;
	if( check_equiv_m(ans0, m_S_red.first, resol ) )
	{
		if( epg_new == C2h_X )
		{
			S_red = ans0;
			trans_mat2 = m_S_red.second;
			putMinkowskiReducedMonoclinicP(1, 2, S_red, trans_mat2);
		}
		else if( epg_new == C2h_Y )
		{
			S_red = transform_sym_matrix(put_matrix_YXZ(), ans0);
			trans_mat2 = mprod(m_S_red.second, put_matrix_YXZ());
			putMinkowskiReducedMonoclinicP(0, 2, S_red, trans_mat2);
		}
		else // if( epg_new == C2h_Z )
		{
			S_red = transform_sym_matrix(put_matrix_YZX(), ans0);
			trans_mat2 = mprod(m_S_red.second, put_matrix_ZXY());
			putMinkowskiReducedMonoclinicP(0, 1, S_red, trans_mat2);
		}
		ans.insert( SymMat43_Double( S_red, trans_mat2) );
	}

	ans0 = m_S_red.first;
	cal_average_crystal_system(C2h_Y, ans0);
	if( check_equiv_m(ans0, m_S_red.first, resol ) )
	{
		if( epg_new == C2h_X )
		{
			S_red = transform_sym_matrix(put_matrix_YXZ(), ans0);
			trans_mat2 = mprod(m_S_red.second, put_matrix_YXZ());
			putMinkowskiReducedMonoclinicP(1, 2, S_red, trans_mat2);
		}
		else if( epg_new == C2h_Y )
		{
			S_red = ans0;
			trans_mat2 = m_S_red.second;
			putMinkowskiReducedMonoclinicP(0, 2, S_red, trans_mat2);
		}
		else // if( epg_new == C2h_Z )
		{
			S_red = transform_sym_matrix(put_matrix_XZY(), ans0);
			trans_mat2 = mprod(m_S_red.second, put_matrix_XZY());
			putMinkowskiReducedMonoclinicP(0, 1, S_red, trans_mat2);
		}
		ans.insert( SymMat43_Double( S_red, trans_mat2) );
	}

	ans0 = m_S_red.first;
	cal_average_crystal_system(C2h_Z, ans0);
	if( check_equiv_m(ans0, m_S_red.first, resol ) )
	{
		if( epg_new == C2h_X )
		{
			S_red = transform_sym_matrix(put_matrix_ZXY(), ans0);
			trans_mat2 = mprod(m_S_red.second, put_matrix_YZX());
			putMinkowskiReducedMonoclinicP(1, 2, S_red, trans_mat2);
		}
		else if( epg_new == C2h_Y )
		{
			S_red = transform_sym_matrix(put_matrix_XZY(), ans0);
			trans_mat2 = mprod(m_S_red.second, put_matrix_XZY());
			putMinkowskiReducedMonoclinicP(0, 2, S_red, trans_mat2);
		}
		else // if( epg_new == C2h_Z )
		{
			S_red = ans0;
			trans_mat2 = m_S_red.second;
			putMinkowskiReducedMonoclinicP(0, 1, S_red, trans_mat2);
		}
		ans.insert( SymMat43_Double( S_red, trans_mat2) );
	}

	return !( ans.empty() );
}


bool LatticeFigureOfMeritToCheckSymmetry::checkIfLatticeIsOrthorhombic(const Double& resol,
							map< SymMat<Double>, NRMat<Int4> >& ans) const
{
	ans.clear();

	const BravaisType& brat = m_latfom.putBravaisType();

	SymMat<Double> ans0 = m_S_red.first;
	cal_average_crystal_system(D2h, ans0);
	if( check_equiv_m(ans0, m_S_red.first, resol ) )
	{
		if( brat.enumBravaisLattice() == BaseX )
		{
			if( ans0(1,1) < ans0(2,2) )
			{
				ans.insert( SymMat43_Double( transform_sym_matrix(put_matrix_ZYX(), ans0), mprod( m_S_red.second, put_matrix_ZYX() ) ) );
			}
			else
			{
				ans.insert( SymMat43_Double( transform_sym_matrix(put_matrix_YZX(), ans0), mprod( m_S_red.second, put_matrix_ZXY() ) ) );
			}
		}
		else if( brat.enumBravaisLattice() == BaseY )
		{
			if( ans0(0,0) < ans0(2,2) )
			{
				ans.insert( SymMat43_Double( transform_sym_matrix(put_matrix_ZXY(), ans0), mprod( m_S_red.second, put_matrix_YZX() ) ) );
			}
			else
			{
				ans.insert( SymMat43_Double( transform_sym_matrix(put_matrix_XZY(), ans0), mprod( m_S_red.second, put_matrix_XZY() ) ) );
			}
		}
		else if( brat.enumBravaisLattice() == BaseZ )
		{
			if( ans0(0,0) < ans0(1,1) )
			{
				ans.insert( SymMat43_Double( transform_sym_matrix(put_matrix_YXZ(), ans0), mprod( m_S_red.second, put_matrix_YXZ() ) ) );
			}
			else
			{
				ans.insert( SymMat43_Double( transform_sym_matrix(put_matrix_XYZ(), ans0), m_S_red.second ) );
			}
		}
		else
		{
			NRMat<Int4> trans_mat = m_S_red.second;
			putMinkowskiReducedOrthorhombic(brat.enumBravaisLattice(), ans0, trans_mat);
			ans.insert( SymMat43_Double(ans0, trans_mat ) );
		}
		return true;
	}
	return false;
}


bool LatticeFigureOfMeritToCheckSymmetry::checkIfLatticeIsTetragonal(const Double& resol,
		map< SymMat<Double>, NRMat<Int4> >& ans) const
{
	ans.clear();

	SymMat<Double> ans0 = m_S_red.first;
	cal_average_crystal_system(D4h_X, ans0);
	if( check_equiv_m(ans0, m_S_red.first, resol ) )
	{
		ans.insert( SymMat43_Double(
				transform_sym_matrix(put_matrix_YZX(), ans0), mprod( m_S_red.second, put_matrix_ZXY() ) ) );
	}

	ans0 = m_S_red.first;
	cal_average_crystal_system(D4h_Y, ans0);
	if( check_equiv_m(ans0, m_S_red.first, resol ) )
	{
		ans.insert( SymMat43_Double(
				transform_sym_matrix(put_matrix_XZY(), ans0), mprod( m_S_red.second, put_matrix_XZY() ) ) );
	}

	ans0 = m_S_red.first;
	cal_average_crystal_system(D4h_Z, ans0);
	if( check_equiv_m(ans0, m_S_red.first, resol ) )
	{
		ans.insert( SymMat43_Double(ans0, m_S_red.second ) );
	}

	return !( ans.empty() );
}




bool LatticeFigureOfMeritToCheckSymmetry::checkIfLatticeIsHexagonal(const ePointGroup& epg_new, const Double& resol,
		map< SymMat<Double>, NRMat<Int4> >& ans) const
{
	ans.clear();
	const BravaisType& brat = m_latfom.putBravaisType();

	SymMat43_Double ans2(SymMat<Double>(3), NRMat<Int4>(3,3));

	if( brat.enumPointGroup() == C2h_X )
	{
		ans2.first = transform_sym_matrix(put_matrix_YZX(), m_S_red.first);
		ans2.second = mprod( m_S_red.second, put_matrix_ZXY() );
	}
	else if( brat.enumPointGroup() == C2h_Y )
	{
		ans2.first = transform_sym_matrix(put_matrix_ZXY(), m_S_red.first);
		ans2.second = mprod( m_S_red.second, put_matrix_YZX() );
	}
	else // if( brat.enumPointGroup() == C2h_Z )
	{
		ans2.first = transform_sym_matrix(put_matrix_XYZ(), m_S_red.first);
		ans2.second = m_S_red.second;
	}

	if( ans2.first(0,1) < Double() )
	{
		ans2.first(0,1) *= -1;
		ans2.second[0][0] *= -1;
		ans2.second[1][0] *= -1;
		ans2.second[2][0] *= -1;
	}

	SymMat<Double> ans0 = ans2.first;
	cal_average_crystal_system(epg_new, ans2.first);
	if( check_equiv_m(ans2.first, ans0, resol ) )
	{
		ans.insert( ans2 );
		return true;
	}
	else return false;
}


bool LatticeFigureOfMeritToCheckSymmetry::checkLatticeSymmetry(const ePointGroup& epg_new, const Double& resol,
		map< SymMat<Double>, NRMat<Int4> >& ans) const
{
	ans.clear();
	const BravaisType& brat = m_latfom.putBravaisType();
	if( epg_new == Ci || epg_new == brat.enumPointGroup() )
	{
		ans.insert( m_S_red );
		return true;
	}

	if( epg_new == C2h_X || epg_new == C2h_Y ||  epg_new == C2h_Z )
	{
		assert( brat.enumPointGroup() == Ci );
		assert( brat.enumBravaisLattice() == Prim );

		return checkIfLatticeIsMonoclinic(epg_new, resol, ans);
	}
	else if( epg_new == D4h_Z )
	{
		assert( brat.enumPointGroup() == D2h );
		assert( brat.enumBravaisLattice() == Prim
				|| brat.enumBravaisLattice() == Inner );

		return checkIfLatticeIsTetragonal(resol, ans);
	}
	else if( epg_new == D2h )
	{
		assert( brat.enumPointGroup() != Ci || brat.enumBravaisLattice() == Prim );
		assert( brat.enumPointGroup() != C2h_Z || brat.enumBravaisLattice() == BaseX );
		assert( brat.enumPointGroup() != C2h_X || brat.enumBravaisLattice() == BaseY );
		assert( brat.enumPointGroup() != C2h_Y || brat.enumBravaisLattice() == BaseZ );
		assert( brat.enumBravaisLattice() != Rhom_hex );

		return checkIfLatticeIsOrthorhombic(resol, ans);
	}
	else if( epg_new == D6h )
	{
		assert( brat.enumBravaisLattice() == Prim );
		assert( brat.enumPointGroup() == C2h_X
				|| brat.enumPointGroup() == C2h_Y
				|| brat.enumPointGroup() == C2h_Z );
		return checkIfLatticeIsHexagonal(epg_new, resol, ans);
	}
	else
	{
		assert( epg_new == Oh );
		assert( brat.enumBravaisLattice() == Prim
				|| brat.enumBravaisLattice() == Inner
				|| brat.enumBravaisLattice() == Face );

		SymMat43_Double ans2 = m_S_red;
		cal_average_crystal_system(epg_new, ans2.first);
		if( check_equiv_m(ans2.first, m_S_red.first, resol ) )
		{
			ans.insert( ans2 );
			return true;
		}
	}
	return !(ans.empty());
}


void LatticeFigureOfMeritToCheckSymmetry::printLatticeInformation(
					const eABCaxis& abc_axis,
					const eRHaxis& rh_axis,
					const Int4& label_start0,
					ostream* os) const
{
	m_latfom.printLatticeInformation(abc_axis, rh_axis, label_start0, os);

  	const SymMat<Double>& dbl_S_init = this->putInitialSellingReducedForm();

  	Int4 label_start = label_start0;
	os->width(label_start); *os << "";
  	*os << "<DeloneReducedMetricTensor>";
  	os->width(14);
   	*os << dbl_S_init(0,0);
  	os->width(14);
   	*os << dbl_S_init(1,1);
  	os->width(14);
   	*os << dbl_S_init(2,2);
  	os->width(14);
   	*os << dbl_S_init(1,2);
  	os->width(14);
   	*os << dbl_S_init(0,2);
  	os->width(14);
   	*os << dbl_S_init(0,1);
  	*os << " </DeloneReducedMetricTensor>\n";
	os->width(label_start); *os << "";

	*os << "<Distance>";
  	os->width(14);
   	*os << m_distance;
  	*os << " </Distance>\n";
}
