/*********************************************************************

    ui_font.c

    Build font for MAME's user interface.
    By NJ and the MAME Plus! Team

*********************************************************************/

#include <windows.h>
#include "driver.h"

#define COLOR_BUTTONS	(90)
#define MAX_GLYPH_FONT	(150)

pen_t uifont_colortable[MAX_COLORTABLE];
static gfx_element *uirotfont_sb;
static gfx_element *uirotfont_db;
static UINT16 _uifonttable[65536];
static int need_font_warning;

/*************************************
 *
 *  Font data
 *
 *************************************/

INLINE void get_ext_font_info(UINT8 *gc1, UINT8 *gc2, UINT8 *gsz)
{
	switch (options.langcode)
	{
	case UI_LANG_ZH_CN:
		*gc1 = 0xaa;
		*gc2 = 0xc0;
		*gsz = 63;
		break;

	case UI_LANG_ZH_TW:
		*gc1 = 0x85;
		*gc2 = 0x40;
		*gsz = 189;
		break;

	case UI_LANG_KO_KR:
		*gc1 = 0xad;
		*gc2 = 0xa1;
		*gsz = 94;
		break;

	case UI_LANG_JA_JP:
		*gc1 = 0x85;
		*gc2 = 0x40;
		*gsz = 189;
		break;

	default:
		*gc1 = 0x8d;
		*gc2 = 0x81;
		*gsz = 126;
	}
}

static unsigned char* load_fontdata(void)
{
	FILE *fd;
	int fontsize = (ui_lang_info[options.langcode].numchars + MAX_GLYPH_FONT) * 24;
	unsigned char *fontdata;

	need_font_warning = 0;

	if (!ui_lang_info[options.langcode].numchars)
		return NULL;

	if ((fd = fopen(_("font\\fontname.fnt"), "rb")) == NULL)
	{
		need_font_warning = 1;
		return NULL;
	}

	fontdata = malloc(fontsize);
	memset(fontdata,0,fontsize);
	fread(fontdata, fontsize, 1, fd);
	return fontdata;
}


int uifont_need_font_warning(void)
{
	return need_font_warning;
}


static UINT16 uifont_buildtable(UINT16 code)
{
	UINT8 c1, c2;
	UINT8 gc1, gc2, gsz;

	get_ext_font_info(&gc1, &gc2, &gsz);

	c1 = (code >> 8) & 0xff;
	c2 = code & 0xff;

	code = ui_lang_info[options.langcode].numchars;
	
	if (gc1 <= c1 && c1 <= gc1 + MAX_GLYPH_FONT / gsz)
	{
		if (gc2 <= c2 && c2 <= gc2 + gsz)
		{
			c1 -= gc1;
			c2 -= gc2;
			code += c1 * gsz + c2;

			return code;
		}
	}

	switch (options.langcode)
	{
	case UI_LANG_JA_JP:
		if (0x40 <= c2 && c2 <= 0xfc)			//Shift-JIS (Japanese) c1: (0x81-0x9F, 0xE0-0xFC) c2: (0x40-0xFC)
		{
			if (0x81 <= c1 && c1 <= 0x9f)
				code = (c1 - 0x81) * 0xc0 + (c2 - 0x40);
			else
			if (0xe0 <= c1 && c1 <= 0xfc)
				code = (c1 - 0xe0 + (0xa0 - 0x81)) * 0xc0 + (c2 - 0x40);
		}
		break;

	case UI_LANG_ZH_CN:
		if(0x81 <= c1 && c1 <= 0xFE)			//GBK c1: (0x81-0xFE)  c2: (0x40-0x7E,0x80-0xFE), Kana code a4a1 - a6fe
		{
			if( 0x40 <= c2 && c2 <= 0x7E )
				code = (c1 - 0x81) * 0xBE + (c2 - 0x40);
			else
			if( 0x80 <= c2 && c2 <= 0xFE )
				code = (c1 - 0x81) * 0xBE + (c2 - 0x80 + 0x3F);
		}
		break;

	case UI_LANG_ZH_TW:
		if(0xA1 <= c1 && c1 <= 0xF9)			//BIG5 (Traditional Chinese) c1: (0xA1-0xF9) c2: (0x40-0x7E, 0xA1-0xFE)
		{
			if( 0x40 <= c2 && c2 <= 0x7E )
				code = (c1 - 0xA1) * 0x9D + (c2 - 0x40);
			else
			if( 0xA1 <= c2 && c2 <= 0xFE )
				code = (c1 - 0xA1) * 0x9D + (c2 - 0xA1 + 0x3F);
		}
		break;

	case UI_LANG_KO_KR:
		if(0x81 <= c1 && c1 <= 0xFE)			//KSC-5601-1987 (Korean), c1: (0x81-0xFE), c2: (0x41-0x5A,0x61-0x7A,0x81-0xFE)
		{
			if( 0x41 <= c2 && c2 <= 0x5A )
				code = (c1 - 0x81) * 0xB2 + (c2 - 0x41);
			else
			if( 0x61 <= c2 && c2 <= 0x7A )
				code = (c1 - 0x81) * 0xB2 + (c2 - 0x61 + 0x1A);
			else
			if( 0x81 <= c2 && c2 <= 0xFE )
				code = (c1 - 0x81) * 0xB2 + (c2 - 0x81 + 0x1A + 0x1A);
		}
		break;
	}

	return code;
}

void uifont_buildfont(int *rotcharwidth, int *rotcharheight)
{
	static UINT8 fontdata6x8[256 * 8] =
	{
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	/* [ 0- 1] */
		0x7c,0x80,0x98,0x90,0x80,0xbc,0x80,0x7c,0xf8,0x04,0x64,0x44,0x04,0xf4,0x04,0xf8,	/* [ 2- 3] tape pos 1 */
		0x7c,0x80,0x98,0x88,0x80,0xbc,0x80,0x7c,0xf8,0x04,0x64,0x24,0x04,0xf4,0x04,0xf8,	/* [ 4- 5] tape pos 2 */
		0x7c,0x80,0x88,0x98,0x80,0xbc,0x80,0x7c,0xf8,0x04,0x24,0x64,0x04,0xf4,0x04,0xf8,	/* [ 6- 7] tape pos 3 */
		0x7c,0x80,0x90,0x98,0x80,0xbc,0x80,0x7c,0xf8,0x04,0x44,0x64,0x04,0xf4,0x04,0xf8,	/* [ 8- 9] tape pos 3 */
		0x30,0x48,0x84,0xb4,0xb4,0x84,0x48,0x30,0x30,0x48,0x84,0x84,0x84,0x84,0x48,0x30,	/* [10-11] */
		0x00,0xfc,0x84,0x8c,0xd4,0xa4,0xfc,0x00,0x00,0xfc,0x84,0x84,0x84,0x84,0xfc,0x00,	/* [12-13] */
		0x00,0x38,0x7c,0x7c,0x7c,0x38,0x00,0x00,0x00,0x30,0x68,0x78,0x78,0x30,0x00,0x00,	/* [14-15] circle & bullet */
		0x80,0xc0,0xe0,0xf0,0xe0,0xc0,0x80,0x00,0x04,0x0c,0x1c,0x3c,0x1c,0x0c,0x04,0x00,	/* [16-17] R/L triangle */
		0x20,0x70,0xf8,0x20,0x20,0xf8,0x70,0x20,0x48,0x48,0x48,0x48,0x48,0x00,0x48,0x00,
		0x00,0x00,0x30,0x68,0x78,0x30,0x00,0x00,0x00,0x30,0x68,0x78,0x78,0x30,0x00,0x00,
		0x70,0xd8,0xe8,0xe8,0xf8,0xf8,0x70,0x00,0x1c,0x7c,0x74,0x44,0x44,0x4c,0xcc,0xc0,
		0x20,0x70,0xf8,0x70,0x70,0x70,0x70,0x00,0x70,0x70,0x70,0x70,0xf8,0x70,0x20,0x00,
		0x00,0x10,0xf8,0xfc,0xf8,0x10,0x00,0x00,0x00,0x20,0x7c,0xfc,0x7c,0x20,0x00,0x00,
		0xb0,0x54,0xb8,0xb8,0x54,0xb0,0x00,0x00,0x00,0x28,0x6c,0xfc,0x6c,0x28,0x00,0x00,
		0x00,0x30,0x30,0x78,0x78,0xfc,0x00,0x00,0xfc,0x78,0x78,0x30,0x30,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x20,0x20,0x20,0x20,0x20,0x00,0x20,0x00,
		0x50,0x50,0x50,0x00,0x00,0x00,0x00,0x00,0x00,0x50,0xf8,0x50,0xf8,0x50,0x00,0x00,
		0x20,0x70,0xc0,0x70,0x18,0xf0,0x20,0x00,0x40,0xa4,0x48,0x10,0x20,0x48,0x94,0x08,
		0x60,0x90,0xa0,0x40,0xa8,0x90,0x68,0x00,0x10,0x20,0x40,0x00,0x00,0x00,0x00,0x00,
		0x20,0x40,0x40,0x40,0x40,0x40,0x20,0x00,0x10,0x08,0x08,0x08,0x08,0x08,0x10,0x00,
		0x20,0xa8,0x70,0xf8,0x70,0xa8,0x20,0x00,0x00,0x20,0x20,0xf8,0x20,0x20,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x60,0x00,0x00,0x00,0xf8,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x08,0x10,0x20,0x40,0x80,0x00,0x00,
		0x70,0x88,0x88,0x88,0x88,0x88,0x70,0x00,0x10,0x30,0x10,0x10,0x10,0x10,0x10,0x00,
		0x70,0x88,0x08,0x10,0x20,0x40,0xf8,0x00,0x70,0x88,0x08,0x30,0x08,0x88,0x70,0x00,
		0x10,0x30,0x50,0x90,0xf8,0x10,0x10,0x00,0xf8,0x80,0xf0,0x08,0x08,0x88,0x70,0x00,
		0x70,0x80,0xf0,0x88,0x88,0x88,0x70,0x00,0xf8,0x08,0x08,0x10,0x20,0x20,0x20,0x00,
		0x70,0x88,0x88,0x70,0x88,0x88,0x70,0x00,0x70,0x88,0x88,0x88,0x78,0x08,0x70,0x00,
		0x00,0x00,0x30,0x30,0x00,0x30,0x30,0x00,0x00,0x00,0x30,0x30,0x00,0x30,0x30,0x60,
		0x10,0x20,0x40,0x80,0x40,0x20,0x10,0x00,0x00,0x00,0xf8,0x00,0xf8,0x00,0x00,0x00,
		0x40,0x20,0x10,0x08,0x10,0x20,0x40,0x00,0x70,0x88,0x08,0x10,0x20,0x00,0x20,0x00,
		0x30,0x48,0x94,0xa4,0xa4,0x94,0x48,0x30,0x70,0x88,0x88,0xf8,0x88,0x88,0x88,0x00,
		0xf0,0x88,0x88,0xf0,0x88,0x88,0xf0,0x00,0x70,0x88,0x80,0x80,0x80,0x88,0x70,0x00,
		0xf0,0x88,0x88,0x88,0x88,0x88,0xf0,0x00,0xf8,0x80,0x80,0xf0,0x80,0x80,0xf8,0x00,
		0xf8,0x80,0x80,0xf0,0x80,0x80,0x80,0x00,0x70,0x88,0x80,0x98,0x88,0x88,0x70,0x00,
		0x88,0x88,0x88,0xf8,0x88,0x88,0x88,0x00,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x00,
		0x08,0x08,0x08,0x08,0x88,0x88,0x70,0x00,0x88,0x90,0xa0,0xc0,0xa0,0x90,0x88,0x00,
		0x80,0x80,0x80,0x80,0x80,0x80,0xf8,0x00,0x88,0xd8,0xa8,0x88,0x88,0x88,0x88,0x00,
		0x88,0xc8,0xa8,0x98,0x88,0x88,0x88,0x00,0x70,0x88,0x88,0x88,0x88,0x88,0x70,0x00,
		0xf0,0x88,0x88,0xf0,0x80,0x80,0x80,0x00,0x70,0x88,0x88,0x88,0x88,0x88,0x70,0x08,
		0xf0,0x88,0x88,0xf0,0x88,0x88,0x88,0x00,0x70,0x88,0x80,0x70,0x08,0x88,0x70,0x00,
		0xf8,0x20,0x20,0x20,0x20,0x20,0x20,0x00,0x88,0x88,0x88,0x88,0x88,0x88,0x70,0x00,
		0x88,0x88,0x88,0x88,0x88,0x50,0x20,0x00,0x88,0x88,0x88,0x88,0xa8,0xd8,0x88,0x00,
		0x88,0x50,0x20,0x20,0x20,0x50,0x88,0x00,0x88,0x88,0x88,0x50,0x20,0x20,0x20,0x00,
		0xf8,0x08,0x10,0x20,0x40,0x80,0xf8,0x00,0x30,0x20,0x20,0x20,0x20,0x20,0x30,0x00,
		0x40,0x40,0x20,0x20,0x10,0x10,0x08,0x08,0x30,0x10,0x10,0x10,0x10,0x10,0x30,0x00,
		0x20,0x50,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xfc,
		0x40,0x20,0x10,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x70,0x08,0x78,0x88,0x78,0x00,
		0x80,0x80,0xf0,0x88,0x88,0x88,0xf0,0x00,0x00,0x00,0x70,0x88,0x80,0x80,0x78,0x00,
		0x08,0x08,0x78,0x88,0x88,0x88,0x78,0x00,0x00,0x00,0x70,0x88,0xf8,0x80,0x78,0x00,
		0x18,0x20,0x70,0x20,0x20,0x20,0x20,0x00,0x00,0x00,0x78,0x88,0x88,0x78,0x08,0x70,
		0x80,0x80,0xf0,0x88,0x88,0x88,0x88,0x00,0x20,0x00,0x20,0x20,0x20,0x20,0x20,0x00,
		0x20,0x00,0x20,0x20,0x20,0x20,0x20,0xc0,0x80,0x80,0x90,0xa0,0xe0,0x90,0x88,0x00,
		0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x00,0x00,0x00,0xf0,0xa8,0xa8,0xa8,0xa8,0x00,
		0x00,0x00,0xb0,0xc8,0x88,0x88,0x88,0x00,0x00,0x00,0x70,0x88,0x88,0x88,0x70,0x00,
		0x00,0x00,0xf0,0x88,0x88,0xf0,0x80,0x80,0x00,0x00,0x78,0x88,0x88,0x78,0x08,0x08,
		0x00,0x00,0xb0,0xc8,0x80,0x80,0x80,0x00,0x00,0x00,0x78,0x80,0x70,0x08,0xf0,0x00,
		0x20,0x20,0x70,0x20,0x20,0x20,0x18,0x00,0x00,0x00,0x88,0x88,0x88,0x98,0x68,0x00,
		0x00,0x00,0x88,0x88,0x88,0x50,0x20,0x00,0x00,0x00,0xa8,0xa8,0xa8,0xa8,0x50,0x00,
		0x00,0x00,0x88,0x50,0x20,0x50,0x88,0x00,0x00,0x00,0x88,0x88,0x88,0x78,0x08,0x70,
		0x00,0x00,0xf8,0x10,0x20,0x40,0xf8,0x00,0x08,0x10,0x10,0x20,0x10,0x10,0x08,0x00,
		0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x40,0x20,0x20,0x10,0x20,0x20,0x40,0x00,
		0x00,0x68,0xb0,0x00,0x00,0x00,0x00,0x00,0x20,0x50,0x20,0x50,0xa8,0x50,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x20,0x20,0x40,0x0c,0x10,0x38,0x10,0x20,0x20,0xc0,0x00,
		0x00,0x00,0x00,0x00,0x00,0x28,0x28,0x50,0x00,0x00,0x00,0x00,0x00,0x00,0xa8,0x00,
		0x70,0xa8,0xf8,0x20,0x20,0x20,0x20,0x00,0x70,0xa8,0xf8,0x20,0x20,0xf8,0xa8,0x70,
		0x20,0x50,0x88,0x00,0x00,0x00,0x00,0x00,0x44,0xa8,0x50,0x20,0x68,0xd4,0x28,0x00,
		0x88,0x70,0x88,0x60,0x30,0x88,0x70,0x00,0x00,0x10,0x20,0x40,0x20,0x10,0x00,0x00,
		0x78,0xa0,0xa0,0xb0,0xa0,0xa0,0x78,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x20,0x20,0x00,0x00,0x00,0x00,0x00,
		0x10,0x10,0x20,0x00,0x00,0x00,0x00,0x00,0x28,0x50,0x50,0x00,0x00,0x00,0x00,0x00,
		0x28,0x28,0x50,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x78,0x78,0x30,0x00,0x00,
		0x00,0x00,0x00,0x78,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xfc,0x00,0x00,0x00,0x00,
		0x68,0xb0,0x00,0x00,0x00,0x00,0x00,0x00,0xf4,0x5c,0x54,0x54,0x00,0x00,0x00,0x00,
		0x88,0x70,0x78,0x80,0x70,0x08,0xf0,0x00,0x00,0x40,0x20,0x10,0x20,0x40,0x00,0x00,
		0x00,0x00,0x70,0xa8,0xb8,0xa0,0x78,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x50,0x88,0x88,0x50,0x20,0x20,0x20,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x20,0x00,0x20,0x20,0x20,0x20,0x20,0x00,
		0x00,0x20,0x70,0xa8,0xa0,0xa8,0x70,0x20,0x30,0x48,0x40,0xe0,0x40,0x48,0xf0,0x00,
		0x00,0x48,0x30,0x48,0x48,0x30,0x48,0x00,0x88,0x88,0x50,0xf8,0x20,0xf8,0x20,0x00,
		0x20,0x20,0x20,0x00,0x20,0x20,0x20,0x00,0x78,0x80,0x70,0x88,0x70,0x08,0xf0,0x00,
		0xd8,0xd8,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x48,0x94,0xa4,0xa4,0x94,0x48,0x30,
		0x60,0x10,0x70,0x90,0x70,0x00,0x00,0x00,0x00,0x28,0x50,0xa0,0x50,0x28,0x00,0x00,
		0x00,0x00,0x00,0xf8,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x78,0x00,0x00,0x00,0x00,
		0x30,0x48,0xb4,0xb4,0xa4,0xb4,0x48,0x30,0x7c,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x60,0x90,0x90,0x60,0x00,0x00,0x00,0x00,0x20,0x20,0xf8,0x20,0x20,0x00,0xf8,0x00,
		0x60,0x90,0x20,0x40,0xf0,0x00,0x00,0x00,0x60,0x90,0x20,0x90,0x60,0x00,0x00,0x00,
		0x10,0x20,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x88,0x88,0x88,0xc8,0xb0,0x80,
		0x78,0xd0,0xd0,0xd0,0x50,0x50,0x50,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x10,0x20,0x00,0x20,0x60,0x20,0x20,0x70,0x00,0x00,0x00,
		0x20,0x50,0x20,0x00,0x00,0x00,0x00,0x00,0x00,0xa0,0x50,0x28,0x50,0xa0,0x00,0x00,
		0x40,0x48,0x50,0x28,0x58,0xa8,0x38,0x08,0x40,0x48,0x50,0x28,0x44,0x98,0x20,0x3c,
		0xc0,0x28,0xd0,0x28,0xd8,0xa8,0x38,0x08,0x20,0x00,0x20,0x40,0x80,0x88,0x70,0x00,
		0x40,0x20,0x70,0x88,0xf8,0x88,0x88,0x00,0x10,0x20,0x70,0x88,0xf8,0x88,0x88,0x00,
		0x70,0x00,0x70,0x88,0xf8,0x88,0x88,0x00,0x68,0xb0,0x70,0x88,0xf8,0x88,0x88,0x00,
		0x50,0x00,0x70,0x88,0xf8,0x88,0x88,0x00,0x20,0x50,0x70,0x88,0xf8,0x88,0x88,0x00,
		0x78,0xa0,0xa0,0xf0,0xa0,0xa0,0xb8,0x00,0x70,0x88,0x80,0x80,0x88,0x70,0x08,0x70,
		0x40,0x20,0xf8,0x80,0xf0,0x80,0xf8,0x00,0x10,0x20,0xf8,0x80,0xf0,0x80,0xf8,0x00,
		0x70,0x00,0xf8,0x80,0xf0,0x80,0xf8,0x00,0x50,0x00,0xf8,0x80,0xf0,0x80,0xf8,0x00,
		0x40,0x20,0x70,0x20,0x20,0x20,0x70,0x00,0x10,0x20,0x70,0x20,0x20,0x20,0x70,0x00,
		0x70,0x00,0x70,0x20,0x20,0x20,0x70,0x00,0x50,0x00,0x70,0x20,0x20,0x20,0x70,0x00,
		0x70,0x48,0x48,0xe8,0x48,0x48,0x70,0x00,0x68,0xb0,0x88,0xc8,0xa8,0x98,0x88,0x00,
		0x40,0x20,0x70,0x88,0x88,0x88,0x70,0x00,0x10,0x20,0x70,0x88,0x88,0x88,0x70,0x00,
		0x70,0x00,0x70,0x88,0x88,0x88,0x70,0x00,0x68,0xb0,0x70,0x88,0x88,0x88,0x70,0x00,
		0x50,0x00,0x70,0x88,0x88,0x88,0x70,0x00,0x00,0x88,0x50,0x20,0x50,0x88,0x00,0x00,
		0x00,0x74,0x88,0x90,0xa8,0x48,0xb0,0x00,0x40,0x20,0x88,0x88,0x88,0x88,0x70,0x00,
		0x10,0x20,0x88,0x88,0x88,0x88,0x70,0x00,0x70,0x00,0x88,0x88,0x88,0x88,0x70,0x00,
		0x50,0x00,0x88,0x88,0x88,0x88,0x70,0x00,0x10,0xa8,0x88,0x50,0x20,0x20,0x20,0x00,
		0x00,0x80,0xf0,0x88,0x88,0xf0,0x80,0x80,0x60,0x90,0x90,0xb0,0x88,0x88,0xb0,0x00,
		0x40,0x20,0x70,0x08,0x78,0x88,0x78,0x00,0x10,0x20,0x70,0x08,0x78,0x88,0x78,0x00,
		0x70,0x00,0x70,0x08,0x78,0x88,0x78,0x00,0x68,0xb0,0x70,0x08,0x78,0x88,0x78,0x00,
		0x50,0x00,0x70,0x08,0x78,0x88,0x78,0x00,0x20,0x50,0x70,0x08,0x78,0x88,0x78,0x00,
		0x00,0x00,0xf0,0x28,0x78,0xa0,0x78,0x00,0x00,0x00,0x70,0x88,0x80,0x78,0x08,0x70,
		0x40,0x20,0x70,0x88,0xf8,0x80,0x70,0x00,0x10,0x20,0x70,0x88,0xf8,0x80,0x70,0x00,
		0x70,0x00,0x70,0x88,0xf8,0x80,0x70,0x00,0x50,0x00,0x70,0x88,0xf8,0x80,0x70,0x00,
		0x40,0x20,0x00,0x60,0x20,0x20,0x70,0x00,0x10,0x20,0x00,0x60,0x20,0x20,0x70,0x00,
		0x20,0x50,0x00,0x60,0x20,0x20,0x70,0x00,0x50,0x00,0x00,0x60,0x20,0x20,0x70,0x00,
		0x50,0x60,0x10,0x78,0x88,0x88,0x70,0x00,0x68,0xb0,0x00,0xf0,0x88,0x88,0x88,0x00,
		0x40,0x20,0x00,0x70,0x88,0x88,0x70,0x00,0x10,0x20,0x00,0x70,0x88,0x88,0x70,0x00,
		0x20,0x50,0x00,0x70,0x88,0x88,0x70,0x00,0x68,0xb0,0x00,0x70,0x88,0x88,0x70,0x00,
		0x00,0x50,0x00,0x70,0x88,0x88,0x70,0x00,0x00,0x20,0x00,0xf8,0x00,0x20,0x00,0x00,
		0x00,0x00,0x68,0x90,0xa8,0x48,0xb0,0x00,0x40,0x20,0x88,0x88,0x88,0x98,0x68,0x00,
		0x10,0x20,0x88,0x88,0x88,0x98,0x68,0x00,0x70,0x00,0x88,0x88,0x88,0x98,0x68,0x00,
		0x50,0x00,0x88,0x88,0x88,0x98,0x68,0x00,0x10,0x20,0x88,0x88,0x88,0x78,0x08,0x70,
		0x80,0xf0,0x88,0x88,0xf0,0x80,0x80,0x80,0x50,0x00,0x88,0x88,0x88,0x78,0x08,0x70
	};

	// support Polish and Hungarian
	static UINT8 cp1250_fontdata6x8[256 * 8] =
	{
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x7c,0x80,0x98,0x90,0x80,0xbc,0x80,0x7c,0xf8,0x04,0x64,0x44,0x04,0xf4,0x04,0xf8,
		0x7c,0x80,0x98,0x88,0x80,0xbc,0x80,0x7c,0xf8,0x04,0x64,0x24,0x04,0xf4,0x04,0xf8,
		0x7c,0x80,0x88,0x98,0x80,0xbc,0x80,0x7c,0xf8,0x04,0x24,0x64,0x04,0xf4,0x04,0xf8,
		0x7c,0x80,0x90,0x98,0x80,0xbc,0x80,0x7c,0xf8,0x04,0x44,0x64,0x04,0xf4,0x04,0xf8,
		0x30,0x48,0x84,0xb4,0xb4,0x84,0x48,0x30,0x30,0x48,0x84,0x84,0x84,0x84,0x48,0x30,
		0x00,0xfc,0x84,0x8c,0xd4,0xa4,0xfc,0x00,0x00,0xfc,0x84,0x84,0x84,0x84,0xfc,0x00,
		0x00,0x38,0x7c,0x7c,0x7c,0x38,0x00,0x00,0x00,0x30,0x68,0x78,0x78,0x30,0x00,0x00,
		0x80,0xc0,0xe0,0xf0,0xe0,0xc0,0x80,0x00,0x04,0x0c,0x1c,0x3c,0x1c,0x0c,0x04,0x00,
		0x20,0x70,0xf8,0x20,0x20,0xf8,0x70,0x20,0x48,0x48,0x48,0x48,0x48,0x00,0x48,0x00,
		0x00,0x00,0x30,0x68,0x78,0x30,0x00,0x00,0x00,0x30,0x68,0x78,0x78,0x30,0x00,0x00,
		0x70,0xd8,0xe8,0xe8,0xf8,0xf8,0x70,0x00,0x1c,0x7c,0x74,0x44,0x44,0x4c,0xcc,0xc0,
		0x20,0x70,0xf8,0x70,0x70,0x70,0x70,0x00,0x70,0x70,0x70,0x70,0xf8,0x70,0x20,0x00,
		0x00,0x10,0xf8,0xfc,0xf8,0x10,0x00,0x00,0x00,0x20,0x7c,0xfc,0x7c,0x20,0x00,0x00,
		0xb0,0x54,0xb8,0xb8,0x54,0xb0,0x00,0x00,0x00,0x28,0x6c,0xfc,0x6c,0x28,0x00,0x00,
		0x00,0x30,0x30,0x78,0x78,0xfc,0x00,0x00,0xfc,0x78,0x78,0x30,0x30,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x20,0x20,0x20,0x20,0x20,0x00,0x20,0x00,
		0x50,0x50,0x50,0x00,0x00,0x00,0x00,0x00,0x00,0x50,0xf8,0x50,0xf8,0x50,0x00,0x00,
		0x20,0x70,0xc0,0x70,0x18,0xf0,0x20,0x00,0x40,0xa4,0x48,0x10,0x20,0x48,0x94,0x08,
		0x60,0x90,0xa0,0x40,0xa8,0x90,0x68,0x00,0x10,0x20,0x40,0x00,0x00,0x00,0x00,0x00,
		0x20,0x40,0x40,0x40,0x40,0x40,0x20,0x00,0x10,0x08,0x08,0x08,0x08,0x08,0x10,0x00,
		0x20,0xa8,0x70,0xf8,0x70,0xa8,0x20,0x00,0x00,0x20,0x20,0xf8,0x20,0x20,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x60,0x00,0x00,0x00,0xf8,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x08,0x10,0x20,0x40,0x80,0x00,0x00,
		0x70,0x88,0x88,0x88,0x88,0x88,0x70,0x00,0x10,0x30,0x10,0x10,0x10,0x10,0x10,0x00,
		0x70,0x88,0x08,0x10,0x20,0x40,0xf8,0x00,0x70,0x88,0x08,0x30,0x08,0x88,0x70,0x00,
		0x10,0x30,0x50,0x90,0xf8,0x10,0x10,0x00,0xf8,0x80,0xf0,0x08,0x08,0x88,0x70,0x00,
		0x70,0x80,0xf0,0x88,0x88,0x88,0x70,0x00,0xf8,0x08,0x08,0x10,0x20,0x20,0x20,0x00,
		0x70,0x88,0x88,0x70,0x88,0x88,0x70,0x00,0x70,0x88,0x88,0x88,0x78,0x08,0x70,0x00,
		0x00,0x00,0x30,0x30,0x00,0x30,0x30,0x00,0x00,0x00,0x30,0x30,0x00,0x30,0x30,0x60,
		0x10,0x20,0x40,0x80,0x40,0x20,0x10,0x00,0x00,0x00,0xf8,0x00,0xf8,0x00,0x00,0x00,
		0x40,0x20,0x10,0x08,0x10,0x20,0x40,0x00,0x70,0x88,0x08,0x10,0x20,0x00,0x20,0x00,
		0x30,0x48,0x94,0xa4,0xa4,0x94,0x48,0x30,0x70,0x88,0x88,0xf8,0x88,0x88,0x88,0x00,
		0xf0,0x88,0x88,0xf0,0x88,0x88,0xf0,0x00,0x70,0x88,0x80,0x80,0x80,0x88,0x70,0x00,
		0xf0,0x88,0x88,0x88,0x88,0x88,0xf0,0x00,0xf8,0x80,0x80,0xf0,0x80,0x80,0xf8,0x00,
		0xf8,0x80,0x80,0xf0,0x80,0x80,0x80,0x00,0x70,0x88,0x80,0x98,0x88,0x88,0x70,0x00,
		0x88,0x88,0x88,0xf8,0x88,0x88,0x88,0x00,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x00,
		0x08,0x08,0x08,0x08,0x88,0x88,0x70,0x00,0x88,0x90,0xa0,0xc0,0xa0,0x90,0x88,0x00,
		0x80,0x80,0x80,0x80,0x80,0x80,0xf8,0x00,0x88,0xd8,0xa8,0x88,0x88,0x88,0x88,0x00,
		0x88,0xc8,0xa8,0x98,0x88,0x88,0x88,0x00,0x70,0x88,0x88,0x88,0x88,0x88,0x70,0x00,
		0xf0,0x88,0x88,0xf0,0x80,0x80,0x80,0x00,0x70,0x88,0x88,0x88,0x88,0x88,0x70,0x08,
		0xf0,0x88,0x88,0xf0,0x88,0x88,0x88,0x00,0x70,0x88,0x80,0x70,0x08,0x88,0x70,0x00,
		0xf8,0x20,0x20,0x20,0x20,0x20,0x20,0x00,0x88,0x88,0x88,0x88,0x88,0x88,0x70,0x00,
		0x88,0x88,0x88,0x88,0x88,0x50,0x20,0x00,0x88,0x88,0x88,0x88,0xa8,0xd8,0x88,0x00,
		0x88,0x50,0x20,0x20,0x20,0x50,0x88,0x00,0x88,0x88,0x88,0x50,0x20,0x20,0x20,0x00,
		0xf8,0x08,0x10,0x20,0x40,0x80,0xf8,0x00,0x30,0x20,0x20,0x20,0x20,0x20,0x30,0x00,
		0x40,0x40,0x20,0x20,0x10,0x10,0x08,0x08,0x30,0x10,0x10,0x10,0x10,0x10,0x30,0x00,
		0x20,0x50,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xfc,
		0x40,0x20,0x10,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x70,0x08,0x78,0x88,0x78,0x00,
		0x80,0x80,0xf0,0x88,0x88,0x88,0xf0,0x00,0x00,0x00,0x70,0x88,0x80,0x80,0x78,0x00,
		0x08,0x08,0x78,0x88,0x88,0x88,0x78,0x00,0x00,0x00,0x70,0x88,0xf8,0x80,0x78,0x00,
		0x18,0x20,0x70,0x20,0x20,0x20,0x20,0x00,0x00,0x00,0x78,0x88,0x88,0x78,0x08,0x70,
		0x80,0x80,0xf0,0x88,0x88,0x88,0x88,0x00,0x20,0x00,0x20,0x20,0x20,0x20,0x20,0x00,
		0x20,0x00,0x20,0x20,0x20,0x20,0x20,0xc0,0x80,0x80,0x90,0xa0,0xe0,0x90,0x88,0x00,
		0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x00,0x00,0x00,0xf0,0xa8,0xa8,0xa8,0xa8,0x00,
		0x00,0x00,0xb0,0xc8,0x88,0x88,0x88,0x00,0x00,0x00,0x70,0x88,0x88,0x88,0x70,0x00,
		0x00,0x00,0xf0,0x88,0x88,0xf0,0x80,0x80,0x00,0x00,0x78,0x88,0x88,0x78,0x08,0x08,
		0x00,0x00,0xb0,0xc8,0x80,0x80,0x80,0x00,0x00,0x00,0x78,0x80,0x70,0x08,0xf0,0x00,
		0x20,0x20,0x70,0x20,0x20,0x20,0x18,0x00,0x00,0x00,0x88,0x88,0x88,0x98,0x68,0x00,
		0x00,0x00,0x88,0x88,0x88,0x50,0x20,0x00,0x00,0x00,0xa8,0xa8,0xa8,0xa8,0x50,0x00,
		0x00,0x00,0x88,0x50,0x20,0x50,0x88,0x00,0x00,0x00,0x88,0x88,0x88,0x78,0x08,0x70,
		0x00,0x00,0xf8,0x10,0x20,0x40,0xf8,0x00,0x08,0x10,0x10,0x20,0x10,0x10,0x08,0x00,
		0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x40,0x20,0x20,0x10,0x20,0x20,0x40,0x00,
		0x00,0x68,0xb0,0x00,0x00,0x00,0x00,0x00,0x20,0x50,0x20,0x50,0xa8,0x50,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x20,0x20,0x40,0x0c,0x10,0x38,0x10,0x20,0x20,0xc0,0x00,
		0x00,0x00,0x00,0x00,0x00,0x28,0x28,0x50,0x00,0x00,0x00,0x00,0x00,0x00,0xa8,0x00,
		0x70,0xa8,0xf8,0x20,0x20,0x20,0x20,0x00,0x70,0xa8,0xf8,0x20,0x20,0xf8,0xa8,0x70,
		0x20,0x50,0x88,0x00,0x00,0x00,0x00,0x00,0x44,0xa8,0x50,0x20,0x68,0xd4,0x28,0x00,
		0x88,0x70,0x88,0x60,0x30,0x88,0x70,0x00,0x00,0x10,0x20,0x40,0x20,0x10,0x00,0x00,
		0x30,0x70,0x88,0x70,0x08,0x88,0x70,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xf8,0x08,0x10,0xf0,0x40,0x80,0xf8,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x20,0x20,0x00,0x00,0x00,0x00,0x00,
		0x10,0x10,0x20,0x00,0x00,0x00,0x00,0x00,0x28,0x50,0x50,0x00,0x00,0x00,0x00,0x00,
		0x28,0x28,0x50,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x78,0x78,0x30,0x00,0x00,
		0x00,0x00,0x00,0x78,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xfc,0x00,0x00,0x00,0x00,
		0x68,0xb0,0x00,0x00,0x00,0x00,0x00,0x00,0xf4,0x5c,0x54,0x54,0x00,0x00,0x00,0x00,
		0x88,0x70,0x78,0x80,0x70,0x08,0xf0,0x00,0x00,0x40,0x20,0x10,0x20,0x40,0x00,0x00,
		0x10,0x20,0x78,0x80,0x70,0x08,0xf0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x20,0xf8,0x10,0x20,0x40,0xf8,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x20,0x00,0x20,0x20,0x20,0x20,0x20,0x00,
		0x00,0x20,0x70,0xa8,0xa0,0xa8,0x70,0x20,0x40,0x50,0x60,0x40,0xc0,0x40,0x78,0x00,
		0x00,0x48,0x30,0x48,0x48,0x30,0x48,0x00,0x70,0x88,0x88,0xf8,0x88,0x88,0x98,0x10,
		0x20,0x20,0x20,0x00,0x20,0x20,0x20,0x00,0x78,0x80,0x70,0x88,0x70,0x08,0xf0,0x00,
		0xd8,0xd8,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x48,0x94,0xa4,0xa4,0x94,0x48,0x30,
		0x60,0x10,0x70,0x90,0x70,0x00,0x00,0x00,0x00,0x28,0x50,0xa0,0x50,0x28,0x00,0x00,
		0x00,0x00,0x00,0xf8,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x78,0x00,0x00,0x00,0x00,
		0x30,0x48,0xb4,0xb4,0xa4,0xb4,0x48,0x30,0x20,0xf8,0x08,0x10,0x20,0x40,0xf8,0x00,
		0x60,0x90,0x90,0x60,0x00,0x00,0x00,0x00,0x20,0x20,0xf8,0x20,0x20,0x00,0xf8,0x00,
		0x60,0x90,0x20,0x40,0xf0,0x00,0x00,0x00,0x20,0x30,0x20,0x60,0x20,0x20,0x20,0x00,
		0x10,0x20,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x88,0x88,0x88,0xc8,0xb0,0x80,
		0x78,0xd0,0xd0,0xd0,0x50,0x50,0x50,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x10,0x20,0x00,0x00,0x00,0x70,0x08,0x78,0x88,0x78,0x10,
		0x20,0x50,0x20,0x00,0x00,0x00,0x00,0x00,0x00,0xa0,0x50,0x28,0x50,0xa0,0x00,0x00,
		0x40,0x48,0x50,0x28,0x58,0xa8,0x38,0x08,0x40,0x48,0x50,0x28,0x44,0x98,0x20,0x3c,
		0xc0,0x28,0xd0,0x28,0xd8,0xa8,0x38,0x08,0x20,0x00,0xf8,0x10,0x20,0x40,0xf8,0x00,
		0x40,0x20,0x70,0x88,0xf8,0x88,0x88,0x00,0x10,0x20,0x70,0x88,0xf8,0x88,0x88,0x00,
		0x70,0x00,0x70,0x88,0xf8,0x88,0x88,0x00,0x68,0xb0,0x70,0x88,0xf8,0x88,0x88,0x00,
		0x50,0x00,0x70,0x88,0xf8,0x88,0x88,0x00,0x20,0x50,0x70,0x88,0xf8,0x88,0x88,0x00,
		0x30,0x70,0x88,0x80,0x80,0x88,0x70,0x00,0x70,0x88,0x80,0x80,0x88,0x70,0x08,0x70,
		0x40,0x20,0xf8,0x80,0xf0,0x80,0xf8,0x00,0x10,0x20,0xf8,0x80,0xf0,0x80,0xf8,0x00,
		0xf8,0x80,0x80,0xf0,0x80,0x80,0xf8,0x30,0x50,0x00,0xf8,0x80,0xf0,0x80,0xf8,0x00,
		0x40,0x20,0x70,0x20,0x20,0x20,0x70,0x00,0x10,0x20,0x70,0x20,0x20,0x20,0x70,0x00,
		0x70,0x00,0x70,0x20,0x20,0x20,0x70,0x00,0x50,0x00,0x70,0x20,0x20,0x20,0x70,0x00,
		0x70,0x48,0x48,0xe8,0x48,0x48,0x70,0x00,0x20,0xa8,0xc8,0xa8,0x98,0x88,0x88,0x00,
		0x40,0x20,0x70,0x88,0x88,0x88,0x70,0x00,0x20,0x70,0x88,0x88,0x88,0x88,0x70,0x00,
		0x70,0x00,0x70,0x88,0x88,0x88,0x70,0x00,0x68,0xb0,0x70,0x88,0x88,0x88,0x70,0x00,
		0x50,0x00,0x70,0x88,0x88,0x88,0x70,0x00,0x00,0x88,0x50,0x20,0x50,0x88,0x00,0x00,
		0x00,0x74,0x88,0x90,0xa8,0x48,0xb0,0x00,0x40,0x20,0x88,0x88,0x88,0x88,0x70,0x00,
		0x10,0x20,0x88,0x88,0x88,0x88,0x70,0x00,0x70,0x00,0x88,0x88,0x88,0x88,0x70,0x00,
		0x50,0x00,0x88,0x88,0x88,0x88,0x70,0x00,0x10,0xa8,0x88,0x50,0x20,0x20,0x20,0x00,
		0x00,0x80,0xf0,0x88,0x88,0xf0,0x80,0x80,0x60,0x90,0x90,0xb0,0x88,0x88,0xb0,0x00,
		0x40,0x20,0x70,0x08,0x78,0x88,0x78,0x00,0x10,0x20,0x70,0x08,0x78,0x88,0x78,0x00,
		0x70,0x00,0x70,0x08,0x78,0x88,0x78,0x00,0x68,0xb0,0x70,0x08,0x78,0x88,0x78,0x00,
		0x50,0x00,0x70,0x08,0x78,0x88,0x78,0x00,0x20,0x50,0x70,0x08,0x78,0x88,0x78,0x00,
		0x10,0x20,0x70,0x88,0x80,0x80,0x78,0x00,0x00,0x00,0x70,0x88,0x80,0x78,0x08,0x70,
		0x40,0x20,0x70,0x88,0xf8,0x80,0x70,0x00,0x10,0x20,0x70,0x88,0xf8,0x80,0x70,0x00,
		0x00,0x00,0x70,0x88,0xf8,0x80,0x70,0x18,0x10,0x20,0x00,0x60,0x20,0x20,0x70,0x00,
		0x40,0x20,0x00,0x60,0x20,0x20,0x70,0x00,0x10,0x20,0x00,0x60,0x20,0x20,0x70,0x00,
		0x20,0x50,0x00,0x60,0x20,0x20,0x70,0x00,0x50,0x00,0x00,0x60,0x20,0x20,0x70,0x00,
		0x50,0x60,0x10,0x78,0x88,0x88,0x70,0x00,0x10,0x20,0xb0,0xc8,0x88,0x88,0x88,0x00,
		0x40,0x20,0x00,0x70,0x88,0x88,0x70,0x00,0x10,0x20,0x00,0x70,0x88,0x88,0x70,0x00,
		0x20,0x50,0x00,0x70,0x88,0x88,0x70,0x00,0x68,0xb0,0x00,0x70,0x88,0x88,0x70,0x00,
		0x00,0x50,0x00,0x70,0x88,0x88,0x70,0x00,0x00,0x20,0x00,0xf8,0x00,0x20,0x00,0x00,
		0x00,0x00,0x68,0x90,0xa8,0x48,0xb0,0x00,0x40,0x20,0x88,0x88,0x88,0x98,0x68,0x00,
		0x10,0x20,0x88,0x88,0x88,0x98,0x68,0x00,0x70,0x00,0x88,0x88,0x88,0x98,0x68,0x00,
		0x50,0x00,0x88,0x88,0x88,0x98,0x68,0x00,0x10,0x20,0x88,0x88,0x88,0x78,0x08,0x70,
		0x80,0xf0,0x88,0x88,0xf0,0x80,0x80,0x80,0x50,0x00,0x88,0x88,0x88,0x78,0x08,0x70
	};

	static unsigned char fontdata6x12[256 * 12] =
	{
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x04,0x14,0x34,0x7c,0x30,0x10,0x00,0x00,0x00,
		0x00,0x00,0x7c,0x80,0x98,0x90,0x80,0xbc,0x80,0x7c,0x00,0x00,
		0x00,0x00,0xf8,0x04,0x64,0x44,0x04,0xf4,0x04,0xf8,0x00,0x00,
		0x00,0x00,0x7c,0x80,0x98,0x88,0x80,0xbc,0x80,0x7c,0x00,0x00,
		0x00,0x00,0xf8,0x04,0x64,0x24,0x04,0xf4,0x04,0xf8,0x00,0x00,
		0x00,0x00,0x7c,0x80,0x88,0x98,0x80,0xbc,0x80,0x7c,0x00,0x00,
		0x00,0x00,0xf8,0x04,0x24,0x64,0x04,0xf4,0x04,0xf8,0x00,0x00,
		0x00,0x00,0x7c,0x80,0x90,0x98,0x80,0xbc,0x80,0x7c,0x00,0x00,
		0x00,0x00,0xf8,0x04,0x44,0x64,0x04,0xf4,0x04,0xf8,0x00,0x00,
		0x00,0x00,0x10,0x28,0x44,0x54,0x54,0x44,0x28,0x10,0x00,0x00,
		0x00,0x00,0x10,0x28,0x44,0x44,0x44,0x44,0x28,0x10,0x00,0x00,
		0x00,0x00,0x00,0x7e,0x42,0x46,0x6a,0x52,0x7e,0x00,0x00,0x00,
		0x00,0x00,0x00,0x7e,0x42,0x42,0x42,0x42,0x7e,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x30,0x68,0x78,0x78,0x30,0x00,0x00,0x00,0x00,
		0x00,0x00,0x80,0xc0,0xe0,0xf0,0xf0,0xe0,0xc0,0x80,0x00,0x00,
		0x00,0x00,0x08,0x18,0x38,0x78,0x78,0x38,0x18,0x08,0x00,0x00,
		0x00,0x00,0x20,0x70,0xf8,0x20,0x20,0xf8,0x70,0x20,0x00,0x00,
		0x50,0x50,0x50,0x50,0x50,0x50,0x50,0x00,0x00,0x50,0x00,0x00,
		0x00,0x00,0x00,0x00,0x60,0xd0,0xf0,0x60,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x60,0xd0,0xf0,0xf0,0x60,0x00,0x00,0x00,0x00,
		0x00,0x00,0x70,0xd8,0xe8,0xe8,0xf8,0xf8,0x70,0x00,0x00,0x00,
		0x00,0x00,0x18,0x78,0x68,0x48,0x48,0x58,0xd8,0xc0,0x00,0x00,
		0x00,0x00,0x30,0x78,0xfc,0x78,0x78,0x78,0x78,0x00,0x00,0x00,
		0x00,0x00,0x78,0x78,0x78,0x78,0xfc,0x78,0x30,0x00,0x00,0x00,
		0x00,0x00,0x00,0x10,0xf8,0xfc,0xfc,0xf8,0x10,0x00,0x00,0x00,
		0x00,0x00,0x00,0x20,0x7c,0xfc,0xfc,0x7c,0x20,0x00,0x00,0x00,
		0x00,0x00,0x00,0xb0,0x54,0xb8,0xb8,0x54,0xb0,0x00,0x00,0x00,
		0x00,0x00,0x00,0x28,0x6c,0xfc,0x6c,0x28,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x30,0x30,0x78,0x78,0xfc,0xfc,0x00,0x00,0x00,
		0x0e,0x00,0x3f,0x80,0x71,0xc0,0x6e,0xc0,0xee,0xe0,0xee,0xe0,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x00,0x00,0x20,0x00,0x00,
		0xa0,0xa0,0xa0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x50,0x50,0xf8,0x50,0x50,0x50,0xf8,0x50,0x50,0x00,0x00,
		0x00,0x20,0x78,0xa0,0xa0,0x70,0x28,0x28,0xf0,0x20,0x00,0x00,
		0x40,0xa0,0xa4,0x48,0x10,0x20,0x48,0x94,0x14,0x08,0x00,0x00,
		0x00,0x60,0x90,0xa0,0x40,0xa8,0xa8,0x90,0x90,0x68,0x00,0x00,
		0x10,0x20,0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x10,0x20,0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x20,0x10,0x00,
		0x40,0x20,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x20,0x40,0x00,
		0x00,0x00,0x20,0xa8,0x70,0xf8,0x70,0xa8,0x20,0x00,0x00,0x00,
		0x00,0x00,0x20,0x20,0x20,0xf8,0x20,0x20,0x20,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x60,0x00,
		0x00,0x00,0x00,0x00,0x00,0xf0,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x60,0x00,0x00,
		0x08,0x08,0x10,0x10,0x20,0x20,0x40,0x40,0x80,0x80,0x00,0x00,
		0x00,0x70,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x70,0x00,0x00,
		0x00,0x20,0x60,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x00,0x00,
		0x00,0x70,0x88,0x88,0x08,0x10,0x20,0x40,0x80,0xf8,0x00,0x00,
		0x00,0x70,0x88,0x88,0x08,0x70,0x08,0x88,0x88,0x70,0x00,0x00,
		0x00,0x30,0x30,0x50,0x50,0x90,0x90,0xf8,0x10,0x10,0x00,0x00,
		0x00,0xf8,0x80,0x80,0xf0,0x08,0x08,0x88,0x88,0x70,0x00,0x00,
		0x00,0x70,0x80,0x80,0xf0,0x88,0x88,0x88,0x88,0x70,0x00,0x00,
		0x00,0xf8,0x08,0x08,0x08,0x10,0x20,0x20,0x20,0x20,0x00,0x00,
		0x00,0x70,0x88,0x88,0x88,0x70,0x88,0x88,0x88,0x70,0x00,0x00,
		0x00,0x70,0x88,0x88,0x88,0x88,0x78,0x08,0x08,0x70,0x00,0x00,
		0x00,0x00,0x00,0x60,0x60,0x00,0x00,0x00,0x60,0x60,0x00,0x00,
		0x00,0x00,0x00,0x60,0x60,0x00,0x00,0x00,0x60,0x20,0x40,0x00,
		0x00,0x08,0x10,0x20,0x40,0x80,0x40,0x20,0x10,0x08,0x00,0x00,
		0x00,0x00,0x00,0xf0,0x00,0x00,0xf0,0x00,0x00,0x00,0x00,0x00,
		0x00,0x80,0x40,0x20,0x10,0x08,0x10,0x20,0x40,0x80,0x00,0x00,
		0x00,0x70,0x88,0x88,0x08,0x10,0x20,0x20,0x00,0x20,0x00,0x00,
		0x00,0x70,0x88,0xe8,0xa8,0xa8,0xe8,0x90,0x88,0x70,0x00,0x00,
		0x00,0x70,0x88,0x88,0x88,0x88,0xf8,0x88,0x88,0x88,0x00,0x00,
		0x00,0xf0,0x88,0x88,0x88,0xf0,0x88,0x88,0x88,0xf0,0x00,0x00,
		0x00,0x70,0x88,0x88,0x80,0x80,0x80,0x88,0x88,0x70,0x00,0x00,
		0x00,0xf0,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0xf0,0x00,0x00,
		0x00,0xf8,0x80,0x80,0x80,0xf0,0x80,0x80,0x80,0xf8,0x00,0x00,
		0x00,0xf8,0x80,0x80,0x80,0xf0,0x80,0x80,0x80,0x80,0x00,0x00,
		0x00,0x70,0x88,0x88,0x80,0x98,0x88,0x88,0x88,0x70,0x00,0x00,
		0x00,0x88,0x88,0x88,0x88,0xf8,0x88,0x88,0x88,0x88,0x00,0x00,
		0x00,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x00,0x00,
		0x00,0x08,0x08,0x08,0x08,0x08,0x08,0x88,0x88,0x70,0x00,0x00,
		0x00,0x88,0x88,0x90,0xa0,0xc0,0xa0,0x90,0x88,0x88,0x00,0x00,
		0x00,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0xf8,0x00,0x00,
		0x00,0x88,0x88,0xd8,0xd8,0xa8,0xa8,0x88,0x88,0x88,0x00,0x00,
		0x00,0x88,0xc8,0xc8,0xa8,0xa8,0x98,0x98,0x88,0x88,0x00,0x00,
		0x00,0x70,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x70,0x00,0x00,
		0x00,0xf0,0x88,0x88,0x88,0xf0,0x80,0x80,0x80,0x80,0x00,0x00,
		0x00,0x70,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x70,0x08,0x00,
		0x00,0xf0,0x88,0x88,0x88,0xf0,0x88,0x88,0x88,0x88,0x00,0x00,
		0x00,0x70,0x88,0x88,0x80,0x70,0x08,0x88,0x88,0x70,0x00,0x00,
		0x00,0xf8,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x00,0x00,
		0x00,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x70,0x00,0x00,
		0x00,0x88,0x88,0x88,0x88,0x88,0x88,0x50,0x50,0x20,0x00,0x00,
		0x00,0x88,0x88,0x88,0x88,0xa8,0xa8,0xd8,0xd8,0x88,0x00,0x00,
		0x00,0x88,0x50,0x50,0x20,0x20,0x20,0x50,0x50,0x88,0x00,0x00,
		0x00,0x88,0x88,0x88,0x50,0x50,0x20,0x20,0x20,0x20,0x00,0x00,
		0x00,0xf8,0x08,0x10,0x10,0x20,0x40,0x40,0x80,0xf8,0x00,0x00,
		0x38,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x38,0x00,
		0x00,0x88,0x88,0x50,0x50,0xf8,0x20,0xf8,0x20,0x20,0x00,0x00,
		0xe0,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0xe0,0x00,
		0x20,0x50,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xfc,0x00,
		0x40,0x20,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x70,0x88,0x78,0x88,0x88,0x78,0x00,0x00,
		0x00,0x80,0x80,0x80,0xf0,0x88,0x88,0x88,0x88,0xf0,0x00,0x00,
		0x00,0x00,0x00,0x00,0x70,0x88,0x80,0x80,0x88,0x70,0x00,0x00,
		0x00,0x08,0x08,0x08,0x78,0x88,0x88,0x88,0x88,0x78,0x00,0x00,
		0x00,0x00,0x00,0x00,0x70,0x88,0xf8,0x80,0x88,0x70,0x00,0x00,
		0x00,0x30,0x40,0x40,0xe0,0x40,0x40,0x40,0x40,0x40,0x00,0x00,
		0x00,0x00,0x00,0x00,0x78,0x88,0x88,0x78,0x08,0x88,0x70,0x00,
		0x00,0x80,0x80,0x80,0xf0,0x88,0x88,0x88,0x88,0x88,0x00,0x00,
		0x00,0x20,0x20,0x00,0x20,0x20,0x20,0x20,0x20,0x20,0x00,0x00,
		0x00,0x20,0x20,0x00,0x20,0x20,0x20,0x20,0x20,0x20,0xc0,0x00,
		0x00,0x80,0x80,0x80,0x80,0x90,0xa0,0xe0,0x90,0x88,0x00,0x00,
		0x00,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x00,0x00,
		0x00,0x00,0x00,0x00,0xf0,0xa8,0xa8,0xa8,0xa8,0xa8,0x00,0x00,
		0x00,0x00,0x00,0x00,0xb0,0xc8,0x88,0x88,0x88,0x88,0x00,0x00,
		0x00,0x00,0x00,0x00,0x70,0x88,0x88,0x88,0x88,0x70,0x00,0x00,
		0x00,0x00,0x00,0x00,0xf0,0x88,0x88,0x88,0xf0,0x80,0x80,0x00,
		0x00,0x00,0x00,0x00,0x78,0x88,0x88,0x88,0x78,0x08,0x08,0x00,
		0x00,0x00,0x00,0x00,0xb0,0xc8,0x80,0x80,0x80,0x80,0x00,0x00,
		0x00,0x00,0x00,0x00,0x70,0x88,0x60,0x10,0x88,0x70,0x00,0x00,
		0x00,0x40,0x40,0x40,0xe0,0x40,0x40,0x40,0x40,0x30,0x00,0x00,
		0x00,0x00,0x00,0x00,0x88,0x88,0x88,0x88,0x98,0x68,0x00,0x00,
		0x00,0x00,0x00,0x00,0x88,0x88,0x88,0x88,0x50,0x20,0x00,0x00,
		0x00,0x00,0x00,0x00,0xa8,0xa8,0xa8,0xa8,0xa8,0x50,0x00,0x00,
		0x00,0x00,0x00,0x00,0x88,0x50,0x20,0x20,0x50,0x88,0x00,0x00,
		0x00,0x00,0x00,0x00,0x88,0x88,0x88,0x88,0x78,0x08,0x70,0x00,
		0x00,0x00,0x00,0x00,0xf8,0x08,0x10,0x20,0x40,0xf8,0x00,0x00,
		0x10,0x20,0x20,0x20,0x20,0x40,0x20,0x20,0x20,0x20,0x10,0x00,
		0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x00,
		0x40,0x20,0x20,0x20,0x20,0x10,0x20,0x20,0x20,0x20,0x40,0x00,
		0x68,0x90,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x20,0x50,0x20,0x50,0xa8,0x50,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xa8,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x20,0x50,0x50,0x20,0x00,0x00,
		0x70,0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x00,0x00,0x00,0x00,
		0x00,0x00,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x70,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x40,0x20,0x00,0x00,
		0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0xf8,0x08,0x08,0xf8,0x08,0x08,0x10,0x20,0xc0,0x00,0x00,
		0x00,0x00,0x00,0x00,0xf8,0x08,0x28,0x30,0x20,0x40,0x00,0x00,
		0x00,0x00,0x00,0x00,0x10,0x20,0x60,0xa0,0x20,0x20,0x00,0x00,
		0x00,0x00,0x00,0x00,0x20,0xf8,0x88,0x08,0x10,0x20,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0xf8,0x20,0x20,0x20,0xf8,0x00,0x00,
		0x00,0x00,0x00,0x10,0x10,0xf8,0x10,0x30,0x50,0x90,0x00,0x00,
		0x00,0x00,0x00,0x40,0x40,0xf8,0x48,0x50,0x40,0x40,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x70,0x10,0x10,0x10,0xf8,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0xf0,0x10,0xf0,0x10,0xf0,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0xa8,0xa8,0x08,0x10,0x60,0x00,0x00,
		0x00,0x00,0x00,0x00,0x80,0x78,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0xf8,0x08,0x28,0x28,0x30,0x20,0x20,0x40,0x80,0x00,0x00,
		0x00,0x08,0x10,0x10,0x20,0x60,0xa0,0x20,0x20,0x20,0x00,0x00,
		0x00,0x20,0x20,0xf8,0x88,0x88,0x10,0x10,0x20,0x40,0x00,0x00,
		0x00,0x00,0xf8,0x20,0x20,0x20,0x20,0x20,0x20,0xf8,0x00,0x00,
		0x00,0x10,0xf8,0x10,0x30,0x30,0x50,0x50,0x90,0x30,0x00,0x00,
		0x00,0x20,0x20,0xf8,0x28,0x28,0x28,0x48,0x48,0x98,0x00,0x00,
		0x00,0x20,0x20,0xf8,0x20,0x20,0xf8,0x20,0x20,0x20,0x00,0x00,
		0x00,0x40,0x78,0x48,0x88,0x88,0x10,0x10,0x20,0x40,0x00,0x00,
		0x00,0x40,0x78,0x50,0x90,0x90,0x10,0x20,0x20,0x40,0x00,0x00,
		0x00,0x00,0xf8,0x08,0x08,0x08,0x08,0x08,0x08,0xf8,0x00,0x00,
		0x00,0x50,0xf8,0x50,0x50,0x50,0x10,0x20,0x20,0x40,0x00,0x00,
		0x00,0x80,0x48,0x08,0x88,0x48,0x08,0x10,0x20,0xc0,0x00,0x00,
		0x00,0xf0,0x10,0x10,0x10,0x20,0x20,0x50,0x48,0x88,0x00,0x00,
		0x00,0x40,0x40,0xf8,0x48,0x48,0x50,0x40,0x40,0x38,0x00,0x00,
		0x00,0x88,0x88,0x48,0x48,0x08,0x10,0x10,0x20,0x40,0x00,0x00,
		0x00,0x40,0x78,0x48,0x48,0xa8,0x98,0x10,0x20,0x40,0x00,0x00,
		0x00,0x10,0x60,0x20,0xf8,0x20,0x20,0x40,0x40,0x80,0x00,0x00,
		0x00,0xa0,0xa8,0xa8,0x08,0x08,0x10,0x10,0x20,0x40,0x00,0x00,
		0x00,0x70,0x00,0x00,0xf8,0x20,0x20,0x20,0x40,0x80,0x00,0x00,
		0x00,0x40,0x40,0x40,0x60,0x50,0x48,0x40,0x40,0x40,0x00,0x00,
		0x00,0x20,0x20,0xf8,0x20,0x20,0x20,0x20,0x40,0x80,0x00,0x00,
		0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x00,0x00,0xf8,0x00,0x00,
		0x00,0x00,0xf8,0x08,0x48,0x30,0x10,0x28,0x48,0x80,0x00,0x00,
		0x00,0x20,0xf8,0x08,0x08,0x10,0x20,0x68,0xa8,0x20,0x00,0x00,
		0x00,0x08,0x08,0x08,0x08,0x10,0x10,0x20,0x20,0x40,0x00,0x00,
		0x00,0x50,0x50,0x50,0x48,0x48,0x48,0x88,0x88,0x88,0x00,0x00,
		0x00,0x80,0x80,0xf8,0x80,0x80,0x80,0x80,0x80,0x78,0x00,0x00,
		0x00,0xf8,0x08,0x08,0x08,0x08,0x10,0x10,0x20,0x40,0x00,0x00,
		0x00,0x40,0x40,0xa0,0xa0,0x10,0x10,0x08,0x08,0x00,0x00,0x00,
		0x00,0x20,0xf8,0x20,0x20,0xa8,0xa8,0xa8,0xa8,0x20,0x00,0x00,
		0x00,0x00,0xf8,0x08,0x08,0x10,0x90,0x60,0x20,0x10,0x00,0x00,
		0x00,0xc0,0x30,0x00,0xc0,0x30,0x00,0xc0,0x30,0x00,0x00,0x00,
		0x00,0x40,0x40,0x40,0x40,0x40,0x50,0x88,0x88,0xf8,0x00,0x00,
		0x00,0x10,0x10,0x90,0x50,0x20,0x30,0x28,0x40,0x80,0x00,0x00,
		0x00,0xf8,0x20,0x20,0x20,0xf8,0x20,0x20,0x20,0x18,0x00,0x00,
		0x00,0x40,0x40,0xf8,0x48,0x48,0x50,0x40,0x40,0x40,0x00,0x00,
		0x00,0x00,0x00,0x70,0x10,0x10,0x10,0x10,0x10,0xf8,0x00,0x00,
		0x00,0x00,0x00,0xf8,0x08,0x08,0xf8,0x08,0x08,0xf8,0x00,0x00,
		0x00,0x70,0x00,0xf8,0x08,0x08,0x08,0x10,0x10,0x20,0x00,0x00,
		0x00,0x90,0x90,0x90,0x90,0x10,0x10,0x20,0x20,0x40,0x00,0x00,
		0x00,0xa0,0xa0,0xa0,0xa0,0xa0,0xa8,0xa8,0xb0,0xa0,0x00,0x00,
		0x00,0x80,0x80,0x80,0x80,0x80,0x88,0x90,0xa0,0xc0,0x00,0x00,
		0x00,0x00,0xf8,0x88,0x88,0x88,0x88,0x88,0x88,0xf8,0x00,0x00,
		0x00,0xf8,0x88,0x88,0x88,0x08,0x10,0x10,0x20,0x40,0x00,0x00,
		0x00,0x00,0x80,0x48,0x08,0x08,0x10,0x10,0x20,0xc0,0x00,0x00,
		0x00,0xa0,0xa0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x40,0xa0,0xa0,0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	};

	static UINT8 fontdata12x8[MAX_GLYPH_FONT * 16] =
	{
#include "font/cmd12x8.dat"
	};

	static UINT8 fontdata12x12[MAX_GLYPH_FONT * 24] =
	{
#include "font/cmd12x12.dat"
	};

	static const gfx_layout fontlayout6x8 =
	{
		6,8,
		256,
		1,
		{ 0 },
		{ 0, 1, 2, 3, 4, 5, 6, 7 },
		{ 0*8, 1*8, 2*8, 3*8, 4*8, 5*8, 6*8, 7*8 },
		8*8
	};

	static const gfx_layout fontlayout12x8 =
	{
		12,8,
		MAX_GLYPH_FONT,
		1,
		{ 0 },
		{ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 },
		{ 0*16, 1*16, 2*16, 3*16, 4*16, 5*16, 6*16, 7*16 },
		8*16
	};

	static gfx_layout fontlayout6x12 =
	{
		6,12,	/* 6*12 characters */
		256,    /* 256 characters */
		1,	/* 1 bit per pixel */
		{ 0 },
		{ 0, 1, 2, 3, 4, 5, 6, 7 },	/* straightforward layout */
		{ 0*8, 1*8, 2*8, 3*8, 4*8, 5*8, 6*8, 7*8, 8*8, 9*8, 10*8, 11*8 },
		8*12	/* every char takes 8 consecutive bytes */
	};
	static gfx_layout fontlayout12x12 =
	{
		12,12,
		0,	// total number set later
		1,
		{ 0 },
		{ 0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15 },
		{ 0*16,1*16,2*16,3*16,4*16,5*16,6*16,7*16,8*16,9*16,10*16,11*16 },
		16*12
	};

	gfx_layout layout;
	gfx_layout layout1;
	gfx_layout layout2;
	UINT32 tempoffset[MAX_GFX_SIZE];
	unsigned char *fontdata_sb;
	static unsigned char *fontdata_db;
	static int fontlangcode = -1;
	static int use_8linefont = 1;
	int uirotwidth, uirotheight;
	int temp, i;

	uifont_freefont();
	*rotcharwidth = 0;
	*rotcharheight = 0;

	if (fontlangcode != options.langcode)
	{
		if (!use_8linefont && fontdata_db)
			free(fontdata_db);

		if ((fontdata_db = load_fontdata()) == NULL)
			use_8linefont = 1;
		else
			use_8linefont = 0;

		fontlangcode = options.langcode;
	}

	if (use_8linefont)
	{
		switch (options.langcode)
		{
		case UI_LANG_PL_PL:
		case UI_LANG_HU_HU:
			fontdata_sb = cp1250_fontdata6x8;
			break;
		default:
			fontdata_sb = fontdata6x8;
			break;
		}

		fontdata_db = fontdata12x8;

		layout = fontlayout6x8;
		layout1 = fontlayout6x8;
		layout2 = fontlayout12x8;
	}
	else
	{
		fontdata_sb = fontdata6x12;
		memcpy(fontdata_db + ui_lang_info[options.langcode].numchars * 24, fontdata12x12, MAX_GLYPH_FONT * 24);

		layout = fontlayout6x12;
		layout1 = fontlayout6x12;
		layout2 = fontlayout12x12;
		layout2.total = ui_lang_info[options.langcode].numchars + MAX_GLYPH_FONT;
	}


	ui_get_bounds(&uirotwidth, &uirotheight);

	if (options.artwork_res == 2)
	{
		/* pixel double horizontally */
		if (uirotwidth >= 420)
		{
			memcpy(tempoffset, layout1.xoffset, sizeof(tempoffset));
			for (i = 0; i < layout1.width; i++)
				layout1.xoffset[i*2+0] = layout1.xoffset[i*2+1] = tempoffset[i];
			layout1.width *= 2;

			memcpy(tempoffset, layout2.xoffset, sizeof(tempoffset));
			for (i = 0; i < layout2.width; i++)
				layout2.xoffset[i*2+0] = layout2.xoffset[i*2+1] = tempoffset[i];
			layout2.width *= 2;
		}

		/* pixel double vertically */
		if (uirotheight >= 420)
		{
			memcpy(tempoffset, layout1.yoffset, sizeof(tempoffset));
			for (i = 0; i < layout1.height; i++)
				layout1.yoffset[i*2+0] = layout1.yoffset[i*2+1] = tempoffset[i];
			layout1.height *= 2;

			memcpy(tempoffset, layout2.yoffset, sizeof(tempoffset));
			for (i = 0; i < layout2.height; i++)
				layout2.yoffset[i*2+0] = layout2.yoffset[i*2+1] = tempoffset[i];
			layout2.height *= 2;
		}
	}

	/* apply swappage */
	if (Machine->ui_orientation & ORIENTATION_SWAP_XY)
	{
		memcpy(tempoffset, layout1.xoffset, sizeof(tempoffset));
		memcpy(layout1.xoffset, layout1.yoffset, sizeof(layout1.xoffset));
		memcpy(layout1.yoffset, tempoffset, sizeof(layout1.yoffset));

		temp = layout1.width;
		layout1.width = layout1.height;
		layout1.height = temp;

		memcpy(tempoffset, layout2.xoffset, sizeof(tempoffset));
		memcpy(layout2.xoffset, layout2.yoffset, sizeof(layout2.xoffset));
		memcpy(layout2.yoffset, tempoffset, sizeof(layout2.yoffset));

		temp = layout2.width;
		layout2.width = layout2.height;
		layout2.height = temp;
	}

	/* apply xflip */
	if (Machine->ui_orientation & ORIENTATION_FLIP_X)
	{
		memcpy(tempoffset, layout1.xoffset, sizeof(tempoffset));
		for (i = 0; i < layout1.width; i++)
			layout1.xoffset[i] = tempoffset[layout1.width - 1 - i];

		memcpy(tempoffset, layout2.xoffset, sizeof(tempoffset));
		for (i = 0; i < layout2.width; i++)
			layout2.xoffset[i] = tempoffset[layout2.width - 1 - i];
	}

	/* apply yflip */
	if (Machine->ui_orientation & ORIENTATION_FLIP_Y)
	{
		memcpy(tempoffset, layout1.yoffset, sizeof(tempoffset));
		for (i = 0; i < layout1.height; i++)
			layout1.yoffset[i] = tempoffset[layout1.height - 1 - i];

		memcpy(tempoffset, layout2.yoffset, sizeof(tempoffset));
		for (i = 0; i < layout2.height; i++)
			layout2.yoffset[i] = tempoffset[layout2.height - 1 - i];
	}

	/* decode rotated font */
	uirotfont_sb = allocgfx(&layout1);
	uirotfont_db = allocgfx(&layout2);
	if (!uirotfont_sb || !uirotfont_db)
		fatalerror("Fatal error: could not allocate memory for UI font!");
	decodegfx(uirotfont_sb, fontdata_sb, 0, uirotfont_sb->total_elements);
	decodegfx(uirotfont_db, fontdata_db, 0, uirotfont_db->total_elements);

	/* set the raw and rotated character width/height */
	*rotcharwidth = (Machine->ui_orientation & ORIENTATION_SWAP_XY) ? layout1.height : layout1.width;
	*rotcharheight = (Machine->ui_orientation & ORIENTATION_SWAP_XY) ? layout1.width : layout1.height;

	/* set up the bogus colortable */
	uirotfont_sb->colortable = uifont_colortable;
	uirotfont_sb->total_colors  = MAX_COLORTABLE;
	uirotfont_db->colortable = uifont_colortable;
	uirotfont_db->total_colors = MAX_COLORTABLE;

	/* build decode table */
	for (i = 0x4081; i < 0xfffe; i++)
		_uifonttable[i] = uifont_buildtable(i) + 256;
}


void uifont_freefont(void)
{
	if (uirotfont_sb)
	{
		freegfx(uirotfont_sb);
		uirotfont_sb = NULL;
	}
	if (uirotfont_db)
	{
		freegfx(uirotfont_db);
		uirotfont_db = NULL;
	}
}


int uifont_decodechar(const unsigned char *s, UINT16 *code)
{
	unsigned char c1 = *(unsigned char *)s;
	unsigned char c2 = *(unsigned char *)(s + 1);
	int isDBC = 0;

#ifdef UI_COLOR_DISPLAY
	#define NULL_SPACE1	'\b'
	// Remove to Null Space Character
	if (c1 == NULL_SPACE1)
		return 3;
#endif /* UI_COLOR_DISPLAY */

	switch (options.langcode)
	{
	case UI_LANG_JA_JP:
		isDBC = (((c1 >= 0x81 && c1 <= 0x9f) || (c1 >= 0xe0 && c1 <= 0xfc)) && (c2 >= 0x40 && c2 <= 0xfc));
		break;
	case UI_LANG_ZH_CN:
		isDBC = ((c1 >= 0x81 && c1 <= 0xfe) && ((c2 >= 0x40 && c2 <= 0x7e) || (c2 >= 0x80 && c2 <= 0xfe)));
		break;
	case UI_LANG_ZH_TW:
		isDBC = ((c1 >= 0xa1 && c1 <= 0xf9) && ((c2 >= 0x40 && c2 <= 0x7e) || (c2 >= 0xa1 && c2 <= 0xfe)));
		break;
	case UI_LANG_KO_KR:
		isDBC = ((c1 >= 0x81 && c1 <= 0xfe) && ((c2 >= 0x41 && c2 <= 0x5a) || (c2 >= 0x61 && c2 <= 0x7a) || (c2 >= 0x81 && c2 <= 0xfe)));
		break;
	}

	if (!isDBC)
	{
		UINT8 gc1, gc2, gsz;

		get_ext_font_info(&gc1, &gc2, &gsz);

		if (gc1 <= c1 && c1 <= gc1 + MAX_GLYPH_FONT / gsz)
		{
			if (gc2 <= c2 && c2 <= gc2 + gsz)
				isDBC = 1;
		}
	}

	if (isDBC)
	{
		*code = _uifonttable[(c1 << 8) | c2];

		if (*code == 0)
			*code = _uifonttable[0x817f];

		return 2;
	}

	*code = c1;
	return 1;
}

void uifont_drawchar(mame_bitmap *dest, UINT16 code, int color, int sx, int sy, const rectangle *bounds)
{
	gfx_element *uifont = uirotfont_sb;
	pen_t back = uifont_colortable[FONT_COLOR_NORMAL];

#ifdef UI_COLOR_DISPLAY
	uifont_colortable[FONT_COLOR_NORMAL] = color;
#else /* UI_COLOR_DISPLAY */
	if (color)
		uifont_colortable[FONT_COLOR_NORMAL] = get_white_pen();
	else
		uifont_colortable[FONT_COLOR_NORMAL] = get_black_pen();
#endif /* UI_COLOR_DISPLAY */

	if (code > 0x00ff && uirotfont_db != NULL)
	{
		uifont = uirotfont_db;
		code -= 256;

#ifdef UI_COLOR_DISPLAY
		{
#include "cmd_table.c"
			int _code = code - ui_lang_info[options.langcode].numchars;

			if (_code > 0 && _code < COLOR_BUTTONS)
			{
				int col = color_table[_code];

				if (col)
					uifont_colortable[FONT_COLOR_NORMAL] = uifont_colortable[col];
			}
		}
#endif /* UI_COLOR_DISPLAY */
	}

	drawgfx(dest,
	        uifont,
	        code,
#ifdef UI_COLOR_DISPLAY
	        0,
#else /* UI_COLOR_DISPLAY */
		color,
#endif /* UI_COLOR_DISPLAY */
	        0,
	        0,
	        sx,
	        sy,
	        bounds,
#ifdef UI_COLOR_DISPLAY
	        TRANSPARENCY_PEN,
#else /* UI_COLOR_DISPLAY */
#ifdef TRANS_UI
	        TRANSPARENCY_PEN,
#else /* TRANS_UI */
	        TRANSPARENCY_NONE,
#endif /* TRANS_UI */
#endif /* UI_COLOR_DISPLAY */
	        0);

	uifont_colortable[FONT_COLOR_NORMAL] = back;
}


#ifdef UI_COLOR_DISPLAY
void convert_command_move(char *buf)
{
	unsigned char *c = (unsigned char *)buf;
	UINT8 gc1, gc2, gsz;

#include "cmd_plus.c"

	get_ext_font_info(&gc1, &gc2, &gsz);

	while (*c)
	{
		UINT16 code;

		if (uifont_decodechar(c, &code) == 2)
		{
			c += 2;
		}
		else
		{
			struct fix_command_t *fixcmd;

			fixcmd = NULL;

#ifdef COMMAND_DEFAULT_TEXT
			if (c[0] == COMMAND_DEFAULT_TEXT)
				fixcmd = default_text;
#endif /* COMMAND_DEFAULT_TEXT */

#ifdef COMMAND_EXPAND_TEXT
			if (c[0] == COMMAND_EXPAND_TEXT)
				fixcmd = expand_text;
#endif /* COMMAND_EXPAND_TEXT */

			if (fixcmd)
			{
				if (c[0] == c[1])
				{
					c[1] = NULL_SPACE1;
					c += 2;
					continue;
				}

				while (fixcmd->glyph_code)
				{
					if (c[1] == fixcmd->glyph_char)
						break;

					fixcmd++;
				}

				if (fixcmd->glyph_code)
				{
					c[0] = gc1 + (fixcmd->glyph_code / gsz);
					c[1] = gc2 + (fixcmd->glyph_code % gsz);
					c += 2;
					continue;
				}
			}

#ifdef COMMAND_CONVERT_TEXT
			if (c[0] == COMMAND_CONVERT_TEXT)
			{
				int len = 0;
				int i;

				if (c[0] == c[1])
				{
					c[1] = NULL_SPACE1;
					c += 2;
					continue;
				}

				for (i = 0; convert_text[i].glyph_str; i++)
				{
					len = strlen(convert_text[i].glyph_str);

					if (strncmp(c + 1, convert_text[i].glyph_str, len) == 0)
						break;
				}

				if (convert_text[i].glyph_str)
				{
					c[0] = gc1 + (convert_text[i].glyph_code / gsz);
					c[1] = gc2 + (convert_text[i].glyph_code % gsz);

					for (i = 2; i < len + 1; i++)
						c[i] = NULL_SPACE1;

					c += len + 1;
					continue;
				}
			}
#endif /* COMMAND_CONVERT_TEXT */

			if (c[0] == '\\' && c[1] == 'n')
			{
				c[0] = '\n';
				c[1] = NULL_SPACE1;
				c+=2;
			}
			else
				c++;
		}
	}
}
#endif /* UI_COLOR_DISPLAY */
