/*
	AramakiOnline
	Copyright (C) 2008-2009 superbacker

	This program is free software; you can redistribute it and/or modify it under the terms
	of the GNU General Public License as published by the Free Software Foundation;
	either version 3 of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful, but WITHOUT ANY
	WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
	FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

	You should have received a copy of the GNU General Public License along with this
	program. If not, see <http://www.gnu.org/licenses/>.
*/

package jp.sourceforge.aramaki.net {
	import flash.events.EventDispatcher;
	import flash.events.NetStatusEvent;
	import flash.media.Camera;
	import flash.media.Microphone;
	import flash.net.NetConnection;
	import flash.net.NetStream;
	
	import jp.sourceforge.aramaki.events.PeerChangeGroupEvent;
	import jp.sourceforge.aramaki.events.PeerEvent;
	import jp.sourceforge.aramaki.events.PeerMessageEvent;
	
	/**
	 * ピア接続イベント
	 * @eventType jp.sourceforge.aramaki.events.PeerEvent.CONNECT
	 */
	[Event(name="peerConnect",type="jp.sourceforge.aramaki.events.PeerEvent")]
	
	/**
	 * ピア接続失敗イベント
	 * @eventType jp.sourceforge.aramaki.events.PeerEvent.CONNECT_FAILED
	 */
	[Event(name="peerConnectFailed",type="jp.sourceforge.aramaki.events.PeerEvent")]
	
	/**
	 * ピア切断イベント
	 * @eventType jp.sourceforge.aramaki.events.PeerEvent.CLOSE
	 */
	[Event(name="peerClose",type="jp.sourceforge.aramaki.events.PeerEvent")]
	
	/**
	 * メッセージ受信イベント
	 * @eventType jp.sourceforge.aramaki.events.PeerMessageEvent.PEER_MESSAGE
	 */
	[Event(name="peerMessage",type="jp.sourceforge.aramaki.events.PeerMessageEvent")]
	
	/**
	 * グループ変更イベント
	 * @eventType jp.sourceforge.aramaki.events.PeerChangeGroupEvent.PEER_CHANGE_GROUP
	 */
	[Event(name="peerChangeGroup",type="jp.sourceforge.aramaki.events.PeerChangeGroupEvent")]
	
	/**
	 * ピア
	 */
	public class Peer extends EventDispatcher {
		/**
		 * NetConnection
		 */
		private var netConnection:NetConnection;
		/**
		 * 接続ストリーム
		 */
		private var controlStream:NetStream;
		/**
		 * 入力ストリーム
		 */
		private var incomingStream:NetStream;
		/**
		 * 出力ストリーム
		 */
		private var outgoingStream:NetStream;
		/**
		 * @private
		 */
		private var _group:String;
		/**
		 * 状態
		 */
		private var state:int=NOT_CONNECTED;
		/**
		 * 未接続
		 */
		private const NOT_CONNECTED:int=0;
		/**
		 * 接続中
		 */
		private const CONNECTING:int=1;
		/**
		 * 接続受付中
		 */
		private const ACCEPTING:int=2;
		/**
		 * 接続完了
		 */
		private const CONNECTED:int=3;
		/**
		 * @private
		 */
		private var _id:String;
		
		/**
		 * ピアへ接続する
		 * 
		 * 接続受付ストリームに接続後、入力ストリーム・出力ストリームを作成する
		 * 
		 * @param netConnection StratusへのNetConnection
		 * @param id 接続するピアのID
		 */
		public function connect(netConnection:NetConnection,id:String):void {
			trace("Peer: connect: "+id);
			
			this.netConnection=netConnection;
			netConnection.addEventListener(NetStatusEvent.NET_STATUS,netConnectionNetStatusHandler);
			_id=id;
			
			state=CONNECTING;
			
			controlStream=new NetStream(netConnection,id);
			controlStream.addEventListener(NetStatusEvent.NET_STATUS,controlStreamNetStatusHandler);
			controlStream.play("listener");
			
			createOutgoingStream();
		}
		
		/**
		 * ピアからの接続を受け付ける
		 * 
		 * @param netConnection StratusへのNetConnection
		 * @param id 接続するピアのID
		 */
		public function accept(netConnection:NetConnection,id:String):void {
			trace("Peer: accept: "+id);
			
			this.netConnection=netConnection;
			netConnection.addEventListener(NetStatusEvent.NET_STATUS,netConnectionNetStatusHandler);
			_id=id;
			
			state=ACCEPTING;
			
			createOutgoingStream();
			createIncomingStream();
		}
		
		/**
		 * ピアのID
		 */
		public function get farID():String {
			return _id;
		}
		
		/**
		 * 所属グループ
		 */
		public function get group():String {
			return _group;
		}
		
		/**
		 * 接続フラグ
		 */
		public function get connected():Boolean {
			return state==CONNECTED;
		}
		
		/**
		 * 入力ストリームを作成
		 */
		private function createIncomingStream():void {
			trace("Peer: createIncomingStream: "+_id);
			
			incomingStream=new NetStream(netConnection,_id);
			incomingStream.addEventListener(NetStatusEvent.NET_STATUS,incomingStreamNetStatusHandler);
			incomingStream.client={message:messageHandler,changeGroup:changeGroupHandler,connectComplete:connectComplete};
			incomingStream.play(netConnection.nearID);
		}
		
		/**
		 * 出力ストリームを作成
		 */
		private function createOutgoingStream():void {
			trace("Peer: createOutgoingStream: "+_id);
			
			outgoingStream=new NetStream(netConnection,NetStream.DIRECT_CONNECTIONS);
			outgoingStream.addEventListener(NetStatusEvent.NET_STATUS,outgoingStreamNetStatusHandler);
			outgoingStream.publish(_id);
		}
		
		/**
		 * 接続完了
		 */
		private function connectComplete():void {
			trace("Peer: connectComplete: "+_id);
			
			state=CONNECTED;
			
			dispatchEvent(new PeerEvent(PeerEvent.PEER_CONNECT,false,false,this));
		}
		
		/**
		 * NetConnectionのNetStatusハンドラ
		 */
		private function netConnectionNetStatusHandler(event:NetStatusEvent):void {
			switch(event.info.code) {
				case "NetStream.Connect.Closed": {
					if (event.info.stream.farID==_id) {
						switch(state) {
							case CONNECTED: {
								//切断
								close();
								break;
							}
							case ACCEPTING: {
								//接続受付失敗
								connectFailed();
								break;
							}
							case CONNECTING:{
								//接続中
								//<再接続>
								break;
							}
						}
					}
					
					break;
				}
			}
		}
		
		/**
		 * 接続失敗
		 */
		private function connectFailed():void {
			trace("Peer: connectFailed: "+_id);
			
			dispose();
			
			dispatchEvent(new PeerEvent(PeerEvent.PEER_CONNECT_FAILED,false,false,this));
		}
		
		/**
		 * 接続ストリームのNetStatusハンドラ
		 */
		private function controlStreamNetStatusHandler(event:NetStatusEvent):void {
			trace("Peer: controlStream: "+event.info.code);
		}
		
		/**
		 * 入力ストリームのNetStatusハンドラ
		 */
		private function incomingStreamNetStatusHandler(event:NetStatusEvent):void {
			trace("Peer: incomingStream: "+event.info.code);
		}
		
		/**
		 * 出力ストリームのNetStatusハンドラ
		 */
		private function outgoingStreamNetStatusHandler(event:NetStatusEvent):void {
			trace("Peer: outgoingStream: "+event.info.code);
			
			switch(event.info.code) {
				case "NetStream.Play.Start": {
					if (state==ACCEPTING) {
						//接続受付完了
						outgoingStream.send("connectComplete");
						connectComplete();
					} else if (state==CONNECTING) {
						createIncomingStream();
					}
					
					break;
				}
			}
		}
		
		/**
		 * メッセージを受信したときに呼ばれる
		 * 
		 * @param handlerName メッセージハンドラの名前
		 * @param args メッセージハンドラの引数
		 */
		private function messageHandler(handlerName:String,args:Array):void {
			trace("Peer: message: "+_id);
			
			dispatchEvent(new PeerMessageEvent(PeerMessageEvent.PEER_MESSAGE,false,false,this,handlerName,args));
		}
		
		/**
		 * グループが変更されたときに呼ばれる
		 * 
		 * @param group グループ
		 */
		private function changeGroupHandler(group:String):void {
			trace("Peer: changeGroup: "+_id);
			
			_group=group;
			dispatchEvent(new PeerChangeGroupEvent(PeerChangeGroupEvent.PEER_CHANGE_GROUP,false,false,this,group));
		}
		
		/**
		 * 接続を閉じる
		 */
		public function close():void {
			trace("Peer: close: "+_id);
			
			if (state==CONNECTED) {
				//切断イベントを発行
				dispatchEvent(new PeerEvent(PeerEvent.PEER_CLOSE,false,false,this));
			}
			
			dispose();
		}
		
		/**
		 * 後始末
		 */
		private function dispose():void {
			if (controlStream) {
				//接続ストリームを閉じる
				controlStream.removeEventListener(NetStatusEvent.NET_STATUS,controlStreamNetStatusHandler);
				controlStream.close();
				controlStream=null;
			}
			
			if (outgoingStream) {
				//出力ストリームを閉じる
				outgoingStream.removeEventListener(NetStatusEvent.NET_STATUS,outgoingStreamNetStatusHandler);
				outgoingStream.close();
				outgoingStream=null;
			}
			
			if (incomingStream) {
				//入力ストリームを閉じる
				incomingStream.removeEventListener(NetStatusEvent.NET_STATUS,incomingStreamNetStatusHandler);
				incomingStream.close();
				incomingStream=null;
			}
			
			netConnection.removeEventListener(NetStatusEvent.NET_STATUS,netConnectionNetStatusHandler);
			netConnection=null;
			
			state=NOT_CONNECTED;
		}
		
		/**
		 * メッセージを送信
		 * 
		 * @param handlerName メッセージハンドラの名前
		 * @param args メッセージハンドラの引数
		 */
		public function send(handlerName:String,...args):void {
			outgoingStream.send("message",handlerName,args);
		}
		
		/**
		 * グループの変更を通知
		 * 
		 * @param group グループ
		 */
		public function notifyChangeGroup(group:String):void {
			trace("Peer: notifyChangeGroup: "+_id);
			
			outgoingStream.send("changeGroup",group);
		}
		
		/**
		 * 映像配信を開始
		 * 
		 * @param camera カメラ
		 */
		public function attachCamera(camera:Camera):void {
			outgoingStream.attachCamera(camera);
		}
		
		/**
		 * 音声配信を開始
		 * 
		 * @param microphone マイク
		 */
		public function attachAudio(microphone:Microphone):void {
			outgoingStream.attachAudio(microphone);
		}
	}
}