/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.util.primitive;

import java.io.IOException;
import java.util.Collection;
import java.util.RandomAccess;

import net.morilib.lang.Hashes;

/**
 *
 * @author MORIGUCHI, Yuichiro 2010/10/11
 */
public class FloatArrayVector extends AbstractFloatVector
implements RandomAccess, java.io.Serializable {
	
	//
	private static final long serialVersionUID = 3332872309405682099L;
	
	//
	private float[] array;
	private transient int size;
	
	public FloatArrayVector(int initialCapacity) {
		super();
		if(initialCapacity < 0) {
			throw new IllegalArgumentException();
		}
		array = new float[initialCapacity];
	}
	
	public FloatArrayVector() {
		this(10);
	}
	
	public FloatArrayVector(FloatCollection a) {
		array = a.toFloatArray();
		size  = array.length;
	}
	
	public FloatArrayVector(FloatCollection... as) {
		size = 0;
		for(FloatCollection a : as) {
			size  += a.size();
		}
		
		int s2 = 0;
		array = new float[size];
		for(FloatCollection a : as) {
			float[] b = a.toFloatArray();
			
			System.arraycopy(b, 0, array, s2, b.length);
			s2 += a.size();
		}
	}
	
	public FloatArrayVector(
			Collection<? extends FloatCollection> as) {
		size = 0;
		for(FloatCollection a : as) {
			size  += a.size();
		}
		
		int s2 = 0;
		array = new float[size];
		for(FloatCollection a : as) {
			float[] b = a.toFloatArray();
			
			System.arraycopy(b, 0, array, s2, b.length);
			s2 += a.size();
		}
	}
	
	/**
	 * 
	 * @param nsize
	 */
	public void ensureCapacity(int nsize) {
		if(nsize > array.length) {
			int ns = nsize;
			float[] b;
			
			while(ns < array.length) {
				ns = (array.length < 1288490186) ?
						(array.length / 3 * 5) + 1 : Integer.MAX_VALUE;
			}
			b = new float[ns];
			System.arraycopy(array, 0, b, 0, size);
			array = b;
		}
	}
	
	/* (non-Javadoc)
	 * @see net.morilib.util.primitive.FloatList#add(int, float)
	 */
	public void addFloat(int index, float v) {
		if(index > size || index < 0) {
			throw new IndexOutOfBoundsException();
		}
		
		ensureCapacity(size + 1);
		modCount++;
		if(index < size) {
			System.arraycopy(
					array, index, array, index + 1, size - index);
		}
		array[index] = v;
		size++;
	}

	/* (non-Javadoc)
	 * @see net.morilib.util.primitive.FloatList#addAll(int, net.morilib.util.primitive.FloatCollection)
	 */
	public boolean addAllFloat(int index, FloatCollection a) {
		if(index > size || index < 0) {
			throw new IndexOutOfBoundsException();
		}
		
		modCount++;
		if(a.isEmpty()) {
			return false;
		} else {
			float[] b = a.toFloatArray();
			
			ensureCapacity(size + b.length);
			System.arraycopy(
					array, index, array, index + b.length,
					size - index);
			System.arraycopy(b, 0, array, index, b.length);
			size += b.length;
			return true;
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.util.primitive.FloatList#get(int)
	 */
	public float getFloat(int index) {
		if(index >= size || index < 0) {
			throw new IndexOutOfBoundsException();
		}
		return array[index];
	}

	/* (non-Javadoc)
	 * @see net.morilib.util.primitive.FloatList#remove(int)
	 */
	public float removeAt(int index) {
		if(index >= size || index < 0) {
			throw new IndexOutOfBoundsException();
		}
		
		float res = array[index];
		modCount++;
		if(index < size - 1) {
			System.arraycopy(
					array, index + 1, array, index,
					size - index - 1);
		}
		size--;
		return res;
	}

	/* (non-Javadoc)
	 * @see net.morilib.util.primitive.FloatList#set(int, float)
	 */
	public float setFloat(int index, float v) {
		if(index >= size || index < 0) {
			throw new IndexOutOfBoundsException();
		}
		
		float res = array[index];
		modCount++;
		array[index] = v;
		return res;
	}

	/* (non-Javadoc)
	 * @see net.morilib.util.primitive.FloatCollection#addAll(net.morilib.util.primitive.FloatCollection)
	 */
	public boolean addAllFloat(FloatCollection a) {
		float[] b = a.toFloatArray();
		
		modCount++;
		ensureCapacity(size + b.length);
		System.arraycopy(b, 0, array, size, b.length);
		size += b.length;
		return true;
	}

	/* (non-Javadoc)
	 * @see net.morilib.util.primitive.FloatCollection#addAll(net.morilib.util.primitive.FloatCollection[])
	 */
	public boolean addAllFloat(FloatCollection... as) {
		int nsize = 0;
		for(FloatCollection a : as) {
			nsize  += a.size();
		}
		
		if(nsize > 0) {
			int s2 = size;
			
			modCount++;
			ensureCapacity(size + nsize);
			for(FloatCollection a : as) {
				float[] b = a.toFloatArray();
				
				System.arraycopy(b, 0, array, s2, b.length);
				s2 += a.size();
			}
			size += nsize;
			return true;
		} else {
			return false;
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.util.primitive.FloatCollection#addAll(java.util.Iterator)
	 */
	public boolean addAllFloat(Collection<? extends FloatCollection> as) {
		int nsize = 0;
		for(FloatCollection a : as) {
			nsize  += a.size();
		}
		
		if(nsize > 0) {
			int s2 = size;
			
			modCount++;
			ensureCapacity(size + nsize);
			for(FloatCollection a : as) {
				float[] b = a.toFloatArray();
				
				System.arraycopy(b, 0, array, s2, b.length);
				s2 += a.size();
			}
			size += nsize;
			return true;
		} else {
			return false;
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.util.primitive.FloatCollection#clear()
	 */
	public void clear() {
		modCount++;
		size = 0;
	}

	/* (non-Javadoc)
	 * @see net.morilib.util.primitive.FloatCollection#size()
	 */
	public int size() {
		return size;
	}

	/* (non-Javadoc)
	 * @see net.morilib.util.primitive.FloatCollection#toArray()
	 */
	public float[] toFloatArray() {
		float[] b = new float[size];
		
		System.arraycopy(array, 0, b, 0, size);
		return b;
	}

	/* (non-Javadoc)
	 * @see net.morilib.util.primitive.FloatCollection#toArray(float[])
	 */
	public float[] toFloatArray(float[] a) {
		if(a.length < size) {
			return toFloatArray();
		}
		System.arraycopy(array, 0, a, 0, size);
		return a;
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	public int hashCode() {
		return Hashes.sumHashCode(array);
	}
	
	//
	private void writeObject(
			java.io.ObjectOutputStream s) throws IOException {
		s.defaultWriteObject();
	}
	
	//
	private void readObject(
			java.io.ObjectInputStream s
			) throws IOException, ClassNotFoundException {
		s.defaultReadObject();
		size = array.length;
	}

}
