/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.math;

import java.math.BigInteger;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2010/11/21
 */
public final class Math2 {

	//
	private Math2() {}

	/**
	 * 
	 */
	public static final double EULER_CONSTANT =
		0.57721566490153286060651209008240243;

	/**
	 * 
	 */
	public static final double GOLDEN_RATIO =
		1.618033988749894848204586834365;

	/**
	 * 
	 */
	public static final int HARDY_RAMANUJAN_NUMBER = 1729;

	//
	private static final BigInteger LIMIT_NRT =
		BigInteger.valueOf(Integer.MAX_VALUE);

	/**
	 * 
	 * @param n
	 * @return
	 */
	public static final int minus1ToThe(int n) {
		return ((n & 1) == 0) ? 1 : -1;
	}

	/**
	 * 
	 * @param alpha
	 * @return
	 */
	public static double decimalPart(double x) {
		double r = Math.IEEEremainder(x, 1.0);

		return (r < 0) ? 1 + r : r;
	}

	/**
	 * 
	 * @param x
	 * @return
	 */
	public static boolean isInteger(double x) {
		return decimalPart(x) == 0.0;
	}

	//
	private static int toInt(byte b) {
		return (b < 0) ? (b & 0x7f) + 128 : b;
	}

	//
	private static final int BASE = 16;
	private static final int LOG2BASE = 4;

	/**
	 * 
	 * @param q
	 * @return
	 */
	public static BigInteger sqrtExact(BigInteger q) {
		byte[] a;
		int i = 0;
		BigInteger s = BigInteger.ZERO;
		BigInteger t = BigInteger.ZERO;
		BigInteger r = BigInteger.ZERO;

		if(q.signum() == 0) {
			return BigInteger.ZERO;
		} else if(q.signum() < 0) {
			throw new IllegalArgumentException();
		}

		a = q.toByteArray();
		while(a[i] == 0);
		for(; i < a.length; i++) {
			int x = 0;

			if(s.signum() == 0) {
				t = BigInteger.valueOf(toInt(a[i]));
				for(; x * x <= t.intValue(); x++);
				x--;
				s = BigInteger.valueOf(x);
			} else {
				BigInteger s0 = s;

				t = t.shiftLeft(LOG2BASE * 2).add(BigInteger.valueOf(
						toInt(a[i])));
				do {
					s = s0.shiftLeft(LOG2BASE).add(
							BigInteger.valueOf(x));
				} while(s.multiply(
						BigInteger.valueOf(x++)).compareTo(t) <= 0);
				x -= 2;
				s = s0.shiftLeft(LOG2BASE).add(
						BigInteger.valueOf(x));
			}
			t = t.subtract(s.multiply(BigInteger.valueOf(x)));
			s = s.add(BigInteger.valueOf(x));
			r = r.shiftLeft(LOG2BASE).add(BigInteger.valueOf(x));
		}
		return (t.signum() == 0) ? r : r.negate();
	}

	//
	private static BigInteger[] split(BigInteger b, int n) {
		byte[] c = b.toByteArray();
		byte[] d;
		BigInteger[] r;
		BigInteger   x;
		int i, l = 0;

		for(; c[l] == 0; l++);
		d = new byte[(c.length - l) * 2];
		for(int k = l; k < c.length; k++) {
			d[(k - l) * 2]     = (byte)(toInt(c[k]) >> 4);
			d[(k - l) * 2 + 1] = (byte)(toInt(c[k]) & 0xf);
		}

		r = new BigInteger[(d.length + n - 1) / n];
		i = d.length % n;
		if(i > 0) {
			x = BigInteger.ZERO;
			for(int k = 0; k < i; k++) {
				x = x.shiftLeft(LOG2BASE).add(BigInteger.valueOf(
						toInt(d[k])));
			}
			r[0] = x;
		}

		for(; i < d.length; i += n) {
			x = BigInteger.ZERO;
			for(int k = 0; k < n; k++) {
				x = x.shiftLeft(LOG2BASE).add(BigInteger.valueOf(
						toInt(d[i + k])));
			}
			r[(i + n - 1) / n] = x;
		}
		return r;
	}

	/**
	 * 
	 * @param q
	 * @param n
	 * @return
	 */
	public static BigInteger nrtExact(BigInteger q, int n) {
		BigInteger[] a;
		BigInteger   r = BigInteger.ZERO;
		BigInteger   t = BigInteger.ZERO;
		int i = 0;

		if(q.signum() == 0) {
			return BigInteger.ZERO;
		} else if(q.signum() < 0) {
			throw new IllegalArgumentException();
		} else if(n < 0) {
			throw new IllegalArgumentException();
		}

		a = split(q, n);
		for(; i < a.length; i++) {
			BigInteger x;
			int b = 0;

			t = t.shiftLeft(LOG2BASE * n).add(a[i]);
			r = r.shiftLeft(LOG2BASE);
			for(; b < BASE; b++) {
				x = r.add(BigInteger.valueOf(b)).pow(n);
				if(x.compareTo(t) > 0) {
					break;
				}
			}
			r = r.add(BigInteger.valueOf(b - 1));
		}
		return (t.subtract(q).signum() == 0) ? r : r.negate();
	}

	/**
	 * 
	 * @param q
	 * @param n
	 * @return
	 */
	public static BigInteger nrtExact(BigInteger q, BigInteger n) {
		if(n.signum() < 0 || n.compareTo(LIMIT_NRT) > 0) {
			throw new IllegalArgumentException();
		}
		return nrtExact(q, n.intValue());
	}

	/**
	 * 
	 * @param x
	 * @param b
	 * @return
	 */
	public static BigInteger integerLog(BigInteger x, BigInteger b) {
		BigInteger r = BigInteger.valueOf(-1);
		BigInteger s;
		BigInteger y = x;

		if(x.signum() <= 0) {
			throw new IllegalArgumentException();
		}

		do {
			y = y.divide(b);
			s = y.remainder(b);
			r = r.add(BigInteger.ONE);
		} while(y.signum() > 0);
		return (s.signum() == 0) ? r : r.negate();
	}

	/**
	 * 
	 * @param b1
	 * @param b2
	 * @return
	 */
	public static BigInteger pow(BigInteger b1, BigInteger b2) {
		byte[] a = b2.toByteArray();
		int i = 0;
		BigInteger r = BigInteger.ONE;

		if(b2.signum() < 0) {
			throw new IllegalArgumentException();
		} else if(b2.signum() == 0) {
			return BigInteger.ONE;
		} else if(b1.signum() == 0) {
			return BigInteger.ZERO;
		} else if(b1.equals(BigInteger.ONE)) {
			return BigInteger.ONE;
		} else if(b1.equals(BigInteger.valueOf(-1))) {
			int p = b2.remainder(BigInteger.valueOf(2)).intValue();

			return BigInteger.valueOf((p == 0) ? 1 : -1);
		}

		for(; a[i] == 0; i++);
		for(; i < a.length; i++) {
			r = r.pow(256);
			r = r.multiply(b1.pow(toInt(a[i])));
		}
		return r;
	}

}
