/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp;

import java.math.BigDecimal;

import net.morilib.lang.number.AbstractNumericalField;
import net.morilib.lang.number.Integer2;
import net.morilib.lang.number.NumericalField;
import net.morilib.lang.number.Rational;
import net.morilib.lisp.sos.LispType;

/**
 * 
 *
 *
 * @author MORIGUCHI, Yuichiro 2009
 */
public abstract class LispExactReal extends LispReal {

	//
	private static final NumericalField<LispReal>
	_FIELDR = new AbstractNumericalField<LispReal>() {

		@Override
		public LispReal valueOf(float v) {
			return LispDouble.toExact(v);
		}

		@Override
		public LispReal valueOf(double v) {
			return LispDouble.toExact(v);
		}

		@Override
		public LispReal valueOf(BigDecimal v) {
			return LispUtils.bigDecimalToRational(v);
		}

		@Override
		public LispReal valueOf(Rational v) {
			return LispRational.valueOf(v);
		}

		@Override
		public LispReal valueOf(int v) {
			return LispInteger.valueOf(v);
		}

		@Override
		public LispReal valueOf(long v) {
			return LispInteger.valueOf(v);
		}

		@Override
		public LispReal valueOf(Integer2 v) {
			return LispInteger.valueOf(v.toBigInteger());
		}

		@Override
		public LispReal getUnit() {
			return LispInteger.ONE;
		}

		@Override
		public LispReal getZero() {
			return LispInteger.ZERO;
		}

	};

	/**
	 * 
	 * @return
	 */
	public abstract Rational toRational();

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lisp.LispReal#toExact()
	 */
	public LispExactReal toExact() {
		return this;
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lisp.LispNumber#isNaN()
	 */
	public boolean isNaN() {
		return false;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.LispReal#isInfinity()
	 */
	@Override
	public boolean isInfinity() {
		return false;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.Datum#getType()
	 */
	@Override
	public LispType getType() {
		return LispType.REAL;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lang.algebra.Addable#multiply(int)
	 */
	@Override
	public LispReal multiply(int n) {
		return multiply(new LispSmallInt(n));
	}

	/* (non-Javadoc)
	 * @see net.morilib.lang.algebra.Multipliable#power(int)
	 */
	@Override
	public LispReal power(int n) {
		if(n < 0) {
			return power(-n).invert();
		} else if(n == 0) {
			return LispInteger.ONE;
		} else if(n == 1) {
			return this;
		} else {
			LispReal r = this;

			for(int i = 1; i < n; i++) {
				r = r.multiply(this);
			}
			return r;
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.lang.number.NumericalRingElement#getUniverse()
	 */
	@Override
	public NumericalField<LispReal> getUniverse() {
		return _FIELDR;
	}

}
