/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp;

import java.util.Map;

import net.morilib.lisp.r6rs.LibraryID;
import net.morilib.lisp.r6rs.LibraryIDException;
import net.morilib.lisp.r6rs.SymbolEnv;
import net.morilib.lisp.subr.BinaryArgs;

/**
 * 
 *
 *
 * @author MORIGUCHI, Yuichiro 2009
 */
public class EnvironmentObject extends Datum {

	/**
	 * 
	 *
	 *
	 * @author MORIGUCHI, Yuichiro 2011/01/22
	 */
	public static class R6RSEnvironment extends Subr {

		/* (non-Javadoc)
		 * @see net.morilib.lisp.Subr#eval(net.morilib.lisp.Datum, net.morilib.lisp.Environment, net.morilib.lisp.LispMessage)
		 */
		@Override
		public Datum eval(
				Datum body, Environment env, LispMessage mesg) {
			Environment env2 = new Environment();
			Map<Symbol, SymbolEnv> mp;

			try {
				mp = LibraryID.compileImport(body);
			} catch (LibraryIDException e) {
				throw e.toLispException(mesg);
			}

			for(Map.Entry<Symbol, SymbolEnv> e : mp.entrySet()) {
				Environment env3 = e.getValue().getEnvironment();

				env2.bindDatumReadOnly(
						e.getValue().getSymbol(),
						env3.getDatum(e.getKey()));
			}
			return new EnvironmentObject(env2, false);
		}

	}

	/**
	 * 
	 *
	 *
	 * @author MORIGUCHI, Yuichiro 2011/01/23
	 */
	public static class BindEnvironment extends BinaryArgs {

		/* (non-Javadoc)
		 * @see net.morilib.lisp.subr.BinaryArgs#execute(net.morilib.lisp.Datum, net.morilib.lisp.Datum, net.morilib.lisp.Environment, net.morilib.lisp.LispMessage)
		 */
		@Override
		protected Datum execute(
				Datum c1a, Datum c2a, Environment env,
				LispMessage mesg) {
			Map<Datum, Datum> maps;
			Environment eo;

			maps = LispUtils.assocToMap(c1a);
			if(maps == null) {
				throw mesg.getError("err.require.assoc", c1a);
			} else if(!(c2a instanceof EnvironmentObject)) {
				throw mesg.getError("err.environment", c2a);
			}

			eo = ((EnvironmentObject)c2a).getEnvironment();
			for(Map.Entry<Datum, Datum> e : maps.entrySet()) {
				Symbol s;

				if(!(e.getKey() instanceof Symbol)) {
					throw mesg.getError(
							"err.require.symbol", e.getKey());
				}
				s = (Symbol)e.getKey();
				eo.bindDatumReadOnly(s, e.getValue());
			}
			return c2a;
		}

	}

	//
	private Environment environment;
	private boolean inherit = false;

	//
	private EnvironmentObject(
			Environment environment, boolean inherit) {
		if(environment == null) {
			throw new NullPointerException();
		}

		this.environment = environment;
		this.inherit = inherit;
	}

	//
	/*package*/ static EnvironmentObject newNullEnv(int ver) {
		return new EnvironmentObject(Scheme.newNullEnv(ver), false);
	}

	//
	/*package*/ static EnvironmentObject newRnRSEnv(int ver) {
		return new EnvironmentObject(Scheme.newRnRSEnv(ver), false);
	}

	//
	/*package*/ static EnvironmentObject newInteractionEnv(
			Environment e) {
		//return new EnvironmentObject(Scheme.newInteractionEnv(e, msg));
		return new EnvironmentObject(e, true);
	}

	//
	/*package*/ Environment getEnvironment() {
		return environment;
	}

	//
	/*package*/ boolean isInherit() {
		return inherit;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.Datum#toDisplayString(java.lang.StringBuilder)
	 */
	@Override
	public void toDisplayString(StringBuilder buf) {
		buf.append("#<environment>");
	}

}
