/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.math.BigInteger;
import java.util.LinkedList;
import java.util.Locale;
import java.util.logging.Logger;

import net.morilib.lisp.util.LogEnv;

public final class Scheme {
	
	/**
	 * 
	 */
	public static final int SCHEME_VERSION = 5;
	
	
	private static Logger _log = LogEnv.init("schlush.main");
	
	private Environment  global;
	private LispCompiler comp;
	private CodeExecutor exec;
	private IntStack     memento;
	private Parser       parser;
	private LispMessage  message;
	private String       stackTrace;
	
	
	private Scheme(LispMessage msg) {
		if(msg == null) {
			throw new NullPointerException();
		}
		
		global  = new Environment();
		message = msg;
		comp    = CompilerFactory.getInstance(message);
		exec    = CodeExecutorFactory.getInstance(message);
		memento = exec.newMemento();
		parser  = new Parser(global, message);
	}
	
	
	/*package*/ Scheme(Environment env, LispMessage msg) {
		if(env == null) {
			throw new NullPointerException();
		} else if(msg == null) {
			throw new NullPointerException();
		}
		
		global  = env;
		message = msg;
		comp    = CompilerFactory.getInstance(message);
		exec    = CodeExecutorFactory.getInstance(message);
		memento = exec.newMemento();
		parser  = new Parser(global, message);
	}
	
	
	public static Scheme newInstance(Locale lc) {
		Scheme res = new Scheme(LispMessage.getInstance(lc));
		
		InitSubrLoader.load(res.global);
		InitVarLoader.load(res.global);
		res.parser.clear();
		InitLispLoader.load(res);
		res.parser.clear();
		return res;
	}
	
	public static Scheme newInstance() {
		return newInstance(Locale.getDefault());
	}
	
	
	/*package*/ static Environment newNullEnv(int ver) {
		Scheme res = new Scheme(LispMessage.getInstance());
		
		InitSubrLoader.loadNullEnv(res.global, ver);
		InitVarLoader.loadNullEnv(res.global, ver);
		res.parser.clear();
		InitLispLoader.loadNullEnv(res, ver);
		res.parser.clear();
		return res.global;
	}
	
	
	/*package*/ static Environment newRnRSEnv(int ver) {
		Scheme res = new Scheme(LispMessage.getInstance());
		
		InitSubrLoader.loadRnRSEnv(res.global, ver);
		InitVarLoader.loadRnRSEnv(res.global, ver);
		res.parser.clear();
		InitLispLoader.loadRnRSEnv(res, ver);
		res.parser.clear();
		return res.global;
	}
	
	
	public Datum input(Datum sexp) {
		try {
			Datum d = sexp;
			CompiledCode.Builder b = new CompiledCode.Builder();
			long t;
			
			// ޥѴ
			t = System.currentTimeMillis();
			d = comp.expandMacro(d, global, exec, memento);
			//System.out.println(LispUtils.getResult(d));
			IntLispUtils.timelog(_log, "Expand Macro : ", t);
			
			// ѥ
			t = System.currentTimeMillis();
			comp.compile(
					d, global, b, true, new Cons(), true,
					new LinkedList<Cons>(), exec, memento,
					new LispCompiler.MiscInfo(null));
			b.addReturnOp();
			IntLispUtils.timelog(_log, "Compile : ", t);
			
			// ¹
			//System.out.println(b.getCodeRef().toString());
			t = System.currentTimeMillis();
			Datum res = exec.exec(b.getCodeRef(), global, memento);
			IntLispUtils.timelog(_log, "Execute : ", t);
			
			// åɽ
			_log.fine(memento.toString());
			
			//parser.clear();
			return res;
		} finally {
			parser.clear();
			stackTrace = memento.getStackTrace();
			memento    = exec.newMemento();
		}
	}
	
	
	public Datum input(String rd) {
		try {
			parser.read(rd);
			if(!parser.parse()) {
				return null;
			}
			
			Datum d = parser.getDatum();
			if(d != null) {
				return input(d);
			} else {
				parser.clear();
				return null;
			}
		} catch(ReadException e) {
			parser.clear();
			throw e;
		} catch(IOException e) {
			throw new RuntimeException();
		}
	}
	
	
	public Datum exec(String rd) {
		Datum res = input(rd);
		
		if(res == null) {
			throw new IllegalArgumentException(
					"S-exp is not completed");
		}
		return res;
	}
	
	
	public void readFile(Reader rd1) throws IOException {
		BufferedReader rd = new BufferedReader(rd1);
		int lineno = 0;
		
		while(true) {
			try {
				String read = rd.readLine();
				lineno++;
				
				if(read == null) {
					break;
				}
				input(read);
			} catch(ReadException e) {
				String ln = message.get("err.lineno");
				throw new ReadFileException(
						lineno + ln + ":" + e.getMessage(), e);
			} catch(LispException e) {
				String ln = message.get("err.lineno");
				throw new ReadFileException(
						lineno + ln + ":" + e.getMessage(), e);
			} catch(JavaException e) {
				String ln = message.get("err.lineno");
				throw new ReadFileException(
						lineno + ln + ":" + e.getMessage(), e);
			}
		}
		parser.clear();
	}
	
	
	public void readEvalPrintLoop(Reader rd1) throws IOException {
		BufferedReader rd = new BufferedReader(rd1);
		//int lineno = 0;
		
		reploop:
		while(true) {
			String prompt = " >";
			Datum res = null;
			
			try {
				while(res == null) {
					System.out.print(prompt);
					String r2 = rd.readLine();
					
					if(r2 == null) {
						break reploop;
					}
					res = input(r2);
					prompt = ">>";
				}
				
				// ɽ
				System.out.println(LispUtils.getResult(res));
				
				// åɽ
				//System.out.println(memento);
			} catch(ReadException e) {
				System.err.println(
						message.get("err.repl.read") +
						e.getMessage());
				//e.printStackTrace();
				//parser.clear();
				//memento = exec.newMemento();
			} catch(LispException e) {
				String tra = stackTrace;
				
				System.err.println(
						message.get("err.repl.err") +
						e.getMessage());
				if(tra != null && !tra.equals("")) {
					System.err.println(message.get("err.stacktrace"));
					System.err.print(tra);
				}
				_log.fine("Stack trace\n" + tra);
				//e.printStackTrace();
				//parser.clear();
				//memento = exec.newMemento();
			} catch(JavaException e) {
				System.err.println(
						message.get("err.repl.err") +
						e.getMessage());
				//e.printStackTrace();
				//parser.clear();
				//memento = exec.newMemento();
			}
		}
	}
	
	
	public static void main(String[] args) throws IOException {
		Scheme eval = Scheme.newInstance();
		
		if(args.length == 0) {
			eval.readEvalPrintLoop(new InputStreamReader(System.in));
			System.exit(0);
		} else {
			InputStream ins;
			
			if(args[0].equals("-")) {
				ins = System.in;
			} else {
				ins = new FileInputStream(args[0]);
			}
			
			try {
				eval.readFile(new InputStreamReader(ins));
				System.exit(0);
			} catch(ReadFileException e) {
				System.err.println(e.getMessage());
				System.exit(2);
			}
		}
	}
	
	
	public void set(String var, Object o) {
		global.bindDatum(Symbol.getSymbol(var), LispUtils.toDatum(o));
	}
	
	
	public void setDotList(String var, Object cdr, Object... lst) {
		ConsListBuilder b = new ConsListBuilder();
		
		for(Object o : lst) {
			b.append(LispUtils.toDatum(o));
		}
		global.bindDatum(
				Symbol.getSymbol(var), b.get(LispUtils.toDatum(cdr)));
	}
	
	
	public void setList(String var, Object... lst) {
		setDotList(var, Nil.NIL, lst);
	}
	
	
	public void setJavaInstance(String var, Object o) {
		global.bindDatum(Symbol.getSymbol(var), new JavaInstance(o));
	}
	
	
	public Datum get(String var) {
		return global.findDatum(Symbol.getSymbol(var));
	}
	
	
	public String getString(String var) {
		Datum d = get(var);
		
		if(d instanceof LispString) {
			return ((LispString)d).getString();
		} else {
			throw new ClassCastException();
		}
	}
	
	
	public BigInteger getExactInteger(String var) {
		Datum d = get(var);
		
		if(d instanceof LispInteger) {
			return d.getBigInteger();
		} else {
			throw new ClassCastException();
		}
	}
	
	
	public int getExactInt(String var) {
		return getExactInteger(var).intValue();
	}
	
	
	public long getExactLong(String var) {
		return getExactInteger(var).longValue();
	}
	
	
	public LispNumber getNumber(String var) {
		Datum d = get(var);
		
		if(d instanceof LispNumber) {
			return (LispNumber)d;
		} else {
			throw new ClassCastException();
		}
	}
	
	
	public LispReal getReal(String var) {
		LispNumber n = getNumber(var);
		
		if(n.isReal()) {
			return n.getReal();
		} else {
			throw new ClassCastException();
		}
	}
	
	
	public double getDouble(String var) {
		return getReal(var).getRealDouble();
	}
	
	
	public int getInt(String var) {
		return getReal(var).getBigInteger().intValue();
	}
	
	
	public long getLong(String var) {
		return getReal(var).getBigInteger().longValue();
	}
	
	
	public Object getJavaInstance(String var) {
		Datum d = get(var);
		
		if(d instanceof JavaInstance) {
			return ((JavaInstance)d).getJavaInstance();
		} else {
			throw new ClassCastException();
		}
	}
	
	
	public Datum call(String var, Object... lst) {
		CompiledCode.Builder cd = new CompiledCode.Builder();
		Datum fn = get(var);
		
		cd.addPush(fn);
		cd.addPush(LispUtils.toConsList(lst));
		cd.addCall();
		cd.addReturnOp();
		return exec.exec(cd.getCodeRef(), global, memento);
	}
	
	
	public String callString(String var, Object... lst) {
		Datum d = call(var, lst);
		
		if(d instanceof LispString) {
			return ((LispString)d).getString();
		} else {
			throw new ClassCastException();
		}
	}
	
	
	public BigInteger callExactInteger(String var, Object... lst) {
		Datum d = call(var, lst);
		
		if(d instanceof LispInteger) {
			return d.getBigInteger();
		} else {
			throw new ClassCastException();
		}
	}
	
	
	public int callExactInt(String var, Object... lst) {
		return callExactInteger(var, lst).intValue();
	}
	
	
	public long callExactLong(String var, Object... lst) {
		return callExactInteger(var, lst).longValue();
	}
	
	
	public LispNumber callNumber(String var, Object... lst) {
		Datum d = call(var, lst);
		
		if(d instanceof LispNumber) {
			return (LispNumber)d;
		} else {
			throw new ClassCastException();
		}
	}
	
	
	public LispReal callReal(String var, Object... lst) {
		LispNumber n = callNumber(var, lst);
		
		if(n.isReal()) {
			return n.getReal();
		} else {
			throw new ClassCastException();
		}
	}
	
	
	public double callDouble(String var, Object... lst) {
		return callReal(var, lst).getRealDouble();
	}
	
	
	public int callInt(String var, Object... lst) {
		return callReal(var, lst).getBigInteger().intValue();
	}
	
	
	public long callLong(String var, Object... lst) {
		return callReal(var, lst).getBigInteger().longValue();
	}
	
}
