/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp;

import java.lang.reflect.Array;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

public final class LispUtils {
	
	private LispUtils() {
		// do nothing
	}
	
	
	private static void printBodyAux1(
			Cons c, StringBuilder buf, boolean met, String disp) {
		if(c.getCdr() instanceof Cons) {
			Cons cd = (Cons)c.getCdr();
			
			if(cd.getCdr() == Nil.NIL) {
				buf.append(disp);
				printBody(cd.getCar(), buf, met);
			}
		}
	}
	
	private static void printBody(
			Datum d, StringBuilder buf, boolean met) {
		if(d instanceof Atom) {
			Atom a = (Atom)d;
			
			buf.append((met ? a.getResult() : a.print()));
		} else if(d == Nil.NIL) {
			buf.append("()");
		} else if(d == Undef.UNDEF) {
			buf.append("#<undef>");
		} else if(d instanceof Cons) {
			Cons c = (Cons)d;
			
			// quote줿ʸΤȤ
			if(Symbol.QUOTE.equals(c.getCar())) {
				printBodyAux1(c, buf, met, "'");
			} else if(Symbol.QUASIQUOTE.equals(c.getCar())) {
				printBodyAux1(c, buf, met, "`");
			} else if(Symbol.UNQUOTE.equals(c.getCar())) {
				printBodyAux1(c, buf, met, ",");
			} else if(Symbol.UNQUOTE_SPLICING.equals(c.getCar())) {
				printBodyAux1(c, buf, met, ",@");
			} else {
				Set<Datum> jn = new HashSet<Datum>();
				
				buf.append("(");
				while(true) {
					// car
					printBody(c.getCar(), buf, met);
					
					// cdr
					if(c.getCdr() instanceof Cons) {
						if(jn.contains(c)) {
							// circular list
							buf.append(" ...");
							break;
						}
						jn.add(c);
						
						buf.append(" ");
						c = (Cons)c.getCdr();
					} else if(c.getCdr() == Nil.NIL) {
						break;
					} else {
						buf.append(" . ");
						printBody(c.getCdr(), buf, met);
						break;
					}
				}
				buf.append(")");
			}
		} else if(d instanceof LispVector) {
			LispVector v = (LispVector)d;
			
			buf.append("#(");
			for(int i = 0; i < v.size(); i++) {
				if(i > 0) {
					buf.append(" ");
				}
				printBody(v.get(i), buf, met);
			}
			buf.append(")");
		} else if(d instanceof ClosureClass) {
			buf.append("#<closureClass" + d + ">");
		} else if(d instanceof Closure) {
			buf.append("#<closure " + ((Closure)d).printName() + ">");
		} else if(d instanceof Continuation) {
			buf.append("#<continuation " +
					Integer.toString(d.hashCode(), 16) + ">");
		} else if(d instanceof Subr) {
			buf.append("#<subr " + ((Subr)d).getSymbolName() + ">");
		} else if(d instanceof Syntax) {
			buf.append("#<syntax " + ((Syntax)d).getSymbolName() + ">");
		} else if(d instanceof UserSyntax) {
			buf.append("#<syntax " + ((UserSyntax)d).getName() + ">");
		} else if(d instanceof Macro) {
			buf.append("#<macro>");
		} else if(d instanceof Promise) {
			buf.append("#<promise " + d + ">");
		} else if(d == EOFObject.EOF) {
			buf.append("#<eof>");
		} else if(d instanceof InputPort) {
			buf.append("#<iport>");
		} else if(d instanceof OutputPort) {
			buf.append("#<oport>");
		} else if(d instanceof MultiValues) {
			List<Datum> lst = ((MultiValues)d).getValues();
			
			for(int i = 0; i < lst.size(); i++) {
				if(i > 0) {
					buf.append("\n");
				}
				//buf.append(lst.get(i));
				printBody(lst.get(i), buf, met);
			}
		} else if(d instanceof EnvironmentObject) {
			buf.append("#<environment>");
		} else if(d instanceof JavaClass) {
			buf.append("#<java-class>");
		} else if(d instanceof JavaInstance) {
			Object o = ((JavaInstance)d).getJavaInstance();
			
			buf.append("#<java-instance ");
			buf.append(o.getClass().getName());
			buf.append(">");
		} else if(d == JavaNull.JAVA_NULL) {
			buf.append("#<java-null>");
		} else if(d instanceof SymbolScope) {
			// ѥǡ:ܥƱ˰
			SymbolScope d2 = (SymbolScope)d;
			Symbol a = d2.getSymbol();
			
			buf.append((met ? a.getResult() : a.print()));
			//buf.append("#<symbolscope " +
			//		d2.getSymbol().getName() + ">");
		} else if(d instanceof RegexPattern) {
			buf.append("#<regexp " +
					((RegexPattern)d).getPatternString() + ">");
		} else if(d instanceof RegexPattern.Match) {
			buf.append("#<rxmatch " +
					((RegexPattern.Match)d).toStringRepl() + ">");
		} else if(d instanceof Keyword) {
			buf.append(":" + ((Keyword)d).getName());
		} else if(d instanceof NamableDatum) {
			buf.append(((NamableDatum)d).display());
		} else {
			// ѥǡ
			buf.append(d);
		}
	}
	
	
	public static String print(Datum d) {
		StringBuilder buf = new StringBuilder();
		
		printBody(d, buf, false);
		return buf.toString();
	}
	
	
	public static String getResult(Datum d) {
		StringBuilder buf = new StringBuilder();
		
		printBody(d, buf, true);
		return buf.toString();
	}
	
	
	public static Datum listToCons(List<Datum> d, Datum dot) {
		Cons res = null;
		Cons c2 = null;
		
		for(int i = 0; i < d.size(); i++) {
			if(res == null) {
				res = c2 = new Cons();
				c2.setCar(d.get(i));
			} else {
				Cons c3 = new Cons(d.get(i), Nil.NIL);
				c2.setCdr(c3);
				c2 = c3;
			}
		}
		
		if(res == null) {
			return dot;
		} else {
			c2.setCdr(dot);
			return res;
		}
	}
	
	
	public static Datum listToCons(List<Datum> d) {
		return listToCons(d, Nil.NIL);
	}
	
	
	public static List<Datum> consToList(Datum d, LispMessage mesg) {
		List<Datum> res = new ArrayList<Datum>();
		Datum dd = d;
		
		while(dd != Nil.NIL) {
			if(dd instanceof Cons) {
				res.add(((Cons)dd).getCar());
				dd = ((Cons)dd).getCdr();
			} else {
				//throw new NotProperException("proper list required");
				throw mesg.getError("err.list");
			}
		}
		return res;
	}
	
	
	public static List<Datum> consToListIgnoreDot(Datum d) {
		List<Datum> res = new ArrayList<Datum>();
		Datum dd = d;
		
		while(dd != Nil.NIL) {
			if(dd instanceof Cons) {
				res.add(((Cons)dd).getCar());
				dd = ((Cons)dd).getCdr();
			} else {
				break;
			}
		}
		return res;
	}
	
	
	public static int consLength(Datum d) {
		Datum p = d;
		int len = 0;
		
		while(p != Nil.NIL) {
			if(p instanceof Cons) {
				Cons c = (Cons)p;
				
				len++;
				p = c.getCdr();
			} else {
				return len;
			}
		}
		return len;
	}
	
	
	public static List<Datum> toList(Object[] arr) {
		List<Datum> lst = new ArrayList<Datum>();
		
		for(Object o : arr) {
			lst.add(toDatum(o));
		}
		return lst;
	}
	
	
	public static Datum toConsList(Object[] arr, Object cdr) {
		ConsListBuilder lst = new ConsListBuilder();
		
		for(Object o : arr) {
			lst.append(toDatum(o));
		}
		return lst.get(toDatum(cdr));
	}
	
	
	public static Datum toConsList(Object[] arr) {
		return toConsList(arr, Nil.NIL);
	}
	
	
	public static LispReal bigDecimalToRational(BigDecimal v) {
		BigInteger n = v.unscaledValue();
		
		if(v.scale() > 0) {
			BigInteger d = BigInteger.TEN.pow(v.scale());
			
			return LispRational.newRational(n, d);
		} else if(v.scale() < 0) {
			BigInteger d = BigInteger.TEN.pow(-v.scale());
			
			return LispInteger.valueOf(n.multiply(d));
		} else {
			return LispInteger.valueOf(n);
		}
	}
	
	
	public static Datum toDatum(Object o) {
		Class<?> cl = (o == null) ? null : o.getClass();
		
		if(o == null) {
			return JavaNull.JAVA_NULL;
		} if(o instanceof Datum) {
			return (Datum)o;
		} else if(o instanceof Integer) {
			return LispInteger.valueOf(((Integer)o).intValue());
		} else if(o instanceof Long) {
			return LispInteger.valueOf(((Long)o).longValue());
		} else if(o instanceof BigInteger) {
			return LispInteger.valueOf((BigInteger)o);
		} else if(o instanceof BigDecimal) {
			return bigDecimalToRational((BigDecimal)o);
		} else if(o instanceof Float) {
			return new LispDouble(((Float)o).doubleValue());
		} else if(o instanceof Double) {
			return new LispDouble(((Double)o).doubleValue());
		} else if(o instanceof String) {
			return new LispString((String)o);
		} else if(o instanceof Character) {
			return new LispCharacter(((Character)o).charValue());
		} else if(o instanceof Boolean) {
			return LispBoolean.getInstance(((Boolean)o).booleanValue());
		} else if(cl.isArray()) {
			ConsListBuilder bld = new ConsListBuilder();
			int len = Array.getLength(o);
			
			for(int i = 0; i < len; i++) {
				bld.append(toDatum(Array.get(o, i)));
			}
			return bld.get();
		} else {
			return new JavaInstance(o);
			//throw new ClassCastException();
		}
	}
	
	
	public static boolean eqv(Datum d1, Datum d2) {
		return !LispBoolean.FALSE.equals(d1.isEqv(d2));
	}
	
	
	public static boolean equals(Datum d, String o) {
		if(d instanceof LispString) {
			return ((LispString)d).getString().equals((String)o);
		} else {
			return false;
		}
	}
	
	
	public static boolean eqvExact(Datum d, BigInteger v) {
		if(d instanceof LispInteger) {
			return d.getBigInteger().equals(v);
		} else {
			return false;
		}
	}
	
	
	public static boolean eqvExact(Datum d, int v) {
		return eqvExact(d, BigInteger.valueOf(v));
	}
	
	
	public static boolean eqvExact(Datum d, long v) {
		return eqvExact(d, BigInteger.valueOf(v));
	}
	
	
	public static boolean eqvInexact(Datum d, double v) {
		if(d instanceof LispReal) {
			return ((LispReal)d).getRealDouble() == v;
		} else {
			return false;
		}
	}
	
	
	public static boolean eqvInexact(Datum d, Number v) {
		return eqvInexact(d, v.doubleValue());
	}
	
	
	public static boolean eqvInexact(Datum d, int v) {
		return eqvInexact(d, (double)v);
	}
	
	
	public static boolean eqvInexact(Datum d, long v) {
		return eqvInexact(d, (double)v);
	}
	
	
	public static Datum listDot(Object d, Object... lst) {
		ConsListBuilder b = new ConsListBuilder();
		
		for(Object o : lst) {
			b.append(toDatum(o));
		}
		return b.get(toDatum(d));
	}
	
	
	public static Datum list(Object... lst) {
		return listDot(Nil.NIL, lst);
	}
	
	
	public static Cons cons(Object car, Object cdr) {
		return new Cons(toDatum(car), toDatum(cdr));
	}
	
	
	public static LispVector vector(Object... lst) {
		List<Datum> v = new ArrayList<Datum>();
		
		for(Object o : lst) {
			v.add(toDatum(o));
		}
		return new LispVector(v);
	}
	
	
	public static boolean equalsVector(LispVector v1, LispVector v2) {
		if(v1.size() != v2.size()) {
			return false;
		} else {
			for(int i = 0; i < v1.size(); i++) {
				if(!equals(v1.get(i), v2.get(i))) {
					return false;
				}
			}
			return true;
		}
	}
	
	
	public static boolean equals(Datum d1, Datum d2) {
		Datum p = d1;
		Datum q = d2;
		
		while(true) {
			if(p instanceof Cons) {
				Cons pc = (Cons)p;
				
				if(q instanceof Cons) {
					Cons qc = (Cons)q;
					
					if(equals(pc.getCar(), qc.getCar())) {
						p = pc.getCdr();
						q = qc.getCdr();
					} else {
						return false;
					}
				} else {
					return false;
				}
			} else if(p instanceof LispString) {
				if(q instanceof LispString) {
					String s1 = ((LispString)p).getString();
					String s2 = ((LispString)q).getString();
					
					return s1.equals(s2);
				} else {
					return false;
				}
			} else if(p instanceof LispVector) {
				if(q instanceof LispVector) {
					return equalsVector((LispVector)p, (LispVector)q);
				} else {
					return false;
				}
			} else {
				return p.isEqv(q); 
			}
		}
	}
	
	
	public static Map<Datum, Datum> assocToMap(Datum d) {
		Map<Datum, Datum> res = new HashMap<Datum, Datum>();
		ConsIterator p = new ConsIterator(d);
		
		while(p.hasNext()) {
			Datum c = p.next();
			
			if(c instanceof Cons) {
				Cons c0 = (Cons)c;
				
				res.put(c0.getCar(), c0.getCdr());
			} else {
				return null;
			}
		}
		
		return (p.getTerminal() == Nil.NIL) ? res : null;
	}
	
	
	public static Map<Symbol, Datum> assocToMapSymbol(Datum d) {
		Map<Symbol, Datum> res = new HashMap<Symbol, Datum>();
		ConsIterator p = new ConsIterator(d);
		
		while(p.hasNext()) {
			Datum c = p.next();
			
			if(c instanceof Cons) {
				Cons c0 = (Cons)c;
				
				if(c0.getCar() instanceof SymbolName) {
					res.put(((SymbolName)c0.getCar()).getSymbol(),
							c0.getCdr());
				} else {
					return null;
				}
			} else {
				return null;
			}
		}
		
		return (p.getTerminal() == Nil.NIL) ? res : null;
	}
	
}
