/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.range;

import java.util.Set;
import java.util.SortedSet;

import net.morilib.util.Objects;

/**
 * This is an abstract class which implements &quot;range&quot;,
 * a topology with sum and overlap operation, supremum and infimum.<br>
 * 
 * Range(ϰ: ֤½礫ʤ뽸)ɽ쥯饹Ǥ.
 * 
 * @author MORIGUCHI, Yuichiro 2008/01/01
 */
public abstract class Range {
	
	/**
	 * The empty range.<br>
	 * Ǥ.
	 */
	public static final Interval O = new Interval(
			Objects.MAXIMUM, true, Objects.MINIMUM, true);
	
	/**
	 * The universal range of all objects.<br>
	 * ֥ȤޤླྀǤ.
	 */
	public static final Interval U = new Interval(
			Objects.MINIMUM, true, Objects.MAXIMUM, true);

	/**
	 * creates a new range with the specified objects.<br>
	 * Ϳ줿ޤϰϤ.
	 * 
	 * @param s  a collection of points
	 * @return  a new range with the objects
	 */
	public static Range newPoints(Set<?> s) {
		if(s.isEmpty()) {
			return O;
		} else {
			RangeAdder a = new RangeAdder();
			
			for(Object t : s) {
				a.add(Interval.newPoint(t));
			}
			return a.toRange();
		}
	}

	/**
	 * creates a new range with the specified objects.<br>
	 * Ϳ줿ޤϰϤ.
	 * 
	 * @param v1  a point
	 * @param v2  another point
	 * @return  a new range with the objects
	 */
	public static Range
	newPointPair(Object v1, Object v2) {
		return Ranges.sum(
				Interval.newPoint(v1), Interval.newPoint(v2));
	}

	/**
	 * Returns the boundary set of this topological set.<br>
	 * ϰϤΤ٤Ƥζޤླྀ.
	 */
	public abstract Range bound();
	
	/**
	 * returns true if this range is a open set.<br>
	 * ϰϤǤtrue.
	 * 
	 * @return  true if this range is open
	 */
	public abstract boolean isOpen();
	
	/**
	 * returns true if this range is a closed set.<br>
	 * ϰϤĽǤtrue.
	 * 
	 * @return  true if this range is closed
	 */
	public abstract boolean isClosed();
	
	/**
	 * returns the bounds of this range by SortedSet.<br>
	 * ϰϤζSortedSet. 
	 * 
	 * @return  the SortedSet which includes all bounds of this range
	 */
	public abstract SortedSet<?> boundElements();

	/**
	 * returns true if the specified object is below the infimum
	 * bound of this range.<br>
	 * ꤵ줿֥ȤϰϤβ¤ζ꾮Ȥ
	 * true.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object below the infimum
	 */
	/*package*/ abstract boolean isInfimumBoundAbove(Object o);

	/**
	 * returns true if the specified object is above the infimum
	 * bound of this range.<br>
	 * ꤵ줿֥ȤϰϤβ¤ζ礭Ȥ
	 * true.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object above the infimum
	 */
	/*package*/ abstract boolean isInfimumBoundBelow(Object o);

	/**
	 * returns true if the specified object is equal to the infimum
	 * bound of this range.<br>
	 * ꤵ줿֥ȤϰϤβ¤ζȤ
	 * true.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object equals the infimum
	 */
	/*package*/ abstract boolean isInfimumBoundEqualTo(Object o);

	/**
	 * returns true if the specified object is below the supremum
	 * bound of this range.<br>
	 * ꤵ줿֥ȤϰϤξ¤ζ꾮Ȥ
	 * true.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object below the supremum
	 */
	/*package*/ abstract boolean isSupremumBoundAbove(Object o);

	/**
	 * returns true if the specified object is above the supremum
	 * bound of this range.<br>
	 * ꤵ줿֥ȤϰϤξ¤ζ礭Ȥ
	 * true.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object above the supremum
	 */
	/*package*/ abstract boolean isSupremumBoundBelow(Object o);

	/**
	 * returns true if the specified object is equal to the infimum
	 * bound of this range.<br>
	 * ꤵ줿֥ȤϰϤξ¤ζȤ
	 * true.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object equals the supremum
	 */
	/*package*/ abstract boolean isSupremumBoundEqualTo(Object o);
	
	/**
	 * returns true if any points of the specified range is below
	 * the infimum bound of this range.<br>
	 * ꤵ줿ϰϤΤϰϤβ¤ζ꾮Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is any points of the range below the infimum
	 */
	/*package*/ abstract boolean isInfimumBoundAboveAnyClosureOf(
			Range r);
	
	/**
	 * returns true if any points of the specified range is above
	 * the infimum bound of this range.<br>
	 * ꤵ줿ϰϤΤϰϤβ¤ζ礭Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is any points of the range above the infimum
	 */
	/*package*/ abstract boolean isInfimumBoundBelowAnyClosureOf(
			Range r);

	/**
	 * returns true if any points of the specified range is below
	 * the supremum bound of this range.<br>
	 * ꤵ줿ϰϤΤϰϤξ¤ζ꾮Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is any points of the range below the supremum
	 */
	/*package*/ abstract boolean isSupremumBoundAboveAnyClosureOf(
			Range r);

	/**
	 * returns true if any points of the specified range is above
	 * the supremum bound of this range.<br>
	 * ꤵ줿ϰϤΤϰϤξ¤ζ礭Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is any points of the range above the supremum
	 */
	/*package*/ abstract boolean isSupremumBoundBelowAnyClosureOf(
			Range r);
	
	/**
	 * returns true if all points of the specified range is below
	 * the infimum bound of this range.<br>
	 * ꤵ줿ϰϤƤϰϤβ¤ζ꾮Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is all points of the range below the infimum
	 */
	/*package*/ abstract boolean isInfimumBoundAboveAllClosureOf(
			Range r);
	
	/**
	 * returns true if all points of the specified range is above
	 * the infimum bound of this range.<br>
	 * ꤵ줿ϰϤƤϰϤβ¤ζ礭Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is all points of the range above the infimum
	 */
	/*package*/ abstract boolean isInfimumBoundBelowAllClosureOf(
			Range r);

	/**
	 * returns true if all points of the specified range is below
	 * the supremum bound of this range.<br>
	 * ꤵ줿ϰϤƤϰϤξ¤ζ꾮Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is all points of the range below the supremum
	 */
	/*package*/ abstract boolean isSupremumBoundAboveAllClosureOf(
			Range r);

	/**
	 * returns true if all points of the specified range is above
	 * the supremum bound of this range.<br>
	 * ꤵ줿ϰϤƤϰϤξ¤ζ礭Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is all points of the range above the supremum
	 */
	/*package*/ abstract boolean isSupremumBoundBelowAllClosureOf(
			Range r);
	
	/**
	 * returns true if the infimum bound of this range contacts
	 * the supremum bound of the specified range.<br>
	 * ϰϤβ¤ζͿ줿ϰϤξ¤ζȰפ
	 * (ܿ)Ȥtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true if the infimum contacts the supremum the range
	 */
	/*package*/ abstract boolean contactInfimumBound(Range r);

	/**
	 * returns true if the supremum bound of this range contacts
	 * the infimum bound of the specified range.<br>
	 * ϰϤξ¤ζͿ줿ϰϤβ¤ζȰפ
	 * (ܿ)Ȥtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true if the supremum contacts the infimum the range
	 */
	/*package*/ abstract boolean contactSupremumBound(Range r);
	
	/**
	 * returns true if the infimum bound of this range is the same
	 * as one of the specified range.<br>
	 * ϰϤβ¤ζͿ줿ϰϤβ¤ζȰפ
	 * (̤Ǥ)Ȥtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true if the infimum bound is the same as one
	 */
	/*package*/ abstract boolean commonInfimumBoundTo(Range r);

	/**
	 * returns true if the supremum bound of this range is the same
	 * as one of the specified range.<br>
	 * ϰϤξ¤ζͿ줿ϰϤξ¤ζȰפ
	 * (̤Ǥ)Ȥtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true if the supremum bound is the same as one
	 */
	/*package*/ abstract boolean commonSupremumBoundTo(Range r);
	
	/**
	 * returns true if the specified object is below the infimum
	 * of this range.<br>
	 * <p>ꤵ줿֥ȤϰϤβ¤꾮Ȥtrue.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object below the infimum
	 */
	/*package*/ abstract boolean isInfimumAbove(Object o);
	
	/**
	 * returns true if the specified object is above the infimum
	 * of this range.<br>
	 * <p>ꤵ줿֥ȤϰϤβ¤礭Ȥtrue.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object above the infimum
	 */
	/*package*/ abstract boolean isInfimumBelow(Object o);
	
	/**
	 * returns true if the specified object is equal to the infimum
	 * of this range.<br>
	 * <p>ꤵ줿֥ȤϰϤβ¤Ȥtrue.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object equals the infimum
	 */
	/*package*/ abstract boolean isInfimumEqualTo(Object o);
	
	/**
	 * returns true if the specified object is below the supremum
	 * of this range.<br>
	 * <p>ꤵ줿֥ȤϰϤξ¤꾮Ȥtrue.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object below the supremum
	 */
	/*package*/ abstract boolean isSupremumAbove(Object o);
	
	/**
	 * returns true if the specified object is above the supremum
	 * of this range.<br>
	 * <p>ꤵ줿֥ȤϰϤξ¤礭Ȥtrue.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object above the supremum
	 */
	/*package*/ abstract boolean isSupremumBelow(Object o);
	
	/**
	 * returns true if the specified object is equal to the infimum
	 * of this range.<br>
	 * <p>ꤵ줿֥ȤϰϤξ¤Ȥtrue.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object equals the supremum
	 */
	/*package*/ abstract boolean isSupremumEqualTo(Object o);
	
	/**
	 * returns true if the infimum bound of this range is contained
	 * in the specified range.<br>
	 * Ϳ줿ϰϤϰϤβ¤ζޤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true if the infimum bound is contained in the range 
	 */
	public abstract boolean infimumBoundIn(Range r);
	
	/**
	 * returns true if the supremum bound of this range is contained
	 * in the specified range.<br>
	 * Ϳ줿ϰϤϰϤξ¤ζޤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true if the supremum bound is contained in the range 
	 */
	public abstract boolean supremumBoundIn(Range r);
	
	/**
	 * Returns the intervals which consists this range.<br>
	 * ϰϤƤζ֤.
	 * 
	 * @return  the intervals which consists this range
	 */
	public abstract SortedSet<Interval> intervals();
	
	/**
	 * returns the closure of this range.<br>
	 * ϰϤ֤.
	 * 
	 * @return  the closure
	 */
	public abstract Range closure();
	
	/**
	 * returns the interior of this range.<br>
	 * ϰϤ֤.
	 * 
	 * @return  the interior
	 */
	public abstract Range interior();
	
	/**
	 * returns true if the smallest edge of this range is open.
	 * <p>ϰϤξ¦üƤȤtrue.<br />
	 * Υ᥽åɤtrueΤȤ{@link #isInfimumClosed()}
	 * falseǤȤϸ¤ʤ(㤨Сü̵ΤȤ).
	 * 
	 * @return  true if the smallest edge of this range is open
	 */
	public abstract boolean isInfimumOpen();
	
	/**
	 * returns true if the smallest edge of this range is closed.
	 * <p>ϰϤξ¦üĤƤȤtrue.<br />
	 * Υ᥽åɤtrueΤȤ{@link #isInfimumOpen()}
	 * falseǤȤϸ¤ʤ(㤨Сü̵ΤȤ).
	 * 
	 * @return  true if the smallest edge of this range is closed
	 */
	public abstract boolean isInfimumClosed();
	
	/**
	 * return true if the smallest edge of this range is finite.
	 * <p>ϰϤξ¦üͭ¤ΤȤtrue.
	 * 
	 * @return  true if the smallest edge of this range is finite
	 */
	public abstract boolean isInfimumFinite();
	
	/**
	 * returns true if the largest edge of this range is open.
	 * <p>ϰϤ礭¦üƤȤtrue.<br />
	 * Υ᥽åɤtrueΤȤ{@link #isSupremumClosed()}
	 * falseǤȤϸ¤ʤ(㤨Сü̵ΤȤ).
	 * 
	 * @return  true if the largest edge of this range is open
	 */
	public abstract boolean isSupremumOpen();
	
	/**
	 * returns true if the largest edge of this range is closed.
	 * <p>ϰϤ礭¦üĤƤȤtrue.<br />
	 * Υ᥽åɤtrueΤȤ{@link #isSupremumOpen()}
	 * falseǤȤϸ¤ʤ(㤨Сü̵ΤȤ).
	 * 
	 * @return  true if the largest edge of this range is closed
	 */
	public abstract boolean isSupremumClosed();
	
	/**
	 * return true if the largest edge of this range is finite.
	 * <p>ϰϤ礭¦üͭ¤ΤȤtrue.
	 * 
	 * @return  true if the largest edge of this range is finite
	 */
	public abstract boolean isSupremumFinite();
	
	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	public abstract int hashCode();

	/**
	 * returns true if all points of the specified range is below
	 * the infimum of this range.<br>
	 * ꤵ줿ϰϤƤϰϤβ¤꾮Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is all points of the range below the infimum
	 */
	/*package*/ boolean isInfimumAboveAllOf(Range r) {
		return (isInfimumBoundAboveAllClosureOf(r) ||
				(contactInfimumBound(r) &&
						(r.isSupremumOpen() || isInfimumOpen())));
	}

	/**
	 * returns true if all points of the specified range is above
	 * the infimum of this range.<br>
	 * ꤵ줿ϰϤƤϰϤβ¤礭Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is all points of the range above the infimum
	 */
	/*package*/ boolean isInfimumBelowAllOf(Range r) {
		return (isInfimumBoundBelowAllClosureOf(r) ||
				(commonInfimumBoundTo(r) && r.isInfimumOpen()));
	}

	/**
	 * returns true if any points of the specified range is below
	 * the infimum of this range.<br>
	 * ꤵ줿ϰϤΤϰϤβ¤꾮Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is any points of the range below the infimum
	 */
	/*package*/ boolean isInfimumAboveAnyOf(Range r) {
		return (isInfimumBoundAboveAnyClosureOf(r) ||
				(commonInfimumBoundTo(r) &&
						(r.isInfimumClosed() && isInfimumOpen())));
	}

	/**
	 * returns true if any points of the specified range is above
	 * the infimum of this range.<br>
	 * ꤵ줿ϰϤΤϰϤβ¤礭Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is any points of the range above the infimum
	 */
	/*package*/ boolean isInfimumBelowAnyOf(Range r) {
		return isInfimumBoundBelowAnyClosureOf(r);
	}

	/**
	 * returns true if the infimum of this range contacts
	 * the supremum of the specified range.<br>
	 * ϰϤβ¤Ϳ줿ϰϤξ¤Ȱפ(ܿ)Ȥtrue
	 * .
	 * 
	 * @param r  the range to be tested
	 * @return  true if the infimum contacts the supremum the range
	 */
	/*package*/ boolean contactInfimum(Range r) {
		return (contactInfimumBound(r) &&
				(r.isSupremumClosed() && isInfimumClosed()));
	}

	/**
	 * returns true if the infimum of this range is the same
	 * as one of the specified range.<br>
	 * ϰϤβ¤Ϳ줿ϰϤβ¤Ȱפ(̤Ǥ)Ȥ
	 * true.
	 * 
	 * @param r  the range to be tested
	 * @return  true if the infimum bound is the same as one
	 */
	/*package*/ boolean commonInfimumTo(Range r) {
		return (commonInfimumBoundTo(r) &&
				(r.isInfimumClosed() && isInfimumClosed()));
	}

	/**
	 * returns true if all points of the specified range is below
	 * the supremum of this range.<br>
	 * ꤵ줿ϰϤƤϰϤξ¤꾮Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is all points of the range below the supremum
	 */
	/*package*/ boolean isSupremumAboveAllOf(Range r) {
		return (isSupremumBoundAboveAllClosureOf(r) ||
				(commonSupremumBoundTo(r) && r.isSupremumOpen()));
	}

	/**
	 * returns true if all points of the specified range is above
	 * the supremum of this range.<br>
	 * ꤵ줿ϰϤƤϰϤξ¤礭Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is all points of the range above the supremum
	 */
	/*package*/ boolean isSupremumBelowAllOf(Range r) {
		return (isSupremumBoundBelowAllClosureOf(r) ||
				(contactSupremumBound(r) &&
						(r.isInfimumOpen() || isSupremumOpen())));
	}

	/**
	 * returns true if any points of the specified range is below
	 * the infimum of this range.<br>
	 * ꤵ줿ϰϤΤϰϤβ¤꾮Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is any points of the range below the infimum
	 */
	/*package*/ boolean isSupremumAboveAnyOf(Range r) {
		return isSupremumBoundAboveAnyClosureOf(r);
	}

	/**
	 * returns true if any points of the specified range is above
	 * the supremum bound of this range.<br>
	 * ꤵ줿ϰϤΤϰϤξ¤礭Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is any points of the range above the supremum
	 */
	/*package*/ boolean isSupremumBelowAnyOf(Range r) {
		return (isSupremumBoundBelowAnyClosureOf(r) ||
				(commonSupremumBoundTo(r) &&
						(r.isSupremumClosed() && isSupremumOpen())));
	}

	/**
	 * returns true if the supremum bound of this range contacts
	 * the infimum of the specified range.<br>
	 * ϰϤξ¤Ϳ줿ϰϤβ¤Ȱפ(ܿ)Ȥtrue
	 * .
	 * 
	 * @param r  the range to be tested
	 * @return  true if the supremum contacts the infimum the range
	 */
	/*package*/ boolean contactSupremum(Range r) {
		return (contactSupremumBound(r) &&
				(r.isSupremumClosed() && isInfimumClosed()));
	}

	/**
	 * returns true if the supremum of this range is the same
	 * as one of the specified range.<br>
	 * ϰϤξ¤Ϳ줿ϰϤξ¤Ȱפ
	 * (̤Ǥ)Ȥtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true if the supremum bound is the same as one
	 */
	/*package*/ boolean commonSupremumTo(Range r) {
		return (commonSupremumBoundTo(r) &&
				(r.isSupremumClosed() && isSupremumClosed()));
	}

	/**
	 * gets the relative complementary of this set in the specified
	 * set.<br>
	 * νͿ줿κ.
	 * 
	 * @param r  the universal interval
	 * @return  the complementary interval
	 */
	public Range complement(Range r) {
		if(isEmpty() || r.isEmpty()) {
			return this;
		} else {
			RangeAdder bld  = new RangeAdder();
			
			for(Interval o : r.intervals()) {
				Interval.limit(o, this, bld);
			}
			return bld.toRange();
		}
	}

	/**
	 * returns the interval which includes the smallest edge of this
	 * object.
	 * <p>Υ֥Ȥξ¦üޤ֤.
	 */
	public Range getInfimumRange() {
		Range res = null;
		
		for(Range t : intervals()) {
			if(res == null ||
					res.isInfimumBoundAboveAnyClosureOf(t)) {
				res = t;
			}
		}
		return res;
	}

	/**
	 * returns the interval which includes the largest edge of this
	 * object.
	 * <p>Υ֥Ȥ礭¦üޤ֤.
	 */
	public Range getSupremumRange() {
		Range res = null;
		
		for(Range t : intervals()) {
			if(res == null ||
					res.isSupremumBoundBelowAnyClosureOf(t)) {
				res = t;
			}
		}
		return res;
	}

	/**
	 * gets the join of this and the specified Range.<br>
	 * <p>ϰϤȰϰϤ½(join).
	 * 
	 * @param t  the Range
	 * @return  the join of this and the argument
	 */
	public Range join(Range t) {
		Range r = t;
		
		if(r.isEmpty()) {
			return this;
		} else if(isEmpty()) {
			return r;
		} else {
			MergedRange u = new MergedRange(this);
			
			u.addAll(t.intervals());
			return u;
		}
	}

	/**
	 * gets the meet of this and the specified Range.<br>
	 * <p>ϰϤȰϰϤζʬ(meet).
	 * 
	 * @param t  the Range
	 * @return  the meet of this and the argument
	 */
	public Range meet(Range t) {
		Range r = t;
		
		if(isEmpty() || r.isEmpty()) {
			return O;
		} else {
			RangeAdder bld  = new RangeAdder();
			
			for(Interval o : r.intervals()) {
				Interval.meet(o, this, bld);
			}
			return bld.toRange();
		}
	}

	/**
	 * Returns true if the closure of this set is independent of
	 * the specified set.<br>
	 * νͿ줿ΩƤȤtrue.
	 * 
	 * @param t  the topology to be tested
	 * @return   true if this is independent of the topology t
	 * @see net.morilib.range.Topology#closureIndependentOf(net.morilib.range.Topology)
	 */
	public boolean closureIndependentOf(Range t) {
		return independentOf(t) && boundsIndependentOf(t);
	}

	/**
	 * returns true if the set contains some points of
	 * the specified set.<br>
	 * ν礬Ϳ줿Τɤ줫ޤȤtrue.
	 * 
	 * @param t  the topology to be tested
	 * @return  true if this contains some points of the specified set
	 * @see net.morilib.range.Topology#containsAny(net.morilib.range.Topology)
	 */
	public boolean containsAny(Range t) {
		return !independentOf(t);
	}

	/**
	 * Returns true if this set contains the given object.<br>
	 * ν礬Ϳ줿֥ȤޤȤtrue.
	 * 
	 * @param o  the object to be tested
	 * @return   true if the object is contained in this
	 */
	public abstract boolean contains(Object o);
	
	/**
	 * Returns true if the given object is neighborhood of
	 * this set.<br>
	 * νζ˵ĤޤͿ줿֥Ȥ򤳤νγ
	 * ޤȤtrue.
	 * 
	 * @param o  the object to be tested
	 * @return   true if the object is neighborhood of this
	 */
	public abstract boolean isNeighborhoodOf(Object o);
	
	/**
	 * Returns true if the closure of this set contains the given
	 * object.<br>
	 * νͿ줿֥ȤޤȤtrue.
	 * 
	 * @param o  the object to be tested
	 * @return   true if the object is neighborhood of this
	 */
	public abstract boolean containsClosure(Object o);
	
	/**
	 * Returns true if this set contains the given object.<br>
	 * νζͿ줿֥ȤޤȤtrue.
	 * 
	 * @param o  the object to be tested
	 * @return   true if the object is contained in this
	 */
	public abstract boolean containsBound(Object o);
	
	/**
	 * Returns true if this is empty.<br>
	 * ν礬ΤȤtrue.
	 * 
	 * @return  true if this is empty
	 */
	public abstract boolean isEmpty();
	
	/**
	 * Returns true if this set is in the specified set.<br>
	 * ν礬Ϳ줿ʬΤȤtrue.
	 * 
	 * @param t  the topology to be tested
	 * @return   true if this is in the topology t
	 */
	public abstract boolean in(Range t);
	
	/**
	 * Returns true if this set is independent of
	 * the specified set.<br>
	 * ν礬Ϳ줿ΩƤȤtrue.
	 * 
	 * @param t  the topology to be tested
	 * @return   true if this is independent of the topology t
	 */
	public abstract boolean independentOf(Range t);
	
	/**
	 * Returns true if the interior of this set is independent of
	 * the specified set.<br>
	 * νͿ줿ΩƤȤtrue.
	 * 
	 * @param t  the topology to be tested
	 * @return   true if this is independent of the topology t
	 */
	public abstract boolean interiorIndependentOf(Range t);
	
	/**
	 * Returns true if the closure of this set is in
	 * the specified set.<br>
	 * νͿ줿ʬΤȤtrue.
	 * 
	 * @param t  the topology to be tested
	 * @return   true if this is in the topology t
	 */
	public abstract boolean closureContains(Range t);
	
	/**
	 * Returns true if the interior of this set is in
	 * the specified set.<br>
	 * νͿ줿ʬΤȤtrue.
	 * 
	 * @param t  the topology to be tested
	 * @return   true if this is in the topology t
	 */
	public abstract boolean interiorContains(Range t);
	
	/**
	 * returns true if the set contains all points of
	 * the specified set.<br>
	 * ν礬Ϳ줿ƴޤ(Ϳ줿礬ν
	 * ʬǤ)Ȥtrue.
	 * 
	 * @param t  the topology to be tested
	 * @return  true if this contains all points of the specified set
	 */
	public abstract boolean containsAll(Range t);
	
	/**
	 * returns true if the bound of this set is a subset of the
	 * specified set.<br>
	 * νζͿ줿ʬΤȤtrue.
	 * 
	 * @param t  the topology to be tested
	 * @return  true if the bound of this set is a subset
	 */
	public abstract boolean boundsIn(Range t);
	
	/**
	 * returns true if the bound of this set is independent of
	 * the specified set.<br>
	 * νζͿ줿ΩǤȤtrue.
	 * 
	 * @param t  the topology to be tested
	 * @return  true if the bound of this set is independent
	 */
	public abstract boolean boundsIndependentOf(Range t);

	/**
	 * returns true if this object equals to the given object.
	 * <p>Υ֥ȤͿ줿֥ȤȤtrue.
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(Object o) {
		if(o instanceof Range) {
			//return containsAll((Range)o) && in((Range)o);
			return in((Range)o) && ((Range)o).in(this);
		}
		return false;
	}

}
