/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.range;

/**
 * Constant objects and utilities for character sets
 * implemented by the Range library.
 * <p>Range饤֥ˤ줿ʸåȤ֥Ȥ
 * 桼ƥƥǤ.
 * 
 * @author MORIGUCHI, Yuichiro 2008/01/01
 */
public final class CharSets {
	
	//
	private CharSets() {}
	
	/**
	 * The character set of the numbers by ASCII characters.
	 * <p>ASCIIʸˤʸåȤǤ.
	 */
	public static final Range NUMBERS = newCharInterval('0', '9');
	
	/**
	 * The character set of the numbers by ASCII characters.
	 * <p>ASCIIʸˤʸåȤǤ.
	 */
	public static final Range ASCII_NUMBERS = newCharInterval('0', '9');
	
	/**
	 * The character set of the capital alphabets.
	 * <p>ʸʸåȤǤ.
	 */
	public static final Range CAPITAL_LETTERS =
		newCharInterval('A', 'Z');
	
	/**
	 * The character set of the small alphabets.
	 * <p>ѾʸʸåȤǤ.
	 */
	public static final Range SMALL_LETTERS = newCharInterval('a', 'z');
	
	/**
	 * The character set of the English alphabets.
	 * <p>ʸʸåȤǤ.
	 */
	public static final Range ENGLISH_ALPHABETS =
		Ranges.sum(CAPITAL_LETTERS, SMALL_LETTERS);
	
	/**
	 * The character set of the English alphabets and numbers.
	 * <p>ѿʸåȤǤ.
	 */
	public static final Range ENGLISH_ALPHABETS_NUMBERS =
		Ranges.sum(ENGLISH_ALPHABETS, ASCII_NUMBERS);
	
	/**
	 * The character set of the Japanese hiraganas.
	 * <p>Ҥ餬ʤʸåȤǤ.
	 */
	public static final Range HIRAGANA =
		newCharInterval('\u3041', '\u3094');
	
	/**
	 * The character set of the Japanese katakanas.
	 * <p>ʤʸåȤǤ.
	 */
	public static final Range KATAKANA =
		newCharInterval('\u30a1', '\u30fa');
	
	/**
	 * The character set of the white spaces by ASCII characters.
	 * <p>ASCIIʸˤʸåȤǤ.
	 */
	public static final Range SPACES = Ranges.sum(
			IntervalsInt.newPoint(' '), IntervalsInt.newPoint('\t'));
	
	/**
	 * The character set of the token separators by ASCII characters.
	 * <p>ASCIIʸˤȡڤʸʸåȤǤ.
	 */
	public static final Range TOKEN = Ranges.sum(
			IntervalsInt.newPoint(' '), IntervalsInt.newPoint('\t'));
	
	/**
	 * creates a new character set of the given characters.
	 * <p>Ϳ줿ʸüȤʸåȤ.
	 * 
	 * @param f  the starting character
	 * @param t  the ending character
	 * @return  a new character set
	 */
	public static final Range newCharInterval(char f, char t) {
		return IntervalsInt.newClosedInterval((int)f, (int)t);
	}

}
