/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp;

import java.math.BigDecimal;
import java.math.BigInteger;

import net.morilib.util.Utils;

public final class LispRational extends LispExactReal {
	
	//
	private BigInteger numer;
	private BigInteger denom;
	
	private LispRational(BigInteger num, BigInteger den) {
		numer = num;
		denom = den;
	}
	
	
	public static final LispExactReal newRational(
			BigInteger num, BigInteger den) {
		BigInteger n = num;
		BigInteger d = den;
		int nsig = num.signum();
		int dsig = den.signum();
		
		if(dsig == 0) {
			throw new LispArithmeticException("denominator is zero");
		} else if(nsig == 0) {
			return LispInteger.ZERO;
		} else if(dsig < 0) {
			n = n.negate();
			d = d.negate();
		}
		
		BigInteger gcd = n.gcd(d);
		n = n.divide(gcd);
		d = d.divide(gcd);
		
		if(d.equals(BigInteger.ONE)) {
			return LispInteger.valueOf(n);
		} else {
			return new LispRational(n, d);
		}
	}
	
	
	public static final LispExactReal newRational(int num, int den) {
		int n1   = num;
		int d1   = den;
		
		if(d1 == 0) {
			throw new LispArithmeticException("denominator is zero");
		} else if(n1 == 0) {
			return LispInteger.ZERO;
		} else if(d1 < 0) {
			n1 = -n1;
			d1 = -d1;
		}
		
		int gcd = Utils.gcd(n1, d1);
		n1  = n1 / gcd;
		d1  = d1 / gcd;
		
		if(d1 == 1) {
			return LispInteger.valueOf(n1);
		} else {
			return new LispRational(
					BigInteger.valueOf(n1), BigInteger.valueOf(d1));
		}
	}
	
	
	public BigInteger getNumerator() {
		return numer;
	}
	
	public BigInteger getDenominator() {
		return denom;
	}
	
	@Override
	public LispNumber add(LispNumber x) {
		if(x instanceof LispComplex) {
			LispComplex c = (LispComplex)x;
			
			return LispComplex.newComplex(
					getRealDouble() + c.getRealDouble(),
					c.getImagDouble());
		} else if(x instanceof LispRational) {
			LispRational n = (LispRational)x;
			BigInteger nd = denom.multiply(n.denom);
			BigInteger nn = numer.multiply(n.denom).add(
					n.numer.multiply(denom));
			
			return LispRational.newRational(nn, nd);
		} else if(x instanceof LispInteger) {
			BigInteger n  = x.getBigInteger();
			BigInteger nd = denom;
			BigInteger nn = numer.add(n.multiply(denom));
			
			return LispRational.newRational(nn, nd);
		} else if(x instanceof LispDouble) {
			double d1 = (numer.doubleValue() / denom.doubleValue());
			double d2 = d1 + ((LispDouble)x).doubleValue();
			
			return new LispDouble(d2);
		}
		throw new IllegalArgumentException(x.toString());
	}

	@Override
	public LispNumber div(LispNumber x) {
		if(x instanceof LispComplex) {
			double xr = ((LispComplex)x).getRealDouble();
			double xi = ((LispComplex)x).getImagDouble();
			double n  = getRealDouble();
			
			if(xr == 0.0) {
				return LispComplex.newComplex(0, -n / xi);
			} else {
				return LispComplex.newComplex(
						(n * xr)    / (xr * xr + xi * xi),
						(-(n * xi)) / (xr * xr + xi * xi));
			}
		} else if(x instanceof LispRational) {
			LispRational n = (LispRational)x;
			BigInteger nd = denom.multiply(n.numer);
			BigInteger nn = numer.multiply(n.denom);
			
			return LispRational.newRational(nn, nd);
		} else if(x instanceof LispInteger) {
			BigInteger n  = x.getBigInteger();
			BigInteger nd = n.multiply(denom);
			BigInteger nn = numer;
			
			return LispRational.newRational(nn, nd);
		} else if(x instanceof LispDouble) {
			double d1 = (numer.doubleValue() / denom.doubleValue());
			double d2 = d1 / ((LispDouble)x).doubleValue();
			
			return new LispDouble(d2);
		}
		throw new IllegalArgumentException(x.toString());
	}

	@Override
	public boolean isEqualTo(LispNumber x) {
		if(x instanceof LispComplex) {
			return false;
		} else if(x instanceof LispRational) {
			LispRational n = (LispRational)x;
			
			return (numer.equals(n.numer) && denom.equals(n.denom));
		} else if(x instanceof LispInteger) {
			return false;
		} else if(x instanceof LispDouble) {
			double d1 = (numer.doubleValue() / denom.doubleValue());
			
			return d1 == ((LispDouble)x).doubleValue();
		}
		throw new IllegalArgumentException(x.toString());
	}

	@Override
	public boolean isLessThan(LispReal x) {
		if(x instanceof LispRational) {
			LispRational n = (LispRational)x;
			BigInteger n1 = numer.multiply(n.denom);
			BigInteger n2 = n.numer.multiply(denom);
			
			return n1.compareTo(n2) < 0;
		} else if(x instanceof LispInteger) {
			BigInteger n  = x.getBigInteger();
			BigInteger n2 = n.multiply(denom);
			
			return numer.compareTo(n2) < 0;
		} else if(x instanceof LispDouble) {
			double d1 = (numer.doubleValue() / denom.doubleValue());
			
			return d1 < x.getRealDouble();
		}
		throw new IllegalArgumentException(x.toString());
	}

	@Override
	public boolean isMoreThan(LispReal x) {
		if(x instanceof LispRational) {
			LispRational n = (LispRational)x;
			BigInteger n1 = numer.multiply(n.denom);
			BigInteger n2 = n.numer.multiply(denom);
			
			return n1.compareTo(n2) > 0;
		} else if(x instanceof LispInteger) {
			BigInteger n  = x.getBigInteger();
			BigInteger n2 = n.multiply(denom);
			
			return numer.compareTo(n2) > 0;
		} else if(x instanceof LispDouble) {
			double d1 = (numer.doubleValue() / denom.doubleValue());
			
			return d1 > x.getRealDouble();
		}
		throw new IllegalArgumentException(x.toString());
	}

	@Override
	public LispNumber mul(LispNumber x) {
		if(x instanceof LispComplex) {
			LispComplex c = (LispComplex)x;
			
			if(c.getRealDouble() == 0.0) {
				return LispComplex.newComplex(
						0, getRealDouble() * c.getImagDouble());
			} else {
				return LispComplex.newComplex(
						getRealDouble() * c.getRealDouble(),
						getRealDouble() * c.getImagDouble());
			}
		} else if(x instanceof LispRational) {
			LispRational n = (LispRational)x;
			BigInteger nd = denom.multiply(n.denom);
			BigInteger nn = numer.multiply(n.numer);
			
			return LispRational.newRational(nn, nd);
		} else if(x instanceof LispInteger) {
			BigInteger n  = x.getBigInteger();
			BigInteger nd = denom;
			BigInteger nn = n.multiply(numer);
			
			return LispRational.newRational(nn, nd);
		} else if(x instanceof LispDouble) {
			double d1 = (numer.doubleValue() / denom.doubleValue());
			double d2 = d1 * ((LispDouble)x).doubleValue();
			
			return new LispDouble(d2);
		}
		throw new IllegalArgumentException(x.toString());
	}

	@Override
	public LispNumber sub(LispNumber x) {
		if(x instanceof LispComplex) {
			LispComplex c = (LispComplex)x;
			
			return LispComplex.newComplex(
					getRealDouble() - c.getRealDouble(),
					-c.getImagDouble());
		} else if(x instanceof LispRational) {
			LispRational n = (LispRational)x;
			BigInteger nd = denom.multiply(n.denom);
			BigInteger nn = numer.multiply(n.denom).subtract(
					n.numer.multiply(denom));
			
			return LispRational.newRational(nn, nd);
		} else if(x instanceof LispInteger) {
			BigInteger n  = x.getBigInteger();
			BigInteger nd = denom;
			BigInteger nn = numer.subtract(n.multiply(denom));
			
			return LispRational.newRational(nn, nd);
		} else if(x instanceof LispDouble) {
			double d1 = (numer.doubleValue() / denom.doubleValue());
			double d2 = d1 - ((LispDouble)x).doubleValue();
			
			return new LispDouble(d2);
		}
		throw new IllegalArgumentException(x.toString());
	}

	@Override
	public LispNumber uminus() {
		return new LispRational(numer.negate(), denom);
	}
	
	
	public int signum() {
		return numer.signum();
	}
	
	
	public LispNumber toInexact() {
		return new LispDouble(numer.doubleValue() / denom.doubleValue());
	}
	
	
	@Override
	public String getResult() {
		return numer.toString() + "/" + denom.toString();
	}

	@Override
	public String print() {
		return numer.toString() + "/" + denom.toString();
	}
	
	
	public boolean equals(Object x) {
		if(x instanceof LispRational) {
			LispRational n = (LispRational)x;
			
			return (numer.equals(n.numer) && denom.equals(n.denom));
		}
		return false;
	}
	
	
	public int hashCode() {
		int l = 17;
		
		l = 37 * l + numer.hashCode();
		l = 37 * l + denom.hashCode();
		return l;
	}
	
	
	public String toString() {
		return numer.toString() + "/" + denom.toString();
	}

	@Override
	public boolean isInteger() {
		return false;
	}

	@Override
	public boolean isRational() {
		return true;
	}

	@Override
	public boolean isReal() {
		return true;
	}
	
	
	public boolean isExact() {
		return true;
	}
	
	
	public LispString toLispString(int radix) {
		if(radix < 2 || radix > 36) {
			throw new IndexOutOfBoundsException("radix is out of range");
		}
		
		return new LispString(
				numer.toString(radix) + "/" + denom.toString(radix));
	}
	
	
	public boolean isOne() {
		return numer.equals(denom);
	}


	@Override
	public BigInteger getBigInteger() {
		return numer.divide(denom);
	}


	@Override
	public int getInt() {
		return getBigInteger().intValue();
	}
	
	
	public long getLong() {
		return getBigInteger().longValue();
	}
	
	
	public BigDecimal getBigDecimal() {
		BigDecimal n, d;
		
		n = new BigDecimal(numer);
		d = new BigDecimal(denom);
		return n.divide(d);
	}


	@Override
	public double getRealDouble() {
		return numer.doubleValue() / denom.doubleValue();
	}

}
