/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp;

import java.math.BigDecimal;
import java.math.BigInteger;

public final class LispComplex extends LispNumber {
	
	
	public static final LispComplex I = new LispComplex(0, 1.0);
	
	
	public static final LispComplex MINUS_I = new LispComplex(0, -1.0);
	
	
	//
	private double real;
	private double imag;
	
	private LispComplex(double r, double i) {
		real = r;
		imag = i;
	}
	
	
	public double getRealDouble() {
		return real;
	}
	
	
	public LispReal getReal() {
		return new LispDouble(real);
	}
	
	
	public double getImagDouble() {
		return imag;
	}
	
	
	public LispReal getImag() {
		return new LispDouble(imag);
	}
	
	
	public static LispNumber newComplex(double r, double i) {
		if(i == 0.0) {
			return new LispDouble(r);
		} else if(Double.isNaN(r) || Double.isNaN(i)) {
			return LispDouble.NaN;
		} else {
			return new LispComplex(r, i);
		}
	}

	@Override
	public LispNumber add(LispNumber x) {
		if(x instanceof LispComplex) {
			LispComplex c = (LispComplex)x;
			
			return newComplex(real + c.real, imag + c.imag);
		} else if(x instanceof LispReal) {
			return newComplex(real + x.getRealDouble(), imag);
		}
		throw new IllegalArgumentException(x.toString());
	}

	@Override
	public LispNumber div(LispNumber x) {
		if(x instanceof LispComplex) {
			double xr = ((LispComplex)x).real;
			double xi = ((LispComplex)x).imag;
			
			if(xr == 0.0) {
				return newComplex(imag / xi, -real / xi);
			} else {
				return newComplex(
						(real * xr + imag * xi) / (xr * xr + xi * xi),
						(imag * xr - real * xi) / (xr * xr + xi * xi));
			}
		} else if(x instanceof LispReal) {
			double r = x.getRealDouble();
			
			return newComplex(real / r, imag / r);
		}
		throw new IllegalArgumentException(x.toString());
	}

	@Override
	public boolean isEqualTo(LispNumber x) {
		if(x instanceof LispComplex) {
			LispComplex c = (LispComplex)x;
			
			return (real == c.real) && (imag == c.imag);
		} else if(x instanceof LispReal) {
			return false;
		}
		throw new IllegalArgumentException(x.toString());
	}

	@Override
	public boolean isInteger() {
		return false;
	}

	@Override
	public boolean isRational() {
		return false;
	}

	@Override
	public boolean isReal() {
		return false;
	}

	@Override
	public LispNumber mul(LispNumber x) {
		if(x instanceof LispComplex) {
			double xr = ((LispComplex)x).real;
			double xi = ((LispComplex)x).imag;
			
			if(xr == 0.0) {
				return newComplex(-imag * xi, real * xi);
			} else {
				return newComplex(
						(real * xr - imag * xi),
						(imag * xr + real * xi));
			}
		} else if(x instanceof LispReal) {
			double r = x.getRealDouble();
			
			return newComplex(real * r, imag * r);
		}
		throw new IllegalArgumentException(x.toString());
	}

	@Override
	public LispNumber sub(LispNumber x) {
		if(x instanceof LispComplex) {
			LispComplex c = (LispComplex)x;
			
			return newComplex(real - c.real, imag - c.imag);
		} else if(x instanceof LispReal) {
			return newComplex(real - x.getRealDouble(), imag);
		}
		throw new IllegalArgumentException(x.toString());
	}

	@Override
	public LispNumber uminus() {
		return newComplex(-real, -imag);
	}

	@Override
	public String getResult() {
		if(imag > 0 && imag != Double.POSITIVE_INFINITY) {
			return (LispNumber.disp(real) + "+" +
					LispNumber.disp(imag) + "i");
		} else {
			return (LispNumber.disp(real) +
					LispNumber.disp(imag) + "i");
		}
	}

	@Override
	public String print() {
		return getResult();
	}
	
	
	public boolean isExact() {
		return false;
	}
	
	
	public LispNumber toExact() {
		throw new LispNotSupportedException(
				"err.notsupported.exactcomplex");
	}
	
	
	public LispNumber toInexact() {
		return this;
	}
	
	
	public boolean equals(Object x) {
		if(x instanceof LispComplex) {
			LispComplex c = (LispComplex)x;
			
			return (real == c.real) && (imag == c.imag);
		}
		return false;
	}
	
	
	public int hashCode() {
		int l = 17;
		long lr = Double.doubleToLongBits(real);
		long li = Double.doubleToLongBits(imag);
		
		l = 37 * l + (int)(lr ^ (lr >>> 32));
		l = 37 * l + (int)(li ^ (li >>> 32));
		return l;
	}
	
	
	public String toString() {
		return getResult();
	}
	
	
	public LispString toLispString(int radix) {
		if(radix < 2 || radix > 36) {
			throw new IndexOutOfBoundsException("radix is out of range");
		} else if(radix != 10) {
			throw new IllegalArgumentException(
					"radix except 10 is not supported");
		}
		
		if(imag > 0 && imag != Double.POSITIVE_INFINITY) {
			return new LispString(
					LispNumber.disp(real) + "+" +
					LispNumber.disp(imag) + "i");
		} else {
			return new LispString(
					LispNumber.disp(real) +
					LispNumber.disp(imag) + "i");
		}
	}
	
	
	public boolean isNaN() {
		return Double.isNaN(real) || Double.isNaN(imag);
	}
	
	
	public boolean isOne() {
		return real == 1.0 && imag == 0.0;
	}


	@Override
	public BigInteger getBigInteger() {
		throw new UnsupportedOperationException();
	}


	@Override
	public BigInteger getDenominator() {
		throw new UnsupportedOperationException();
	}


	@Override
	public int getInt() {
		throw new UnsupportedOperationException();
	}


	@Override
	public long getLong() {
		throw new UnsupportedOperationException();
	}
	
	
	public BigDecimal getBigDecimal() {
		throw new UnsupportedOperationException();
	}


	@Override
	public BigInteger getNumerator() {
		throw new UnsupportedOperationException();
	}
	
}
