/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp;

import java.io.IOException;
import java.io.PushbackReader;
import java.io.StringReader;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Stack;

public class Parser {
	
	private static enum State {
		INIT,                           // 
		NUMBER_PREFIX,                  // ͤprefix
		HEAD_SIGN,                      // Ƭ
		NUMBER,                         // 
		FLOAT_NUMBER,                   // ư
		FLOAT_NUMBER_D,                 // ư
		FLOAT_NUMBER_E,                 // ưؿ
		FLOAT_NUMBER_E_SIGN,            // ưؿ
		FLOAT_NUMBER_E_D,               // ưؿ
		RATIONAL_NUMBER,                // ͭ
		RATIONAL_NUMBER_2,              // ͭ
		INF_I,                          // ̵
		INF_N,
		INF_F,
		INF_DOT,
		INF_0,
		COMPLEX_I,                      // 
		INIT_POLAR,                     // (˷)
		HEAD_SIGN_POLAR,                // Ƭ(˷)
		NUMBER_POLAR,                   // (˷)
		FLOAT_NUMBER_POLAR,             // ư(˷)
		FLOAT_NUMBER_D_POLAR,           // ư(˷)
		FLOAT_NUMBER_E_POLAR,           // ưؿ(˷)
		FLOAT_NUMBER_E_SIGN_POLAR,      // ưؿ(˷)
		FLOAT_NUMBER_E_D_POLAR,         // ưؿ(˷)
		RATIONAL_NUMBER_POLAR,          // ͭ(˷)
		RATIONAL_NUMBER_2_POLAR,        // ͭ(˷)
		HEAD_SIGN_COMPLEX,              // Ƭ(ʣǿ)
		NUMBER_COMPLEX,                 // (ʣǿ)
		FLOAT_NUMBER_COMPLEX,           // ư(ʣǿ)
		FLOAT_NUMBER_D_COMPLEX,         // ư(ʣǿ)
		FLOAT_NUMBER_E_COMPLEX,         // ưؿ(ʣǿ)
		FLOAT_NUMBER_E_SIGN_COMPLEX,    // ưؿ(ʣǿ)
		FLOAT_NUMBER_E_D_COMPLEX,       // ưؿ(ʣǿ)
		RATIONAL_NUMBER_COMPLEX,        // ͭ(ʣǿ)
		RATIONAL_NUMBER_2_COMPLEX,      // ͭ(ʣǿ)
		INF_I_COMPLEX,                  // ̵(ʣǿ)
		INF_N_COMPLEX,
		INF_F_COMPLEX,
		INF_DOT_COMPLEX,
		INF_0_COMPLEX,
		COMPLEX_I2,                     // ʣǿñ(i)
		STRING,                         // ʸ
		STRING_ESCAPE,                  // ʸ()
		SHARP_SYNTAX_I,                 // #()
		SHARP_SYNTAX,                   // #
		CHARACTER_EXP,                  // ʸ
		CHARACTER_EXP_NAME,             // ʸ()
		SYMBOL,                         // ܥ
		SYMBOL2,                        // ܥ(ʣǿ)
		LIST,                           // ꥹ(Ƭ)
		LIST2,                          // ꥹ()
		LIST_DOT_INIT,                  // ɥåȥꥹ(INIT)
		LIST_DOT,                       // ɥåȥꥹ
		QUOTE,                          // quote("'", "`", ",", ",@")
		VECTOR,                         // ٥(Ƭ)
		VECTOR2,                        // ٥()
		COMMENT,                        //
		BLOCK_COMMENT,                  //
		S_EXP_COMMENT,                  //
		//END_TOKEN,
	};
	
	private enum Exact {
		DEFAULT,
		EXACT,
		INEXACT,
	}
	
	
	// marker
	private static final Datum _VOID = new Datum() {};
	private static final Symbol _DOT = Symbol.getSymbol(".");
	
	private static final int EOF = -1;
	
	private LispMessage message;
	private PushbackReader rd = null;
	private State state;
	private StringBuilder buf  = new StringBuilder();
	private StringBuilder buf2 = new StringBuilder();  // ʣǿ
	private Stack<Datum>       resultStack = new Stack<Datum>();
	private Stack<State>       stateStack  = new Stack<State>();
	private Stack<Cons>        consStack   = new Stack<Cons>();
	private Stack<List<Datum>> vectorStack = new Stack<List<Datum>>();
	private Stack<Integer>     parenStack  = new Stack<Integer>();
	private boolean prefixed = false;
	private Exact exactness = Exact.DEFAULT;
	private int radix = -1;
	private boolean signed = false;
	private boolean readBefore = false;
	private boolean consoleMode;
	
	
	/*package*/ Parser(LispMessage msg) {
		message = msg;
		consoleMode = true;
		clear();
	}
	
	
	/*package*/ Parser(PushbackReader read, LispMessage msg) {
		message = msg;
		consoleMode = false;
		rd = read;
		clear();
	}
	
	
	public Parser() {
		this(LispMessage.getInstance());
	}
	
	
	public Parser(Locale lc) {
		this(LispMessage.getInstance(lc));
	}
	
	
	public Parser(PushbackReader read) {
		this(read, LispMessage.getInstance());
	}
	
	
	public Parser(PushbackReader read, Locale lc) {
		this(read, LispMessage.getInstance(lc));
	}
	
	
	private void clearBuf() {
		buf  = new StringBuilder();
		buf2 = new StringBuilder();
		prefixed = false;
		exactness = Exact.DEFAULT;
		radix = -1;
		signed = false;
	}
	
	private int read() throws IOException {
		int res = rd.read();
		
		if(res < 0) {
			return EOF;
		} else {
			readBefore = true;
			return res;
		}
	}
	
	private int readNext2() throws IOException {
		int res;
		
		while((res = rd.read()) >= 0) {
			readBefore = true;
			if(!Character.isWhitespace(res)) {
				return res;
			}
		}
		return EOF;
	}
	
	private int readNext() throws IOException {
		clearBuf();
		return readNext2();
	}
	
	private void appendBuf(int c) {
		buf.append((char)c);
	}
	
	private void appendBuf2(int c) {
		buf2.append((char)c);
	}
	
	private String getToken() {
		String res = buf.toString();
		return res;
	}
	
	private String getToken2() {
		String res = buf2.toString();
		return res;
	}
	
	
	private static double parseDouble(String str) {
		if("+inf.0".equals(str)) {
			return Double.POSITIVE_INFINITY;
		} else if("-inf.0".equals(str)) {
			return Double.NEGATIVE_INFINITY;
		} else if("+nan.0".equals(str)) {
			return Double.NaN;
		} else if("-nan.0".equals(str)) {
			return Double.NaN;
		} else {
			return Double.parseDouble(str);
		}
	}
	
	private double getValueByDouble(String str) {
		double i;
		String[] ind = str.split("/");
		
		if(ind.length == 2) {
			double n = parseDouble(ind[0]);
			double d = parseDouble(ind[1]);
			
			i = n / d;
		} else if(ind.length == 1) {
			i = parseDouble(ind[0]);
		} else {
			throw new RuntimeException("internal error");
		}
		return i;
	}
	
	private BigInteger getValueByBigInt(String str, int r) {
		return new BigInteger(str.replaceFirst("^\\+", ""), r);
	}
	
	
	private boolean isExpsym(int r) {
		return (r == 'e' || r == 'E' ||
				r == 's' || r == 'S' ||
				r == 'f' || r == 'F' ||
				r == 'd' || r == 'D' ||
				r == 'l' || r == 'L');
	}
	
	private boolean isDigit09az(int r) {
		return ((r >= '0' && r <= '9') ||
				(radix == 16 && ((r >= 'A' && r <= 'F') ||
						(r >= 'a' && r <= 'f'))));
	}
	
	private boolean isDigit09(int r) {
		return (r >= '0' && r <= '9');
	}
	
	private static String reverseNumFormat(Exact exactness, int radix) {
		String res = "";
		
		if(Exact.EXACT.equals(exactness)) {
			res += "#e";
		} else if(Exact.INEXACT.equals(exactness)) {
			res += "#i";
		}
		
		if(radix == 2) {
			res += "#b";
		} else if(radix == 8) {
			res += "#o";
		} else if(radix == 10) {
			res += "#d";
		} else if(radix == 16) {
			res += "#x";
		}
		return res;
	}
	
	private String reverseNumFormat() {
		return reverseNumFormat(exactness, radix);
	}
	
	private State symbolNum() {
		if(prefixed) {
			//String s =
			//	reverseNumFormat() + getToken() + getToken2();
			
			throw message.getReadError(
					"err.read.format.numeric");
			//throw new ReadException("bad numeric format");
		}
		return State.SYMBOL;
	}
	
	private State symbol2Num() {
		if(prefixed) {
			//String s =
			//	reverseNumFormat() + getToken() + getToken2();
			
			throw message.getReadError(
					"err.read.format.numeric");
			//throw new ReadException("bad numeric format");
		}
		return State.SYMBOL2;
	}
	
	private State popStateSymbolNum() {
		if(prefixed) {
			throw message.getReadError(
					"err.read.format.numeric",
					reverseNumFormat() + getToken() + getToken2());
			//throw new ReadException("bad numeric format");
		}
		resultStack.push(getSymbol());
		return popStateStack();
	}
	
	
	private LispNumber getInteger() {
		String str = getToken();
		BigInteger val;
		LispReal atom;
		int r = (radix < 0) ? 10 : radix;
		Exact ex = exactness;
		
		clearBuf();
		try {
			val = getValueByBigInt(str, r);
			if(Exact.INEXACT.equals(ex)) {
				atom = new LispDouble(val.doubleValue());
			} else {
				atom = LispInteger.valueOf(val);
			}
			return atom;
		} catch(NumberFormatException e) {
			throw message.getReadError(
					"err.read.format.numeric",
					reverseNumFormat(ex, radix) + str);
			//throw new ReadException("number format error");
		}
	}
	
	private LispNumber getRational() {
		BigInteger num, den;
		String str = getToken();
		String[] nd = str.split("/");
		int r = (radix < 0) ? 10 : radix;
		Exact ex = exactness;
		
		clearBuf();
		try {
			num = getValueByBigInt(nd[0], r);
			den = getValueByBigInt(nd[1], r);
			if(Exact.INEXACT.equals(ex)) {
				double numd = num.doubleValue();
				double dend = den.doubleValue();
				
				return new LispDouble(numd / dend);
			} else {
				return LispRational.newRational(num, den);
			}
		} catch(NumberFormatException e) {
			throw message.getReadError(
					"err.read.format.numeric",
					reverseNumFormat(ex, r) + str);
			//throw new ReadException("number format error");
		}
	}
	
	private LispNumber getDouble() {
		String str = getToken();
		double val;
		LispDouble atom;
		int r = (radix < 0) ? 10 : radix;
		Exact ex = exactness;
		
		clearBuf();
		if(r != 10) {
			throw message.getReadError(
					"err.read.notsupported.radix10",
					reverseNumFormat(ex, r) + str);
			//throw new ReadException("only radix 10 is supported");
		}
		
		val = parseDouble(str);
		atom = new LispDouble(val);
		if(Exact.EXACT.equals(ex)) {
			return atom.toExact();
		} else {
			return atom;
		}
	}
	
	private LispNumber getImaginary() {
		String gtok1 = getToken();
		int r = (radix < 0) ? 10 : radix;
		Exact ex = exactness;
		
		clearBuf();
		if(r > 0 && r != 10) {
			throw message.getReadError(
					"err.read.notsupported.radix10",
					reverseNumFormat(ex, r) + gtok1);
			//throw new ReadException("only radix 10 is supported");
		}
		
		// 
		double i;
		String   iii = gtok1.replaceFirst("i$", "");
		if("+".equals(iii)) {
			i = 1.0;
		} else if("-".equals(iii)) {
			i = -1.0;
		} else {
			i = getValueByDouble(iii);
		}
		
		return LispComplex.newComplex(0.0, i);
	}
	
	private LispNumber getComplex() {
		String gtok1 = getToken();
		String gtok2 = getToken2();
		int rx = (radix < 0) ? 10 : radix;
		Exact ex = exactness;
	
		clearBuf();
		if(rx > 0 && rx != 10) {
			throw message.getReadError(
					"err.read.notsupported.radix10",
					reverseNumFormat(ex, rx) + gtok1 + gtok2);
			//throw new ReadException("only radix 10 is supported");
		}
		
		// 
		double i;
		String   iii = gtok2.replaceFirst("i$", "");
		if("+".equals(iii)) {
			i = 1.0;
		} else if("-".equals(iii)) {
			i = -1.0;
		} else {
			i = getValueByDouble(iii);
		}
		
		if(i != 0.0) {
			// 
			double r = getValueByDouble(gtok1);
			
			return LispComplex.newComplex(r, i);
		} else {
			if(gtok1.indexOf('/') > 0) {
				// ͭ
				String[] nd = gtok1.split("/");
				BigInteger num = new BigInteger(nd[0]);
				BigInteger den = new BigInteger(nd[1]);
				
				clearBuf();
				return LispRational.newRational(num, den);
			} else if(gtok1.matches("^[0-9]+$")) {
				// 
				BigInteger val = new BigInteger(gtok1);
				LispInteger atom = LispInteger.valueOf(val);
				
				clearBuf();
				return atom;
			} else {
				// ư
				double val = parseDouble(gtok1);
				LispDouble atom = new LispDouble(val);
				
				clearBuf();
				return atom;
			}
		}
	}
	
	private LispNumber getPolar() {
		String str = getToken();
		double r;
		int rx = (radix < 0) ? 10 : radix;
		Exact ex = exactness;
		
		clearBuf();
		if(rx > 0 && rx != 10) {
			throw message.getReadError(
					"err.read.notsupported.radix10",
					reverseNumFormat(ex, rx) + str);
			//throw new ReadException("only radix 10 is supported");
		}
		
		String[] ra = str.split("@");
		r = getValueByDouble(ra[0]);    // Ⱦ
		if(r == 0.0) {
			return new LispDouble(0.0);
		} else {
			double a = getValueByDouble(ra[1]);
			
			return LispComplex.newComplex(
					r * Math.cos(a),
					r * Math.sin(a));
		}
	}
	
	private Datum getSharpSyntax() {
		String str = getToken();
		
		clearBuf();
		if("t".equalsIgnoreCase(str)) {
			return LispBoolean.TRUE;
		} else if("f".equalsIgnoreCase(str)) {
			return LispBoolean.FALSE;
		} else {
			throw message.getReadError("err.read.sharp.unknown", str);
			//throw new ReadException("unsupported #-syntax");
		}
	}
	
	private LispCharacter getCharacter() {
		String str = getToken();
		
		clearBuf();
		if(str.length() == 1) {
			return new LispCharacter(str.charAt(0));
		} else {
			LispCharacter r = LispCharacter.getByName(str);
			
			if(r == null) {
				throw message.getReadError(
						"err.read.character.unknown", str);
				//throw new ReadException(
				//		"unknown character name:" + str);
			} else {
				return r;
			}
		}
	}
	
	private LispString getString() {
		String str = getToken();
		
		clearBuf();
		return new LispString(str);		
	}
	
	private Datum getSymbol() {
		String str = getToken() + getToken2();
		
		clearBuf();
		if("+nan.0".equals(str)) {
			return LispDouble.NaN;
		}
		return Symbol.getSymbol(str);
	}
	
	private Symbol getQuote() {
		String str = getToken();
		
		clearBuf();
		if("\'".equals(str)) {
			return Symbol.QUOTE;
		} else if("`".equals(str)) {
			return Symbol.QUASIQUOTE;
		} else if(",".equals(str)) {
			return Symbol.UNQUOTE;
		} else if(",@".equals(str)) {
			return Symbol.UNQUOTE_SPLICING;
		} else {
			throw new RuntimeException("unknown quote:" + str);
		}
	}
	
	private State popStateStack() {
		while(true) {
			State poped;
			
			if(stateStack.empty()) {
				return State.INIT;
			}
			
			poped = stateStack.pop();
			if(State.QUOTE.equals(poped)) {
				Datum c31 = resultStack.pop();
				Cons  c32 = consStack.pop();
				
				if(c31 == _VOID) {
					stateStack.push(poped);
					consStack.push(c32);
					return State.INIT;
				} else {
					c32.setCar(c31);
				}
			} else if(State.S_EXP_COMMENT.equals(poped)) {
				Datum dd = resultStack.pop();
				
				if(_DOT.equals(dd)) {
					throw message.getReadError("err.read.dot");
					//throw new ReadException("bad dot syntax");
				}
				resultStack.push(_VOID);
			} else {
				return poped;
			}
		}
	}
	
	private void checkParenthesis(int r) {
		int p = parenStack.pop();
		
		if(p == '(' && r == ')') {
			// ok
		} else if(p == '[' && r == ']') {
			// ok
		} else {
			throw message.getReadError(
					"err.read.parenthesis.mismatch");
		}
	}
	
	
	public boolean readParse(String exp) throws IOException {
		read(exp);
		return parse();
	}
	
	
	public void read(String read) {
		if(rd != null) {
			try {
				rd.close();
			} catch(IOException e) {
				throw new RuntimeException();
			}
		}
		rd = new PushbackReader(new StringReader(read));
	}
	
	
	public Datum getDatum() {
		while(!resultStack.empty() && resultStack.get(0) == _VOID) {
			resultStack.remove(0);
		}
		
		if(stateStack.empty() && !resultStack.empty()) {
			return resultStack.get(0);
		} else {
			return null;
		}
	}
	
	
	public List<Datum> getData() {
		return Collections.unmodifiableList(resultStack);
	}
	
	
	public boolean isReadBefore() {
		return readBefore;
	}
	
	private boolean isStackOK() {
		return (stateStack.empty() || (stateStack.size() == 1 &&
				stateStack.peek().equals(State.INIT)));
	}
	
	
	public void clear() {
		resultStack.clear();
		stateStack.clear();
		consStack.clear();
		
		clearBuf();
		stateStack.push(State.INIT);
		state = State.INIT;
		readBefore = false;
	}
	
	public boolean parse() throws IOException {
		int r;
		
		if(rd == null) {
			throw new IllegalStateException();
		}
		
		if(stateStack.empty() && State.INIT.equals(state)) {
			stateStack.push(State.INIT);
		}
		
		if(State.STRING.equals(state)) {
			r = read();
		} else {
			readBefore = false;
			r = readNext();
		}
		
		while(true) {
			if(!consoleMode && readBefore && stateStack.empty()) {
				if(r >= 0) {
					rd.unread(r);
				}
				return true;
			}
			
			switch(state) {
			case INIT:
				if(r == '(' || r == '[') {
					parenStack.push(r);
					state = State.LIST;
					r = readNext();
				} else if(r == ')' || r == ']') {
					throw message.getReadError(
							"err.read.closeparenthesis");
					//throw new ReadException("extra close parenthesis");
				} else if(r == '\'' || r == '`' || r == ',') {
					appendBuf(r);
					state = State.QUOTE;
					r = readNext2();
				} else if(r == '-' || r == '+') {
					appendBuf(r);
					state = State.HEAD_SIGN;
					r = read();
				} else if(r == '\"') {
					state = State.STRING;
					r = read();
				} else if(isDigit09(r)) {
					appendBuf(r);
					state = State.NUMBER;
					r = read();
				} else if(r == '#') {
					state = State.SHARP_SYNTAX_I;
					r = read();
				} else if(r == EOF) {
					//return stateStack.empty();
					return isStackOK();
				} else if(r == '.') {
					appendBuf(r);
					state = State.FLOAT_NUMBER;
					r = read();
				} else if(r == ';') {
					stateStack.push(state);
					state = State.COMMENT;
					r = read();
				} else {
					appendBuf(r);
					state = State.SYMBOL;
					r = read();
				}
				break;
			case NUMBER_PREFIX:
				if(r == EOF) {
					throw message.getReadError("err.read.sharp.unknown");
					//throw new ReadException("bad #-syntax");
				} else if(r == '#') {
					r = read();
					
					if(r == 'e' || r == 'E') {
						if(!Exact.DEFAULT.equals(exactness)) {
							throw message.getReadError(
									"err.read.numberprefix");
						}
						exactness = Exact.EXACT;
						state = State.NUMBER_PREFIX;
					} else if(r == 'i' || r == 'I') {
						if(!Exact.DEFAULT.equals(exactness)) {
							throw message.getReadError(
									"err.read.numberprefix");
						}
						exactness = Exact.INEXACT;
						state = State.NUMBER_PREFIX;
					} else if(r == 'b' || r == 'B') {
						if(radix > 0) {
							throw message.getReadError(
									"err.read.numberprefix");
						}
						radix = 2;
						state = State.NUMBER_PREFIX;
					} else if(r == 'o' || r == 'O') {
						if(radix > 0) {
							throw message.getReadError(
									"err.read.numberprefix");
						}
						radix = 8;
						state = State.NUMBER_PREFIX;
					} else if(r == 'd' || r == 'D') {
						if(radix > 0) {
							throw message.getReadError(
									"err.read.numberprefix");
						}
						radix = 10;
						state = State.NUMBER_PREFIX;
					} else if(r == 'x' || r == 'X') {
						if(radix > 0) {
							throw message.getReadError(
									"err.read.numberprefix");
						}
						radix = 16;
						state = State.NUMBER_PREFIX;
					}
					r = read();
				} else if(r == '-' || r == '+') {
					appendBuf(r);
					state = State.HEAD_SIGN;
					r = read();
				} else if(isDigit09az(r)) {
					appendBuf(r);
					state = State.NUMBER;
					r = read();
				} else if(r == '.') {
					appendBuf(r);
					state = State.FLOAT_NUMBER;
					r = read();
				} else if(r == 'i') {
					appendBuf(r);
					state = State.COMPLEX_I;
					//state = State.INF_I;
					r = read();
				} else {
					throw message.getReadError(
							"err.read.numberprefix");
					//throw new ReadException("bad numeric format");
				}
				break;
			case HEAD_SIGN:
				signed = true;
				
				if(isDigit09az(r)) {
					state = State.NUMBER;
				} else if(r == '.') {
					appendBuf(r);
					state = State.FLOAT_NUMBER;
					r = read();
				} else if(r == 'i') {
					appendBuf(r);
					//state = State.COMPLEX_I;
					state = State.INF_I;
					r = read();
				} else {
					state = symbolNum();
				}
				break;
			case NUMBER:
				if(isDigit09az(r)) {
					appendBuf(r);
					r = read();
				} else if(r == '.') {
					appendBuf(r);
					state = State.FLOAT_NUMBER;
					r = read();
				} else if(isExpsym(r)) {
					appendBuf(r);
					state = State.FLOAT_NUMBER_E;
					r = read();
				} else if(r == '(' || r == '[' || r == '[') {
					resultStack.push(getInteger());
					state = popStateStack();
				} else if(r == ')' || r == ']') {
					resultStack.push(getInteger());
					state = popStateStack();
				} else if(Character.isWhitespace(r)) {
					resultStack.push(getInteger());
					state = popStateStack();
					r = readNext();
				} else if(r == ';') {
					resultStack.push(getInteger());
					state = popStateStack();
				} else if(r == EOF) {
					resultStack.push(getInteger());
					state = popStateStack();
					return stateStack.empty();
				} else if(r == '/') {
					appendBuf(r);
					state = State.RATIONAL_NUMBER;
					r = read();
				} else if(r == '+' || r == '-') {
					appendBuf2(r);
					state = State.HEAD_SIGN_COMPLEX;
					r = read();
				} else if(r == 'i') {
					appendBuf(r);
					state = State.COMPLEX_I;
					r = read();
				} else if(r == '@') {
					appendBuf(r);
					state = State.INIT_POLAR;
					r = read();
				} else {
					state = symbolNum();
				}
				break;
			case FLOAT_NUMBER:
				if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(isDigit09az(r)) {
					state = State.FLOAT_NUMBER_D;
				} else {
					state = symbolNum();
				}
				break;
			case FLOAT_NUMBER_D:
				if(r == EOF) {
					resultStack.push(getDouble());
					state = popStateStack();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					resultStack.push(getDouble());
					state = popStateStack();
				} else if(r == ')' || r == ']') {
					resultStack.push(getDouble());
					state = popStateStack();
				} else if(r == ';') {
					resultStack.push(getDouble());
					state = popStateStack();
				} else if(Character.isWhitespace(r)) {
					resultStack.push(getDouble());
					state = popStateStack();
					r = readNext();
				} else if(isDigit09az(r)) {
					appendBuf(r);
					r = read();
				} else if(isExpsym(r)) {
					appendBuf(r);
					state = State.FLOAT_NUMBER_E;
					r = read();
				} else if(r == '+' || r == '-') {
					appendBuf2(r);
					state = State.HEAD_SIGN_COMPLEX;
					r = read();
				} else if(r == 'i') {
					appendBuf(r);
					state = State.COMPLEX_I;
					r = read();
				} else if(r == '@') {
					appendBuf(r);
					state = State.INIT_POLAR;
					r = read();
				} else {
					state = symbolNum();
				}
				break;
			case FLOAT_NUMBER_E:
				if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(r == '+' || r == '-') {
					appendBuf(r);
					state = State.FLOAT_NUMBER_E_SIGN;
					r = read();
				} else if(isDigit09az(r)) {
					state = State.FLOAT_NUMBER_E_D;
				} else {
					state = symbolNum();
				}
				break;
			case FLOAT_NUMBER_E_SIGN:
				if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(isDigit09az(r)) {
					state = State.FLOAT_NUMBER_E_D;
				} else {
					state = symbolNum();
				}
				break;
			case FLOAT_NUMBER_E_D:
				if(r == EOF) {
					resultStack.push(getDouble());
					state = popStateStack();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					resultStack.push(getDouble());
					state = popStateStack();
				} else if(r == ')' || r == ']') {
					resultStack.push(getDouble());
					state = popStateStack();
				} else if(r == ';') {
					resultStack.push(getDouble());
					state = popStateStack();
				} else if(Character.isWhitespace(r)) {
					resultStack.push(getDouble());
					state = popStateStack();
					r = readNext();
				} else if(isDigit09az(r)) {
					appendBuf(r);
					r = read();
				} else if(r == '+' || r == '-') {
					appendBuf2(r);
					state = State.HEAD_SIGN_COMPLEX;
					r = read();
				} else if(r == 'i') {
					appendBuf(r);
					state = State.COMPLEX_I;
					r = read();
				} else if(r == '@') {
					appendBuf(r);
					state = State.INIT_POLAR;
					r = read();
				} else {
					state = symbolNum();
				}
				break;
			case RATIONAL_NUMBER:
				if(isDigit09az(r)) {
					appendBuf(r);
					r = read();
					state = State.RATIONAL_NUMBER_2;
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else {
					state = symbolNum();
				}
				break;
			case RATIONAL_NUMBER_2:
				if(isDigit09az(r)) {
					appendBuf(r);
					r = read();
				} else if(r == '(' || r == '[') {
					resultStack.push(getRational());
					state = popStateStack();
				} else if(r == ')' || r == ']') {
					resultStack.push(getRational());
					state = popStateStack();
				} else if(r == ';') {
					resultStack.push(getRational());
					state = popStateStack();
				} else if(Character.isWhitespace(r)) {
					resultStack.push(getRational());
					state = popStateStack();
					r = readNext();
				} else if(r == EOF) {
					resultStack.push(getRational());
					state = popStateStack();
					return stateStack.empty();
				} else if(r == '+' || r == '-') {
					appendBuf2(r);
					state = State.HEAD_SIGN_COMPLEX;
					r = read();
				} else if(r == 'i') {
					appendBuf(r);
					state = State.COMPLEX_I;
					r = read();
				} else if(r == '@') {
					appendBuf(r);
					state = State.INIT_POLAR;
					r = read();
				} else {
					state = symbolNum();
				}
				break;
			case INIT_POLAR:
				if(r == '-' || r == '+') {
					appendBuf(r);
					state = State.HEAD_SIGN_POLAR;
					r = read();
				} else if(isDigit09az(r)) {
					appendBuf(r);
					state = State.NUMBER_POLAR;
					r = read();
				} else if(r == '.') {
					appendBuf(r);
					state = State.FLOAT_NUMBER_POLAR;
					r = read();
				} else {
					state = symbolNum();
				}
				break;
			case HEAD_SIGN_POLAR:
				if(isDigit09az(r)) {
					state = State.NUMBER_POLAR;
				} else if(r == '.') {
					appendBuf(r);
					state = State.FLOAT_NUMBER_POLAR;
					r = read();
				} else {
					state = symbolNum();
				}
				break;
			case NUMBER_POLAR:
				if(isDigit09az(r)) {
					appendBuf(r);
					r = read();
				} else if(r == '.') {
					appendBuf(r);
					state = State.FLOAT_NUMBER_POLAR;
					r = read();
				} else if(isExpsym(r)) {
					appendBuf(r);
					state = State.FLOAT_NUMBER_E_POLAR;
					r = read();
				} else if(r == '(' || r == '[') {
					resultStack.push(getPolar());
					state = popStateStack();
				} else if(r == ')' || r == ']') {
					resultStack.push(getPolar());
					state = popStateStack();
				} else if(r == ';') {
					resultStack.push(getPolar());
					state = popStateStack();
				} else if(Character.isWhitespace(r)) {
					resultStack.push(getPolar());
					state = popStateStack();
					r = readNext();
				} else if(r == EOF) {
					resultStack.push(getPolar());
					state = popStateStack();
					return stateStack.empty();
				} else if(r == '/') {
					appendBuf(r);
					state = State.RATIONAL_NUMBER_POLAR;
					r = read();
				} else {
					state = symbolNum();
				}
				break;
			case FLOAT_NUMBER_POLAR:
				if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(isDigit09az(r)) {
					state = State.FLOAT_NUMBER_D_POLAR;
				} else {
					state = symbolNum();
				}
				break;
			case FLOAT_NUMBER_D_POLAR:
				if(r == EOF) {
					resultStack.push(getPolar());
					state = popStateStack();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					resultStack.push(getPolar());
					state = popStateStack();
				} else if(r == ')' || r == ']') {
					resultStack.push(getPolar());
					state = popStateStack();
				} else if(r == ';') {
					resultStack.push(getPolar());
					state = popStateStack();
				} else if(Character.isWhitespace(r)) {
					resultStack.push(getPolar());
					state = popStateStack();
					r = readNext();
				} else if(isDigit09az(r)) {
					appendBuf(r);
					r = read();
				} else if(isExpsym(r)) {
					appendBuf(r);
					state = State.FLOAT_NUMBER_E_POLAR;
					r = read();
				} else {
					state = symbolNum();
				}
				break;
			case FLOAT_NUMBER_E_POLAR:
				if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(r == '+' || r == '-') {
					appendBuf(r);
					state = State.FLOAT_NUMBER_E_SIGN_POLAR;
					r = read();
				} else if(isDigit09az(r)) {
					state = State.FLOAT_NUMBER_E_D_POLAR;
				} else {
					state = symbolNum();
				}
				break;
			case FLOAT_NUMBER_E_SIGN_POLAR:
				if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(isDigit09az(r)) {
					state = State.FLOAT_NUMBER_E_D_POLAR;
				} else {
					state = symbolNum();
				}
				break;
			case FLOAT_NUMBER_E_D_POLAR:
				if(r == EOF) {
					resultStack.push(getPolar());
					state = popStateStack();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					resultStack.push(getPolar());
					state = popStateStack();
				} else if(r == ')' || r == ']') {
					resultStack.push(getPolar());
					state = popStateStack();
				} else if(r == ';') {
					resultStack.push(getPolar());
					state = popStateStack();
				} else if(Character.isWhitespace(r)) {
					resultStack.push(getPolar());
					state = popStateStack();
					r = readNext();
				} else if(isDigit09az(r)) {
					appendBuf(r);
					r = read();
				} else {
					state = symbolNum();
				}
				break;
			case RATIONAL_NUMBER_POLAR:
				if(isDigit09az(r)) {
					appendBuf(r);
					r = read();
					state = State.RATIONAL_NUMBER_2_POLAR;
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else {
					state = symbolNum();
				}
				break;
			case RATIONAL_NUMBER_2_POLAR:
				if(isDigit09az(r)) {
					appendBuf(r);
					r = read();
				} else if(r == '(' || r == '[') {
					resultStack.push(getPolar());
					state = popStateStack();
				} else if(r == ')' || r == ']') {
					resultStack.push(getPolar());
					state = popStateStack();
				} else if(r == ';') {
					resultStack.push(getPolar());
					state = popStateStack();
				} else if(Character.isWhitespace(r)) {
					resultStack.push(getPolar());
					state = popStateStack();
					r = readNext();
				} else if(r == EOF) {
					resultStack.push(getPolar());
					state = popStateStack();
					return stateStack.empty();
				} else {
					state = symbolNum();
				}
				break;
			case INF_I:
				if(r == EOF) {
					resultStack.push(getImaginary());
					state = popStateStack();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					resultStack.push(getImaginary());
					state = popStateStack();
				} else if(r == ')' || r == ']') {
					resultStack.push(getImaginary());
					state = popStateStack();
				} else if(r == ';') {
					resultStack.push(getImaginary());
					state = popStateStack();
				} else if(Character.isWhitespace(r)) {
					resultStack.push(getImaginary());
					state = popStateStack();
					r = readNext();
				} else if(r == 'n') {
					appendBuf(r);
					state = State.INF_N;
					r = read();
				} else {
					state = symbolNum();
				}
				break;
			case INF_N:
				if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(r == 'f') {
					appendBuf(r);
					state = State.INF_F;
					r = read();
				} else {
					state = symbolNum();
				}
				break;
			case INF_F:
				if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(r == '.') {
					appendBuf(r);
					state = State.INF_DOT;
					r = read();
				} else {
					state = symbolNum();
				}
				break;
			case INF_DOT:
				if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(r == '0') {
					appendBuf(r);
					state = State.INF_0;
					r = read();
				} else {
					state = symbolNum();
				}
				break;
			case INF_0:
				if(r == EOF) {
					resultStack.push(getDouble());
					state = popStateStack();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					resultStack.push(getDouble());
					state = popStateStack();
				} else if(r == ')' || r == ']') {
					resultStack.push(getDouble());
					state = popStateStack();
				} else if(r == ';') {
					resultStack.push(getDouble());
					state = popStateStack();
				} else if(Character.isWhitespace(r)) {
					resultStack.push(getDouble());
					state = popStateStack();
					r = readNext();
				} else if(r == '+' || r == '-') {
					appendBuf2(r);
					state = State.HEAD_SIGN_COMPLEX;
					r = read();
				} else if(r == 'i') {
					appendBuf(r);
					state = State.COMPLEX_I;
					r = read();
				} else if(r == '@') {
					appendBuf(r);
					state = State.INIT_POLAR;
					r = read();
				} else {
					state = symbolNum();
				}
				break;
			case COMPLEX_I:
				if(r == '(' || r == '[') {
					if(signed) {
						resultStack.push(getImaginary());
						state = popStateStack();
					} else {
						state = popStateSymbolNum();
					}
				} else if(r == ')' || r == ']') {
					if(signed) {
						resultStack.push(getImaginary());
						state = popStateStack();
					} else {
						state = popStateSymbolNum();
					}
				} else if(r == ';') {
					if(signed) {
						resultStack.push(getImaginary());
						state = popStateStack();
					} else {
						state = popStateSymbolNum();
					}
				} else if(Character.isWhitespace(r)) {
					if(signed) {
						resultStack.push(getImaginary());
						state = popStateStack();
					} else {
						state = popStateSymbolNum();
					}
					r = readNext();
				} else if(r == EOF) {
					if(signed) {
						resultStack.push(getImaginary());
						state = popStateStack();
					} else {
						state = popStateSymbolNum();
					}
					return stateStack.empty();
				} else {
					state = symbolNum();
				}
				break;
			case HEAD_SIGN_COMPLEX:
				if(isDigit09az(r)) {
					state = State.NUMBER_COMPLEX;
				} else if(r == '.') {
					appendBuf2(r);
					state = State.FLOAT_NUMBER_COMPLEX;
					r = read();
				} else if(r == 'i') {
					appendBuf2(r);
					//state = State.COMPLEX_I2;
					state = State.INF_I_COMPLEX;
					r = read();
				} else {
					state = symbol2Num();
				}
				break;
			case NUMBER_COMPLEX:
				if(isDigit09az(r)) {
					appendBuf2(r);
					r = read();
				} else if(r == '.') {
					appendBuf2(r);
					state = State.FLOAT_NUMBER_COMPLEX;
					r = read();
				} else if(isExpsym(r)) {
					appendBuf2(r);
					state = State.FLOAT_NUMBER_E_COMPLEX;
					r = read();
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else if(r == '/') {
					appendBuf2(r);
					state = State.RATIONAL_NUMBER_COMPLEX;
					r = read();
				} else if(r == 'i') {
					appendBuf2(r);
					state = State.COMPLEX_I2;
					r = read();
				} else {
					state = symbol2Num();
				}
				break;
			case FLOAT_NUMBER_COMPLEX:
				if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(isDigit09az(r)) {
					state = State.FLOAT_NUMBER_D_COMPLEX;
				} else {
					state = symbol2Num();
				}
				break;
			case FLOAT_NUMBER_D_COMPLEX:
				if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(isDigit09az(r)) {
					appendBuf2(r);
					r = read();
				} else if(isExpsym(r)) {
					appendBuf2(r);
					state = State.FLOAT_NUMBER_E_COMPLEX;
					r = read();
				} else if(r == 'i') {
					appendBuf2(r);
					state = State.COMPLEX_I2;
					r = read();
				} else {
					state = symbol2Num();
				}
				break;
			case FLOAT_NUMBER_E_COMPLEX:
				if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(r == '+' || r == '-') {
					appendBuf2(r);
					state = State.FLOAT_NUMBER_E_SIGN_COMPLEX;
					r = read();
				} else if(isDigit09az(r)) {
					state = State.FLOAT_NUMBER_E_D_COMPLEX;
				} else {
					state = symbol2Num();
				}
				break;
			case FLOAT_NUMBER_E_SIGN_COMPLEX:
				if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(isDigit09az(r)) {
					state = State.FLOAT_NUMBER_E_D_COMPLEX;
				} else {
					state = symbol2Num();
				}
				break;
			case FLOAT_NUMBER_E_D_COMPLEX:
				if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(isDigit09az(r)) {
					appendBuf2(r);
					r = read();
				} else if(r == 'i') {
					appendBuf2(r);
					state = State.COMPLEX_I2;
					r = read();
				} else {
					state = symbol2Num();
				}
				break;
			case RATIONAL_NUMBER_COMPLEX:
				if(isDigit09az(r)) {
					appendBuf2(r);
					r = read();
					state = State.RATIONAL_NUMBER_2_COMPLEX;
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else {
					state = symbol2Num();
				}
				break;
			case RATIONAL_NUMBER_2_COMPLEX:
				if(isDigit09az(r)) {
					appendBuf2(r);
					r = read();
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else if(r == 'i') {
					appendBuf2(r);
					state = State.COMPLEX_I2;
					r = read();
				} else {
					state = symbol2Num();
				}
				break;
			case INF_I_COMPLEX:
				if(r == EOF) {
					resultStack.push(getComplex());
					state = popStateStack();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					resultStack.push(getComplex());
					state = popStateStack();
				} else if(r == ')' || r == ']') {
					resultStack.push(getComplex());
					state = popStateStack();
				} else if(r == ';') {
					resultStack.push(getComplex());
					state = popStateStack();
				} else if(Character.isWhitespace(r)) {
					resultStack.push(getComplex());
					state = popStateStack();
					r = readNext();
				} else if(r == 'n') {
					appendBuf2(r);
					state = State.INF_N_COMPLEX;
					r = read();
				} else {
					state = symbol2Num();
				}
				break;
			case INF_N_COMPLEX:
				if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(r == 'f') {
					appendBuf2(r);
					state = State.INF_F_COMPLEX;
					r = read();
				} else {
					state = symbol2Num();
				}
				break;
			case INF_F_COMPLEX:
				if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(r == '.') {
					appendBuf2(r);
					state = State.INF_DOT_COMPLEX;
					r = read();
				} else {
					state = symbol2Num();
				}
				break;
			case INF_DOT_COMPLEX:
				if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(r == '0') {
					appendBuf2(r);
					state = State.INF_0_COMPLEX;
					r = read();
				} else {
					state = symbol2Num();
				}
				break;
			case INF_0_COMPLEX:
				if(r == EOF) {
					state = popStateSymbolNum();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					state = popStateSymbolNum();
				} else if(r == ')' || r == ']') {
					state = popStateSymbolNum();
				} else if(r == ';') {
					state = popStateSymbolNum();
				} else if(Character.isWhitespace(r)) {
					state = popStateSymbolNum();
					r = readNext();
				} else if(r == 'i') {
					appendBuf2(r);
					state = State.COMPLEX_I2;
					r = read();
				} else {
					state = symbol2Num();
				}
				break;
			case COMPLEX_I2:
				if(r == '(' || r == '[') {
					resultStack.push(getComplex());
					state = popStateStack();
				} else if(r == ')' || r == ']') {
					resultStack.push(getComplex());
					state = popStateStack();
				} else if(r == ';') {
					resultStack.push(getComplex());
					state = popStateStack();
				} else if(Character.isWhitespace(r)) {
					resultStack.push(getComplex());
					state = popStateStack();
					r = readNext();
				} else if(r == EOF) {
					resultStack.push(getComplex());
					state = popStateStack();
					return stateStack.empty();
				} else {
					state = symbol2Num();
				}
				break;
			case STRING:
				if(r == EOF) {
					if(consoleMode) {
						appendBuf('\n');
						return stateStack.empty();
					} else {
						throw message.getReadError("err.read.eof");
					}
				} else if(r == '\"') {
					resultStack.push(getString());
					state = popStateStack();
					r = readNext();
				} else if(r == '\\') {
					state = State.STRING_ESCAPE;
					r = read();
				} else {
					appendBuf(r);
					r = read();
				}
				break;
			case STRING_ESCAPE:
				if(r == EOF) {
					if(consoleMode) {
						//appendBuf('\n');
						state = State.STRING;
						return stateStack.empty();
						//throw message.getReadError("err.read.eol");
					} else {
						throw message.getReadError("err.read.eof");
					}
				} else {
					appendBuf(r);
					state = State.STRING;
					r = read();
				}
				break;
			case SHARP_SYNTAX_I:
				if(r == EOF) {
					//resultStack.push(getSharpSyntax());
					//state = popStateStack();
					//return stateStack.empty();
					throw message.getReadError("err.read.sharp.unknown");
					//throw new ReadException("unsupported #-syntax");
				} else if(r == '(') {
					state = State.VECTOR;
					r = readNext();
				} else if(r == ')' || r == ']') {
					//resultStack.push(getSharpSyntax());
					//state = popStateStack();
					throw message.getReadError("err.read.sharp.unknown");
					//throw new ReadException("unsupported #-syntax");
				} else if(r == ';') {
					// s-expression comment
					state = State.S_EXP_COMMENT;
					r = readNext();
				} else if(Character.isWhitespace(r)) {
					//resultStack.push(getSharpSyntax());
					//state = popStateStack();
					//r = readNext();
					throw message.getReadError("err.read.sharp.unknown");
					//throw new ReadException("unsupported #-syntax");
				} else if(r == '|') {
					// block comment
					stateStack.push(State.INIT);
					state = State.BLOCK_COMMENT;
					r = readNext();
				} else if(r == '\\') {
					state = State.CHARACTER_EXP;
					r = read();
				} else if(r == 'e' || r == 'E') {
					prefixed = true;
					exactness = Exact.EXACT;
					state = State.NUMBER_PREFIX;
					r = read();
				} else if(r == 'i' || r == 'I') {
					prefixed = true;
					exactness = Exact.INEXACT;
					state = State.NUMBER_PREFIX;
					r = read();
				} else if(r == 'b' || r == 'B') {
					prefixed = true;
					radix = 2;
					state = State.NUMBER_PREFIX;
					r = read();
				} else if(r == 'o' || r == 'O') {
					prefixed = true;
					radix = 8;
					state = State.NUMBER_PREFIX;
					r = read();
				} else if(r == 'd' || r == 'D') {
					prefixed = true;
					radix = 10;
					state = State.NUMBER_PREFIX;
					r = read();
				} else if(r == 'x' || r == 'X') {
					prefixed = true;
					radix = 16;
					state = State.NUMBER_PREFIX;
					r = read();
				} else {
					state = State.SHARP_SYNTAX;
					appendBuf(r);
					r = read();
				}
				break;
			case SHARP_SYNTAX:
				if(r == EOF) {
					resultStack.push(getSharpSyntax());
					state = popStateStack();
					return stateStack.empty();
				} else if(r == '(' || r == '[') {
					resultStack.push(getSharpSyntax());
					state = popStateStack();
				} else if(r == ')' || r == ']') {
					resultStack.push(getSharpSyntax());
					state = popStateStack();
				} else if(r == ';') {
					resultStack.push(getSharpSyntax());
					state = popStateStack();
				} else if(Character.isWhitespace(r)) {
					resultStack.push(getSharpSyntax());
					state = popStateStack();
					r = readNext();
				} else if(r == '\\') {
					//state = State.CHARACTER_EXP;
					//r = read();
					throw message.getReadError("err.read.sharp.unknown");
					//throw new ReadException("unsupported #-syntax");
				} else {
					appendBuf(r);
					r = read();
				}
				break;
			case CHARACTER_EXP:
				if(r == EOF) {
					throw message.getReadError("err.read.eol");
					//throw new ReadException("unexpected end of line");
				} else if(r == ';') {
					throw message.getReadError(
							"err.read.character.unknown");
					//throw new ReadException("bad character name");
				} else {
					appendBuf(r);
					state = State.CHARACTER_EXP_NAME;
					r = read();
				}
				break;
			case CHARACTER_EXP_NAME:
				if(r == '(' || r == '[') {
					resultStack.push(getCharacter());
					state = popStateStack();
				} else if(r == ')' || r == ']') {
					resultStack.push(getCharacter());
					state = popStateStack();
				} else if(r == ';') {
					resultStack.push(getCharacter());
					state = popStateStack();
				} else if(Character.isWhitespace(r)) {
					resultStack.push(getCharacter());
					state = popStateStack();
					r = readNext();
				} else if(r == EOF) {
					resultStack.push(getCharacter());
					state = popStateStack();
					return stateStack.empty();
				} else {
					appendBuf(r);
					r = read();
				}
				break;
			case SYMBOL:
				if(r == '(' || r == '[') {
					resultStack.push(getSymbol());
					state = popStateStack();
				} else if(r == ')' || r == ']') {
					resultStack.push(getSymbol());
					state = popStateStack();
				} else if(r == ';') {
					resultStack.push(getSymbol());
					state = popStateStack();
				} else if(Character.isWhitespace(r)) {
					resultStack.push(getSymbol());
					state = popStateStack();
					r = readNext();
				} else if(r == EOF) {
					resultStack.push(getSymbol());
					state = popStateStack();
					return stateStack.empty();
				} else {
					appendBuf(r);
					r = read();
				}
				break;
			case SYMBOL2:
				if(r == '(' || r == '[') {
					resultStack.push(getSymbol());
					state = popStateStack();
				} else if(r == ')' || r == ']') {
					resultStack.push(getSymbol());
					state = popStateStack();
				} else if(r == ';') {
					resultStack.push(getSymbol());
					state = popStateStack();
				} else if(Character.isWhitespace(r)) {
					resultStack.push(getSymbol());
					state = popStateStack();
					r = readNext();
				} else if(r == EOF) {
					resultStack.push(getSymbol());
					state = popStateStack();
					return stateStack.empty();
				} else {
					appendBuf2(r);
					r = read();
				}
				break;
			case LIST:
				if(r == EOF) {
					return stateStack.empty();
				} else if(r == ';') {
					stateStack.push(state);
					state = State.COMMENT;
					r = read();
				} else if(r == ')' || r == ']') {
					checkParenthesis(r);
					
					resultStack.push(Nil.NIL);
					state = popStateStack();
					r = readNext();
				} else {
					Cons n = new Cons(_VOID, Nil.NIL);
					
					resultStack.push(n);
					stateStack.push(State.LIST2);
					consStack.push(n);
					state = State.INIT;
				}
				break;
			case LIST2:
				if(r == EOF) {
					return stateStack.empty();
				} else if(r == ';') {
					stateStack.push(state);
					state = State.COMMENT;
					r = read();
				} else {
					Cons c1 = consStack.peek();
					Datum d1 = resultStack.pop();
					
					// s-expression comment
					if(d1 != _VOID) {
						c1.setCar(d1);
					}
					
					if(r == ')' || r == ']') {
						Datum d2 = resultStack.pop();
						
						checkParenthesis(r);
						
						// ꥹȤȽꤹ
						if(d2 instanceof Cons) {
							Cons cd2 = (Cons)d2;
							
							if(cd2.getCar() == _VOID) {
								resultStack.push(Nil.NIL);
							} else {
								resultStack.push(d2);
							}
						} else {
							throw new RuntimeException();
						}
						
						consStack.pop();
						state = popStateStack();
						r = readNext();
					} else if(r == '.') {
						appendBuf(r);
						state = State.LIST_DOT_INIT;
						r = read();
					} else {
						if(d1 != _VOID) {
							Cons n = new Cons();
							
							c1.setCdr(n);
							consStack.pop();
							consStack.push(n);
						}
						stateStack.push(State.LIST2);
						state = State.INIT;
					}
				}
				break;
			case LIST_DOT_INIT:
				Cons c3 = consStack.peek();
				
				if(c3.getCar() == _VOID) {
					throw message.getReadError("err.read.list.invalid");
					//throw new ReadException("invalid list");
				} else if(r == '(' || r == '[' ||
						r == ')' || r == ']' || r == EOF) {
					stateStack.push(State.LIST_DOT);
					state = State.INIT;
					clearBuf();
					c3.setCdr(_VOID);
				} else if(r == ';') {
					stateStack.push(State.LIST_DOT);
					stateStack.push(State.INIT);
					state = State.COMMENT;
					clearBuf();
					c3.setCdr(_VOID);
				} else if(Character.isWhitespace(r)) {
					stateStack.push(State.LIST_DOT);
					state = State.INIT;
					r = readNext();
					c3.setCdr(_VOID);
				} else if(isDigit09(r)) {
					Cons n = new Cons();
					
					c3.setCdr(n);
					consStack.pop();
					stateStack.push(State.LIST2);
					consStack.push(n);
					state = State.FLOAT_NUMBER_D;
				} else {
					Cons n = new Cons();
					
					c3.setCdr(n);
					consStack.pop();
					stateStack.push(State.LIST2);
					consStack.push(n);
					state = State.SYMBOL;
				}
				break;
			case LIST_DOT:
				if(r == EOF) {
					return stateStack.empty();
				} else if(r == ';') {
					stateStack.push(state);
					state = State.COMMENT;
					r = read();
				} else {
					Cons c2 = consStack.peek();
					Datum d1 = resultStack.pop();
					
					// s-expression comment
					if(d1 != _VOID) {
						c2.setCdr(d1);
					}
					
					if(r == ')' || r == ']') {
						checkParenthesis(r);
						
						if(c2.getCdr() == _VOID) {
							throw message.getReadError("err.read.dot");
							//throw new ReadException("Bad dot syntax");
						}
						consStack.pop();
						state = popStateStack();
						r = readNext();
					} else if(d1 == _VOID) {
						stateStack.push(State.LIST_DOT);
						state = State.INIT;
						clearBuf();
					} else {
						//state = State.INIT;
						//resultStack.clear();
						//stateStack.clear();
						//consStack.clear();
						throw message.getReadError("err.read.dot");
						//throw new ReadException("Bad dot syntax");
					}
				}
				break;
			case QUOTE:
				Cons n1 = new Cons();
				Cons n2 = new Cons();
				
				if(r == '@') {
					appendBuf(r);
					r = readNext2();
				}
				
				n1.setCar(getQuote());
				n1.setCdr(n2);
				resultStack.push(n1);
				consStack.push(n2);
				stateStack.push(State.QUOTE);
				state = State.INIT;
				break;
			case VECTOR:
				if(r == EOF) {
					return stateStack.empty();
				} else if(r == ')') {
					resultStack.push(LispVector.EMPTY);
					state = popStateStack();
					r = readNext();
				} else if(r == ']') {
					throw message.getReadError(
							"err.read.parenthesis.mismatch");
				} else if(r == ';') {
					stateStack.push(state);
					state = State.COMMENT;
					r = read();
				} else {
					vectorStack.push(new ArrayList<Datum>());
					stateStack.push(State.VECTOR2);
					state = State.INIT;
				}
				break;
			case VECTOR2:
				if(r == EOF) {
					return stateStack.empty();
				} else if(r == ';') {
					stateStack.push(state);
					state = State.COMMENT;
					r = read();
				} else {
					List<Datum> v1 = vectorStack.peek();
					Datum d1 = resultStack.pop();
					
					// s-expression comment
					if(d1 != _VOID) {
						v1.add(d1);
					}
					
					if(r == ')') {
						vectorStack.pop();
						resultStack.push(new LispVector(v1));
						state = popStateStack();
						r = readNext();
					} else if(r == ']') {
						throw message.getReadError(
								"err.read.parenthesis.mismatch");
					} else {
						stateStack.push(State.VECTOR2);
						state = State.INIT;
					}
				}
				break;
			case COMMENT:
				if(r == EOF) {
					state = stateStack.pop();
					//return stateStack.empty();
				} else if(r == '\n') {
					state = stateStack.pop();
					r = readNext();
				} else {
					r = read();
				}
				break;
			case BLOCK_COMMENT:
				if(r == EOF) {
					return stateStack.empty();
				} else if(r == '|') {
					r = read();
					if(r == '#') {
						state = stateStack.pop();
						r = readNext();
					} else {
						r = readNext();
					}
				} else if(r == '#') {
					r = read();
					if(r == '|') {
						stateStack.push(State.BLOCK_COMMENT);
						r = readNext();
					} else {
						r = readNext();
					}
				} else {
					r = readNext();
				}
				break;
			case S_EXP_COMMENT:
				stateStack.push(State.S_EXP_COMMENT);
				state = State.INIT;
				break;
			}
		}
	}
	
}
