#include "sender_check_record.h"

/**
 * args:
 *  domain    : 送信元メールアドレスから取得したドメイン
 *  from_addr : data構造体に含まれているin_addr構造体
 * ret:
 *  SENDER_CHECK_OK        : Aレコードを取得し、送信元IPアドレスが一致した場合
 *  SENDER_CHECK_NG        : AレコードとIPアドレスが一致しなかった場合
 *  SENDER_CHECK_ERR       : レコード取得時にエラーが発生した場合
 *  SENDER_CHECK_NO_RECORD : Aレコードを1件も引けなかった場合
 */
int comp_ipaddr_ipv4(char *domain, struct in_addr *from_addr)
{
    int len, ans_num, chk_addr, loop_num;
    u_char nsbuf[EDNS0_MAXPACKET];
    ns_msg msg;
    ns_rr rr;

    /* 渡されたドメインを使ってres_queryでAレコードを引く */
    len = res_query(domain, ns_c_any, ns_t_a, nsbuf, sizeof(nsbuf));

    /* res_queryの返り値判断 */
    if (len < 0) {
        if (errno == 0) {
            return SENDER_CHECK_NO_RECORD;
        }
        DEBUGLOG("check_senderdomain:Error %s\n", strerror(errno));
        return SENDER_CHECK_ERR;
    }

    /* 結果を分析 */
    ns_initparse(nsbuf, len, &msg);
    
    /* 応答数を取得 */
    ans_num = ns_msg_count(msg, ns_s_an);
    
    /* 応答数だけループして、Aレコードと送信元アドレスを比較 */
    for (loop_num = 0; loop_num < ans_num; loop_num++) {
        /* &rrにres_queryの結果を整形して格納 */
        ns_parserr(&msg, ns_s_an, loop_num, &rr);

        /* struct in_addrの構造体へのポインタであるrr.rdataと比較 */
        chk_addr = memcmp(from_addr, rr.rdata, sizeof(struct in_addr));

        if (chk_addr == 0) {
            return SENDER_CHECK_OK;
        }
    }
    return SENDER_CHECK_NG;
}

/**
 * args:
 *  domain    : 送信元メールアドレスから取得したドメイン
 *  from_addr : data構造体に含まれているin6_addr構造体
 * ret:
 *  SENDER_CHECK_OK        : AAAAレコードを取得し、送信元IPアドレスが一致した場合
 *  SENDER_CHECK_NG        : AAAAレコードとIPアドレスが一致しなかった場合
 *  SENDER_CHECK_ERR       : レコード取得時にエラーが発生した場合
 *  SENDER_CHECK_NO_RECORD : AAAAレコードを1件も引けなかった場合
 */
int comp_ipaddr_ipv6(char *domain, struct in6_addr *from_addr)
{
    int len, ans_num, chk_addr, loop_num;
    u_char nsbuf[EDNS0_MAXPACKET];
    ns_msg msg;
    ns_rr rr;

    /* 渡されたドメインを使ってres_queryでAAAAレコードを引く */
    len = res_query(domain, ns_c_any, ns_t_aaaa, nsbuf, sizeof(nsbuf));

    /* res_queryの返り値判断 */
    if (len < 0) {
        if (errno == 0) {
            return SENDER_CHECK_NO_RECORD;
        }
        DEBUGLOG("check_senderdomain:Error %s\n", strerror(errno));
        return SENDER_CHECK_ERR;
    }

    /* 結果を分析 */
    ns_initparse(nsbuf, len, &msg);
    
    /* 応答数を取得 */
    ans_num = ns_msg_count(msg, ns_s_an);
    
    /* 応答数だけループして、AAAAレコードと送信元アドレスを比較 */
    for (loop_num = 0; loop_num < ans_num; loop_num++) {
        /* &rrにres_queryの結果を整形して格納 */
        ns_parserr(&msg, ns_s_an, loop_num, &rr);

        /* struct in6_addrの構造体へのポインタであるrr.rdataと比較 */
        chk_addr = memcmp(from_addr, rr.rdata, sizeof(struct in6_addr));

        if (chk_addr == 0) {
            return SENDER_CHECK_OK;
        }
    }
    return SENDER_CHECK_NG;
}


/**
 * args:
 *  domain : 送信元メールアドレスから取得したドメイン
 *  data   : data構造体
 * ret:
 *  SENDER_CHECK_OK        : レコードを取得し、送信元IPアドレスが一致した場合
 *  SENDER_CHECK_NG        : レコードとIPアドレスが一致しなかった場合
 *  SENDER_CHECK_ERR       : レコード取得時にエラーが発生した場合
 *  SENDER_CHECK_NO_RECORD : レコードを1件も引けなかった場合
 */
int comp_ipaddr(char *domain, sender_check_arg_t *data)
{
    int res_comp;

    /* 渡されたdata構造体のafを元にAを引くかAAAAを引くか判断 */
    if (data->af == AF_INET) {
        res_comp = comp_ipaddr_ipv4(domain, &(data->sa).sa_in.sin_addr);
    } else {
        res_comp = comp_ipaddr_ipv6(domain, &(data->sa).sa_in6.sin6_addr);
    }
    return res_comp;
}

/**
 * args:
 *  domain : 送信元メールアドレスから取得したドメイン
 *  data   : data構造体
 * ret:
 *  SENDER_CHECK_OK        : Aレコードを取得し、送信元IPアドレスが一致した場合
 *  SENDER_CHECK_NG        : AレコードとIPアドレスが一致しなかった場合
 *  SENDER_CHECK_ERR       : レコード取得時にエラーが発生した場合
 *  SENDER_CHECK_NO_RECORD : Aレコードを1件も引けなかった場合
 */
int comp_mx(char *domain, sender_check_arg_t *data)
{
    int len, len_dom, ans_num, loop_num, res_mx;
    u_char nsbuf[EDNS0_MAXPACKET];
    u_char *conts;
    ns_msg msg;
    char dom[NS_MAXDNAME];

    /* 渡されたドメインを使ってres_queryでMXレコードを引く */
    len = res_query(domain, ns_c_any, ns_t_mx, nsbuf, sizeof(nsbuf));

    /* res_queryの返り値判断 */
    if (len < 0) {
        if (errno == 0) {
            return SENDER_CHECK_NG;
        }
        DEBUGLOG("check_senderdomain:Error %s\n", strerror(errno));
        return SENDER_CHECK_ERR;
    }

    /* 結果を分析 */
    ns_initparse(nsbuf, len, &msg);

    /* 応答数を取得 */
    ans_num = ns_msg_count(msg, ns_s_an);

    /* 応答のヘッダ以降をcontsへ格納 */
    conts = nsbuf + sizeof(HEADER);

    /* 問い合わせ部の圧縮されたドメインを展開 */
    len_dom = dn_expand(nsbuf, nsbuf + len, conts, dom, sizeof(dom));

    if (len_dom < 0) {
        return SENDER_CHECK_ERR;
    }

    /* 問い合わせ部のドメイン分contsの参照先を進める */
    conts += len_dom;

    /* 不要な分の読み捨て */
    conts += NS_INT16SZ * 2;
    
    /* 応答数だけループして、Aレコードと送信元アドレスを比較 */
    for (loop_num = 0; loop_num < ans_num; loop_num++) {
        /* 回答部の圧縮された問い合わせドメインを展開 */
        len_dom = dn_expand(nsbuf, nsbuf + len, conts, dom, sizeof(dom));

        if (len_dom < 0) {
            return SENDER_CHECK_ERR;
        }

        conts += len_dom;

        /* 不要な分の読み捨て */
        conts += NS_INT16SZ * 4 + NS_INT32SZ;

        /* 回答部の圧縮された回答ドメインを展開 */
        len_dom = dn_expand(nsbuf, nsbuf + len, conts, dom, sizeof(dom));

        if (len_dom < 0) {
            return SENDER_CHECK_ERR;
        }

        /* 取得できたMXレコードをcomp_ipaddrへ渡す */
        res_mx = comp_ipaddr(dom, data);
        switch (res_mx) {
            case SENDER_CHECK_ERR:
                return SENDER_CHECK_ERR;
            case SENDER_CHECK_OK:
                return SENDER_CHECK_OK;
        }
    }
    return SENDER_CHECK_NG;
}
