/*
 * samma
 * harmless.c
 *
 * switch convert, encrypt, delete
 */


#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "exec_command.h"
#include "msg_convert.h"
#include "log.h"

#define MAX_CONFIG_LINE 1024


// 変換プログラムの一覧
typedef struct _conv_func {
    char *item;
    int (*function)();
} conv_func_t;

conv_func_t msgconv_func_list[] = {
    { "alternative", convert_alternative },
    { "multipart", convert_multipart },
    { "none", convert_none },
};
#define CONVFUNC_LIST_SIZE (sizeof(msgconv_func_list) / sizeof(conv_func_t))

/**
 * static function decraretion
 */
int exec_command_with_gmime(GMimeObject *, GMimeObject **, char **, harmless_proc_arg_t *);

void output_conv_log(char *fmt, char *command, sender_check_arg_t *data);


/**
 * functions
 */
void output_conv_log(char *fmt, char *command, sender_check_arg_t *data)
{
    log(
        fmt,
        command,
        data->ip,
        data->message_id,
        data->envelope_from,
        data->rcpt_to
    );

}

void free_harmlessconf(command_list_t *list)
{
    command_list_t *tmp, *del;

    for (tmp = list; tmp != NULL; tmp = tmp->next, free(del)) {
        free(tmp->mime_before);
        free(tmp->mime_after);
        free(tmp->file_extension);
        free(tmp->command_name);
        free_arg_list(tmp->arg_list);
        del = tmp;
    }
}


char *read_harmlessconf(char *file, void **pointer)
{
    static char errbuf[1024] = "";
    char *errmsg;
    command_list_t *new_list = NULL, *list_end = NULL, *making_list = NULL;

    FILE *fp = NULL;

    char line[MAX_CONFIG_LINE + 1];
    int nline, i;
    char *tail, *save;
    char *fst, *snd, *trd, *fth;

    errmsg = is_readable_file(file);
    if (errmsg) {
        sprintf(errbuf, CONV_ERR_READ, file);
        goto error;
    }

    fp = fopen(file, "r");
    if (fp == NULL) {
        sprintf(errbuf, CONV_ERR_OPEN, file, strerror(errno));
        goto error;
    }

    for (nline = 1; fgets(line, MAX_CONFIG_LINE + 1, fp) != NULL; nline++) {
        command_list_t current_list = {NULL, NULL, NULL, NULL, -1, NULL, NULL, NULL};

        tail = strchr(line, '\n');
        if (tail == NULL) {
            sprintf(errbuf, CONV_ERR_TOOLONGLINE, file, nline);
            goto error;
        }
        *tail = '\0';

        if ((line[0] == '#') || (line[0] == '\0')) {
            /* comment or null line */
            continue;
        }

        // 4つに分割
        fst = strtok_r(line, " \t", &save);
        if (fst == NULL) {
            sprintf(errbuf, CONV_ERR_FEW_ITEM, file, nline);
            goto error;
        }
        snd = strtok_r(NULL, " \t", &save);
        if (snd == NULL) {
            sprintf(errbuf, CONV_ERR_FEW_ITEM, file, nline);
            goto error;
        }
        trd = strtok_r(NULL, " \t", &save);
        if (trd == NULL) {
            sprintf(errbuf, CONV_ERR_FEW_ITEM, file, nline);
            goto error;
        }
        // - だったら拡張子を加工しない
        if (strcmp(trd, "-") == 0) {
            trd = "";
        }

        while (isblank(*save)) save++;
        if (save == tail) {
            sprintf(errbuf, CONV_ERR_FEW_ITEM, file, nline);
            goto error;
        }
        fth = save;

        // 指定されたコマンドのチェック
        if (*fth != '/') {
            current_list.command_type = FUNCTION;
            if ((current_list.command_name = strdup(fth)) == NULL) {
                sprintf(errbuf, CONV_ERR_MEM);
                goto error;
            }

            // 内部関数をセット
            for (i = 0; i < CONVFUNC_LIST_SIZE; i++) {
                if (strcasecmp(msgconv_func_list[i].item, fth)) {
                    continue;
                }

                current_list.in_func = msgconv_func_list[i].function;
                break;
            }

            if (i == CONVFUNC_LIST_SIZE) {
                sprintf(errbuf, CONV_ERR_WRONG_FUNC, file, nline, fth);
                goto error;
            }

        } else {
            current_list.command_type = COMMAND;
            if ((current_list.command_name = strdup(fth)) == NULL) {
                sprintf(errbuf, CONV_ERR_MEM);
                goto error;
            }

            errmsg = parse_command(fth, &(current_list.arg_list));
            if (errmsg) {
                sprintf(errbuf, CONV_ERR_WITH_LINE, file, nline, errmsg);
                goto error;
            }
        }

        new_list = calloc(1, sizeof(command_list_t));
        if (new_list == NULL) {
            sprintf(errbuf, CONV_ERR_MEM);
            free_arg_list(current_list.arg_list);
            free(current_list.command_name);
            goto error;
        }
        *new_list = current_list;

        if (making_list == NULL) {
            making_list = new_list;
            list_end = new_list;
        } else {
            list_end->next = new_list;
            list_end = new_list;
        }

        new_list->mime_before = strdup(fst);
        if (new_list->mime_before == NULL) {
            sprintf(errbuf, CONV_ERR_MEM);
            goto error;
        }

        new_list->mime_after = strdup(snd);
        if (new_list->mime_after == NULL) {
            sprintf(errbuf, CONV_ERR_MEM);
            goto error;
        }

        new_list->file_extension = strdup(trd);
        if (new_list->file_extension == NULL) {
            sprintf(errbuf, CONV_ERR_MEM);
            goto error;
        }
    }

    *pointer = making_list;

    goto end;


// 終了処理
error:
    free_harmlessconf(making_list);

end:
    if (fp != NULL) {
        fclose(fp);
    }

    return errbuf[0] == '\0' ? NULL : errbuf;
}

#define MIME_BEFORE 0
#define MIME_AFTER 1
#define FILE_BEFORE 2
#define FILE_AFTER 3
int msg_convert(GMimeObject *part, GMimeObject **new, harmless_proc_arg_t *arg)
{
    int ret;
    char *content_type, *filename;
    struct strtag taglist[] = {
          { "mime-before", 11, NULL }
        , { "mime-after" , 10, NULL }
        , { "file-before", 11, NULL }
        , { "file-after" , 10, NULL }
    };

    command_list_t *c_p;
    command_list_t *command_list = arg->cfg->cf_harmlessconf;

    GMimeContentType *top_content_type_object = g_mime_object_get_content_type(part);

    content_type = g_mime_content_type_to_string(top_content_type_object);
    if (content_type == NULL) {
        log("Content-Type is missing.");
        return HARMLESS_NG;
    }

    taglist[MIME_BEFORE].st_str = content_type;
    taglist[FILE_BEFORE].st_str = filename = estimate_filename(part);

    // mimeタイプ判定のループ
    for (c_p = command_list; c_p != NULL; c_p = c_p->next) {

        // 元のMIMEタイプと受信メールのMIMEタイプが一致したら0でif文に入らない
        // 元のMIMEタイプに*が指定されていた場合はif文に入らない
        // 違っていたら入る
        if ((strcasecmp(c_p->mime_before, "*") != 0) && (strcasecmp(c_p->mime_before, content_type) != 0)) {
            continue;
        }

        switch (c_p->command_type) {
            int _ret = 0;
            case FUNCTION:
                _ret = (c_p->in_func)(part, new, arg);
                if (_ret == HARMLESS_NG) {
                    output_conv_log(CONV_ERR_EXEC_IN, c_p->command_name, arg->maildata);
                    free(content_type);
                    return HARMLESS_NG;
                }

                break;
            case COMMAND:
                _ret = exec_command_with_gmime(part, new, c_p->arg_list, arg);
                if (_ret == HARMLESS_NG) {
                    output_conv_log(CONV_ERR_EXEC_EXT, c_p->command_name, arg->maildata);
                    free(content_type);
                    return HARMLESS_NG;
                }
                // mimetypeを更新
                if (GMIME_IS_OBJECT(*new)) {
                    GMimeContentType *newtype = g_mime_content_type_new_from_string(c_p->mime_after);
                    g_mime_object_set_content_type((GMimeObject *)*new, newtype);
                    g_object_unref(newtype);
                }
                break;
        }

        // チェック関数が実行されたら以下をやってから処理を抜ける
        // ファイル名に拡張子をつける
        {
            char *newfilename = NULL;
            if (*filename != '\0') {
                newfilename =  malloc(
                    strlen(filename)
                    + strlen(c_p->file_extension)
                    + 1
                );
                if (newfilename == NULL) {
                    free(content_type);
                    return HARMLESS_NG;
                }
                sprintf(
                    newfilename,
                    "%s%s",
                    filename,
                    c_p->file_extension
                );
                taglist[FILE_AFTER].st_str = newfilename;
            } else {
                taglist[FILE_AFTER].st_str = taglist[FILE_BEFORE].st_str = arg->cfg->cf_attachmentfilealias;
            }

            taglist[MIME_AFTER].st_str = c_p->mime_after;

            ret = strcat_proc_message(
                &(arg->message),
                &(arg->message_length),
                arg->cfg->cf_harmlessmessageconvert,
                taglist,
                4
            );

            free(content_type);
            if (newfilename != NULL) {
                //g_mime_part_set_filename((GMimePart *)*new, (const char *)newfilename);
                g_mime_object_set_content_type_parameter((GMimeObject *)*new, "filename", newfilename);
                g_mime_object_set_content_disposition_parameter((GMimeObject *)*new, "filename", newfilename);
                free(newfilename);
            }
            return ret;
        }
    }

    // ループが回りきったということは、指定されていないMimeタイプだった
    log("content-type(%s) is not in 'harmless.conf'", content_type);
    free(content_type);
    return HARMLESS_NG;
}

int convert_alternative(GMimeObject *part, GMimeObject **new, harmless_proc_arg_t *arg)
{
    int index, i;
    int alt_part = -1;
    char *content_type = NULL;
    GMimeMultipart *top;

    command_list_t *cp;
    command_list_t *clist = arg->cfg->cf_harmlessconf;

    if (!GMIME_IS_MULTIPART(part)) {
        return HARMLESS_NG;
    }

    // いちいちキャストがめんどくさいので変数で受ける
    top = (GMimeMultipart *)part;

    index = g_mime_multipart_get_count(top);

    for (i = 0; i < index; i++) {
        GMimeObject *target = NULL;
        GMimeContentType *content_type_object = NULL;

        target = g_mime_multipart_get_part(top, i);

        // dont free this
        content_type_object = g_mime_object_get_content_type(target);

        content_type = g_mime_content_type_to_string(content_type_object);
        if (content_type == NULL) {
            continue;
        }

        // 最初に見つかったtext/plainを処理して返す
        if (!strcasecmp(content_type, "text/plain")) {
            int ret;
            // HARMLESS_OK or NG
            ret = harmless_proc_single_part(target, new, arg);
            return ret;
        }

        // text/plainに変換出来るものかどうかをチェック.
        // もう見つかっていたら更新しない.
        // もし見つかっても、この先にtext/plainがあるかもしれないので
        // ループを続ける
        if (alt_part == -1) {
            for (cp = clist; cp != NULL; cp = cp->next) {
                if (strcasecmp(content_type, cp->mime_before)) {
                    continue;
                }

                if (strcasecmp(cp->mime_after, "text/plain")) {
                    continue;
                }

                alt_part = i;
            }
        }
    }

    if (alt_part != -1) {
        GMimeObject *target;

        target = g_mime_multipart_get_part(top, alt_part);
        return harmless_proc_single_part(target, new, arg);
    }

    log("Alternative part has no convert-able part to text/plain.");
    return HARMLESS_NG;
}

int convert_multipart(GMimeObject *part, GMimeObject **new, harmless_proc_arg_t *arg)
{
    int ret;

    if (!GMIME_IS_MULTIPART(part)) {
        return HARMLESS_NG;
    }

    ret = harmless_proc((GMimeMultipart *)part, arg);
    if (ret != HARMLESS_OK) {
        return ret;
    }

    // OKだったら加工されたパートを返す
    *new = part;

    return HARMLESS_OK;
}

int convert_none(GMimeObject *part, GMimeObject **new, harmless_proc_arg_t *arg)
{
    *new = part;
    return HARMLESS_OK;
}

/**
 * すごい適当なので、だれかうまいこと直して下さい
 * （exec_command.cに移行すべきです）
 */
int exec_command_with_gmime(GMimeObject *part, GMimeObject **new, char **cmd_arg, harmless_proc_arg_t *arg)
{
    char *after = NULL;
    size_t aftersize;

    if (!GMIME_IS_PART(part)) {
        log("exec_command_with_gmime: Given part is not a single part");
        return HARMLESS_NG;
    }


    int eret = exec_external(part, cmd_arg, &after, &aftersize, arg->maildata, arg->cfg->cf_harmlesscommandtimeout);

    if (eret != EXEC_EXTERNAL_SUCCESS) {
        return HARMLESS_NG;
    }


    GMimeStream *r_stream = g_mime_stream_mem_new_with_buffer(after, aftersize);
    free(after);
    GMimeStream *f_r_stream = g_mime_stream_filter_new(r_stream);
    g_object_unref(r_stream);


    GMimeContentEncoding enctype = g_mime_part_get_content_encoding((GMimePart *)part);

    GMimeFilter *r_filter = g_mime_filter_basic_new(enctype, TRUE);
    g_mime_stream_filter_add((GMimeStreamFilter *)f_r_stream, r_filter);
    g_object_unref(r_filter);

    GMimeDataWrapper *converted = g_mime_data_wrapper_new_with_stream(f_r_stream, enctype);
    g_object_unref(f_r_stream);

    g_mime_part_set_content_object((GMimePart *)part, converted);
    g_object_unref(converted);

    *new = part;
    return HARMLESS_OK;
}
