#!/usr/local/bin/ruby
# $Id: test_wpm_xml_with_xmlparser.rb,v 1.1.1.1 2004/04/04 15:22:51 toki Exp $
# test of XML processing for Page Maker.

require 'rubyunit'
require 'wpm'
require 'wpm/xmlparser'

module TestWebPageMaker
  class TestXMLNamespaceMapByXMLParser < RUNIT::TestCase
    def setup
      @ns_map = WPM::XMLNamespaceMap.new
    end

    def test_ns
      # nest level 1
      @ns_map.start_element
      @ns_map.add_ns('', 'http://www.w3.org/1999/xhtml')
      @ns_map.add_ns('x', 'test://foo')
      @ns_map.add_ns('y', 'test://bar')

      assert_equal('http://www.w3.org/1999/xhtml', @ns_map[''])
      assert_equal('test://foo', @ns_map['x'])
      assert_equal('test://bar', @ns_map['y'])
      assert_nil(@ns_map['z'])

      # nest level 2
      @ns_map.start_element
      @ns_map.add_ns('', 'test://foo')
      @ns_map.add_ns('x', 'http://www.w3.org/1999/xhtml')
      @ns_map.add_ns('y', 'test://foo')
      @ns_map.add_ns('z', 'test://bar')

      assert_equal('test://foo', @ns_map[''])
      assert_equal('http://www.w3.org/1999/xhtml', @ns_map['x'])
      assert_equal('test://foo', @ns_map['y'])
      assert_equal('test://bar', @ns_map['z'])

      # end of nest level 2
      @ns_map.end_element

      assert_equal('http://www.w3.org/1999/xhtml', @ns_map[''])
      assert_equal('test://foo', @ns_map['x'])
      assert_equal('test://bar', @ns_map['y'])
      assert_nil(@ns_map['z'])

      # end of nest level1
      @ns_map.end_element
    end
  end

  class TestXMLAttributeMapByXMLParser < RUNIT::TestCase
    def setup
      @attr_map = WPM::XMLAttributeMap.new
    end

    def test_attr
      @attr_map.add_attr('', '', 'id', '001')
      @attr_map.add_attr('test://foo', 'foo', 'name', 'foo')
      @attr_map.add_attr('test://bar', 'bar', 'name', 'bar')

      assert_equal('001', @attr_map['id'])
      assert_equal('001', @attr_map['', 'id'])
      assert_equal('foo', @attr_map['test://foo', 'name'])
      assert_equal('bar', @attr_map['test://bar', 'name'])
      assert_nil(@attr_map['nothing'])
      assert_nil(@attr_map['', 'nothing'])
      assert_nil(@attr_map['test://foo', 'nothing'])
      assert_nil(@attr_map['test://bar', 'nothing'])
      assert_nil(@attr_map['test://detarame', 'nothing'])

      attr_list = [
	# ns            prefix name    value
	[ '',           '',    'id',   '001' ],
	[ 'test://foo', 'foo', 'name', 'foo' ],
	[ 'test://bar', 'bar', 'name', 'bar' ]
      ]
      assert_equal(attr_list.sort, @attr_map.to_a.sort)
    end
  end

  class TestXMLReaderByXMLParser < RUNIT::TestCase
    def setup
      @reader = WPM::XMLReader.new
      @reader.extend(WPM::XMLAssistByXMLParser)
      @reader.instance_eval{ @alist = Array.new }
      def @reader.alist
	@alist
      end
      def @reader.start_element(ns_uri, prefix, name, attr_map)
	@alist.push([ :start_element, ns_uri, prefix, name, attr_map.to_a.sort ])
      end
      def @reader.end_element(ns_uri, prefix, name)
	@alist.push([ :end_element, ns_uri, prefix, name ])
      end
      def @reader.processing_instruction(target, data)
	@alist.push([ :pi, target, data ])
      end
      def @reader.character(data)
	if (! @alist.empty? && @alist[-1][0] == :text) then
	  @alist[-1][1].concat(data)
	else
	  @alist.push([ :text, data ])
	end
      end
      def @reader.comment(data)
	@alist.push([ :comment, data ])
      end
    end

    def test_parse
      filename = 'text.xml'
      begin
	File.open(filename, 'w') { |output|
	  output.print "<?xml version=\"1.0\"?>\n"
	  output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
	  output.print "<head>\n"
	  output.print "<title>test</title>\n"
	  output.print "</head>\n"
	  output.print "<body>\n"
	  output.print "<p>Hello world.</p>\n"
	  output.print "</body>\n"
	  output.print "</html>\n"
	}
	File.open(filename, 'r') { |input|
	  @reader.read(input)
	}
      ensure
	File.delete(filename)
      end

      xml_alist = [
	[ :start_element, 'http://www.w3.org/1999/xhtml', '', 'html', [] ],
	[ :text, "\n" ],
	[ :start_element, 'http://www.w3.org/1999/xhtml', '', 'head', [] ],
	[ :text, "\n" ],
	[ :start_element, 'http://www.w3.org/1999/xhtml', '', 'title', [] ],
	[ :text, 'test' ],
	[ :end_element, 'http://www.w3.org/1999/xhtml', '', 'title' ],
	[ :text, "\n" ],
	[ :end_element, 'http://www.w3.org/1999/xhtml', '', 'head' ],
	[ :text, "\n" ],
	[ :start_element, 'http://www.w3.org/1999/xhtml', '', 'body', [] ],
	[ :text, "\n" ],
	[ :start_element, 'http://www.w3.org/1999/xhtml', '', 'p', [] ],
	[ :text, "Hello world." ],
	[ :end_element, 'http://www.w3.org/1999/xhtml', '', 'p' ],
	[ :text, "\n" ],
	[ :end_element, 'http://www.w3.org/1999/xhtml', '', 'body' ],
	[ :text, "\n" ],
	[ :end_element, 'http://www.w3.org/1999/xhtml', '', 'html' ]
      ]
      assert_equal(xml_alist, @reader.alist)
    end
  end
end
