#!/usr/local/bin/ruby
# $Id: test_wpm_messg_manip.rb,v 1.1.1.1 2004/04/04 15:22:50 toki Exp $
# test of Message Manipulation

require 'rubyunit'
require 'wpm'

module TestWebPageMaker
  class TestMessageManip < RUNIT::TestCase
    def setup
      # for WPM::Driver class
      @handler_call = 0
      @env_call = 0
      @env = Hash.new
      @params_call = 0
      @params = Hash.new
      @header_call = 0
      @set_header_call = 0
      @header = Hash.new
      @write_call = 0
      @write_messg_body = ''
      @close_call = 0

      @messg_manip = WPM::MessageManipulator.new(self)
    end

    def handler
      @handler_call += 1
      self
    end

    def env
      @env_call += 1
      @env.dup
    end

    def params
      @params_call += 1
      @params
    end

    def header(name)
      @header_call += 1
      @header[name]
    end

    def set_header(name, value)
      @set_header_call += 1
      @header[name] = value
      nil
    end

    def write(messg_body)
      @write_call += 1
      @write_messg_body << messg_body
      nil
    end

    def close
      @close_call += 1
      nil
    end

    def test_handler
      assert_equal(self, @messg_manip.handler)
      assert_equal(1, @handler_call)
    end

    def test_env
      @env['CONTENT_TYPE'] = 'application/x-www-form-urlencoded'
      environ = @messg_manip.env
      assert_equal(1, @env_call)
      assert_equal(1, environ.size)
      assert_equal('application/x-www-form-urlencoded', environ['CONTENT_TYPE'])
    end

    def test_params
      @params['HALO'] = 'Hello world.'
      form_params = @messg_manip.params
      assert_equal(1, @params_call)
      assert_equal(1, form_params.size)
      assert_equal('Hello world.', @params['HALO'])
    end

    def test_header
      @header['Content-Type'] = 'text/html'
      assert_equal('text/html', @messg_manip.header('Content-Type'))
      assert_equal(1, @header_call)
    end

    def test_set_header
      @messg_manip.set_header('Content-Type', 'text/html')
      assert_equal(1, @set_header_call)
      assert_equal(1, @header.size)
      assert_equal('text/html', @header['Content-Type'])
    end

    def test_write
      @messg_manip.write("<html>\n")
      @messg_manip.write("<head><title>HALO</title></head>\n")
      @messg_manip.write("<body><p>Hello world.</p></body>\n")
      @messg_manip.write("</html>\n")
      assert_equal(4, @write_call)
      messg_body = "<html>\n"
      messg_body << "<head><title>HALO</title></head>\n"
      messg_body << "<body><p>Hello world.</p></body>\n"
      messg_body << "</html>\n"
      assert_equal(messg_body, @write_messg_body)
    end

    def test_close
      @messg_manip.close
      assert_equal(1, @close_call)
    end

    def test_status
      assert_equal(200, @messg_manip.status)
      assert_equal(1, @header_call)
      @header['Status'] = '404 Not Found'
      assert_equal(404, @messg_manip.status)
      assert_equal(2, @header_call)
    end

    def test_set_header
      @messg_manip.set_status(200)
      assert_equal(1, @set_header_call)
      assert_equal('200 OK', @header['Status'])
    end

    def test_location
      assert_nil(@messg_manip.location)
      assert_equal(1, @header_call)
      @header['Location'] = 'http://localhost/hello/world'
      assert_equal('http://localhost/hello/world', @messg_manip.location)
      assert_equal(2, @header_call)
    end

    def test_set_location
      @messg_manip.set_location('http://localhost/hello/world')
      assert_equal(2, @set_header_call)
      assert_equal(2, @header.size)
      assert_equal('302 Found', @header['Status'])
      assert_equal('http://localhost/hello/world', @header['Location'])
    end

    def test_set_location_with_status
      @messg_manip.set_location('http://localhost/hello/world', 301)
      assert_equal(2, @set_header_call)
      assert_equal(2, @header.size)
      assert_equal('301 Moved Permanently', @header['Status'])
      assert_equal('http://localhost/hello/world', @header['Location'])
    end

    def test_curr_page
      @env['PATH_INFO'] = '/this_page'
      assert_equal('this_page', @messg_manip.curr_page)
      assert_equal(1, @env_call)
    end

    def test_curr_page2
      @env['PATH_INFO'] = '/this_page/path/info'
      assert_equal('this_page', @messg_manip.curr_page)
      assert_equal(1, @env_call)
    end

    def test_path_info
      @env['PATH_INFO'] = '/this_page/path/info'
      assert_equal('/path/info', @messg_manip.path_info)
      assert_equal(1, @env_call)
    end

    def test_path_info_with_empty_string
      @env['PATH_INFO'] = '/this_page'
      assert_equal('', @messg_manip.path_info)
      assert_equal(1, @env_call)
    end

    def test_path_info_with_nil
      assert_equal('', @messg_manip.path_info)
      assert_equal(1, @env_call)
    end

    def test_page_path
      @env['SCRIPT_NAME'] = '/script.cgi'
      @env['PATH_INFO'] = '/this_page/path/info'
      assert_equal('/script.cgi/this_page/path/info', @messg_manip.page_path)
      assert_equal(3, @env_call)
    end

    def test_page_path_with_page_name
      @env['SCRIPT_NAME'] = '/script.cgi'
      @env['PATH_INFO'] = '/this_page/path/info'
      assert_equal('/script.cgi/that_page/path/info', @messg_manip.page_path('that_page'))
      assert_equal(2, @env_call)
    end

    def test_page_with_page_name_and_path_info
      @env['SCRIPT_NAME'] = '/script.cgi'
      @env['PATH_INFO'] = '/this_page/path/info'
      assert_equal('/script.cgi/that_page/foo/bar', @messg_manip.page_path('that_page', '/foo/bar'))
      assert_equal(1, @env_call)
    end

    def test_redirect
      @env['SERVER_NAME'] = 'server_name'
      @env['SERVER_PORT'] = '8888'
      @env['SCRIPT_NAME'] = '/script.cgi'
      @env['PATH_INFO'] = '/this_page/path/info'
      @messg_manip.redirect('that_page')
      assert_equal(3, @env_call)
      assert_equal(2, @set_header_call)
      assert_equal(2, @header.size)
      assert_equal('303 See Other', @header['Status'])
      assert_equal('http://server_name:8888/script.cgi/that_page/path/info', @header['Location'])
    end

    def test_redirect_with_query
      @env['SERVER_NAME'] = 'server_name'
      @env['SERVER_PORT'] = '8888'
      @env['SCRIPT_NAME'] = '/script.cgi'
      @env['PATH_INFO'] = '/this_page/path/info'
      @messg_manip.redirect('that_page', { 'HALO' => 'Hello world.' })
      assert_equal(3, @env_call)
      assert_equal(2, @set_header_call)
      assert_equal(2, @header.size)
      assert_equal('303 See Other', @header['Status'])
      assert_equal('http://server_name:8888/script.cgi/that_page/path/info?HALO=Hello+world.', @header['Location'])
    end

    def test_redirect_with_query2
      @env['SERVER_NAME'] = 'server_name'
      @env['SERVER_PORT'] = '8888'
      @env['SCRIPT_NAME'] = '/script.cgi'
      @env['PATH_INFO'] = '/this_page/path/info'
      @messg_manip.redirect('that_page', [ %w[ foo 1 ], %w[ bar 2 ]])
      assert_equal(3, @env_call)
      assert_equal(2, @set_header_call)
      assert_equal(2, @header.size)
      assert_equal('303 See Other', @header['Status'])
      assert_equal('http://server_name:8888/script.cgi/that_page/path/info?foo=1&bar=2', @header['Location'])
    end
  end
end
