#!/usr/local/bin/ruby
# $Id: test_server.rb,v 1.4 2004/10/02 01:00:25 toki Exp $

require 'rubyunit'
require 'thread'
require 'socket'
require 'sockutils'
require 'rucy/logger'
require 'rucy/server'
require 'rucy/document'
require 'rucy/response'

Thread.abort_on_exception = true

module TestRucy
  class TestMultiThreadServer < RUNIT::TestCase
    include SocketUtils

    SERVER_PORT = 9876

    def setup
      @restart_called = 0
      @server_open_call = 0
      @server_close_call = 0
      @document_open_call = 0
      @document_close_call = 0
      @server = Rucy::Server.new
      @server.open_hook{ |s|
	@server_open_call += 1
	s.bind_address = 'localhost'
	s.port = SERVER_PORT
	s.mount(Rucy::Page.new("<html>Hello world.</html>\n"), '/')
	s.mount(self, '/test_document_open_close')
	# logger = Rucy::Logger.new(STDERR)
	# logger.log_debug = true
	# s.add_logger(logger)
      }
      @server.close_hook{ |s|
	@server_close_call += 1
      }
      @thread = Thread.new{
	@server.accept
      }
      sleep(0.1)
    end

    def open
      @document_open_call += 1
      nil
    end

    def close
      @document_close_call += 1
      nil
    end

    def each(traversed={})
      yield(self)
      nil
    end

    def teardown
      @server.close
      @thread.join
      assert_equal(1 + @restart_called, @server_open_call)
      assert_equal(1 + @restart_called, @server_close_call)
      assert_equal(1 + @restart_called, @document_open_call)
      assert_equal(1 + @restart_called, @document_close_call)
    end

    def test_restart
      @server.restart
      @restart_called += 1
      sleep(1)
    end

    def test_GET
      socket = TCPSocket.new('localhost', SERVER_PORT)
      setsockopt(socket)
      begin
	socket.print "GET / HTTP/1.1\r\n"
	socket.print "Host: localhost:#{SERVER_PORT}\r\n"
	socket.print "\r\n"
	socket.flush

	response = Rucy::Response.new
	response.parse(socket)
	assert_equal('HTTP/1.1 200 OK', response.line)
	assert(! response.headers('Connection').find{ |v| v =~ /close/i })
	assert_equal('26', response.header('Content-Length'))
	assert_equal('text/html', response.header('Content-Type'))
	assert_match(response.header('Date'), /^(Mon|Tue|Wed|Thu|Fri|Sat|Sun), \d{2} (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) \d{4} \d{2}:\d{2}:\d{2} GMT$/)
	assert_equal(Rucy::SERVER_TOKEN_LIST, response.header('Server'))
	assert_equal("<html>Hello world.</html>\n", socket.read(26))

	socket.print "GET / HTTP/1.1\r\n"
	socket.print "Host: localhost:#{SERVER_PORT}\r\n"
	socket.print "Connection: close\r\n"
	socket.print "\r\n"
	socket.flush

	response = Rucy::Response.new
	response.parse(socket)
	assert_equal('HTTP/1.1 200 OK', response.line)
	assert_equal('close', response.header('Connection'))
	assert_equal('26', response.header('Content-Length'))
	assert_equal('text/html', response.header('Content-Type'))
	assert_match(response.header('Date'), /^(Mon|Tue|Wed|Thu|Fri|Sat|Sun), \d{2} (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) \d{4} \d{2}:\d{2}:\d{2} GMT$/)
	assert_equal(Rucy::SERVER_TOKEN_LIST, response.header('Server'))
	assert_equal("<html>Hello world.</html>\n", socket.read(26))
      ensure
	socket.close
      end
    end

    def make_access_thread(nreqs, wait)
      Thread.new{
	socket = TCPSocket.new('localhost', SERVER_PORT)
	setsockopt(socket)
	wait.call
	begin
	  (nreqs - 1).times do
	    socket.print "GET / HTTP/1.1\r\n"
	    socket.print "Host: localhost:#{SERVER_PORT}\r\n"
	    socket.print "\r\n"

	    response = Rucy::Response.new
	    response.parse(socket)
	    assert_equal('HTTP/1.1 200 OK', response.line)
	    assert(! response.headers('Connection').find{ |v| v =~ /close/i })
	    assert_equal('26', response.header('Content-Length'))
	    assert_equal('text/html', response.header('Content-Type'))
	    assert_match(response.header('Date'), /^(Mon|Tue|Wed|Thu|Fri|Sat|Sun), \d{2} (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) \d{4} \d{2}:\d{2}:\d{2} GMT$/)
	    assert_equal(Rucy::SERVER_TOKEN_LIST, response.header('Server'))
	    assert_equal("<html>Hello world.</html>\n", socket.read(26))
	  end

	  socket.print "GET / HTTP/1.1\r\n"
	  socket.print "Host: localhost:#{SERVER_PORT}\r\n"
	  socket.print "Connection: close\r\n"
	  socket.print "\r\n"

	  response = Rucy::Response.new
	  response.parse(socket)
	  assert_equal('HTTP/1.1 200 OK', response.line)
	  assert_equal('close', response.header('Connection'))
	  assert_equal('26', response.header('Content-Length'))
	  assert_equal('text/html', response.header('Content-Type'))
	  assert_match(response.header('Date'), /^(Mon|Tue|Wed|Thu|Fri|Sat|Sun), \d{2} (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) \d{4} \d{2}:\d{2}:\d{2} GMT$/)
	  assert_equal(Rucy::SERVER_TOKEN_LIST, response.header('Server'))
	  assert_equal("<html>Hello world.</html>\n", socket.read(26))
	ensure
	  socket.close
	end
      }
    end
    private :make_access_thread

    def test_multi_access
      start = false
      lock = Mutex.new
      cond = ConditionVariable.new
      wait = proc{
	lock.synchronize{
	  until (start)
	    cond.wait(lock)
	  end
	}
      }

      th_grp = ThreadGroup.new
      nconns = 16; nreqs = 32
      nconns.times do
	th_grp.add(make_access_thread(nreqs, wait))
      end

      lock.synchronize{
	start = true
	cond.broadcast
      }

      for th in th_grp.list
	th.join
      end
    end
  end

  class TestMultiThreadServerTimeout < RUNIT::TestCase
    include SocketUtils

    SERVER_PORT = 9876

    def setup
      @server_open_call = 0
      @server_close_call = 0
      @document_open_call = 0
      @document_close_call = 0
      @server = Rucy::Server.new
      @server.open_hook{ |s|
	@server_open_call += 1
	s.bind_address = 'localhost'
	s.port = SERVER_PORT
	s.timeout = 0.01	# short timeout for test
	s.mount(Rucy::Page.new("<html>Hello world.</html>\n"), '/')
	s.mount(self, '/test_document_open_close')
	# logger = Rucy::Logger.new(STDERR)
	# logger.log_debug = true
	# s.add_logger(logger)
      }
      @server.close_hook{ |s|
	@server_close_call += 1
      }
      @thread = Thread.new{
	@server.accept
      }
      sleep(0.1)
    end

    def open
      @document_open_call += 1
      nil
    end

    def close
      @document_close_call += 1
      nil
    end

    def each(traversed={})
      yield(self)
      nil
    end

    def teardown
      @server.close
      @thread.join
      assert_equal(1, @server_open_call)
      assert_equal(1, @server_close_call)
      assert_equal(1, @document_open_call)
      assert_equal(1, @document_close_call)
    end

    def test_timeout
      socket = TCPSocket.new('localhost', SERVER_PORT)
      setsockopt(socket)
      begin
	sleep(0.1)		# wait for timeout
	assert_nil(socket.gets)
      ensure
	socket.close
      end
    end
  end
end
