#!/usr/local/bin/ruby
# $Id: test_redirect.rb,v 1.3 2004/09/29 15:14:01 toki Exp $

require 'rubyunit'
require 'pseudo_req_res'
require 'rucy/loader'

module TestRucy
  class TestRedirectDocument < RUNIT::TestCase
    include PseudoLogger
    include PseudoRequestResponse

    def setup
      super
      @loader = Rucy::SimpleLoader.new
      @loader.load('../mod_docs/redirect.rb')
      @RedirectDocument = @loader.const_get('RedirectDocument')
    end

    def test_URL
      redirect = @RedirectDocument.new('http://www.foo.bar/index.html')
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      redirect.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(302, @messg_head.status)
      assert_equal('Found', @messg_head.reason)
      assert_equal('http://www.foo.bar/index.html', @messg_head.header('Location'))
      assert((@messg_head.has_header? 'Content-Type'))
      assert(! @messg_body.empty?)
    end

    def test_URL_with_status
      redirect = @RedirectDocument.new('http://www.foo.bar/index.html', 307)
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      redirect.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(307, @messg_head.status)
      assert_equal('Temporary Redirect', @messg_head.reason)
      assert_equal('http://www.foo.bar/index.html', @messg_head.header('Location'))
      assert((@messg_head.has_header? 'Content-Type'))
      assert(! @messg_body.empty?)
    end

    def test_URL_and_HEAD
      redirect = @RedirectDocument.new('http://www.foo.bar/index.html')
      @request.method = 'HEAD'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      redirect.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(302, @messg_head.status)
      assert_equal('Found', @messg_head.reason)
      assert_equal('http://www.foo.bar/index.html', @messg_head.header('Location'))
      assert((@messg_head.has_header? 'Content-Type'))
      assert(@messg_body.empty?)
    end

    def test_URL_and_method_not_allowed
      redirect = @RedirectDocument.new('http://www.foo.bar/index.html')
      @request.method = 'DELETE'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      done = false
      begin
	redirect.publish('', @request, @response, self)
      rescue Rucy::HTTPError
	assert_equal(405, $!.status) #  Method Not Allowed
	assert_equal('GET, HEAD', $!.header('Allow'))
	done = true
      end
      assert(done)
      assert(@messg_body.empty?)
    end

    def test_rel_URL_abs_path
      redirect = @RedirectDocument.new('/foo/bar/index.html')
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      redirect.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(302, @messg_head.status)
      assert_equal('Found', @messg_head.reason)
      assert_equal('http://server:8080/foo/bar/index.html', @messg_head.header('Location'))
      assert((@messg_head.has_header? 'Content-Type'))
      assert(! @messg_body.empty?)
    end
  end

  class TestRedirectDocumentFactory < RUNIT::TestCase
    def setup
      @loader = Rucy::SimpleLoader.new
      @loader.load('../mod_docs/redirect.rb')
      @RedirectDocumentFactory = @loader.const_get('RedirectDocumentFactory')
      @factory = @RedirectDocumentFactory.instance
    end

    def test_set_doc_option
      @factory.set_doc_option(Hash.new)
    end

    def test_doc_name
      assert_equal('Redirect', @factory.doc_name)
    end

    def test_doc_args
      args = @factory.doc_args
      assert_equal([ 'location', :string, nil ], args[0])
      assert_equal([ 'status', :select, [
		       '302 Found',
		       '303 See Other',
		       '301 Moved Permanently',
		       '307 Temporary Redirect'
		     ]
		   ], args[1])
    end

    def test_new
      doc = @factory.new('/foo', '302 Found')
      assert_instance_of(@loader.const_get('RedirectDocument'), doc)
    end
  end

  class TestRegexpRedirectDocument < RUNIT::TestCase
    include PseudoLogger
    include PseudoRequestResponse

    def setup
      super
      @loader = Rucy::SimpleLoader.new
      @loader.load('../mod_docs/redirect.rb')
      @RegexpRedirectDocument = @loader.const_get('RegexpRedirectDocument')
    end

    def test_URL
      redirect = @RegexpRedirectDocument.new(%r"^.*/bar*?$", 'http://www.foo.bar/index.html')
      @request.method = 'GET'
      @request.path = '/foo/bar'
      @request.version = 'HTTP/1.1'
      redirect.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(302, @messg_head.status)
      assert_equal('Found', @messg_head.reason)
      assert_equal('http://www.foo.bar/index.html', @messg_head.header('Location'))
      assert((@messg_head.has_header? 'Content-Type'))
      assert(! @messg_body.empty?)
    end

    def test_URL_with_replacing
      redirect = @RegexpRedirectDocument.new(%r"^.*/bar/(.*)$", 'http://www.foo.bar/\1/index.html')
      @request.method = 'GET'
      @request.path = '/foo/bar/baz'
      @request.version = 'HTTP/1.1'
      redirect.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(302, @messg_head.status)
      assert_equal('Found', @messg_head.reason)
      assert_equal('http://www.foo.bar/baz/index.html', @messg_head.header('Location'))
      assert((@messg_head.has_header? 'Content-Type'))
      assert(! @messg_body.empty?)
    end

    def test_URL_with_query
      redirect = @RegexpRedirectDocument.new(%r"^.*/bar/(.*)\?(.*)?$", 'http://www.foo.bar/\1/index.html?\2')
      @request.method = 'GET'
      @request.path = '/foo/bar/baz'
      @request.query = 'alpha=1&beta=2'
      @request.version = 'HTTP/1.1'
      redirect.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(302, @messg_head.status)
      assert_equal('Found', @messg_head.reason)
      assert_equal('http://www.foo.bar/baz/index.html?alpha=1&beta=2', @messg_head.header('Location'))
      assert((@messg_head.has_header? 'Content-Type'))
      assert(! @messg_body.empty?)
    end

    def test_URL_with_status
      redirect = @RegexpRedirectDocument.new(%r"^.*/bar*?$", 'http://www.foo.bar/index.html', 307)
      @request.method = 'GET'
      @request.path = '/foo/bar'
      @request.version = 'HTTP/1.1'
      redirect.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(307, @messg_head.status)
      assert_equal('Temporary Redirect', @messg_head.reason)
      assert_equal('http://www.foo.bar/index.html', @messg_head.header('Location'))
      assert((@messg_head.has_header? 'Content-Type'))
      assert(! @messg_body.empty?)
    end

    def test_URL_and_HEAD
      redirect = @RegexpRedirectDocument.new(%r"^.*/bar.*?$", 'http://www.foo.bar/index.html')
      @request.method = 'HEAD'
      @request.path = '/foo/bar'
      @request.version = 'HTTP/1.1'
      redirect.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(302, @messg_head.status)
      assert_equal('Found', @messg_head.reason)
      assert_equal('http://www.foo.bar/index.html', @messg_head.header('Location'))
      assert((@messg_head.has_header? 'Content-Type'))
      assert(@messg_body.empty?)
    end

    def test_URL_and_not_found
      redirect = @RegexpRedirectDocument.new(%r"^.*/bar*?$", 'http://www.foo.bar/index.html')
      @request.method = 'GET'
      @request.path = '/baz'
      @request.version = 'HTTP/1.1'
      done = false
      begin
	redirect.publish('', @request, @response, self)
      rescue Rucy::HTTPError
	assert_equal(404, $!.status) # Not Found
	done = true
      end
      assert(done)
      assert(@messg_body.empty?)
    end

    def test_URL_and_method_not_allowed
      redirect = @RegexpRedirectDocument.new(%r"^.*/bar*?$", 'http://www.foo.bar/index.html')
      @request.method = 'DELETE'
      @request.path = '/foo/bar'
      @request.version = 'HTTP/1.1'
      done = false
      begin
	redirect.publish('', @request, @response, self)
      rescue Rucy::HTTPError
	assert_equal(405, $!.status) # Method Not Allowed
	assert_equal('GET, HEAD', $!.header('Allow'))
	done = true
      end
      assert(done)
      assert(@messg_body.empty?)
    end

    def test_rel_URL_abs_path
      redirect = @RegexpRedirectDocument.new(%r"^.*/bar*?$", '/index.html')
      @request.method = 'GET'
      @request.path = '/foo/bar'
      @request.version = 'HTTP/1.1'
      redirect.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(302, @messg_head.status)
      assert_equal('Found', @messg_head.reason)
      assert_equal('http://server:8080/index.html', @messg_head.header('Location'))
      assert((@messg_head.has_header? 'Content-Type'))
      assert(! @messg_body.empty?)
    end
  end

  class TestRegexpRedirectDocumentFactory < RUNIT::TestCase
    def setup
      @loader = Rucy::SimpleLoader.new
      @loader.load('../mod_docs/redirect.rb')
      @RegexpRedirectDocumentFactory = @loader.const_get('RegexpRedirectDocumentFactory')
      @factory = @RegexpRedirectDocumentFactory.instance
    end

    def test_set_doc_option
      @factory.set_doc_option(Hash.new)
    end

    def test_doc_name
      assert_equal('RegexpRedirect', @factory.doc_name)
    end

    def test_doc_args
      args = @factory.doc_args
      assert_equal([ 'request path pattern (regexp)', :regexp, nil ], args[0])
      assert_equal([ %q"replace location (usable \&, \0, \1 ... \9, \`, \', \+)", :string, nil ], args[1])
      assert_equal([ 'status', :select, [
		       '302 Found',
		       '303 See Other',
		       '301 Moved Permanently',
		       '307 Temporary Redirect'
		     ]
		   ], args[2])
    end

    def test_new
      doc = @factory.new(%r'/foo(.*?)\.html$', '\`/bar\1.html', '302 Found')
      assert_instance_of(@loader.const_get('RegexpRedirectDocument'), doc)
    end
  end
end
