#!/usr/local/bin/ruby
# $Id: test_params.rb,v 1.43 2004/10/20 13:05:42 toki Exp $

require 'rubyunit'
require 'pseudo_props'
require 'rucy/params'
require 'rucy/properties'

module TestRucy
  class TestAdminParams < RUNIT::TestCase
    include PseudoProperties

    def setup
      super

      # test target
      @admin_params = Rucy::AdminParams.new
    end

    def test_modified_count
      @admin_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(0, @admin_params.modified_count)
      @admin_params.modified_count = 1
      assert_equal(1, @admin_params.modified_count)
      assert_equal(0, @modified_count)
      @admin_params.save(self)
      assert_equal(1, @set_params_call)
      assert_equal(1, @modified_count)
    end

    def test_load_default_settings
      @admin_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(%w[ admin_user admin_password localhost_only japanese_handling ].sort,
		   @params_names.sort)
      assert_equal('admin', @admin_params.admin_user)
      assert_equal('', @admin_params.admin_password)
      assert_equal(true, @admin_params.localhost_only)
      assert_equal(false, @admin_params.japanese_handling)
    end

    def test_load_admin_user
      @store['admin_user'] = 'alice'
      @admin_params.load(self)
      assert_equal(1, @params_call)
      assert((@params_names.include? 'admin_user'))
      assert_equal('alice', @admin_params.admin_user)
    end

    def test_load_admin_password
      @store['admin_password'] = 'himitsu'
      @admin_params.load(self)
      assert_equal(1, @params_call)
      assert((@params_names.include? 'admin_password'))
      assert_equal('himitsu', @admin_params.admin_password)
    end

    def test_load_localhost_only
      @store['localhost_only'] = false
      @admin_params.load(self)
      assert_equal(1, @params_call)
      assert((@params_names.include? 'localhost_only'))
      assert_equal(false, @admin_params.localhost_only)
    end

    def test_load_japanese_handling
      @store['japanese_handling'] = true
      @admin_params.load(self)
      assert_equal(1, @params_call)
      assert((@params_names.include? 'japanese_handling'))
      assert_equal(true, @admin_params.japanese_handling)
    end

    def test_save_admin_user
      @admin_params.load(self)
      assert_equal(1, @params_call)
      assert_equal('admin', @admin_params.admin_user)
      @admin_params.admin_user = 'alice'
      assert_equal('alice', @admin_params.admin_user)
      @admin_params.save(self)
      assert_equal(1, @set_params_call)
      assert((@set_params_names.include? 'admin_user'))
      assert_equal('alice', @store['admin_user'])
    end

    def test_save_admin_password
      @admin_params.load(self)
      assert_equal(1, @params_call)
      assert_equal('', @admin_params.admin_password)
      @admin_params.admin_password = 'himitsu'
      assert_equal('himitsu', @admin_params.admin_password)
      @admin_params.save(self)
      assert_equal(1, @set_params_call)
      assert((@set_params_names.include? 'admin_password'))
      assert_equal('himitsu', @store['admin_password'])
    end

    def test_save_localhost_only
      @store['admin_password'] = 'himitsu'
      @admin_params.load(self)
      assert_equal(1, @params_call)
      assert(! @admin_params.admin_password.empty?)
      assert_equal(true, @admin_params.localhost_only)
      @admin_params.localhost_only = false
      assert_equal(false, @admin_params.localhost_only)
      @admin_params.save(self)
      assert_equal(1, @set_params_call)
      assert((@set_params_names.include? 'localhost_only'))
      assert_equal(false, @store['localhost_only'])
    end

    def test_save_japanese_handling
      @admin_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(false, @admin_params.japanese_handling)
      @admin_params.japanese_handling = true
      assert_equal(true, @admin_params.japanese_handling)
      @admin_params.save(self)
      assert_equal(1, @set_params_call)
      assert((@set_params_names.include? 'japanese_handling'))
      assert_equal(true, @store['japanese_handling'])
    end

    def test_null_admin_user
      @admin_params.load(self)
      assert_equal(1, @params_call)
      assert_equal('admin', @admin_params.admin_user)
      @admin_params.admin_user = nil
      assert_equal('admin', @admin_params.admin_user)
      @admin_params.admin_user = ' '
      assert_equal('admin', @admin_params.admin_user)
    end

    def test_admin_user_with_white_space
      @admin_params.load(self)
      assert_equal(1, @params_call)
      assert_equal('admin', @admin_params.admin_user)
      @admin_params.admin_user = ' alice '
      assert_equal('alice', @admin_params.admin_user)
    end

    def test_null_admin_password
      @store['admin_password'] = 'himitsu'
      @admin_params.load(self)
      assert_equal(1, @params_call)
      assert_equal('himitsu', @admin_params.admin_password)
      @admin_params.admin_password = nil
      assert_equal('', @admin_params.admin_password)
    end

    def test_no_effect_network_access
      @admin_params.load(self)
      assert_equal(1, @params_call)
      assert(@admin_params.admin_password.empty?)
      assert_equal(true, @admin_params.localhost_only)
      @admin_params.localhost_only = false
      assert_equal(true, @admin_params.localhost_only)
    end
  end

  class TestServerParams < RUNIT::TestCase
    include PseudoProperties

    def setup
      super

      # test target
      @server_params = Rucy::ServerParams.new
    end

    def test_modified_count
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(0, @server_params.modified_count)
      @server_params.modified_count = 1
      assert_equal(1, @server_params.modified_count)
      assert_equal(0, @modified_count)
      @server_params.save(self)
      assert_equal(1, @set_params_call)
      assert_equal(1, @modified_count)
    end

    def test_load_default_settings
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal([ 'server_type',
		     'port',
		     'bind_address',
		     'timeout',
		     'keep_alive',
		     'max_requests',
		     'queue_length',
		     'messengers',
		     'messenger_threads',
		     'messenger_queue_length',
		     'subprocess_user',
		     'subprocess_group',
		     'do_not_reverse_lookup'
                   ].sort,
		   @params_names.sort)
      assert_equal('multithread', @server_params.server_type)
      assert_equal(8888, @server_params.port)
      assert_equal(nil, @server_params.bind_address)
      assert_equal(300, @server_params.timeout)
      assert_equal(8, @server_params.keep_alive)
      assert_equal(32, @server_params.max_requests)
      assert_equal(16, @server_params.queue_length)
      assert_equal(8, @server_params.messengers)
      assert_equal(4, @server_params.messenger_threads)
      assert_equal(4, @server_params.messenger_queue_length)
      assert_equal('nobody', @server_params.subprocess_user)
      assert_equal('nobody', @server_params.subprocess_group)
      assert_equal(false, @server_params.do_not_reverse_lookup)
    end

    def test_load_server_type
      @store['server_type'] = 'multiprocess'
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert((@params_names.include? 'server_type'))
      assert_equal('multiprocess', @server_params.server_type)
    end

    def test_load_port
      @store['port'] = 80
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert((@params_names.include? 'port'))
      assert_equal(80, @server_params.port)
    end

    def test_load_bind_address
      @store['bind_address'] = 'localhost'
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert((@params_names.include? 'bind_address'))
      assert_equal('localhost', @server_params.bind_address)
    end

    def test_load_timeout
      @store['timeout'] = 60
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert((@params_names.include? 'timeout'))
      assert_equal(60, @server_params.timeout)
    end

    def test_load_keep_alive
      @store['keep_alive'] = 10
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert((@params_names.include? 'keep_alive'))
      assert_equal(10, @server_params.keep_alive)
    end

    def test_load_max_requests
      @store['max_requests'] = 256
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert((@params_names.include? 'max_requests'))
      assert_equal(256, @server_params.max_requests)
    end

    def test_load_queue_length
      @store['queue_length'] = 64
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert((@params_names.include? 'queue_length'))
      assert_equal(64, @server_params.queue_length)
    end

    def test_load_messengers
      @store['messengers'] = 128
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert((@params_names.include? 'messengers'))
      assert_equal(128, @server_params.messengers)
    end

    def test_load_messenger_threads
      @store['messenger_threads'] = 512
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert((@params_names.include? 'messenger_threads'))
      assert_equal(512, @server_params.messenger_threads)
    end

    def test_load_messenger_queue_length
      @store['messenger_queue_length'] = 256
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert((@params_names.include? 'messenger_queue_length'))
      assert_equal(256, @server_params.messenger_queue_length)
    end

    def test_load_subprocess_user
      @store['subprocess_user'] = 'httpd'
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert((@params_names.include? 'subprocess_user'))
      assert_equal('httpd', @server_params.subprocess_user)
    end

    def test_load_subprocess_group
      @store['subprocess_group'] = 'httpd'
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert((@params_names.include? 'subprocess_group'))
      assert_equal('httpd', @server_params.subprocess_group)
    end

    def test_load_do_not_reverse_lookup
      @store['do_not_reverse_lookup'] = true
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert((@params_names.include? 'do_not_reverse_lookup'))
      assert_equal(true, @server_params.do_not_reverse_lookup)
    end

    def test_save_server_type
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal('multithread', @server_params.server_type)
      @server_params.server_type = 'multiprocess'
      assert_equal('multiprocess', @server_params.server_type)
      @server_params.save(self)
      assert((@set_params_names.include? 'server_type'))
      assert_equal('multiprocess', @store['server_type'])
    end

    def test_save_port
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(8888, @server_params.port)
      @server_params.port = 80
      assert_equal(80, @server_params.port)
      @server_params.save(self)
      assert((@set_params_names.include? 'port'))
      assert_equal(80, @store['port'])
    end

    def test_save_bind_address
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(nil, @server_params.bind_address)
      @server_params.bind_address = 'localhost'
      assert_equal('localhost', @server_params.bind_address)
      @server_params.save(self)
      assert((@set_params_names.include? 'bind_address'))
      assert_equal('localhost', @store['bind_address'])
    end

    def test_save_timeout
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(300, @server_params.timeout)
      @server_params.timeout = 60
      assert_equal(60, @server_params.timeout)
      @server_params.save(self)
      assert((@set_params_names.include? 'timeout'))
      assert_equal(60, @store['timeout'])
    end

    def test_save_keep_alive
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(8, @server_params.keep_alive)
      @server_params.keep_alive = 30
      assert_equal(30, @server_params.keep_alive)
      @server_params.save(self)
      assert((@set_params_names.include? 'keep_alive'))
      assert_equal(30, @store['keep_alive'])
    end

    def test_save_max_requests
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(32, @server_params.max_requests)
      @server_params.max_requests = 256
      assert_equal(256, @server_params.max_requests)
      @server_params.save(self)
      assert((@set_params_names.include? 'max_requests'))
      assert_equal(256, @store['max_requests'])
    end

    def test_save_queue_length
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(16, @server_params.queue_length)
      @server_params.queue_length = 64
      assert_equal(64, @server_params.queue_length)
      @server_params.save(self)
      assert((@set_params_names.include? 'queue_length'))
      assert_equal(64, @store['queue_length'])
    end

    def test_save_messengers
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(8, @server_params.messengers)
      @server_params.messengers = 16
      assert_equal(16, @server_params.messengers)
      @server_params.save(self)
      assert((@set_params_names.include? 'messengers'))
      assert_equal(16, @store['messengers'])
    end

    def test_save_messenger_threads
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(4, @server_params.messenger_threads)
      @server_params.messenger_threads = 32
      assert_equal(32, @server_params.messenger_threads)
      @server_params.save(self)
      assert((@set_params_names.include? 'messenger_threads'))
      assert_equal(32, @store['messenger_threads'])
    end

    def test_save_messenger_queue_length
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(4, @server_params.messenger_queue_length)
      @server_params.messenger_queue_length = 16
      assert_equal(16, @server_params.messenger_queue_length)
      @server_params.save(self)
      assert((@set_params_names.include? 'messenger_queue_length'))
      assert_equal(16, @store['messenger_queue_length'])
    end

    def test_save_subprocess_user
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal('nobody', @server_params.subprocess_user)
      @server_params.subprocess_user = 'httpd'
      assert_equal('httpd', @server_params.subprocess_user)
      @server_params.save(self)
      assert((@set_params_names.include? 'subprocess_user'))
      assert_equal('httpd', @store['subprocess_user'])
    end

    def test_save_subprocess_group
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal('nobody', @server_params.subprocess_group)
      @server_params.subprocess_group = 'httpd'
      assert_equal('httpd', @server_params.subprocess_group)
      @server_params.save(self)
      assert((@set_params_names.include? 'subprocess_group'))
      assert_equal('httpd', @store['subprocess_group'])
    end

    def test_save_do_not_reverse_lookup
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(false, @server_params.do_not_reverse_lookup)
      @server_params.do_not_reverse_lookup = true
      assert_equal(true, @server_params.do_not_reverse_lookup)
      @server_params.save(self)
      assert((@set_params_names.include? 'do_not_reverse_lookup'))
      assert_equal(true, @store['do_not_reverse_lookup'])
    end

    def test_invalid_server_type
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal('multithread', @server_params.server_type)
      assert_exception(RuntimeError) {
	@server_params.server_type = 'foo'
      }
      assert_equal('multithread', @server_params.server_type)
    end

    def test_invalid_port_number
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(8888, @server_params.port)
      assert_exception(RuntimeError) {
	@server_params.port = -1
      }
      assert_equal(8888, @server_params.port)
    end

    def test_empty_bind_address
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(nil, @server_params.bind_address)
      @server_params.bind_address = ''
      assert_equal(nil, @server_params.bind_address)
      @server_params.bind_address = ' '
      assert_equal(nil, @server_params.bind_address)
    end

    def test_invalid_timeout
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(300, @server_params.timeout)
      assert_exception(RuntimeError) {
	@server_params.timeout = 0
      }
      assert_exception(RuntimeError) {
	@server_params.timeout = -1
      }
      assert_equal(300, @server_params.timeout)
    end

    def test_invalid_keep_alive
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(8, @server_params.keep_alive)
      assert_exception(RuntimeError) {
	@server_params.keep_alive = -1
      }
      assert_equal(8, @server_params.keep_alive)
    end

    def test_invalid_max_requests
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(32, @server_params.max_requests)
      assert_exception(RuntimeError) {
	@server_params.max_requests = 0
      }
      assert_exception(RuntimeError) {
	@server_params.max_requests = -1
      }
      assert_equal(32, @server_params.max_requests)
    end

    def test_invalid_queue_length
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(16, @server_params.queue_length)
      assert_exception(RuntimeError) {
	@server_params.queue_length = 0
      }
      assert_exception(RuntimeError) {
	@server_params.queue_length = -1
      }
      assert_equal(16, @server_params.queue_length)
    end

    def test_invalid_messengers
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(8, @server_params.messengers)
      assert_exception(RuntimeError) {
	@server_params.messengers = 0
      }
      assert_exception(RuntimeError) {
	@server_params.messengers = -1
      }
      assert_equal(8, @server_params.messengers)
    end

    def test_invalid_messenger_threads
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(4, @server_params.messenger_threads)
      assert_exception(RuntimeError) {
	@server_params.messenger_threads = 0
      }
      assert_exception(RuntimeError) {
	@server_params.messenger_threads = -1
      }
      assert_equal(4, @server_params.messenger_threads)
    end

    def test_invalid_messenger_queue_length
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(4, @server_params.messenger_queue_length)
      assert_exception(RuntimeError) {
	@server_params.messenger_queue_length = 0
      }
      assert_exception(RuntimeError) {
	@server_params.messenger_queue_length = -1
      }
      assert_equal(4, @server_params.messenger_queue_length)
    end

    def test_invalid_subprocess_user
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal('nobody', @server_params.subprocess_user)
      assert_exception(RuntimeError) {
	@server_params.subprocess_user = -1
      }
      assert_exception(RuntimeError) {
	@server_params.subprocess_user = :nobody
      }
      assert_equal('nobody', @server_params.subprocess_user)
    end

    def test_invalid_subprocess_group
      @server_params.load(self)
      assert_equal(1, @params_call)
      assert_equal('nobody', @server_params.subprocess_group)
      assert_exception(RuntimeError) {
	@server_params.subprocess_group = -1
      }
      assert_exception(RuntimeError) {
	@server_params.subprocess_group = :nobody
      }
      assert_equal('nobody', @server_params.subprocess_group)
    end
  end

  class TestLoggingParams < RUNIT::TestCase
    include PseudoProperties

    def setup
      super

      # test target
      @logging_params = Rucy::LoggingParams.new
    end

    def test_modified_count
      @logging_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(0, @logging_params.modified_count)
      @logging_params.modified_count = 1
      assert_equal(1, @logging_params.modified_count)
      assert_equal(0, @modified_count)
      @logging_params.save(self)
      assert_equal(1, @set_params_call)
      assert_equal(1, @modified_count)
    end

    def test_load_default_settings
      @logging_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(%w[ stdout_logging_level logfiles ].sort, @params_names.sort)
      assert_equal('notice', @logging_params.stdout_logging_level)
      assert([], @logging_params.logfiles)
    end

    def test_load_stdout_logging_level
      @store['stdout_logging_level'] = 'debug'
      @logging_params.load(self)
      assert_equal(1, @params_call)
      assert((@params_names.include? 'stdout_logging_level'))
      assert_equal('debug', @logging_params.stdout_logging_level)
    end

    def test_save_stdout_logging_level
      @logging_params.load(self)
      assert_equal(1, @params_call)
      assert_equal('notice', @logging_params.stdout_logging_level)
      @logging_params.stdout_logging_level = 'debug'
      assert_equal('debug', @logging_params.stdout_logging_level)
      @logging_params.save(self)
      assert_equal(1, @set_params_call)
      assert((@set_params_names.include? 'stdout_logging_level'))
      assert_equal('debug', @store['stdout_logging_level'])
    end

    def test_load_logfiles
      @store['logfiles'] = [
	{ 'path' => 'rucy.log',
	  'logging_level' => 'info',
	}
      ]
      @logging_params.load(self)
      assert_equal(1, @params_call)
      assert((@params_names.include? 'logfiles'))
      assert_equal(1, @logging_params.logfiles.length)
      assert_equal('rucy.log', @logging_params.logfiles[0].path)
      assert_equal('info', @logging_params.logfiles[0].logging_level)
    end

    def test_save_logfiles
      @logging_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(0, @logging_params.logfiles.length)
      @logging_params.add_logfile
      assert_equal(1, @logging_params.logfiles.length)
      assert_equal(nil, @logging_params.logfiles[0].path)
      assert_equal('notice', @logging_params.logfiles[0].logging_level)

      @logging_params.logfiles[0].path = 'debug.log'
      assert_equal('debug.log', @logging_params.logfiles[0].path)
      @logging_params.logfiles[0].logging_level = 'debug'
      assert_equal('debug', @logging_params.logfiles[0].logging_level)

      @logging_params.save(self)
      assert_equal(1, @set_params_call)
      assert((@set_params_names.include? 'logfiles'))
      assert_equal(1, @store['logfiles'].length)
      assert_equal([ { 'path' => 'debug.log',
		       'logging_level' => 'debug'
		     }
		   ], @store['logfiles'])
    end

    def test_delete_logfile_at
      @store['logfiles'] = [
	{ 'path' => 'rucy.log',
	  'logging_level' => 'info',
	}
      ]
      @logging_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(1, @logging_params.logfiles.length)
      @logging_params.del_logfile_at(0)
      assert_equal(0, @logging_params.logfiles.length)
      @logging_params.save(self)
      assert_equal(1, @set_params_call)
      assert_equal([], @store['logfiles'])
    end

    def test_invalid_logfiles
      @logging_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(0, @logging_params.logfiles.length)
      @logging_params.add_logfile

      @logging_params.logfiles[0].path = nil
      assert_equal(nil, @logging_params.logfiles[0].path)
      assert_exception(RuntimeError) {
	@logging_params.save(self)
      }

      @logging_params.logfiles[0].path = ' '
      assert_equal(' ', @logging_params.logfiles[0].path)
      assert_exception(RuntimeError) {
	@logging_params.save(self)
      }

      assert_exception(RuntimeError) {
	@logging_params.logfiles[0].logging_level = 'detarame'
      }
    end
  end

  class TestAccessLogParams < RUNIT::TestCase
    include PseudoProperties

    def setup
      super

      # target
      @access_log_params = Rucy::AccessLogParams.new
    end

    def test_modified_count
      @access_log_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(0, @access_log_params.modified_count)
      @access_log_params.modified_count = 1
      assert_equal(1, @access_log_params.modified_count)
      assert_equal(0, @modified_count)
      @access_log_params.save(self)
      assert_equal(1, @set_params_call)
      assert_equal(1, @modified_count)
    end

    def test_load_default_settings
      @access_log_params.load(self)
      assert_equal(1, @params_call)
      assert_equal([ 'access_log_logging',
		     'access_log_format',
		     'access_logs'
		   ].sort, @params_names.sort)
      assert_equal(true, @access_log_params.logging)
      assert_equal(Rucy::AccessLog::COMMON_LOG_FORMAT, @access_log_params.format)
      assert_equal(0, @access_log_params.logfiles.length)
    end

    def test_load_logging
      @store['access_log_logging'] = false
      @access_log_params.load(self)
      assert_equal(1, @params_call)
      assert((@params_names.include? 'access_log_logging'))
      assert_equal(false, @access_log_params.logging)
    end

    def test_load_format
      @store['access_log_format'] = 'HALO'
      @access_log_params.load(self)
      assert_equal(1, @params_call)
      assert((@params_names.include? 'access_log_format'))
      assert_equal('HALO', @access_log_params.format)
    end

    def test_load_logfiles
      @store['access_logs'] = [ { 'path' => 'access.log', 'format' => 'HALO' } ]
      @access_log_params.load(self)
      assert_equal(1, @params_call)
      assert((@params_names.include? 'access_logs'))
      assert_equal(1, @access_log_params.logfiles.length)
      assert_equal('access.log', @access_log_params.logfiles[0].path)
      assert_equal('HALO', @access_log_params.logfiles[0].format)
    end

    def test_save_logging
      @access_log_params.load(self)
      assert_equal(true, @access_log_params.logging)
      @access_log_params.logging = false
      assert_equal(false, @access_log_params.logging)
      @access_log_params.save(self)
      assert_equal(1, @set_params_call)
      assert((@set_params_names.include? 'access_log_logging'))
      assert_equal(false, @store['access_log_logging'])
    end

    def test_save_format
      @store['access_log_format'] = 'foo'
      @access_log_params.load(self)
      assert_equal('foo', @access_log_params.format)
      @access_log_params.format = 'bar'
      assert_equal('bar', @access_log_params.format)
      @access_log_params.save(self)
      assert_equal(1, @set_params_call)
      assert((@set_params_names.include? 'access_log_format'))
      assert_equal('bar', @store['access_log_format'])
    end

    def test_save_logfiles
      @store['access_logs'] = [ { 'path' => nil, 'format' => nil } ]
      @access_log_params.load(self)
      assert_equal(1, @access_log_params.logfiles.length)
      assert_equal(nil, @access_log_params.logfiles[0].path)
      assert_equal(nil, @access_log_params.logfiles[0].format)
      @access_log_params.logfiles[0].path = 'access.log'
      @access_log_params.logfiles[0].format = 'HALO'
      assert_equal('access.log', @access_log_params.logfiles[0].path)
      assert_equal('HALO', @access_log_params.logfiles[0].format)
      @access_log_params.save(self)
      assert_equal(1, @set_params_call)
      assert((@set_params_names.include? 'access_logs'))
      assert_equal([ { 'path' => 'access.log', 'format' => 'HALO' } ], @store['access_logs'])
    end

    def test_add_logfile
      @access_log_params.load(self)
      assert_equal(0, @access_log_params.logfiles.length)
      @access_log_params.add_logfile
      assert_equal(1, @access_log_params.logfiles.length)
      assert_equal(nil, @access_log_params.logfiles[0].path)
      assert_equal(Rucy::AccessLog::COMMON_LOG_FORMAT, @access_log_params.logfiles[0].format)
    end

    def test_del_logfile_at
      @store['access_logs'] = [ { 'path' => nil, 'format' => nil } ]
      @access_log_params.load(self)
      assert_equal(1, @access_log_params.logfiles.length)
      @access_log_params.del_logfile_at(0)
      assert_equal(0, @access_log_params.logfiles.length)
    end

    def test_invalid_logfiles
      @access_log_params.load(self)
      assert_equal(1, @params_call)
      assert_equal(0, @access_log_params.logfiles.length)

      @access_log_params.format = nil
      assert_exception(RuntimeError) {
	@access_log_params.save(self)
      }
      @access_log_params.format = ''
      assert_exception(RuntimeError) {
	@access_log_params.save(self)
      }
      @access_log_params.format = ' '
      assert_exception(RuntimeError) {
	@access_log_params.save(self)
      }

      @access_log_params.format = 'test'
      @access_log_params.add_logfile

      @access_log_params.logfiles[0].format = 'foo'
      @access_log_params.logfiles[0].path = nil
      assert_exception(RuntimeError) {
	@access_log_params.save(self)
      }
      @access_log_params.logfiles[0].path = ''
      assert_exception(RuntimeError) {
	@access_log_params.save(self)
      }
      @access_log_params.logfiles[0].path = ' '
      assert_exception(RuntimeError) {
	@access_log_params.save(self)
      }

      @access_log_params.logfiles[0].path = 'foo.log'
      @access_log_params.logfiles[0].format = nil
      assert_exception(RuntimeError) {
	@access_log_params.save(self)
      }
      @access_log_params.logfiles[0].format = ''
      assert_exception(RuntimeError) {
	@access_log_params.save(self)
      }
      @access_log_params.logfiles[0].format = ' '
      assert_exception(RuntimeError) {
	@access_log_params.save(self)
      }
    end
  end

  class TestAliasList < RUNIT::TestCase
    include PseudoProperties

    def setup
      super

      # test target
      @alias_list = Rucy::AliasList.new
    end

    def test_modified_count
      @alias_list.load(self)
      assert_equal(1, @list_call)
      assert_equal(0, @alias_list.modified_count)
      @alias_list.modified_count = 1
      assert_equal(1, @alias_list.modified_count)
      assert_equal(0, @modified_count)
      @alias_list.save(self)
      assert_equal(1, @set_list_call)
      assert_equal(1, @modified_count)
    end

    def test_load_default_settings
      @alias_list.load(self)
      assert_equal(1, @list_call)
      assert_equal('aliases', @list_name)
      assert(@alias_list.empty?)
      assert_equal(0, @alias_list.size)
      assert_equal(0, @alias_list.length)
      for alias_params in @alias_list
	assert_fail('found an element in empty list.')
      end
    end

    def test_load
      @store['aliases'] = [
	{ 'alias_path' => '/',
	  'orig_path' => '/index.rb',
	  'virtual_host' => 'foo'
	}
      ]
      @alias_list.load(self)
      assert_equal(1, @list_call)
      assert_equal('aliases', @list_name)
      assert(! @alias_list.empty?)
      assert_equal(1, @alias_list.size)
      assert_equal(1, @alias_list.length)

      # test of operator `[]'
      assert_equal('/', @alias_list[0].alias_path)
      assert_equal('/index.rb', @alias_list[0].orig_path)
      assert_equal('foo', @alias_list[0].virtual_host)

      # test of each
      count = 0
      for params in @alias_list
	count += 1
	assert_equal('/', params.alias_path)
	assert_equal('/index.rb', params.orig_path)
	assert_equal('foo', params.virtual_host)
      end
      assert_equal(1, count)
    end

    def test_save
      @store['aliases'] = [
	{ 'alias_path' => nil,
	  'orig_path' => nil,
	  'virtual_host' => nil
	}
      ]
      @alias_list.load(self)
      @alias_list[0].alias_path = '/'
      @alias_list[0].orig_path = '/index.rb'
      @alias_list[0].virtual_host = 'foo'
      @alias_list.save(self)
      assert_equal(1, @set_list_call)
      assert_equal('aliases', @set_list_name)
      assert_equal([ { 'alias_path' => '/',
		       'orig_path' => '/index.rb',
		       'virtual_host' => 'foo'
		     }
		   ], @store['aliases'])
    end

    def test_add_entry
      @alias_list.load(self)
      assert(@alias_list.empty?)
      assert_equal(0, @alias_list.size)
      assert_equal(0, @alias_list.length)
      @alias_list.add_entry
      assert(! @alias_list.empty?)
      assert_equal(1, @alias_list.size)
      assert_equal(1, @alias_list.length)
      assert_equal(nil, @alias_list[0].alias_path)
      assert_equal(nil, @alias_list[0].orig_path)
      assert_equal(nil, @alias_list[0].virtual_host)
    end

    def test_delete_at
      @store['aliases'] = [
	{ 'alias_path' => '/foo',
	  'orig_path' => '/foo.rb',
	  'virtual_host' => nil
	}
      ]
      @alias_list.load(self)
      assert(! @alias_list.empty?)
      assert_equal(1, @alias_list.size)
      assert_equal(1, @alias_list.length)
      assert_equal('/foo', @alias_list[0].alias_path)
      assert_equal('/foo.rb', @alias_list[0].orig_path)
      assert_equal(nil, @alias_list[0].virtual_host)
      @alias_list.delete_at(0)
      assert(@alias_list.empty?)
      assert_equal(0, @alias_list.size)
      assert_equal(0, @alias_list.length)
    end

    def test_swap
      @store['aliases'] = [
	{ 'alias_path' => '/foo',
	  'orig_path' => '/foo.rb',
	  'virtual_host' => nil
	},
	{ 'alias_path' => '/bar',
	  'orig_path' => '/bar.rb',
	  'virtual_host' => 'bar'
	}
      ]
      @alias_list.load(self)
      assert_equal('/foo', @alias_list[0].alias_path)
      assert_equal('/foo.rb', @alias_list[0].orig_path)
      assert_equal(nil, @alias_list[0].virtual_host)
      assert_equal('/bar', @alias_list[1].alias_path)
      assert_equal('/bar.rb', @alias_list[1].orig_path)
      assert_equal('bar', @alias_list[1].virtual_host)
      @alias_list.swap(0, 1)
      assert_equal('/bar', @alias_list[0].alias_path)
      assert_equal('/bar.rb', @alias_list[0].orig_path)
      assert_equal('bar', @alias_list[0].virtual_host)
      assert_equal('/foo', @alias_list[1].alias_path)
      assert_equal('/foo.rb', @alias_list[1].orig_path)
      assert_equal(nil, @alias_list[1].virtual_host)
    end

    def test_inavlid_alias_path
      @store['aliases'] = [
	{ 'alias_path' => '/foo',
	  'orig_path' => '/foo.rb',
	  'virtual_host' => nil
	}
      ]
      @alias_list.load(self)
      assert_equal('/foo', @alias_list[0].alias_path)
      assert_equal('/foo.rb', @alias_list[0].orig_path)
      assert_equal(nil, @alias_list[0].virtual_host)

      @alias_list[0].alias_path = nil
      assert_exception(RuntimeError) {
	@alias_list.save(self)
      }

      @alias_list[0].alias_path = ''
      assert_exception(RuntimeError) {
	@alias_list.save(self)
      }

      @alias_list[0].alias_path = ' '
      assert_exception(RuntimeError) {
	@alias_list.save(self)
      }

      assert_exception(RuntimeError) {
	@alias_list[0].alias_path = 123
      }
      assert_exception(RuntimeError) {
	@alias_list[0].alias_path = false
      }
    end

    def test_inavlid_orig_path
      @store['aliases'] = [
	{ 'alias_path' => '/foo',
	  'orig_path' => '/foo.rb',
	  'virtual_host' => nil
	}
      ]
      @alias_list.load(self)
      assert_equal('/foo', @alias_list[0].alias_path)
      assert_equal('/foo.rb', @alias_list[0].orig_path)
      assert_equal(nil, @alias_list[0].virtual_host)

      @alias_list[0].orig_path = nil
      assert_exception(RuntimeError) {
	@alias_list.save(self)
      }

      @alias_list[0].orig_path = ''
      assert_exception(RuntimeError) {
	@alias_list.save(self)
      }

      @alias_list[0].orig_path = ' '
      assert_exception(RuntimeError) {
	@alias_list.save(self)
      }

      assert_exception(RuntimeError) {
	@alias_list[0].orig_path = 123
      }
      assert_exception(RuntimeError) {
	@alias_list[0].orig_path = false
      }
    end

    def test_invalid_virtual_host
      @store['aliases'] = [
	{ 'alias_path' => '/foo',
	  'orig_path' => '/foo.rb',
	  'virtual_host' => nil
	}
      ]
      @alias_list.load(self)
      assert_equal('/foo', @alias_list[0].alias_path)
      assert_equal('/foo.rb', @alias_list[0].orig_path)
      assert_equal(nil, @alias_list[0].virtual_host)

      @alias_list[0].virtual_host = nil	# allowed
      assert_exception(RuntimeError) {
	@alias_list[0].virtual_host = 123
      }
      assert_exception(RuntimeError) {
	@alias_list[0].virtual_host = false
      }
    end

    def test_empty_virtual_host
      @store['aliases'] = [
	{ 'alias_path' => nil,
	  'orig_path' => nil,
	  'virtual_host' => nil
	}
      ]
      @alias_list.load(self)
      assert_equal(nil, @alias_list[0].virtual_host)
      @alias_list[0].virtual_host = ''
      assert_equal(nil, @alias_list[0].virtual_host)
      @alias_list[0].virtual_host = ' '
      assert_equal(nil, @alias_list[0].virtual_host)
    end
  end

  class TestDocumentList < RUNIT::TestCase
    include PseudoProperties

    def setup
      super

      # test target
      @doc_list = Rucy::DocumentList.new
    end

    def test_modified_count
      @doc_list.load(self)
      assert_equal(1, @list_call)
      assert_equal(0, @doc_list.modified_count)
      @doc_list.modified_count = 1
      assert_equal(1, @doc_list.modified_count)
      assert_equal(0, @modified_count)
      @doc_list.save(self)
      assert_equal(1, @set_list_call)
      assert_equal(1, @modified_count)
    end

    def test_load_default_settings
      @doc_list.load(self)
      assert_equal(1, @list_call)
      assert_equal('documents', @list_name)
      assert(@doc_list.empty?)
      assert_equal(0, @doc_list.size)
      assert_equal(0, @doc_list.length)
      for mount_params in @doc_list
	assert_fail('found an element in empty list.')
      end
    end

    def test_load
      @store['documents'] = [
	{ 'document' => 'Page',
	  'arguments' => [
	    "Hello world.\n",
	    'text/plain'
          ],
	  'mount_path' => '/halo',
	  'mount_mask' => nil,
	  'virtual_host' => nil,
	  'comment' => 'HALO'
	}
      ]
      @doc_list.load(self)
      assert_equal(1, @list_call)
      assert_equal('documents', @list_name)
      assert(! @doc_list.empty?)
      assert_equal(1, @doc_list.size)
      assert_equal(1, @doc_list.length)

      # test of operator `[]'
      assert_equal('Page', @doc_list[0].name)
      assert_equal([ "Hello world.\n",
		     'text/plain'
		   ], @doc_list[0].args)
      assert_equal('/halo', @doc_list[0].path)
      assert_equal(nil, @doc_list[0].mask)
      assert_equal(nil, @doc_list[0].virtual_host)
      assert_equal('HALO', @doc_list[0].comment)

      # test of arguments modification
      @doc_list[0].args.clear
      assert_equal([ "Hello world.\n",
		     'text/plain'
		   ], @doc_list[0].args)

      # test of each
      count = 0
      for doc_params in @doc_list
	count += 1
	assert_equal('Page', doc_params.name)
	assert_equal([ "Hello world.\n",
		       'text/plain'
		     ], doc_params.args)
	assert_equal('/halo', doc_params.path)
	assert_equal(nil, doc_params.mask)
	assert_equal(nil, doc_params.virtual_host)
	assert_equal('HALO', doc_params.comment)
      end
      assert_equal(1, count)
    end

    def test_save
      @store['documents'] = [
	{ 'document' => nil,
	  'arguments' => [],
	  'mount_path' => nil,
	  'mount_mask' => nil,
	  'virtual_host' => nil,
	  'comment' => nil
	}
      ]
      @doc_list.load(self)
      @doc_list[0].name = 'Page'
      @doc_list[0].set_args_at(0, "Hello world.\n")
      @doc_list[0].set_args_at(1, 'text/plain')
      @doc_list[0].path = '/halo'
      @doc_list[0].mask = nil
      @doc_list[0].virtual_host = nil
      @doc_list[0].comment = 'HALO'
      @doc_list.save(self)
      assert_equal(1, @set_list_call)
      assert_equal('documents', @set_list_name)
      assert_equal([ { 'document' => 'Page',
		       'arguments' => [
			 "Hello world.\n",
			 'text/plain'
		       ],
		       'mount_path' => '/halo',
		       'mount_mask' => nil,
		       'virtual_host' => nil,
		       'comment' => 'HALO'
		     }
		   ], @store['documents'])
    end

    def test_add_entry
      @doc_list.load(self)
      assert(@doc_list.empty?)
      assert_equal(0, @doc_list.size)
      assert_equal(0, @doc_list.length)
      @doc_list.add_entry
      assert(! @doc_list.empty?)
      assert_equal(1, @doc_list.size)
      assert_equal(1, @doc_list.length)
      assert_equal(nil, @doc_list[0].name)
      assert(@doc_list[0].args.empty?)
      assert_equal(nil, @doc_list[0].path)
      assert_equal(nil, @doc_list[0].mask)
      assert_equal(nil, @doc_list[0].virtual_host)
      assert_equal(nil, @doc_list[0].comment)
    end

    def test_delete_at
      @store['documents'] = [
	{ 'document' => 'Foo',
	  'arguments' => [],
	  'mount_path' => nil,
	  'mount_mask' => nil,
	  'virtual_host' => nil,
	  'comment' => nil
	}
      ]
      @doc_list.load(self)
      assert(! @doc_list.empty?)
      assert_equal(1, @doc_list.size)
      assert_equal(1, @doc_list.length)
      assert_equal('Foo', @doc_list[0].name)
      @doc_list.delete_at(0)
      assert(@doc_list.empty?)
      assert_equal(0, @doc_list.size)
      assert_equal(0, @doc_list.length)
    end

    def test_swap
      @store['documents'] = [
	{ 'document' => 'Foo',
	  'arguments' => [],
	  'mount_path' => nil,
	  'mount_mask' => nil,
	  'virtual_host' => nil,
	  'comment' => nil
	},
	{ 'document' => 'Bar',
	  'arguments' => [],
	  'mount_path' => nil,
	  'mount_mask' => nil,
	  'virtual_host' => nil,
	  'comment' => nil
	}
      ]
      @doc_list.load(self)
      assert_equal('Foo', @doc_list[0].name)
      assert_equal('Bar', @doc_list[1].name)
      @doc_list.swap(0, 1)
      assert_equal('Bar', @doc_list[0].name)
      assert_equal('Foo', @doc_list[1].name)
    end
  end

  class TestDocumentListBackwardCompatibility < RUNIT::TestCase
    include PseudoProperties

    def setup
      super

      # test target
      @doc_list = Rucy::DocumentList.new
    end

    def test_attributes
      @store['documents'] = [
	{ 'document' => 'Foo',
	  'arguments' => [ 'foo', 'bar' ],
	  'mount_path' => '/mount_path',
	  'mount_mask' => /mount_mask/,
	  'virtual_host' => 'virtual_host',
	  'comment' => "Comment.\n"
	}
      ]
      @doc_list.load(self)
      assert_equal('Foo', @doc_list[0]['document'])
      assert_equal([ 'foo', 'bar' ], @doc_list[0]['arguments'])
      assert_equal('/mount_path', @doc_list[0]['mount_path'])
      assert_equal(/mount_mask/, @doc_list[0]['mount_mask'])
      assert_equal('virtual_host', @doc_list[0]['virtual_host'])
      assert_equal("Comment.\n", @doc_list[0]['comment'])
    end

    def test_set_attributes
      @store['documents'] = [
	{ 'document' => nil,
	  'arguments' => [],
	  'mount_path' => nil,
	  'mount_mask' => nil,
	  'virtual_host' => nil,
	  'comment' => nil
	}
      ]
      @doc_list.load(self)
      @doc_list[0]['document'] = 'Foo'
      @doc_list[0]['arguments'] = [ 'foo', 'bar' ]
      @doc_list[0]['mount_path'] = '/mount_path'
      @doc_list[0]['mount_mask'] = /mount_mask/
      @doc_list[0]['virtual_host'] = 'virtual_host'
      @doc_list[0]['comment'] = "Comment.\n"
      @doc_list.save(self)
      assert_equal([ { 'document' => 'Foo',
		       'arguments' => [ 'foo', 'bar' ],
		       'mount_path' => '/mount_path',
		       'mount_mask' => /mount_mask/,
		       'virtual_host' => 'virtual_host',
		       'comment' => "Comment.\n"
		     }
		   ], @store['documents'])
    end

    def test_to_hash
      @store['documents'] = [
	{ 'document' => 'Foo',
	  'arguments' => [ 'foo', 'bar' ],
	  'mount_path' => '/mount_path',
	  'mount_mask' => /mount_mask/,
	  'virtual_host' => 'virtual_host',
	  'comment' => "Comment.\n"
	}
      ]
      @doc_list.load(self)
      assert_equal({ 'document' => 'Foo',
		     'arguments' => [ 'foo', 'bar' ],
		     'mount_path' => '/mount_path',
		     'mount_mask' => /mount_mask/,
		     'virtual_host' => 'virtual_host',
		     'comment' => "Comment.\n"
		   }, @doc_list[0].to_hash)
    end

    def test_push
      @store['documents'] = [
	{ 'document' => 'Foo',
	  'arguments' => [ 'foo', 'bar' ],
	  'mount_path' => '/mount_path',
	  'mount_mask' => /mount_mask/,
	  'virtual_host' => 'virtual_host',
	  'comment' => "Comment.\n"
	}
      ]
      @doc_list.load(self)
      assert_equal(1, @doc_list.length)
      @doc_list[1] = Hash.new
      assert_equal(2, @doc_list.length)
      assert_equal(nil, @doc_list[1]['document'])
      assert_equal([], @doc_list[1]['arguments'])
      assert_equal(nil, @doc_list[1]['mount_path'])
      assert_equal(nil, @doc_list[1]['mount_mask'])
      assert_equal(nil, @doc_list[1]['virtual_host'])
      assert_equal(nil, @doc_list[1]['comment'])
    end

    def test_push_with_attrs
      @store['documents'] = []
      @doc_list.load(self)
      assert_equal(0, @doc_list.length)
      @doc_list[0] = {
	'document' => 'Foo',
	'arguments' => [ 'foo', 'bar' ],
	'mount_path' => '/mount_path',
	'mount_mask' => /mount_mask/,
	'virtual_host' => 'virtual_host',
	'comment' => "Comment.\n"
      }
      assert_equal(1, @doc_list.length)
      assert_equal('Foo', @doc_list[0]['document'])
      assert_equal([ 'foo', 'bar' ], @doc_list[0]['arguments'])
      assert_equal('/mount_path', @doc_list[0]['mount_path'])
      assert_equal(/mount_mask/, @doc_list[0]['mount_mask'])
      assert_equal('virtual_host', @doc_list[0]['virtual_host'])
      assert_equal("Comment.\n", @doc_list[0]['comment'])
    end
  end

  class TestFilterList < RUNIT::TestCase
    include PseudoProperties

    def setup
      super

      # test target
      @filter_list = Rucy::FilterList.new
    end

    def test_modified_count
      @filter_list.load(self)
      assert_equal(1, @list_call)
      assert_equal(0, @filter_list.modified_count)
      @filter_list.modified_count = 1
      assert_equal(1, @filter_list.modified_count)
      assert_equal(0, @modified_count)
      @filter_list.save(self)
      assert_equal(1, @set_list_call)
      assert_equal(1, @modified_count)
    end

    def test_load_default_settings
      @filter_list.load(self)
      assert_equal(1, @list_call)
      assert_equal('filters', @list_name)
      assert(@filter_list.empty?)
      assert_equal(0, @filter_list.size)
      assert_equal(0, @filter_list.length)
      for mount_params in @filter_list
	assert_fail('found an element in empty list.')
      end
    end

    def test_load
      @store['filters'] = [
	{ 'filter' => 'Foo',
	  'arguments' => [
	    'apple',
	    'banana'
          ],
	  'attach_path' => '/foo',
	  'attach_mask' => %r"\.html?($|/)",
	  'virtual_host' => 'www.foo.org',
	  'comment' => 'test'
	}
      ]
      @filter_list.load(self)
      assert_equal(1, @list_call)
      assert_equal('filters', @list_name)
      assert(! @filter_list.empty?)
      assert_equal(1, @filter_list.size)
      assert_equal(1, @filter_list.length)

      # test of operator `[]'
      assert_equal('Foo', @filter_list[0].name)
      assert_equal([ 'apple', 'banana' ], @filter_list[0].args)
      assert_equal('/foo', @filter_list[0].path)
      assert_equal(%r"\.html?($|/)", @filter_list[0].mask)
      assert_equal('www.foo.org', @filter_list[0].virtual_host)
      assert_equal('test', @filter_list[0].comment)

      # test of arguments modification
      @filter_list[0].args.clear
      assert_equal([ 'apple', 'banana' ], @filter_list[0].args)

      # test of each
      count = 0
      for filter_params in @filter_list
	count += 1
	assert_equal('Foo', filter_params.name)
	assert_equal([ 'apple', 'banana' ], filter_params.args)
	assert_equal('/foo', filter_params.path)
	assert_equal(%r"\.html?($|/)", filter_params.mask)
	assert_equal('www.foo.org', filter_params.virtual_host)
	assert_equal('test', filter_params.comment)
      end
      assert_equal(1, count)
    end

    def test_save
      @store['filters'] = [
	{ 'filter' => nil,
	  'arguments' => [],
	  'attach_path' => nil,
	  'attach_mask' => nil,
	  'virtual_host' => nil,
	  'comment' => nil
	}
      ]
      @filter_list.load(self)
      @filter_list[0].name = 'Foo'
      @filter_list[0].set_args_at(0, 'apple')
      @filter_list[0].set_args_at(1, 'banana')
      @filter_list[0].path = '/foo'
      @filter_list[0].mask = %r"\.html($|/)"
      @filter_list[0].virtual_host = 'www.foo.org'
      @filter_list[0].comment = 'test'
      @filter_list.save(self)
      assert_equal(1, @set_list_call)
      assert_equal('filters', @set_list_name)
      assert_equal([ { 'filter' => 'Foo',
		       'arguments' => [ 'apple', 'banana' ],
		       'attach_path' => '/foo',
		       'attach_mask' => %r"\.html($|/)",
		       'virtual_host' => 'www.foo.org',
		       'comment' => 'test'
		     }
		   ], @store['filters'])
    end

    def test_add_entry
      @filter_list.load(self)
      assert(@filter_list.empty?)
      assert_equal(0, @filter_list.size)
      assert_equal(0, @filter_list.length)
      @filter_list.add_entry
      assert(! @filter_list.empty?)
      assert_equal(1, @filter_list.size)
      assert_equal(1, @filter_list.length)
      assert_equal(nil, @filter_list[0].name)
      assert(@filter_list[0].args.empty?)
      assert_equal(nil, @filter_list[0].path)
      assert_equal(nil, @filter_list[0].mask)
      assert_equal(nil, @filter_list[0].virtual_host)
      assert_equal(nil, @filter_list[0].comment)
    end

    def test_delete_at
      @store['filters'] = [
	{ 'filter' => 'Foo',
	  'arguments' => [],
	  'attach_path' => nil,
	  'attach_mask' => nil,
	  'virtual_host' => nil,
	  'comment' => nil
	}
      ]
      @filter_list.load(self)
      assert(! @filter_list.empty?)
      assert_equal(1, @filter_list.size)
      assert_equal(1, @filter_list.length)
      assert_equal('Foo', @filter_list[0].name)
      @filter_list.delete_at(0)
      assert(@filter_list.empty?)
      assert_equal(0, @filter_list.size)
      assert_equal(0, @filter_list.length)
    end

    def test_swap
      @store['filters'] = [
	{ 'filter' => 'Foo',
	  'arguments' => [],
	  'attach_path' => nil,
	  'attach_mask' => nil,
	  'virtual_host' => nil,
	  'comment' => nil
	},
	{ 'filter' => 'Bar',
	  'arguments' => [],
	  'attach_path' => nil,
	  'attach_mask' => nil,
	  'virtual_host' => nil,
	  'comment' => nil
	}
      ]
      @filter_list.load(self)
      assert_equal('Foo', @filter_list[0].name)
      assert_equal('Bar', @filter_list[1].name)
      @filter_list.swap(0, 1)
      assert_equal('Bar', @filter_list[0].name)
      assert_equal('Foo', @filter_list[1].name)
    end
  end

  class TestFilterListBackwardCompatibility < RUNIT::TestCase
    include PseudoProperties

    def setup
      super

      # test target
      @filter_list = Rucy::FilterList.new
    end

    def test_attributes
      @store['filters'] = [
	{ 'filter' => 'Foo',
	  'arguments' => [ 'foo', 'bar' ],
	  'attach_path' => '/attach_path',
	  'attach_mask' => /attach_mask/,
	  'virtual_host' => 'www.foo.org',
	  'comment' => "Comment.\n"
	}
      ]
      @filter_list.load(self)
      assert_equal('Foo', @filter_list[0]['filter'])
      assert_equal([ 'foo', 'bar' ], @filter_list[0]['arguments'])
      assert_equal('/attach_path', @filter_list[0]['attach_path'])
      assert_equal(/attach_mask/, @filter_list[0]['attach_mask'])
      assert_equal('www.foo.org', @filter_list[0]['virtual_host'])
      assert_equal("Comment.\n", @filter_list[0]['comment'])
    end

    def test_set_attributes
      @store['filters'] = [
	{ 'filter' => nil,
	  'arguments' => [],
	  'attach_path' => nil,
	  'attach_mask' => nil,
	  'virtual_host' => nil,
	  'comment' => nil
	}
      ]
      @filter_list.load(self)
      @filter_list[0]['filter'] = 'Foo'
      @filter_list[0]['arguments'] = [ 'foo', 'bar' ]
      @filter_list[0]['attach_path'] = '/attach_path'
      @filter_list[0]['attach_mask'] = /attach_mask/
      @filter_list[0]['virtual_host'] = 'www.foo.org'
      @filter_list[0]['comment'] = "Comment.\n"
      @filter_list.save(self)
      assert_equal([ { 'filter' => 'Foo',
		       'arguments' => [ 'foo', 'bar' ],
		       'attach_path' => '/attach_path',
		       'attach_mask' => /attach_mask/,
		       'virtual_host' => 'www.foo.org',
		       'comment' => "Comment.\n"
		     }
		   ], @store['filters'])
    end

    def test_to_hash
      @store['filters'] = [
	{ 'filter' => 'Foo',
	  'arguments' => [ 'foo', 'bar' ],
	  'attach_path' => '/attach_path',
	  'attach_mask' => /attach_mask/,
	  'virtual_host' => 'www.foo.org',
	  'comment' => "Comment.\n"
	}
      ]
      @filter_list.load(self)
      assert_equal({ 'filter' => 'Foo',
		     'arguments' => [ 'foo', 'bar' ],
		     'attach_path' => '/attach_path',
		     'attach_mask' => /attach_mask/,
		     'virtual_host' => 'www.foo.org',
		     'comment' => "Comment.\n"
		   }, @filter_list[0].to_hash)
    end

    def test_push
      @store['filters'] = [
	{ 'filter' => 'Foo',
	  'arguments' => [ 'foo', 'bar' ],
	  'attach_path' => '/attach_path',
	  'attach_mask' => /attach_mask/,
	  'virtual_host' => 'www.foo.org',
	  'comment' => "Comment.\n"
	}
      ]
      @filter_list.load(self)
      assert_equal(1, @filter_list.length)
      @filter_list[1] = Hash.new
      assert_equal(2, @filter_list.length)
      assert_equal(nil, @filter_list[1]['filter'])
      assert_equal([], @filter_list[1]['arguments'])
      assert_equal(nil, @filter_list[1]['attach_path'])
      assert_equal(nil, @filter_list[1]['attach_mask'])
      assert_equal(nil, @filter_list[1]['virtual_host'])
      assert_equal(nil, @filter_list[1]['comment'])
    end

    def test_push_with_attrs
      @store['filters'] = []
      @filter_list.load(self)
      assert_equal(0, @filter_list.length)
      @filter_list[0] = {
	'filter' => 'Foo',
	'arguments' => [ 'foo', 'bar' ],
	'attach_path' => '/attach_path',
	'attach_mask' => /attach_mask/,
	'virtual_host' => 'www.foo.org',
	'comment' => "Comment.\n"
      }
      assert_equal(1, @filter_list.length)
      assert_equal('Foo', @filter_list[0]['filter'])
      assert_equal([ 'foo', 'bar' ], @filter_list[0]['arguments'])
      assert_equal('/attach_path', @filter_list[0]['attach_path'])
      assert_equal(/attach_mask/, @filter_list[0]['attach_mask'])
      assert_equal('www.foo.org', @filter_list[0]['virtual_host'])
      assert_equal("Comment.\n", @filter_list[0]['comment'])
    end
  end
end
