#!/usr/local/bin/ruby
# $Id: test_monitor.rb,v 1.10 2004/10/12 14:36:40 toki Exp $

require 'rubyunit'
require 'rucy/monitor'

module TestRucy
  class TestRuntimeErrorSample < RUNIT::TestCase
    def build_sample
      Rucy::RuntimeErrorSample.new(self, @timestamp)
    end

    def setup
      # for Exception class
      @error_type = self.class
      @message_call = 0
      @class_call = 0
      @backtrace_call = 0

      # target
      @timestamp = Time.mktime(2004, 10, 8, 2, 10, 58) # Fri Oct 08 02:10:58 2004
      @sample = build_sample
    end

    def message
      @message_call += 1
      'HALO'
    end

    def class
      @class_call += 1 if @class_call # check for testing framework.
      super
    end

    def backtrace
      @backtrace_call += 1
      []
    end

    def test_error_message
      assert_equal('HALO', @sample.error_message)
      assert_equal(1, @message_call)
    end

    def test_error_type
      assert_equal(@error_type, @sample.error_type)
      assert_equal(1, @class_call)
    end

    def test_backtrace
      assert_equal([], @sample.backtrace)
      assert_equal(1, @backtrace_call)
    end

    def test_timestamp
      assert_instance_of(Time, @sample.timestamp)
      assert_equal(2004, @sample.timestamp.year)
      assert_equal(10, @sample.timestamp.mon)
      assert_equal(8, @sample.timestamp.day)
      assert_equal(2, @sample.timestamp.hour)
      assert_equal(10, @sample.timestamp.min)
      assert_equal(58, @sample.timestamp.sec)
    end
  end

  class TestLoadErrorSample < TestRuntimeErrorSample
    def build_sample
      Rucy::LoadErrorSample.new(self, 'mod_docs/sample.rb', @timestamp)
    end

    def test_mod_path
      assert_equal('mod_docs/sample.rb', @sample.mod_path)
    end
  end

  class TestLoggingErrorSample < TestRuntimeErrorSample
    def build_sample
      Rucy::LoggingErrorSample.new(self, 'rucy.log', @timestamp)
    end

    def test_log_path
      assert_equal('rucy.log', @sample.log_path)
    end
  end

  class TestAliasErrorSample < TestRuntimeErrorSample
    def build_sample
      Rucy::AliasErrorSample.new(self, '/alias/path', '/orig/path', 'virtual_host', @timestamp)
    end

    def test_alias_path
      assert_equal('/alias/path', @sample.alias_path)
    end

    def test_orig_path
      assert_equal('/orig/path', @sample.orig_path)
    end

    def test_virutal_host
      assert_equal('virtual_host', @sample.virtual_host)
    end
  end

  class TestDocumentErrorSample < TestRuntimeErrorSample
    def build_sample
      Rucy::DocumentErrorSample.new(self,
				    'Document',
				    [ [ 'foo', 'alice' ], [ 'bar', 'bob' ] ],
				    '/mount_path',
				    /mount_mask/,
				    'virtual_host',
				    @timestamp)
    end

    def test_document
      assert_equal('Document', @sample.document)
    end

    def test_arguments
      assert_equal([ [ 'foo', 'alice' ], [ 'bar', 'bob' ] ], @sample.arguments)
      @sample.arguments.each{|pair| pair.clear }
      assert_equal([ [ 'foo', 'alice' ], [ 'bar', 'bob' ] ], @sample.arguments)
      @sample.arguments.clear
      assert_equal([ [ 'foo', 'alice' ], [ 'bar', 'bob' ] ], @sample.arguments)
    end

    def test_mount_path
      assert_equal('/mount_path', @sample.mount_path)
    end

    def tset_mount_mask
      assert_equal(/mount_mask/, @sample.mount_mask)
    end

    def test_virtual_host
      assert_equal('virtual_host', @sample.virtual_host)
    end
  end

  class TestFilterErrorSample < TestRuntimeErrorSample
    def build_sample
      Rucy::FilterErrorSample.new(self,
				  'Filter',
				  [ [ 'foo', 'alice' ], [ 'bar', 'bob' ] ],
				  '/attach_path',
				  /attach_mask/,
				  'virtual_host',
				  @timestamp)
    end

    def test_filter
      assert_equal('Filter', @sample.filter)
    end

    def test_arguments
      assert_equal([ [ 'foo', 'alice' ], [ 'bar', 'bob' ] ], @sample.arguments)
      @sample.arguments.each{|pair| pair.clear }
      assert_equal([ [ 'foo', 'alice' ], [ 'bar', 'bob' ] ], @sample.arguments)
      @sample.arguments.clear
      assert_equal([ [ 'foo', 'alice' ], [ 'bar', 'bob' ] ], @sample.arguments)
    end

    def test_attach_path
      assert_equal('/attach_path', @sample.attach_path)
    end

    def test_attach_mask
      assert_equal(/attach_mask/, @sample.attach_mask)
    end

    def test_virtual_host
      assert_equal('virtual_host', @sample.virtual_host)
    end
  end

  class TestSetupErrorMonitor < RUNIT::TestCase
    def setup
      @monitor = Rucy::SetupErrorMonitor.new
    end

    def test_load_erros
      @monitor.capture_load_error(RuntimeError.new('load error'), 'mod_docs/test.rb')
      assert_equal(1, @monitor.load_errors.length)
      assert_equal('load error', @monitor.load_errors[0].error_message)
      assert_equal(RuntimeError, @monitor.load_errors[0].error_type)
      assert_equal(nil, @monitor.load_errors[0].backtrace)
      assert_instance_of(Time, @monitor.load_errors[0].timestamp)
      assert_equal('mod_docs/test.rb', @monitor.load_errors[0].mod_path)
    end

    def test_logging_errors
      @monitor.capture_logging_error(RuntimeError.new('logging error'), 'rucy.log')
      assert_equal(1, @monitor.logging_errors.length)
      assert_equal('logging error', @monitor.logging_errors[0].error_message)
      assert_equal(RuntimeError, @monitor.logging_errors[0].error_type)
      assert_equal(nil, @monitor.logging_errors[0].backtrace)
      assert_instance_of(Time, @monitor.logging_errors[0].timestamp)
      assert_equal('rucy.log', @monitor.logging_errors[0].log_path)
    end

    def test_alias_errors
      @monitor.capture_alias_error(RuntimeError.new('alias error'), '/alias_path', '/orig_path', 'virtual_host')
      assert_equal(1, @monitor.alias_errors.length)
      assert_equal('alias error', @monitor.alias_errors[0].error_message)
      assert_equal(RuntimeError, @monitor.alias_errors[0].error_type)
      assert_equal(nil, @monitor.alias_errors[0].backtrace)
      assert_instance_of(Time, @monitor.alias_errors[0].timestamp)
      assert_equal('/alias_path', @monitor.alias_errors[0].alias_path)
      assert_equal('/orig_path', @monitor.alias_errors[0].orig_path)
      assert_equal('virtual_host', @monitor.alias_errors[0].virtual_host)
    end

    def test_document_errors
      @monitor.capture_document_error(RuntimeError.new('document error'),
				      'Document',
				      [ [ 'foo', 'apple' ], [ 'bar', 'banana' ] ],
				      '/mount_path',
				      /mount_mask/,
				      'virtual_host')
      assert_equal(1, @monitor.document_errors.length)
      assert_equal('document error', @monitor.document_errors[0].error_message)
      assert_equal(RuntimeError, @monitor.document_errors[0].error_type)
      assert_equal(nil, @monitor.document_errors[0].backtrace)
      assert_instance_of(Time, @monitor.document_errors[0].timestamp)
      assert_equal('Document', @monitor.document_errors[0].document)
      assert_equal([ [ 'foo', 'apple' ], [ 'bar', 'banana' ] ], @monitor.document_errors[0].arguments)
      assert_equal('/mount_path', @monitor.document_errors[0].mount_path)
      assert_equal(/mount_mask/, @monitor.document_errors[0].mount_mask)
      assert_equal('virtual_host', @monitor.document_errors[0].virtual_host)
    end

    def test_filter_errors
      @monitor.capture_filter_error(RuntimeError.new('filter error'),
				    'Filter',
				    [ [ 'foo', 'apple' ], [ 'bar', 'banana' ] ],
				    '/attach_path',
				    /attach_mask/,
				    'virtual_host')
      assert_equal(1, @monitor.filter_errors.length)
      assert_equal('filter error', @monitor.filter_errors[0].error_message)
      assert_equal(RuntimeError, @monitor.filter_errors[0].error_type)
      assert_equal(nil, @monitor.filter_errors[0].backtrace)
      assert_instance_of(Time, @monitor.filter_errors[0].timestamp)
      assert_equal('Filter', @monitor.filter_errors[0].filter)
      assert_equal([ [ 'foo', 'apple' ], [ 'bar', 'banana' ] ], @monitor.filter_errors[0].arguments)
      assert_equal('/attach_path', @monitor.filter_errors[0].attach_path)
      assert_equal(/attach_mask/, @monitor.filter_errors[0].attach_mask)
      assert_equal('virtual_host', @monitor.filter_errors[0].virtual_host)
    end
  end
end
