#!/usr/local/bin/ruby
# $Id: test_message.rb,v 1.1.1.1 2004/04/04 15:22:50 toki Exp $

require 'rubyunit'
require 'pseudo_io'
require 'rucy/message'

module TestRucy
  class TestMessage < RUNIT::TestCase
    def setup
      @messg = Rucy::Message.new
    end

    def test_header
      assert_nil(@messg.header('Content-Type'))
      assert_nil(@messg.header('CONTENT-TYPE'))
      assert_nil(@messg.header('content-type'))

      @messg.set_header('Content-Type', 'text/html')
      assert_equal('text/html', @messg.header('Content-Type'))
      assert_equal('text/html', @messg.header('CONTENT-TYPE'))
      assert_equal('text/html', @messg.header('content-type'))

      @messg.set_header('CONTENT-TYPE', 'text/html; charset=utf-8')
      assert_equal('text/html; charset=utf-8', @messg.header('Content-Type'))
      assert_equal('text/html; charset=utf-8', @messg.header('CONTENT-TYPE'))
      assert_equal('text/html; charset=utf-8', @messg.header('content-type'))

      assert_nil(@messg.header('Accept-Charset', 0))
      assert_nil(@messg.header('Accept-Charset', 1))
      assert_nil(@messg.header('Accept-Charset', 2))

      @messg.set_header('Accept-Charset', 'utf-8', true)
      @messg.set_header('Accept-Charset', 'euc-jp', true)
      @messg.set_header('Accept-Charset', 'shift_jis', true)

      assert_equal('utf-8', @messg.header('Accept-Charset', 0))
      assert_equal('euc-jp', @messg.header('Accept-Charset', 1))
      assert_equal('shift_jis', @messg.header('Accept-Charset', 2))
    end

    def test_headers
      assert(@messg.headers('Accept-Charset').empty?)
      assert(@messg.headers('ACCEPT-CHARSET').empty?)
      assert(@messg.headers('accept-charset').empty?)

      @messg.set_header('Accept-Charset', 'utf-8', true)
      @messg.set_header('Accept-Charset', 'euc-jp', true)
      @messg.set_header('Accept-Charset', 'shift_jis', true)

      assert_equal(%w[ utf-8 euc-jp shift_jis ], @messg.headers('Accept-Charset'))
      assert_equal(%w[ utf-8 euc-jp shift_jis ], @messg.headers('ACCEPT-CHARSET'))
      assert_equal(%w[ utf-8 euc-jp shift_jis ], @messg.headers('accept-charset'))
    end

    def test_delete_header
      assert_nil(@messg.delete_header('Accept-Charset', 1))
      assert_nil(@messg.delete_header('Accept-Charset'))

      @messg.set_header('Accept-Charset', 'utf-8', true)
      @messg.set_header('Accept-Charset', 'euc-jp', true)
      @messg.set_header('Accept-Charset', 'shift_jis', true)

      assert_equal(%w[ utf-8 euc-jp shift_jis ], @messg.headers('Accept-Charset'))
      assert_equal('euc-jp', @messg.delete_header('Accept-Charset', 1))
      assert_equal(%w[ utf-8 shift_jis ], @messg.headers('Accept-Charset'))
      assert_equal(%w[ utf-8 shift_jis ], @messg.delete_header('Accept-Charset'))
      assert(@messg.headers('Accept-Charset').empty?)
    end

    def test_delete_header_if
      assert(! @messg.delete_header_if('Accept-Charset') { |value| false })
      assert(! @messg.delete_header_if('Accept-Charset') { |value| true })

      @messg.set_header('Accept-Charset', 'utf-8', true)
      @messg.set_header('Accept-Charset', 'euc-jp', true)
      @messg.set_header('Accept-Charset', 'shift_jis', true)

      assert(! @messg.delete_header_if('Accept-Charset') { |value| false })
      assert_equal('utf-8', @messg.header('Accept-Charset', 0))
      assert_equal('euc-jp', @messg.header('Accept-Charset', 1))
      assert_equal('shift_jis', @messg.header('Accept-Charset', 2))

      assert(@messg.delete_header_if('Accept-Charset') { |value| value == 'euc-jp' })
      assert_equal('utf-8', @messg.header('Accept-Charset', 0))
      assert_equal('shift_jis', @messg.header('Accept-Charset', 1))
    end

    def test_has_header
      assert(! (@messg.has_header? 'Content-Type'))
      assert(! (@messg.has_header? 'CONTENT-TYPE'))
      assert(! (@messg.has_header? 'content-type'))
      assert_nil(@messg.header('Content-Type'))
      assert(@messg.headers('Content-Type').empty?)

      @messg.set_header('Content-Type', 'text/html')

      assert((@messg.has_header? 'Content-Type'))
      assert((@messg.has_header? 'CONTENT-TYPE'))
      assert((@messg.has_header? 'content-type'))
      assert_equal('text/html', @messg.header('Content-Type'))
      assert(! @messg.headers('Content-Type').empty?)
    end

    def test_each_header
      @messg.each_header do
	assert_fail('must not reached.')
      end

      @messg.set_header('Content-Type', 'text/html')
      @messg.set_header('Accept-Charset', 'utf-8', true)
      @messg.set_header('Accept-Charset', 'euc-jp', true)
      @messg.set_header('Accept-Charset', 'shift_jis', true)

      header_list = [
	%w[ Accept-Charset utf-8 ],
	%w[ Accept-Charset euc-jp ],
	%w[ Accept-Charset shift_jis ],
	%w[ Content-Type text/html ]
      ]

      @messg.each_header do |name, value|
	assert_equal(header_list[0][0], name)
	assert_equal(header_list[0][1], value)
	header_list.shift
      end
      assert(header_list.empty?)
    end

    def test_parse_header
      pio = PseudoIO.new
      pio << "Content-Type: text/html\r\n"
      pio << "Content-Length: 26\r\n"
      pio << "\r\n"
      pio << "<html>Hello world.</html>\n"

      @messg.parse_header(pio)
      assert_equal('text/html', @messg.header('Content-Type'))
      assert_equal('26', @messg.header('Content-Length'))

      assert_equal("<html>Hello world.</html>\n", pio.read)
    end

    def test_parse_header_with_LF
      pio = PseudoIO.new
      pio << "Content-Type: text/html\n"
      pio << "Content-Length: 26\n"
      pio << "\n"
      pio << "<html>Hello world.</html>\n"

      @messg.parse_header(pio)
      assert_equal('text/html', @messg.header('Content-Type'))
      assert_equal('26', @messg.header('Content-Length'))

      assert_equal("<html>Hello world.</html>\n", pio.read)
    end
  end
end
