#!/usr/local/bin/ruby
# $Id: test_control.rb,v 1.41 2004/10/20 14:37:36 toki Exp $

require 'rubyunit'
require 'wpm'
require 'wpm/rucydriver'
require 'rucy/properties'
require 'rucy/control'
require 'rucy/logger'
require 'rucy/document'
require 'rucy/passwd'

module TestRucy
  class TestControl < RUNIT::TestCase
    include Rucy::LoggingLevel

    def setup
      # for Rucy::Properties class
      @properties = Hash.new
      @list_call = 0
      @list_last_name = nil
      @set_list_call = 0
      @set_list_last_name = nil
      @map_call = 0
      @map_last_name = nil
      @set_map_call = 0
      @set_map_last_name = nil
      @params_call = 0
      @params_last_names = nil
      @set_params_call = 0
      @set_params_last_names = nil

      # for Rucy::Server class
      @start_time_call = 0
      @restart_count_call = 0
      @restart_time_call = 0
      @restart_call = 0
      @close_call = 0
      @restart_signal_call = 0
      @restart_signal = nil
      @port_call = 0
      @port = nil
      @bind_address_call = 0
      @bind_address = nil
      @timeout_call = 0
      @timeout = nil
      @keep_alive_call = 0
      @keep_alive = nil
      @max_requests_call = 0
      @max_requests = nil
      @queue_length_call = 0
      @queue_length = nil
      @messengers_call = 0
      @messengers = nil
      @messenger_factory_call = 0
      @messenger_factory = nil
      @messenger_threads_call = 0
      @messenger_threads = nil
      @messenger_queue_length_call = 0
      @messenger_queue_length = nil
      @subprocess_user_call = 0
      @subprocess_user = nil
      @subprocess_group_call = 0
      @subprocess_group = nil
      @add_logger_call = 0
      @add_access_log_call = 0
      @logging_access_call = 0
      @messg_call = 0
      @set_alias_call = 0
      @set_alias_alist = Array.new
      @set_virtual_alias_call = 0
      @set_virtual_alias_alist = Array.new
      @mount_call = 0
      @virtual_mount_call = 0
      @attach_call = 0
      @virtual_attach_call = 0
      @folder = Rucy::FolderDocument.new
      @close_hook_call = 0

      # for Rucy::PasswordEncryptor
      @pw_enc = Rucy::PasswordEncryptor.new
      @crypt_call = 0

      # for Rucy::DocumentLoader class
      @loader = Rucy::DocumentLoader.new('../mod_docs')
      @load_call = 0
      @load_errors_call = 0
      @factory_call = 0
      @RemoteAddressAllowAccessFilter_call = 0
      @BasicAuthFilter_call = 0

      @control = nil
    end

    def list(name)
      @list_call += 1
      @list_last_name = name
      if (@properties[name]) then
	assert_instance_of(Array, @properties[name])
	return @properties[name].dup
      else
	return Array.new
      end
    end

    def set_list(name, list)
      @set_list_call += 1
      @set_list_last_name = name
      @properties[name] = list.dup
      nil
    end

    def map(name)
      @map_call += 1
      @map_last_name = name
      if (@properties[name]) then
	assert_instance_of(Hash, @properties[name])
	return @properties[name].dup
      else
	return Hash.new
      end
    end

    def set_map(name, map)
      @set_map_call += 1
      @set_map_last_name = name
      @properties[name] = map.dup
      nil
    end

    def params(*names)
      @params_call += 1
      @params_last_names = names
      params = Hash.new
      for name in names
	if (@properties.include? name) then
	  params[name] = @properties[name]
	end
      end
      params
    end

    def set_params(params)
      @set_params_call += 1
      @set_params_last_names = params.keys
      for name, value in params
	@properties[name] = value
      end
      nil
    end

    def start_time
      @start_time_call += 1
      Time.mktime(2004, 10, 13, 0, 44, 31)
    end

    def restart_count
      @restart_count_call += 1
      256
    end

    def restart_time
      @restart_time_call += 1
      Time.mktime(2004, 10, 13, 0, 45, 57)
    end

    def restart
      @restart_call += 1
      nil
    end

    def close
      @close_call += 1
      nil
    end

    def restart_signal=(restart_signal)
      @restart_signal_call += 1
      @restart_signal = restart_signal
      @restart_signal.server = self
      restart_signal
    end

    def port
      @port
    end

    def port=(port)
      @port_call += 1
      @port = port
    end

    def bind_address=(new_address)
      @bind_address_call += 1
      @bind_address = new_address
    end

    def timeout=(timeout)
      @timeout_call += 1
      @timeout = timeout
    end

    def keep_alive=(keep_alive)
      @keep_alive_call += 1
      @keep_alive = keep_alive
    end

    def max_requests=(max_requests)
      @max_requests_call += 1
      @max_requests = max_requests
    end

    def queue_length=(queue_length)
      @queue_length_call += 1
      @queue_length = queue_length
    end

    def messengers=(messengers)
      @messengers_call += 1
      @messengers = messengers
    end

    def messenger_factory=(messenger_factory)
      @messenger_factory_call += 1
      @messenger_factory = messenger_factory
    end

    def messenger_threads=(messenger_threads)
      @messenger_threads_call += 1
      @messenger_threads = messenger_threads
    end

    def messenger_queue_length=(messenger_queue_length)
      @messenger_queue_length_call += 1
      @messenger_queue_length = messenger_queue_length
    end

    def subprocess_user=(subprocess_user)
      @subprocess_user_call += 1
      @subprocess_user = subprocess_user
    end

    def subprocess_group=(subprocess_group)
      @subprocess_group_call += 1
      @subprocess_group = subprocess_group
    end

    def add_logger(logger)
      @add_logger_call += 1
      nil
    end

    def add_access_log(access_log)
      @add_access_log_call += 1
      nil
    end

    def logging_access(format)
      @logging_access_call += 1
      nil
    end

    def messg(level, messg, format=true)
      @messg_call += 1
      nil
    end

    def set_alias(alias_path, orig_path)
      @set_alias_call += 1
      @set_alias_alist.push([ alias_path, orig_path ])
      @folder.set_alias(alias_path, orig_path)
      nil
    end

    def set_virtual_alias(host, alias_path, orig_path)
      @set_virtual_alias_call += 1
      @set_virtual_alias_alist.push([ host, alias_path, orig_path ])
      @folder.set_virtual_alias(host, alias_path, orig_path)
      nil
    end

    def mount(document, path, mask=nil)
      @mount_call += 1
      @folder.mount(document, path, mask)
      nil
    end

    def virtual_mount(host, document, path, mask=nil)
      @virtual_mount_call += 1
      @folder.virtual_mount(host, document, path, mask)
      nil
    end

    def attach(filter, path, mask=nil)
      @attach_call += 1
      @folder.attach(filter, path, mask)
      nil
    end

    def virtual_attach(host, document, path, mask=nil)
      @virtual_attach_call += 1
      @folder.virtual_attach(host, document, path, mask)
      nil
    end

    def close_hook(&block)
      @close_hook_call += 1
      nil
    end

    def crypt(password)
      @crypt_call += 1
      @pw_enc.crypt(password)
    end

    def load
      @load_call += 1
      @loader.load
      nil
    end

    def load_errors
      @load_errors_call += 1
      [ [ 'foo.rb', RuntimeError.new('HALO') ] ]
    end

    def factory
      @factory_call += 1
      @loader.factory
    end

    def RemoteAddressAllowAccessFilter
      @RemoteAddressAllowAccessFilter_call += 1
      @loader.RemoteAddressAllowAccessFilter
    end

    def BasicAuthFilter
      @BasicAuthFilter_call += 1
      @loader.BasicAuthFilter
    end

    def build_control
      @control = Rucy::Control.new
      @control.loader = self
      @control.pw_enc = self
      @control.properties = self
      @control.page_path = 'nothing'
    end
    private :build_control

    def test_load_errors
      build_control
      err_alist = @control.load_errors
      assert_equal(1, @load_errors_call)
      assert_equal(1, err_alist.length)
      assert_equal('foo.rb', err_alist[0][0])
      assert_instance_of(RuntimeError, err_alist[0][1])
      assert_equal('HALO', err_alist[0][1].message)
    end

    def test_server_restart
      build_control
      @control.server_setup(self)
      assert_equal(0, @restart_call)
      @control.server_restart
      @restart_signal.wait
      assert_equal(1, @restart_call)
    end

    def test_server_close
      build_control
      @control.server_setup(self)
      assert_equal(0, @close_call)
      @control.server_close
      @restart_signal.wait
      assert_equal(1, @close_call)
    end

    def test_server_params
      build_control
      server_params = @control.server_params
      assert_equal(1, @params_call)
      @control.set_server_params(server_params)
      assert_equal(1, @set_params_call)
      assert_exception(RuntimeError) {
	@control.set_server_params(Hash.new)
      }
    end

    def test_logging_params
      build_control
      logging_params = @control.logging_params
      assert_equal(1, @params_call)
      @control.set_logging_params(logging_params)
      assert_equal(1, @set_params_call)
      assert_exception(RuntimeError) {
	@control.set_logging_params(Hash.new)
      }
    end

    def test_aliases
      build_control
      alias_list = @control.aliases
      assert_equal(1, @list_call)
      @control.set_aliases(alias_list)
      assert_equal(1, @set_list_call)
      assert_exception(RuntimeError) {
	@control.set_aliases(Array.new)
      }
    end

    def test_documents
      build_control
      document_list = @control.documents
      assert_equal(1, @list_call)
      assert_equal('documents', @list_last_name)
      assert(document_list.empty?)
      document_list.push({ 'document' => 'Page',
			   'arguments' => [
			     "Hello world.\n",
			     'text/plain'
			   ],
			   'mount_path' => '/hello',
			   'mount_mask' => nil,
			   'virtual_host' => nil,
			   'comment' => 'foo'
			 })
      document_list.push({ 'document' => 'LocalFile',
			   'arguments' => [
			     '/home/alice/public_html'
			   ],
			   'mount_path' => '/~alice',
			   'mount_mask' => nil,
			   'virtual_host' => 'foo',
			   'comment' => 'bar'
			 })
      @control.set_documents(document_list)
      assert_equal(1, @set_list_call)
      assert_equal('documents', @set_list_last_name)
      assert_equal(2, @properties['documents'].length)
      assert_equal({ 'document' => 'Page',
		     'arguments' => [
		       "Hello world.\n",
		       'text/plain'
		     ],
		     'mount_path' => '/hello',
		     'mount_mask' => nil,
		     'virtual_host' => nil,
		     'comment' => 'foo'
		   }, @properties['documents'][0])
      assert_equal({ 'document' => 'LocalFile',
		     'arguments' => [
		       '/home/alice/public_html'
		     ],
		     'mount_path' => '/~alice',
		     'mount_mask' => nil,
		     'virtual_host' => 'foo',
		     'comment' => 'bar'
		   }, @properties['documents'][1])
    end

    def test_filters
      build_control
      filter_list = @control.filters
      assert_equal(1, @list_call)
      assert_equal('filters', @list_last_name)
      assert(filter_list.empty?)
      filter_list.push({ 'filter' => 'ERB',
			 'arguments' => [
			   1
			 ],
			 'attach_path' => '/foo',
			 'attach_mask' => %r"\.rhtml?($|/)",
			 'virtual_host' => nil,
			 'comment' => 'apple'
		       })
      filter_list.push({ 'filter' => 'SSI',
			 'arguments' => [
			   { 'config' => true,
			     'include' => true,
			     'echo' => true,
			     'fsize' => true,
			     'flastmod' => true,
			     'exec' => true
			   }
			 ],
			 'attach_path' => '/bar',
			 'attach_mask' => %r"\.shtml?($|/)",
			 'virtual_host' => 'baz',
			 'comment' => 'banana'
		       })
      @control.set_filters(filter_list)
      assert_equal(1, @set_list_call)
      assert_equal('filters', @set_list_last_name)
      assert_equal(2, @properties['filters'].length)
      assert_equal({ 'filter' => 'ERB',
		     'arguments' => [
		       1
		     ],
		     'attach_path' => '/foo',
		     'attach_mask' => %r"\.rhtml?($|/)",
		     'virtual_host' => nil,
		     'comment' => 'apple'
		   }, @properties['filters'][0])
      assert_equal({ 'filter' => 'SSI',
		     'arguments' => [
		       { 'config' => true,
			 'include' => true,
			 'echo' => true,
			 'fsize' => true,
			 'flastmod' => true,
			 'exec' => true
		       }
		     ],
		     'attach_path' => '/bar',
		     'attach_mask' => %r"\.shtml?($|/)",
		     'virtual_host' => 'baz',
		     'comment' => 'banana'
		   }, @properties['filters'][1])
    end

    def test_default_server_settings
      build_control
      @control.server_setup(self)
      assert_equal(4, @params_call) # for server parameters and logging parameters and access log parameters and admin parameters
      assert_equal(3, @list_call) # for (1) alias list and (2) document list and (3) filter list
      assert_equal(1, @load_call)
      assert_equal(1, @factory_call)
      assert_equal(1, @restart_signal_call)
      assert_instance_of(Rucy::MultiThreadRestartSignal, @restart_signal)
      assert_equal(1, @port_call)
      assert_equal(8888, @port)
      assert_equal(1, @bind_address_call)
      assert_equal(nil, @bind_address)
      assert_equal(1, @timeout_call)
      assert_equal(300, @timeout)
      assert_equal(1, @keep_alive_call)
      assert_equal(8, @keep_alive)
      assert_equal(1, @max_requests_call)
      assert_equal(32, @max_requests)
      assert_equal(1, @queue_length_call)
      assert_equal(16, @queue_length)
      assert_equal(1, @messengers_call)
      assert_equal(8, @messengers)
      assert_equal(1, @messenger_factory_call)
      assert_equal(Rucy::MultiThreadMessenger, @messenger_factory)
      assert_equal(1, @messenger_threads_call)
      assert_equal(4, @messenger_threads)
      assert_equal(1, @messenger_queue_length_call)
      assert_equal(4, @messenger_queue_length)
      assert_equal(1, @subprocess_user_call)
      assert_equal('nobody', @subprocess_user)
      assert_equal(1, @subprocess_group_call)
      assert_equal('nobody', @subprocess_group)
      assert_equal(1, @add_logger_call)
      assert_equal(1, @close_hook_call)
      assert_equal(1, @start_time_call)
      assert_equal(2004, @control.start_time.year)
      assert_equal(10, @control.start_time.month)
      assert_equal(13, @control.start_time.day)
      assert_equal(0, @control.start_time.hour)
      assert_equal(44, @control.start_time.min)
      assert_equal(31, @control.start_time.sec)
      assert_equal(1, @restart_count_call)
      assert_equal(256, @control.restart_count)
      assert_equal(1, @restart_time_call)
      assert_equal(2004, @control.restart_time.year)
      assert_equal(10, @control.restart_time.month)
      assert_equal(13, @control.restart_time.day)
      assert_equal(0, @control.restart_time.hour)
      assert_equal(45, @control.restart_time.min)
      assert_equal(57, @control.restart_time.sec)
    end

    def test_modified_server_settings
      @properties['server_type'] = 'multiprocess'
      @properties['port'] = 80
      @properties['bind_address'] = 'localhost'
      @properties['timeout'] = 60
      @properties['keep_alive'] = 4
      @properties['max_requests'] = 24
      @properties['queue_length'] = 32
      @properties['messengers'] = 16
      @properties['messenger_threads'] = 8
      @properties['messenger_queue_length'] = 8
      @properties['subprocess_user'] = 'httpd'
      @properties['subprocess_group'] = 'httpd'
      build_control
      @control.server_setup(self)
      assert_equal(4, @params_call) # for server parameters and logging parameters and access log parameters and admin parameters
      assert_equal(3, @list_call) # for (1) alias list and (2) document list and (3) filter list
      assert_equal(1, @load_call)
      assert_equal(1, @factory_call)
      assert_equal(1, @restart_signal_call)
      assert_instance_of(Rucy::MultiProcessRestartSignal, @restart_signal)
      assert_equal(1, @port_call)
      assert_equal(80, @port)
      assert_equal(1, @bind_address_call)
      assert_equal('localhost', @bind_address)
      assert_equal(1, @timeout_call)
      assert_equal(60, @timeout)
      assert_equal(1, @keep_alive_call)
      assert_equal(4, @keep_alive)
      assert_equal(1, @max_requests_call)
      assert_equal(24, @max_requests)
      assert_equal(1, @queue_length_call)
      assert_equal(32, @queue_length)
      assert_equal(1, @messengers_call)
      assert_equal(16, @messengers)
      assert_equal(1, @messenger_factory_call)
      assert_equal(Rucy::MultiProcessMessenger, @messenger_factory)
      assert_equal(1, @messenger_threads_call)
      assert_equal(8, @messenger_threads)
      assert_equal(1, @messenger_queue_length_call)
      assert_equal(8, @messenger_queue_length)
      assert_equal(1, @subprocess_user_call)
      assert_equal('httpd', @subprocess_user)
      assert_equal(1, @subprocess_group_call)
      assert_equal('httpd', @subprocess_group)
      assert_equal(1, @add_logger_call)
      assert_equal(1, @close_hook_call)
      assert_equal(1, @start_time_call)
      assert_equal(2004, @control.start_time.year)
      assert_equal(10, @control.start_time.month)
      assert_equal(13, @control.start_time.day)
      assert_equal(0, @control.start_time.hour)
      assert_equal(44, @control.start_time.min)
      assert_equal(31, @control.start_time.sec)
      assert_equal(1, @restart_count_call)
      assert_equal(256, @control.restart_count)
      assert_equal(1, @restart_time_call)
      assert_equal(2004, @control.restart_time.year)
      assert_equal(10, @control.restart_time.month)
      assert_equal(13, @control.restart_time.day)
      assert_equal(0, @control.restart_time.hour)
      assert_equal(45, @control.restart_time.min)
      assert_equal(57, @control.restart_time.sec)
    end

    def test_admin_params
      build_control
      admin_params = @control.admin_params
      assert_equal(1, @params_call)
      assert_equal(%w[ admin_password admin_user japanese_handling localhost_only ], @params_last_names.sort)
      assert_equal('admin', admin_params.admin_user)
      assert_equal('', admin_params.admin_password)
      assert_equal(true, admin_params.localhost_only)
      assert_equal(false, admin_params.japanese_handling)
    end

    def test_set_admin_params
      build_control
      admin_params = @control.admin_params
      assert_equal(1, @params_call)
      assert_equal(%w[ admin_password admin_user japanese_handling localhost_only ], @params_last_names.sort)
      assert_equal('admin', admin_params.admin_user)
      assert_equal('', admin_params.admin_password)
      assert_equal(true, admin_params.localhost_only)
      admin_params.localhost_only = false
      @control.set_admin_params(admin_params)
      assert_equal(1, @set_params_call)
      assert_equal(%w[ admin_password admin_user japanese_handling localhost_only ], @set_params_last_names.sort)
      assert_equal('admin', @properties['admin_user'])
      assert_equal('', @properties['admin_password'])
      assert_equal(true, @properties['localhost_only'])
      admin_params.admin_password = @pw_enc.crypt('nazo')
      @control.set_admin_params(admin_params)
      assert_equal(2, @set_params_call)
      assert_equal(%w[ admin_password admin_user japanese_handling localhost_only ], @set_params_last_names.sort)
      assert_equal('admin', @properties['admin_user'])
      assert(@properties['admin_password'] != 'nazo')
      assert(@properties['admin_password'] == @pw_enc.crypt('nazo', @properties['admin_password']))
      assert_equal(true, @properties['localhost_only'])
      admin_params.localhost_only = false
      @control.set_admin_params(admin_params)
      assert_equal(3, @set_params_call)
      assert_equal(%w[ admin_password admin_user japanese_handling localhost_only ], @set_params_last_names.sort)
      assert_equal('admin', @properties['admin_user'])
      assert(@properties['admin_password'] != 'nazo')
      assert(@properties['admin_password'] == @pw_enc.crypt('nazo', @properties['admin_password']))
      assert_equal(false, @properties['localhost_only'])
    end

    def test_ControlPanel_document
      build_control
      @control.server_setup(self)
      assert_equal(1, @load_call)
      assert_equal(1, @factory_call)
      assert_equal(1, @RemoteAddressAllowAccessFilter_call)
      assert_equal(0, @BasicAuthFilter_call)
      assert_equal(1, @mount_call)
      assert_equal(1, @attach_call)
      assert_equal(0, @virtual_mount_call)
      assert_equal(0, @virtual_attach_call)
      document = @folder.find('/control')[0]
      assert_instance_of(Rucy::FilterDocument, document)
      assert_instance_of(@loader.RemoteAddressAllowAccessFilter, document.filter)
      assert_instance_of(Rucy::WebPageMakerDocument, document.document)
      assert_equal('/control', @folder.find('/control')[1])
    end

    def test_ControlPanel_with_BasicAuth
      @properties['admin_user'] = 'admin'
      @properties['admin_password'] = @pw_enc.crypt('nazo')
      @properties['localhost_only'] = false
      build_control
      @control.server_setup(self)
      assert_equal(1, @load_call)
      assert_equal(1, @factory_call)
      assert_equal(0, @RemoteAddressAllowAccessFilter_call)
      assert_equal(1, @BasicAuthFilter_call)
      assert_equal(1, @mount_call)
      assert_equal(1, @attach_call)
      assert_equal(0, @virtual_mount_call)
      assert_equal(0, @virtual_attach_call)
      document = @folder.find('/control')[0]
      assert_instance_of(Rucy::FilterDocument, document)
      assert_instance_of(@loader.BasicAuthFilter, document.filter)
      assert_instance_of(Rucy::WebPageMakerDocument, document.document)
      assert_equal('/control', @folder.find('/control')[1])
    end

    def test_ControlPanel_with_BasicAuth_and_LocalhostOnly
      @properties['admin_user'] = 'admin'
      @properties['admin_password'] = @pw_enc.crypt('nazo')
      @properties['localhost_only'] = true
      build_control
      @control.server_setup(self)
      assert_equal(1, @load_call)
      assert_equal(1, @factory_call)
      assert_equal(1, @RemoteAddressAllowAccessFilter_call)
      assert_equal(1, @BasicAuthFilter_call)
      assert_equal(1, @mount_call)
      assert_equal(2, @attach_call)
      assert_equal(0, @virtual_mount_call)
      assert_equal(0, @virtual_attach_call)
      document = @folder.find('/control')[0]
      assert_instance_of(Rucy::FilterDocument, document)
      assert_instance_of(@loader.BasicAuthFilter, document.filter)
      document = document.document
      assert_instance_of(Rucy::FilterDocument, document)
      assert_instance_of(@loader.RemoteAddressAllowAccessFilter, document.filter)
      assert_instance_of(Rucy::WebPageMakerDocument, document.document)
      assert_equal('/control', @folder.find('/control')[1])
    end

    def test_logging_setup
      @properties['stdout_logging_level'] = 'info'
      @properties['logfiles'] = [
	{ 'path' => 'server.log',
	  'logging_level' => 'notice'
	},
	{ 'path' => 'access.log',
	  'logging_level' => 'info'
	},
	{ 'path' => 'debug.log',
	  'logging_level' => 'debug'
	}
      ]

      begin
	build_control
	@control.server_setup(self)
	assert_equal(4, @add_logger_call)
	assert((File.file? 'server.log'))
	assert((File.file? 'access.log'))
	assert((File.file? 'debug.log'))
	@control.logfiles_close
      ensure
	[ 'server.log',
	  'access.log',
	  'debug.log'
	].each do |filename|
	  if (File.exist? filename) then
	    File.delete(filename)
	  end
	end
      end
    end

    def test_logging_errors
      @properties['logfiles'] = [
	{ 'path' => 'detarame/server.log',
	  'logging_level' => 'notice'
	}
      ]
      build_control
      @control.server_setup(self)
      assert_equal(1, @add_logger_call)
      assert_equal(1, @control.logging_errors.length)
      assert_equal('detarame/server.log', @control.logging_errors[0][:logfile])
      assert_instance_of(Errno::ENOENT, @control.logging_errors[0][:exception])
    end

    def test_alias_setup
      @properties['aliases'] = [
	{ 'virtual_host' => nil,
	  'alias_path' => '/bar',
	  'orig_path' => '/foo'
	},
	{ 'virtual_host' => 'foo',
	  'alias_path' => '/fuge',
	  'orig_path' => '/hoge'
	}
      ]
      build_control
      @control.server_setup(self)
      assert_equal(1, @set_alias_call)
      assert_equal([ [ '/bar', '/foo' ] ], @set_alias_alist)
      assert_equal(1, @set_virtual_alias_call)
      assert_equal([ [ 'foo:8888', '/fuge', '/hoge' ] ], @set_virtual_alias_alist)
    end

    def test_alias_setup_error
      @properties['aliases'] = [
	{ 'virtual_host' => nil,
	  'alias_path' => 'bar',
	  'orig_path' => 'foo'
	}
      ]
      build_control
      @control.server_setup(self)
      assert_equal(1, @set_alias_call)
      assert_equal(1, @control.alias_errors.length)
      assert_equal(nil, @control.alias_errors[0][:virtual_host])
      assert_equal('bar', @control.alias_errors[0][:alias_path])
      assert_equal('foo', @control.alias_errors[0][:orig_path])
      assert_instance_of(RuntimeError, @control.alias_errors[0][:exception])
    end

    def test_document_setup
      @properties['documents'] = [
	{ 'document' => 'Page',
	  'arguments' => [
	    "Hello world.\n",
	    'text/html'
	  ],
	  'mount_path' => '/hello',
	  'mount_mask' => nil,
	  'virtual_host' => nil
	},
	{ 'document' => 'LocalFile',
	  'arguments' => [
	    '/home/alice/public_html',
	    nil
	  ],
	  'mount_path' => '/~alice',
	  'mount_mask' => nil,
	  'virtual_host' => 'foo'
	}
      ]
      build_control
      @control.server_setup(self)
      assert_equal(2, @mount_call)
      assert_equal(1, @virtual_mount_call)
      assert_instance_of(@loader.PageDocument, @folder.find('/hello')[0])
      assert_equal('/hello', @folder.find('/hello')[1])
      assert_instance_of(@loader.LocalFileDocument, @folder.virtual_find('foo:8888', '/~alice')[0])
      assert_equal('/~alice', @folder.virtual_find('foo:8888', '/~alice')[1])
    end

#     def test_document_setup_error
#       error = Object.new
#       class << error
# 	def set_doc_option(option)
# 	end

# 	def doc_name
# 	  'TestError'
# 	end

# 	def doc_args
# 	  []
# 	end

# 	def new
# 	  raise 'exception on creating a document.'
# 	end
#       end
#       @factory.add_document(error)
#       @properties['documents'] = [
# 	{ 'document' => 'TestError',
# 	  'arguments' => [],
# 	  'mount_path' => '/error',
# 	  'mount_mask' => nil,
# 	  'virtual_host' => 'foo'
# 	}
#       ]
#       build_control
#       @control.server_setup(self)
#       assert_equal(1, @mount_call)
#       assert_equal(0, @virtual_mount_call)
#       assert_nil(@folder.find('/error'))
#       assert_equal(1, @control.doc_errors.length)
#       assert_equal('TestError', @control.doc_errors[0][:document])
#       assert_equal([], @control.doc_errors[0][:arguments])
#       assert_equal('/error', @control.doc_errors[0][:mount_path])
#       assert_nil(@control.doc_errors[0][:mount_mask])
#       assert_equal('foo', @control.doc_errors[0][:virtual_host])
#       assert_instance_of(RuntimeError, @control.doc_errors[0][:exception])
#       assert_equal('exception on creating a document.', @control.doc_errors[0][:exception].message)
#     end

    def test_filter_setup
      @properties['documents'] = [
	{ 'document' => 'Page',
	  'arguments' => [
	    "Hello world.\n",
	    'text/html'
	  ],
	  'mount_path' => '/',
	  'mount_mask' => nil,
	  'virtual_host' => nil
	},
	{ 'document' => 'Page',
	  'arguments' => [
	    "Hello virtual world.\n",
	    'text/html'
	  ],
	  'mount_path' => '/',
	  'mount_mask' => nil,
	  'virtual_host' => 'baz'
	}
      ]
      @properties['filters'] = [
	{ 'filter' => 'ERB',
	  'arguments' => [ '1', false ],
	  'attach_path' => '/foo',
	  'attach_mask' => %r"\.rhtml?($|/)",
	  'virtual_host' => nil
	},
	{ 'filter' => 'ERB',
	  'arguments' => [ '1', true ],
	  'attach_path' => '/bar',
	  'attach_mask' => %r"\.rhtml?($|/)",
	  'virtual_host' => 'baz'
	}
      ]
      build_control
      @control.server_setup(self)
      assert_equal(2, @attach_call)
      assert_equal(1, @virtual_attach_call)
      assert_instance_of(@loader.ERBFilter, @folder.find('/foo/index.rhtml')[0].filter)
      assert_equal('/foo', @folder.find('/foo/index.rhtml')[1])
      assert_instance_of(@loader.ERBFilter, @folder.virtual_find('baz:8888', '/bar/index.rhtml')[0].filter)
      assert_equal('/bar', @folder.virtual_find('baz:8888', '/bar/index.shtml')[1])
    end

#     def test_filter_setup_error
#       @properties['documents'] = [
# 	{ 'document' => 'Page',
# 	  'arguments' => [
# 	    "Hello world.\n",
# 	    'text/html'
# 	  ],
# 	  'mount_path' => '/',
# 	  'mount_mask' => nil,
# 	  'virtual_host' => nil
# 	},
# 	{ 'document' => 'Page',
# 	  'arguments' => [
# 	    "Hello virtual world.\n",
# 	    'text/html'
# 	  ],
# 	  'mount_path' => '/',
# 	  'mount_mask' => nil,
# 	  'virtual_host' => 'baz'
# 	}
#       ]
#       error = Object.new
#       class << error
# 	def set_filter_option(option)
# 	end

# 	def filter_name
# 	  'TestError'
# 	end

# 	def filter_args
# 	  []
# 	end

# 	def new
# 	  raise 'exception on creating a filter.'
# 	end
#       end
#       @factory.add_filter(error)
#       @properties['filters'] = [
# 	{ 'filter' => 'TestError',
# 	  'arguments' => [],
# 	  'attach_path' => '/error',
# 	  'attach_mask' => nil,
# 	  'virtual_host' => 'foo'
# 	}
#       ]
#       build_control
#       @control.server_setup(self)
#       assert_equal(1, @attach_call)
#       assert_equal(0, @virtual_attach_call)
#       assert_equal(1, @control.filter_errors.length)
#       assert_equal('TestError', @control.filter_errors[0][:filter])
#       assert_equal([], @control.filter_errors[0][:arguments])
#       assert_equal('/error', @control.filter_errors[0][:attach_path])
#       assert_nil(@control.filter_errors[0][:attach_mask])
#       assert_equal('foo', @control.filter_errors[0][:virtual_host])
#       assert_instance_of(RuntimeError, @control.filter_errors[0][:exception])
#       assert_equal('exception on creating a filter.', @control.filter_errors[0][:exception].message)
#     end
  end
end
