# $Id: test_cgi.rb,v 1.4 2004/09/29 14:43:08 toki Exp $

require 'rubyunit'
require 'pseudo_req_res'
require 'pseudo_io'
require 'rucy/request'
require 'rucy/response'
require 'rucy/loader'

module TestRucy
  module TestCGI
    CGI_DIR = 'cgi'
    TEST_CGI = 'cgi/test.cgi'

    def setup
      super
      Dir.mkdir(CGI_DIR)
    end

    def make_cgi
      File.open(TEST_CGI, 'w') { |output|
	output.binmode
	yield(output)
      }
      nil
    end
    private :make_cgi

    def teardown
      File.delete(TEST_CGI) if (File.exist? TEST_CGI)
      Dir.delete(CGI_DIR)
    end
  end

  class TestCGIDocument < RUNIT::TestCase
    include PseudoLogger
    include PseudoRequestResponse
    include TestCGI

    def setup
      super
      @loader = Rucy::SimpleLoader.new
      @loader.load('../mod_docs/cgi.rb')
      @CGIDocument = @loader.const_get('CGIDocument')
      @cgi = @CGIDocument.new(CGI_DIR, 'ruby')
    end

    def test_CGIDocument_quote_sh
      assert_equal('"Hello world."', @CGIDocument.quote_sh('Hello world.'))
      assert_equal('"\\""', @CGIDocument.quote_sh('"'))
      assert_equal('"\\$"', @CGIDocument.quote_sh('$'))
      assert_equal('"\\`"', @CGIDocument.quote_sh('`'))
      assert_equal('"\\\\"', @CGIDocument.quote_sh('\\'))
    end

    def test_cgi
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print \"Hello world.\\n\"\n"
      }
      @request.method = 'GET'
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("Hello world.\n", @messg_body)
    end

    def test_cgi_status
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"Status: 201 Created\r\\n\"\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print \"Hello world.\\n\"\n"
      }
      @request.method = 'GET'
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(201, @messg_head.status)
      assert_equal('Created', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("Hello world.\n", @messg_body)
    end

    def test_cgi_location
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"Location: /foo/bar\\r\\n\"\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print \"Hello world.\\n\"\n"
      }
      @request.method = 'GET'
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(302, @messg_head.status)
      assert_equal('Found', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('http://server:8080/foo/bar', @messg_head.header('Location'))
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("Hello world.\n", @messg_body)
    end

    def test_cgi_args
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "for arg in ARGV\n"
	output.print "  print arg, \"\n\"\n"
	output.print "end\n"
      }
      @request.method = 'GET'
      @request.uri = 
      @request.uri = '/test.cgi?foo+bar'
      @request.version = 'HTTP/1.1'
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('', @messg_body)
    end

    def test_cgi_args2
      @cgi.pass_args = true
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "for arg in ARGV\n"
	output.print "  print arg, \"\n\"\n"
	output.print "end\n"
      }
      @request.method = 'GET'
      @request.uri = 
      @request.uri = '/test.cgi?foo+bar'
      @request.version = 'HTTP/1.1'
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("foo\nbar\n", @messg_body)
    end

    def test_cgi_auth
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print ENV['HTTP_AUTHORIZATION'].to_s, \"\n\"\n"
      }
      @request.method = 'GET'
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      @request.set_header('Authorization', 'Basic YWxpY2U6Zm9v')
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("\n", @messg_body)
    end

    def test_cgi_auth2
      @cgi.pass_auth = true
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print ENV['HTTP_AUTHORIZATION'].to_s, \"\n\"\n"
      }
      @request.method = 'GET'
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      @request.set_header('Authorization', 'Basic YWxpY2U6Zm9v')
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("Basic YWxpY2U6Zm9v\n", @messg_body)
    end

    def test_cgi_path_info
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print ENV['PATH_INFO'].to_s, \"\n\"\n"
      }
      @request.method = 'GET'
      @request.uri = 
      @request.uri = '/test.cgi/foo/bar'
      @request.version = 'HTTP/1.1'
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("/foo/bar\n", @messg_body)
    end

    def test_cgi_env
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"Content-Type: application/octet-stream\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "Marshal.dump(ENV.to_hash, STDOUT)\n"
      }
      @cgi.env['FOO'] = 'Alice'
      @request.method = 'GET'
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      @request.set_header('User-Agent', 'TestRucy')
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('application/octet-stream', @messg_head.header('Content-Type'))
      env = Marshal.load(@messg_body)
      assert(! (ENV.include? 'FOO'))
      assert(! (ENV.include? 'HTTP_USER_AGENT'))
      assert((env.include? 'FOO'))
      assert_equal('Alice', env['FOO'])
      assert((env.include? 'HTTP_USER_AGENT'))
      assert_equal('TestRucy', env['HTTP_USER_AGENT'])
    end

    def test_cgi_post
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print STDIN.read\n"
      }
      @request.method = 'POST'
      @request.uri = 
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      messg = "Hello world.\n"
      @request.set_header('Content-Type', 'text/plain')
      @request.set_header('Content-Length', messg.length.to_s)
      post_reader = PseudoIO.new
      post_reader.write(messg)
      @request.set_reader(post_reader)
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("Hello world.\n", @messg_body)
    end
  end

  class TestNonParsedHeaderCGIDocument < RUNIT::TestCase
    include PseudoLogger
    include PseudoRequestResponse
    include TestCGI

    def setup
      super
      @loader = Rucy::SimpleLoader.new
      @loader.load('../mod_docs/cgi.rb')
      @CGIDocument = @loader.const_get('CGIDocument')
      @cgi = @CGIDocument.new('cgi', 'ruby', true)
    end

    def test_cgi
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"HTTP/1.1 200 OK\\r\\n\"\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print \"Hello world.\\n\"\n"
      }
      @request.method = 'GET'
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("Hello world.\n", @messg_body)
    end

    def test_cgi_location
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"HTTP/1.1 302 Found\\r\\n\"\n"
	output.print "print \"Location: /foo/bar\\r\\n\"\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print \"Hello world.\\n\"\n"
      }
      @request.method = 'GET'
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(302, @messg_head.status)
      assert_equal('Found', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('/foo/bar', @messg_head.header('Location')) # not converted to absolute-URI
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("Hello world.\n", @messg_body)
    end

    def test_cgi_args
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"HTTP/1.1 200 OK\\r\\n\"\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "for arg in ARGV\n"
	output.print "  print arg, \"\n\"\n"
	output.print "end\n"
      }
      @request.method = 'GET'
      @request.uri = 
      @request.uri = '/test.cgi?foo+bar'
      @request.version = 'HTTP/1.1'
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('', @messg_body)
    end

    def test_cgi_args2
      @cgi.pass_args = true
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"HTTP/1.1 200 OK\\r\\n\"\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "for arg in ARGV\n"
	output.print "  print arg, \"\n\"\n"
	output.print "end\n"
      }
      @request.method = 'GET'
      @request.uri = 
      @request.uri = '/test.cgi?foo+bar'
      @request.version = 'HTTP/1.1'
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("foo\nbar\n", @messg_body)
    end

    def test_cgi_auth
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"HTTP/1.1 200 OK\\r\\n\"\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print ENV['HTTP_AUTHORIZATION'].to_s, \"\n\"\n"
      }
      @request.method = 'GET'
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      @request.set_header('Authorization', 'Basic YWxpY2U6Zm9v')
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("\n", @messg_body)
    end

    def test_cgi_auth2
      @cgi.pass_auth = true
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"HTTP/1.1 200 OK\\r\\n\"\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print ENV['HTTP_AUTHORIZATION'].to_s, \"\n\"\n"
      }
      @request.method = 'GET'
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      @request.set_header('Authorization', 'Basic YWxpY2U6Zm9v')
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("Basic YWxpY2U6Zm9v\n", @messg_body)
    end

    def test_cgi_path_info
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"HTTP/1.1 200 OK\\r\\n\"\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print ENV['PATH_INFO'].to_s, \"\n\"\n"
      }
      @request.method = 'GET'
      @request.uri = 
      @request.uri = '/test.cgi/foo/bar'
      @request.version = 'HTTP/1.1'
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("/foo/bar\n", @messg_body)
    end

    def test_cgi_env
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"HTTP/1.1 200 OK\\r\\n\"\n"
	output.print "print \"Content-Type: application/octet-stream\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "Marshal.dump(ENV.to_hash, STDOUT)\n"
      }
      @cgi.env['FOO'] = 'Alice'
      @request.method = 'GET'
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      @request.set_header('User-Agent', 'TestRucy')
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('application/octet-stream', @messg_head.header('Content-Type'))
      env = Marshal.load(@messg_body)
      assert(! (ENV.include? 'FOO'))
      assert(! (ENV.include? 'HTTP_USER_AGENT'))
      assert((env.include? 'FOO'))
      assert_equal('Alice', env['FOO'])
      assert((env.include? 'HTTP_USER_AGENT'))
      assert_equal('TestRucy', env['HTTP_USER_AGENT'])
    end

    def test_cgi_post
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"HTTP/1.1 200 OK\\r\\n\"\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print STDIN.read\n"
      }
      @request.method = 'POST'
      @request.uri = 
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      messg = "Hello world.\n"
      @request.set_header('Content-Type', 'text/plain')
      @request.set_header('Content-Length', messg.length.to_s)
      post_reader = PseudoIO.new
      post_reader.write(messg)
      @request.set_reader(post_reader)
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("Hello world.\n", @messg_body)
    end
  end

  class TestCGIDocumentFactory < RUNIT::TestCase
    def setup
      @loader = Rucy::SimpleLoader.new
      @loader.load('../mod_docs/cgi.rb')
      @CGIDocumentFactory = @loader.const_get('CGIDocumentFactory')
      @factory = @CGIDocumentFactory.instance
    end

    def test_set_doc_option
      @factory.set_doc_option(Hash.new)
    end

    def test_doc_name
      assert_equal('CGI', @factory.doc_name)
    end

    def test_doc_args
      args = @factory.doc_args
      assert_equal([ 'CGI directory', :string, nil ], args[0])
      assert_equal([ 'CGI run command (optional)', :string, nil ], args[1])
      assert_equal([ 'none parsed header', :bool, false ], args[2])
      assert_equal([ 'pass CGI arguments', :bool, false ], args[3])
      assert_equal([ 'pass authorization header', :bool, false ], args[4])
      assert_equal([ 'environment variable name 1 (optional)', :string, nil ], args[5])
      assert_equal([ 'environment variable value 1 (optional)', :string, nil ], args[6])
      assert_equal([ 'environment variable name 2 (optional)', :string, nil ], args[7])
      assert_equal([ 'environment variable value 2 (optional)', :string, nil ], args[8])
      assert_equal([ 'environment variable name 3 (optional)', :string, nil ], args[9])
      assert_equal([ 'environment variable value 3 (optional)', :string, nil ], args[10])
      assert_equal([ 'environment variable name 4 (optional)', :string, nil ], args[11])
      assert_equal([ 'environment variable value 4 (optional)', :string, nil ], args[12])
      assert_equal([ 'environment variable name 5 (optional)', :string, nil ], args[13])
      assert_equal([ 'environment variable value 5 (optional)', :string, nil ], args[14])
    end

    def test_new
      doc = @factory.new('/', nil, false, false, false,
			 nil, nil,
			 nil, nil,
			 nil, nil,
			 nil, nil,
			 nil, nil)
      assert_instance_of(@loader.const_get('CGIDocument'), doc)
    end
  end
end
