# $Id: params.rb,v 1.59 2004/10/20 14:51:53 toki Exp $

require 'wpm'
require 'wpm/rucydriver'
require 'wpm/rexml'
require 'rucy/restart'
require 'rucy/document'
require 'rucy/messenger'
require 'rucy/logger'
require 'rucy/passwd'
require 'forwardable'
require 'socket'

module Rucy
  module LoadProperties
    def load(properties)
      params = self.new
      params.load(properties)
      params
    end
  end

  class AdminParams
    extend Forwardable
    extend LoadProperties

    def initialize
      @params = nil
    end

    def load(properties)
      @params = properties.params('admin_user', 
				  'admin_password',
				  'localhost_only',
				  'japanese_handling')

      @params['admin_user'] = 'admin' unless (@params.include? 'admin_user')
      @params['admin_password'] = '' unless (@params.include? 'admin_password')
      @params['localhost_only'] = true unless (@params.include? 'localhost_only')
      @params['japanese_handling'] = false unless (@params.include? 'japanese_handling')

      nil
    end

    def save(properties)
      properties.set_params(@params)
      nil
    end

    def server_setup(control, server, loader, doc_factory, pw_enc, page_path)
      page_maker = WebPageMakerDocument.new(WPM::XMLAssistByREXML, page_path, 'ControlPanel')
      page_maker.debug = true
      page_maker.set_page_option(:control, control)
      page_maker.set_page_option(:factory, doc_factory)
      page_maker.set_page_option(:pw_enc, pw_enc)
      page_maker.set_page_option(:base_dir, File.dirname($0))
      if (@params['japanese_handling']) then
	require 'rucy/jconv'
	page_maker.set_input_filter{|data| Jconv.jconv(data) }
	page_maker.set_output_filter{|data| Uconv.u8toeuc(data) }
	page_maker.charset = 'euc-jp'
      end
      server.mount(page_maker, '/control')
      if (@params['localhost_only']) then
	access = loader.RemoteAddressAllowAccessFilter.new
	access.add_allow_address('127.0.0.1')
	server.attach(access, '/control')
      end
      if (@params['admin_user'] && ! @params['admin_user'].empty? &&
	  @params['admin_password'] && ! @params['admin_password'].empty?)
      then
	passwd = PasswordVerifier.new(pw_enc)
	passwd.add_encrypted_user(@params['admin_user'], @params['admin_password'])
	auth = loader.BasicAuthFilter.new(passwd, "Administrator's password")
	server.attach(auth, '/control')
      end
      nil
    end

    def_delegator :@params, :modified_count
    def_delegator :@params, :modified_count=

    def admin_user
      @params['admin_user']
    end

    def admin_user=(new_user)
      return new_user unless new_user
      return new_user if new_user.strip.empty?
      @params['admin_user'] = new_user.strip
    end

    def admin_password
      @params['admin_password']
    end

    def admin_password=(new_password)
      if (new_password) then
	@params['admin_password'] = new_password
      else
	@params['admin_password'] = ''
      end
    end

    def localhost_only
      @params['localhost_only']
    end

    def localhost_only=(boolean)
      if (@params['admin_password'].empty?) then
	@params['localhost_only'] = true
      else
	@params['localhost_only'] = boolean ? true : false
      end
    end

    def japanese_handling
      @params['japanese_handling']
    end

    def japanese_handling=(boolean)
      @params['japanese_handling'] = boolean ? true : false
    end
  end

  class ServerParams
    extend Forwardable
    extend LoadProperties

    def initialize
      @params = nil
    end

    def load(properties)
      @params = properties.params('server_type',
				  'port',
				  'bind_address',
				  'timeout',
				  'keep_alive',
				  'max_requests',
				  'queue_length',
				  'messengers',
				  'messenger_threads',
				  'messenger_queue_length',
				  'subprocess_user',
				  'subprocess_group',
				  'do_not_reverse_lookup')

      @params['server_type'] = 'multithread' unless (@params.include? 'server_type')
      @params['port'] = 8888 unless (@params.include? 'port')
      @params['bind_address'] = nil unless (@params.include? 'bind_address')
      @params['timeout'] = 300 unless (@params.include? 'timeout')
      @params['keep_alive'] = 8 unless (@params.include? 'keep_alive')
      @params['max_requests'] = 32 unless (@params.include? 'max_requests')
      @params['queue_length'] = 16 unless (@params.include? 'queue_length')
      @params['messengers'] = 8 unless (@params.include? 'messengers')
      @params['messenger_threads'] = 4 unless (@params.include? 'messenger_threads')
      @params['messenger_queue_length'] = 4 unless (@params.include? 'messenger_queue_length')
      @params['subprocess_user'] = 'nobody' unless (@params.include? 'subprocess_user')
      @params['subprocess_group'] = 'nobody' unless (@params.include? 'subprocess_group')
      @params['do_not_reverse_lookup'] = false unless (@params.include? 'do_not_reverse_lookup')

      nil
    end

    def save(properties)
      properties.set_params(@params)
      nil
    end

    def server_setup(control, server)
      case (@params['server_type'])
      when 'multithread'
	restart_signal = MultiThreadRestartSignal.new
	control.restart_signal = restart_signal
	server.restart_signal = restart_signal
	server.messenger_factory = MultiThreadMessenger
      when 'multiprocess'
	restart_signal = MultiProcessRestartSignal.new
	control.restart_signal = restart_signal
	server.restart_signal = restart_signal
	server.messenger_factory = MultiProcessMessenger
      else
	control.emerg("unknown server type: #{@params['server_type'].inspect}")
	raise "unknown server type: #{@params['server_type'].inspect}"
      end
      server.port = @params['port']
      server.bind_address = @params['bind_address']
      server.timeout = @params['timeout']
      server.keep_alive = @params['keep_alive']
      server.max_requests = @params['max_requests']
      server.queue_length = @params['queue_length']
      server.messengers = @params['messengers']
      server.messenger_threads = @params['messenger_threads']
      server.messenger_queue_length = @params['messenger_queue_length']
      server.subprocess_user = @params['subprocess_user']
      server.subprocess_group = @params['subprocess_group']
      BasicSocket.do_not_reverse_lookup = @params['do_not_reverse_lookup']
      nil
    end

    def_delegator :@params, :modified_count
    def_delegator :@params, :modified_count=

    def server_type
      @params['server_type']
    end

    def server_type=(new_type)
      case (new_type)
      when 'multithread', 'multiprocess'
	@params['server_type'] = new_type
      else
	raise "unknown server type: #{new_type.inspect}"
      end
    end

    def port
      @params['port']
    end

    def port=(new_port)
      if (new_port < 0) then
	raise "not allowed negative port number: #{new_port.inspect}"
      end
      @params['port'] = new_port
    end

    def bind_address
      @params['bind_address']
    end

    def bind_address=(new_address)
      if (! new_address || new_address.strip.empty?) then
	@params['bind_address'] = nil
      else
	@params['bind_address'] = new_address
      end
    end

    def timeout
      @params['timeout']
    end

    def timeout=(new_timeout)
      if (new_timeout == 0) then
	raise "not allowed zero number of timeout: #{new_timeout.inspect}"
      end
      if (new_timeout < 0) then
	raise "not allowed negative number of timeout: #{new_timeout.inspect}"
      end
      @params['timeout'] = new_timeout
    end

    def keep_alive
      @params['keep_alive']
    end

    def keep_alive=(new_keep_alive)
      if (new_keep_alive < 0) then
	raise "not allowed negative number of keep-alive: #{new_keep_alive.inspect}"
      end
      @params['keep_alive'] = new_keep_alive
    end

    def max_requests
      @params['max_requests']
    end

    def max_requests=(new_reqs)
      if (new_reqs == 0) then
	raise "not allowed zero number of max-requests: #{new_reqs.inspect}"
      end
      if (new_reqs < 0) then
	raise "not allowed negative number of max-requests: #{new_reqs.inspect}"
      end
      @params['max_requests'] = new_reqs
    end

    def queue_length
      @params['queue_length']
    end

    def queue_length=(new_len)
      if (new_len == 0) then
	raise "not allowed zero number of queue length: #{new_len.inspect}"
      end
      if (new_len < 0) then
	raise "not allowed negative number of queue length: #{new_len.inspect}"
      end
      @params['queue_length'] = new_len
    end

    def messengers
      @params['messengers']
    end

    def messengers=(new_messengers)
      if (new_messengers == 0) then
	raise "not allowed zero number of messengers: #{new_messengers.inspect}"
      end
      if (new_messengers < 0) then
	raise "not allowed negative number of messengers: #{new_messengers.inspect}"
      end
      @params['messengers'] = new_messengers
    end

    def messenger_threads
      @params['messenger_threads']
    end

    def messenger_threads=(new_threads)
      if (new_threads == 0) then
	raise "not allowed zero number of messenger threads: #{new_threads.inspect}"
      end
      if (new_threads < 0) then
	raise "not allowed negative number of messenger threads: #{new_threads.inspect}"
      end
      @params['messenger_threads'] = new_threads
    end

    def messenger_queue_length
      @params['messenger_queue_length']
    end

    def messenger_queue_length=(new_len)
      if (new_len == 0) then
	raise "not allowed zero number of messenger queue length: #{new_len.inspect}"
      end
      if (new_len < 0) then
	raise "not allowed negative number of messenger queue length: #{new_len.inspect}"
      end
      @params['messenger_queue_length'] = new_len
    end

    def subprocess_user
      @params['subprocess_user']
    end

    def subprocess_user=(new_user)
      case (new_user)
      when Numeric
	if (new_user < 0) then
	  raise "not allowed negative number of subprocess user: #{new_user.inspect}"
	end
      when String
	# nothing to do.
      else
	raise "not allowed object type of subprocess user: #{new_user.class}"
      end
      @params['subprocess_user'] = new_user
    end

    def subprocess_group
      @params['subprocess_group']
    end

    def subprocess_group=(new_group)
      case (new_group)
      when Numeric
	if (new_group < 0) then
	  raise "not allowed negative number of subprocess group: #{subprocess_group.inspect}"
	end
      when String
	# nothing to do.
      else
	raise "not allowed object type of subprocess group: #{new_group.class}"
      end
      @params['subprocess_group'] = new_group
    end

    def do_not_reverse_lookup
      @params['do_not_reverse_lookup']
    end

    def do_not_reverse_lookup=(boolean)
      @params['do_not_reverse_lookup'] = boolean ? true : false
    end
  end

  class LoggingParams
    extend Forwardable
    extend LoadProperties

    def initialize
      @params = nil
    end

    def load(properties)
      @params = properties.params('stdout_logging_level',
				  'logfiles')

      @params['stdout_logging_level'] = 'notice' unless (@params.include? 'stdout_logging_level')
      @params['logfiles'] = [] unless (@params.include? 'logfiles')

      nil
    end

    def save(properties)
      case (@params['stdout_logging_level'])
      when 'emerg', 'alert', 'crit', 'err', 'warning', 'notice', 'info', 'debug'
	# nothing to do.
      else
	raise "not allowed logging level: #{@params['stdout_logging_level'].inspect}"
      end

      for attrs in @params['logfiles']
	if (attrs['path'] == nil) then
	  raise 'not allowed nil'
	end
	if (attrs['path'].strip.empty?) then
	  raise 'not allowed empty string'
	end
	case (attrs['logging_level'])
	when 'emerg', 'alert', 'crit', 'err', 'warning', 'notice', 'info', 'debug'
	  # nothing to do.
	else
	  raise "not allowed logging level: #{attrs['logging_level'].inspect}"
	end
      end

      properties.set_params(@params)
      nil
    end

    def server_setup(control, server)
      unless ($RUCY_DAEMON) then
	stdout_logger = Logger.new(STDOUT)
	case (@params['stdout_logging_level'])
	when 'emerg'
	  stdout_logger.level = LoggingLevel::LOG_EMERG
	when 'alert'
	  stdout_logger.level = LoggingLevel::LOG_ALERT
	when 'crit'
	  stdout_logger.level = LoggingLevel::LOG_CRIT
	when 'err'
	  stdout_logger.level = LoggingLevel::LOG_ERR
	when 'warning'
	  stdout_logger.level = LoggingLevel::LOG_WARNING
	when 'notice'
	  stdout_logger.level = LoggingLevel::LOG_NOTICE
	when 'info'
	  stdout_logger.level = LoggingLevel::LOG_INFO
	when 'debug'
	  stdout_logger.level = LoggingLevel::LOG_DEBUG
	else
	  raise "unknown logging level: #{@params['stdout_logging_level'].inspect}"
	end
	server.add_logger(stdout_logger)
	server.debug('add stdout logger.')
      end
      for logfile in @params['logfiles']
	begin
	  output = File.open(logfile['path'], 'a')
	  control.add_logfile(output)
	  file_logger = Logger.new(output)
	  case (logfile['logging_level'])
	  when 'emerg'
	    file_logger.level = LoggingLevel::LOG_EMERG
	  when 'alert'
	    file_logger.level = LoggingLevel::LOG_ALERT
	  when 'crit'
	    file_logger.level = LoggingLevel::LOG_CRIT
	  when 'err'
	    file_logger.level = LoggingLevel::LOG_ERR
	  when 'warning'
	    file_logger.level = LoggingLevel::LOG_WARNING
	  when 'notice'
	    file_logger.level = LoggingLevel::LOG_NOTICE
	  when 'info'
	    file_logger.level = LoggingLevel::LOG_INFO
	  when 'debug'
	    file_logger.level = LoggingLevel::LOG_DEBUG
	  else
	    raise "unknown logging level: #{logfile['logging_level'].inspect}"
	  end
	  server.add_logger(file_logger)
	  server.debug("open #{logfile['path'].inspect} logger.")
	rescue StandardError, ScriptError
	  server.err("error: not opened #{logfile['path'].inspect}: (#{$!.class}) #{$!}")
	  control.logging_errors.push({ :logfile => logfile['path'],
					:exception => $!
				      })
	end
      end
      nil
    end

    def_delegator :@params, :modified_count
    def_delegator :@params, :modified_count=

    def stdout_logging_level
      @params['stdout_logging_level']
    end

    def stdout_logging_level=(new_level)
      case (new_level)
      when 'emerg', 'alert', 'crit', 'err', 'warning', 'notice', 'info', 'debug'
	# nothing to do.
      else
	raise "not allowed logging level: #{new_level.inspect}"
      end
      @params['stdout_logging_level'] = new_level
    end

    class LogFile
      def initialize(attrs)
	@attrs = attrs
      end

      def path
	@attrs['path']
      end

      def path=(new_path)
	@attrs['path'] = new_path
      end

      def logging_level
	@attrs['logging_level']
      end

      def logging_level=(new_level)
	case (new_level)
	when 'emerg', 'alert', 'crit', 'err', 'warning', 'notice', 'info', 'debug'
	  # nothing to do.
	else
	  raise "not allowed logging level: #{new_level.inspect}"
	end
	@attrs['logging_level'] = new_level
      end
    end

    def logfiles
      @params['logfiles'].map{|attrs|
	LogFile.new(attrs)
      }
    end

    def add_logfile
      @params['logfiles'].push({ 'path' => nil, 'logging_level' => 'notice' })
      nil
    end

    def del_logfile_at(pos)
      @params['logfiles'].delete_at(pos)
      nil
    end
  end

  class AccessLogParams
    extend Forwardable
    extend LoadProperties

    def initialize
      @params = nil
    end

    def load(properties)
      @params = properties.params('access_log_logging',
				  'access_log_format',
				  'access_logs')

      @params['access_log_logging'] = true unless (@params.include? 'access_log_logging')
      @params['access_log_format'] = AccessLog::COMMON_LOG_FORMAT unless (@params.include? 'access_log_format')
      @params['access_logs'] = [] unless (@params.include? 'access_logs')

      nil
    end

    def save(properties)
      if (@params['access_log_format'] == nil) then
	raise 'not allowed nil'
      end
      if (@params['access_log_format'].strip.empty?) then
	raise 'not allowd empty string'
      end
      for attrs in @params['access_logs']
	if (attrs['path'] == nil) then
	  raise 'not allowed nil'
	end
	if (attrs['path'].strip.empty?) then
	  raise 'not allowed empty string'
	end
	if (attrs['format'] == nil) then
	  raise 'not allowed nil'
	end
	if (attrs['format'].strip.empty?) then
	  raise 'not allowed empty string'
	end
      end
      properties.set_params(@params)
      nil
    end

    def server_setup(control, server)
      if (@params['access_log_logging']) then
	server.logging_access(@params['access_log_format'])
      end
      for logfile in @params['access_logs']
	begin
	  output = File.open(logfile['path'], 'a')
	  control.add_logfile(output)
	  access_log = AccessLog.new(output, logfile['format'])
	  server.add_access_log(access_log)
	  server.debug("open #{logfile['path'].inspect} access log.")
	rescue StandardError, ScriptError
	  server.err("error: not opened #{logfile['path'].inspect}: (#{$!.class}) #{$!}")
	  control.logging_errors.push({ :logfile => logfile['path'],
					:exception => $!
				      })
	end
      end
      nil
    end

    def_delegator :@params, :modified_count
    def_delegator :@params, :modified_count=

    def logging
      @params['access_log_logging']
    end

    def logging=(boolean)
      @params['access_log_logging'] = boolean ? true : false
    end

    def format
      @params['access_log_format']
    end

    def format=(new_format)
      @params['access_log_format'] = new_format
    end

    class LogFiles
      def initialize(attrs)
	@attrs = attrs
      end

      def path
	@attrs['path']
      end

      def path=(new_path)
	@attrs['path'] = new_path
      end

      def format
	@attrs['format']
      end

      def format=(new_format)
	@attrs['format'] = new_format
      end
    end

    def logfiles
      @params['access_logs'].map{|attrs| LogFiles.new(attrs) }
    end

    def add_logfile
      @params['access_logs'].push({ 'path' => nil,
				    'format' => AccessLog::COMMON_LOG_FORMAT
				  })
      nil
    end

    def del_logfile_at(pos)
      @params['access_logs'].delete_at(pos)
      nil
    end
  end

  class AliasList
    extend Forwardable
    extend LoadProperties
    include Enumerable

    def initialize
      @list = nil
    end

    def load(properties)
      @list = properties.list('aliases')
      nil
    end

    def save(properties)
      for attrs in @list
	alias_path = attrs['alias_path']
	if (alias_path == nil) then
	  raise 'not allowed nil for alias_path'
	end
	if (alias_path.strip.empty?) then
	  raise 'not allowed empty string for alias_path'
	end

	orig_path = attrs['orig_path']
	if (orig_path == nil) then
	  raise 'not allowed nil for orig_path'
	end
	if (orig_path.strip.empty?) then
	  raise 'not allowed empty string for orig_path'
	end
      end
      properties.set_list('aliases', @list)
      nil
    end

    def server_setup(control, server)
      for attrs in @list
	alias_path = attrs['alias_path']
	orig_path = attrs['orig_path']
	host = attrs['virtual_host']
	begin
	  if (host) then
	    host += ':' + server.port.to_s
	    server.set_virtual_alias(host, alias_path, orig_path)
	  else
	    server.set_alias(alias_path, orig_path)
	  end
	rescue
	  server.err("error: unavaiable alias: (#{$!.class}) #{$!}")
	  control.alias_errors.push({ :alias_path => alias_path,
				      :orig_path => orig_path,
				      :virtual_host => host,
				      :exception => $!
				    })
	end
      end
      nil
    end

    def_delegator :@list, :modified_count
    def_delegator :@list, :modified_count=
    def_delegator :@list, :empty?
    def_delegator :@list, :size
    def_delegator :@list, :length

    class AliasParams
      def initialize(attrs)
	@attrs = attrs
      end

      def alias_path
	@attrs['alias_path']
      end

      def alias_path=(new_path)
	case (new_path)
	when String, NilClass
	  # nothing to do.
	else
	  raise "not a string for alias_path: #{new_path.inspect}"
	end
	@attrs['alias_path'] = new_path
      end

      def orig_path
	@attrs['orig_path']
      end

      def orig_path=(new_path)
	case (new_path)
	when String, NilClass
	  # nothing to do.
	else
	  raise "not a string for orig_path: #{new_path.inspect}"
	end
	@attrs['orig_path'] = new_path
      end

      def virtual_host
	@attrs['virtual_host']
      end

      def virtual_host=(new_host)
	case (new_host)
	when NilClass
	  @attrs['virtual_host'] = nil
	when String
	  if (new_host.strip.empty?) then
	    @attrs['virtual_host'] = nil
	  else
	    @attrs['virtual_host'] = new_host
	  end
	else
	  raise "not a string for virtual_host: #{new_host.inspect}"
	end
      end

      # for backward compatibility

      def [](name)
	case (name)
	when 'alias_path'
	  self.alias_path
	when 'orig_path'
	  self.orig_path
	when 'virtual_host'
	  self.virtual_host
	else
	  raise "unknown parameter: #{name.inspect}"
	end
      end

      def []=(name, value)
	case (name)
	when 'alias_path'
	  self.alias_path = value
	when 'orig_path'
	  self.orig_path = value
	when 'virtual_host'
	  self.virtual_host = value
	else
	  raise "unknown parameter: #{name.inspect}"
	end
      end
    end

    def [](pos)
      AliasParams.new(@list[pos])
    end

    def add_entry
      @list.push({ 'alias_path' => nil,
		   'orig_path' => nil,
		   'virtual_host' => nil
		 })
      nil
    end

    def delete_at(pos)
      @list.delete_at(pos)
      nil
    end

    def swap(i, j)
      @list[i], @list[j] = @list[j], @list[i]
      nil
    end

    def each
      for attrs in @list
	yield(AliasParams.new(attrs))
      end
      nil
    end

    # for backward compatibility

    def push(attrs)
      add_entry
      [ 'alias_path',
	'orig_path',
	'virtual_host'
      ].each do |key|
	if (attrs.include? key) then
	  @list.last[key] = attrs[key]
	end
      end
      nil
    end
  end

  class DocumentList
    extend Forwardable
    extend LoadProperties
    include Enumerable

    def initialize
      @list = nil
    end

    def load(properties)
      @list = properties.list('documents')
      nil
    end

    def save(properties)
      properties.set_list('documents', @list)
      nil
    end

    def server_setup(control, server, doc_factory)
      for mount_info in @list
	name = mount_info['document']
	args = mount_info['arguments']
	path = mount_info['mount_path']
	mask = mount_info['mount_mask']
	host = mount_info['virtual_host']
	begin
	  document = doc_factory.doc_build(name, args)
	  if (host) then
	    host += ':' + server.port.to_s
	    server.virtual_mount(host, document, path, mask)
	  else
	    server.mount(document, path, mask)
	  end
	rescue StandardError, ScriptError
	  server.err("error: not mounted #{name.inspect} document: (#{$!.class}) #{$!}")
	  control.doc_errors.push({ :document => name,
				    :arguments => args,
				    :mount_path => path,
				    :mount_mask => mask,
				    :virtual_host => host,
				    :exception => $!
				  })
	end
      end
      nil
    end

    def_delegator :@list, :modified_count
    def_delegator :@list, :modified_count=
    def_delegator :@list, :empty?
    def_delegator :@list, :size
    def_delegator :@list, :length

    class MountParams
      def initialize(attrs)
	@attrs = attrs
      end

      def name
	@attrs['document']
      end

      def name=(new_name)
	@attrs['document'] = new_name
      end

      def args
	@attrs['arguments'].dup
      end

      def set_args_at(pos, value)
	@attrs['arguments'][pos] = value
      end

      def path
	@attrs['mount_path']
      end

      def path=(new_path)
	@attrs['mount_path'] = new_path
      end

      def mask
	@attrs['mount_mask']
      end

      def mask=(new_mask)
	@attrs['mount_mask'] = new_mask
      end

      def virtual_host
	@attrs['virtual_host']
      end

      def virtual_host=(new_host)
	@attrs['virtual_host'] = new_host
      end

      def comment
	@attrs['comment']
      end

      def comment=(new_comment)
	@attrs['comment'] = new_comment
      end

      # for backward compatibility

      def [](name)
	case (name)
	when 'document'
	  self.name
	when 'arguments'
	  self.args
	when 'mount_path'
	  self.path
	when 'mount_mask'
	  self.mask
	when 'virtual_host'
	  self.virtual_host
	when 'comment'
	  self.comment
	else
	  raise "unknown key word: #{name.inspect}"
	end
      end

      def []=(name, value)
	case (name)
	when 'document'
	  self.name = value
	when 'arguments'
	  value.each_with_index do |v, i|
	    set_args_at(i, v)
	  end
	when 'mount_path'
	  self.path = value
	when 'mount_mask'
	  self.mask = value
	when 'virtual_host'
	  self.virtual_host = value
	when 'comment'
	  self.comment = value
	else
	  raise "unknown key word: #{name.inspect}"
	end
      end

      def to_hash
	{ 'document' => self.name,
	  'arguments' => self.args,
	  'mount_path' => self.path,
	  'mount_mask' => self.mask,
	  'virtual_host' => self.virtual_host,
	  'comment' => self.comment
	}
      end
    end

    def [](pos)
      MountParams.new(@list[pos])
    end

    def add_entry
      @list.push({ 'document' => nil,
		   'arguments' => [],
		   'mount_path' => nil,
		   'mount_mask' => nil,
		   'virtual_host' => nil,
		   'comment' => nil
		 })
      nil
    end

    def delete_at(pos)
      @list.delete_at(pos)
      nil
    end

    def swap(i, j)
      @list[i], @list[j] = @list[j], @list[i]
      nil
    end

    def each
      for attrs in @list
	yield(MountParams.new(attrs))
      end
      nil
    end

    # for backward compatibility

    def push(attrs)
      self.add_entry
      params = self[-1]
      params.name = attrs['document'] if (attrs.include? 'document')
      if (attrs.include? 'arguments') then
	attrs['arguments'].each_with_index do |v, i|
	  params.set_args_at(i, v)
	end
      end
      params.path = attrs['mount_path'] if (attrs.include? 'mount_path')
      params.mask = attrs['mount_mask'] if (attrs.include? 'mount_mask')
      params.virtual_host = attrs['virtual_host'] if (attrs.include? 'virtual_host')
      params.comment = attrs['comment'] if (attrs.include? 'comment')
      nil
    end

    def []=(pos, attrs)
      if (pos == @list.length && (attrs.kind_of? Hash)) then
	push(attrs)
	return self[-1]
      else
	raise 'failed to add entry of document list.'
      end
      nil
    end
  end

  class FilterList
    extend Forwardable
    extend LoadProperties
    include Enumerable

    def initialize
      @list = nil
    end

    def load(properties)
      @list = properties.list('filters')
      nil
    end

    def save(properties)
      properties.set_list('filters', @list)
      nil
    end

    def server_setup(control, server, doc_factory)
      for filter_info in @list
	name = filter_info['filter']
	args = filter_info['arguments']
	path = filter_info['attach_path']
	mask = filter_info['attach_mask']
	host = filter_info['virtual_host']
	begin
	  filter = doc_factory.filter_build(name, args)
	  if (host) then
	    host += ':' + server.port.to_s
	    server.virtual_attach(host, filter, path, mask)
	  else
	    server.attach(filter, path, mask)
	  end
	rescue StandardError, ScriptError
	  server.err("error: not attached #{name.inspect} filter: (#{$!.class}) #{$!}")
	  @filter_errors.push({ :filter => name,
				:arguments => args,
				:attach_path => path,
				:attach_mask => mask,
				:virtual_host => host,
				:exception => $!
			      })
	end
      end
      nil
    end

    def_delegator :@list, :modified_count
    def_delegator :@list, :modified_count=
    def_delegator :@list, :empty?
    def_delegator :@list, :size
    def_delegator :@list, :length

    class AttachParams
      def initialize(attrs)
	@attrs = attrs
      end

      def name
	@attrs['filter']
      end

      def name=(new_name)
	@attrs['filter'] = new_name
      end

      def args
	@attrs['arguments'].dup
      end

      def set_args_at(pos, value)
	@attrs['arguments'][pos] = value
	nil
      end

      def path
	@attrs['attach_path']
      end

      def path=(new_path)
	@attrs['attach_path'] = new_path
      end

      def mask
	@attrs['attach_mask']
      end

      def mask=(new_mask)
	@attrs['attach_mask'] = new_mask
      end

      def virtual_host
	@attrs['virtual_host']
      end

      def virtual_host=(new_host)
	@attrs['virtual_host'] = new_host
      end

      def comment
	@attrs['comment']
      end

      def comment=(new_comment)
	@attrs['comment'] = new_comment
      end

      # for backward compatibility

      def [](name)
	case (name)
	when 'filter'
	  self.name
	when 'arguments'
	  self.args
	when 'attach_path'
	  self.path
	when 'attach_mask'
	  self.mask
	when 'virtual_host'
	  self.virtual_host
	when 'comment'
	  self.comment
	else
	  raise "unknown key word: #{name.inspect}"
	end
      end

      def []=(name, value)
	case (name)
	when 'filter'
	  self.name = value
	when 'arguments'
	  value.each_with_index do |v, i|
	    set_args_at(i, v)
	  end
	when 'attach_path'
	  self.path = value
	when 'attach_mask'
	  self.mask = value
	when 'virtual_host'
	  self.virtual_host = value
	when 'comment'
	  self.comment = value
	else
	  raise "unknown key word: #{name.inspect}"
	end
      end

      def to_hash
	{ 'filter' => self.name,
	  'arguments' => self.args,
	  'attach_path' => self.path,
	  'attach_mask' => self.mask,
	  'virtual_host' => self.virtual_host,
	  'comment' => self.comment
	}
      end
    end

    def [](pos)
      AttachParams.new(@list[pos])
    end

    def add_entry
      @list.push({ 'filter' => nil,
		   'arguments' => [],
		   'attach_path' => nil,
		   'attach_mask' => nil,
		   'virtual_host' => nil,
		   'comment' => nil
		 })
      nil
    end

    def delete_at(pos)
      @list.delete_at(pos)
      nil
    end

    def swap(i, j)
      @list[i], @list[j] = @list[j], @list[i]
      nil
    end

    def each
      for attrs in @list
	yield(AttachParams.new(attrs))
      end
      nil
    end

    def push(attrs)
      self.add_entry
      params = self[-1]
      params.name = attrs['filter'] if (attrs.include? 'filter')
      if (attrs.include? 'arguments') then
	attrs['arguments'].each_with_index do |v, i|
	  params.set_args_at(i, v)
	end
      end
      params.path = attrs['attach_path'] if (attrs.include? 'attach_path')
      params.mask = attrs['attach_mask'] if (attrs.include? 'attach_mask')
      params.virtual_host = attrs['virtual_host'] if (attrs.include? 'virtual_host')
      params.comment = attrs['comment'] if (attrs.include? 'comment')
      nil
    end

    def []=(pos, attrs)
      if (pos == @list.length && (attrs.kind_of? Hash)) then
	push(attrs)
	return self[-1]
      else
	raise 'failed to add entry of document list.'
      end
      nil
    end
  end
end
