# $Id: restart.rb,v 1.1.1.1 2004/04/04 15:22:49 toki Exp $

require 'thread'

module Rucy
  class RestartSignal
    def initialize
      @server = nil
    end

    attr_writer :server
  end

  class MultiThreadRestartSignal < RestartSignal
    def initialize
      super
      @state = nil
      @lock = Mutex.new
      @cond = ConditionVariable.new
    end

    def notify_restart
      @lock.synchronize{
	unless (@state) then
	  @state = :notify
	  @cond.signal
	end
      }
      nil
    end

    def notify_close
      @lock.synchronize{
	unless (@state) then
	  @state = :close
	  @cond.signal
	end
      }
      nil
    end

    def cancel
      @lock.synchronize{
	unless (@state) then
	  @state = :cancel
	  @cond.signal
	end
      }
      nil
    end

    def wait
      @lock.synchronize{
	until (@state)
	  @cond.wait(@lock)
	end
      }
      case (@state)
      when :notify
	@server.restart
      when :close
	@server.close
      end
      nil
    end
  end

  class MultiProcessRestartSignal < RestartSignal
    CMD_LEN = 1
    RESTART = 'R'
    CLOSE = 'X'
    CANCEL = 'C'

    def initialize
      super
      @pin, @pout = IO.pipe
      @pin.sync = true
      @pout.sync = true
      @closed = false
      @lock = Mutex.new
    end

    def notify_restart
      # call on child process
      @pin.close
      @pout.write(RESTART) rescue Errno::EPIPE
      @pout.close
      nil
    end

    def notify_close
      # call on child process
      @pin.close
      @pout.write(CLOSE) rescue Errno::EPIPE
      @pout.close
      nil
    end

    def cancel
      # call on parent process
      @lock.synchronize{
	unless (@closed) then
	  @pout.write(CANCEL)
	end
      }
      nil
    end

    def wait
      # call on parent process
      case (@pin.read(CMD_LEN))
      when RESTART
	@server.restart
      when CLOSE
	@server.close
      end
      @lock.synchronize{
	@pout.close
	@pin.close
	@closed = true
      }
      nil
    end
  end
end
