# $Id: logger.rb,v 1.20 2004/10/03 12:43:49 toki Exp $

require 'time'
require 'thread'

module Rucy
  module LoggingLevel
    LOG_EMERG   = 7
    LOG_ALERT   = 6
    LOG_CRIT    = 5
    LOG_ERR     = 4
    LOG_WARNING = 3
    LOG_NOTICE  = 2
    LOG_INFO    = 1
    LOG_DEBUG   = 0

    def emerg(messg)
      messg(LOG_EMERG, messg)
      nil
    end

    def alert(messg)
      messg(LOG_ALERT, messg)
      nil
    end

    def crit(messg)
      messg(LOG_CRIT, messg)
      nil
    end

    def err(messg)
      messg(LOG_ERR, messg)
      nil
    end

    def warning(messg)
      messg(LOG_WARNING, messg)
      nil
    end

    alias warn warning

    def notice(messg)
      messg(LOG_NOTICE, messg)
      nil
    end

    def info(messg)
      messg(LOG_INFO, messg)
      nil
    end

    def debug(messg)
      messg(LOG_DEBUG, messg)
      nil
    end
  end

  class Logger
    include LoggingLevel

    def initialize(output, level=LOG_NOTICE)
      @output = output
      @level = level
    end

    attr_accessor :level

    def messg(level, messg, format=true)
      if (level >= @level) then
	if (format) then
	  @output.write("#{Time.now.httpdate}: #{messg}\n")
	else
	  @output.write(messg)
	end
	@output.flush
      end
      nil
    end
  end

  class MultiLogger
    include LoggingLevel

    def initialize
      @logger_list = Array.new
    end

    def add(logger)
      @logger_list.push(logger)
      nil
    end

    def messg(level, messg, format=true)
      for logger in @logger_list
	logger.messg(level, messg, format)
      end
      nil
    end
  end

  class SyncLogger
    include LoggingLevel

    def initialize(logger)
      @logger = logger
      @lock = Mutex.new
    end

    def messg(level, messg, format=true)
      @lock.synchronize{
	@logger.messg(level, messg, format)
      }
      nil
    end
  end

  class AccessLog
    def self.format(messg, request, response, curr_time)
      messg.gsub(/%{.*?}[io]|%./) { |fmt|
	case (fmt)
	when '%%'		# escape
	  '%'
	when '%a'		# remote IP address
	  request.client_address
	when '%A'		# local IP address
	  request.server_address
	when '%B'		# bytes of response body
	  response.send_size.to_s
	when '%b'		# bytes of response body
	  b = response.send_size
	  (b > 0) ? b.to_s : '-'
	when '%D'		# elapsed time (milliseconds)
	  '-'
	when '%f'		# local path
	  response.local_path || '-'
	when '%h'		# remote host
	  request.client_name
	when '%l'		# ident
	  '-'
	when '%m'		# request method
	  request.method
	when '%p'		# server port
	  request.server_port.to_s
	when '%P'		# server process id
	  $$.to_s
	when '%q'		# query string
	  q = request.query
	  (q) ? "?#{q}" : ''
	when '%r'		# request line
	  request.line
	when '%s'		# response status
	  response.status.to_s
	when '%t'		# time
	  curr_time.httpdate
	when '%T'		# elapsed time (seconds)
	  '-'
	when '%u'		# remote user
	  '-'
	when '%U'		# request URL path
	  request.path
	when '%v'		# server name
	  request.host
	when /^%{.*}i$/	# input header
	  name = $&[2...-2]
	  request.header(name) || '-'
	when /^%{.*}o$/		# output header
	  name = $&[2...-2]
	  response.header(name) || '-'
	else
	  fmt
	end
      }
    end

    COMMON_LOG_FORMAT = '%h %l %u %t "%r" %s %b'

    def initialize(output, fmt=COMMON_LOG_FORMAT)
      @output = output
      @fmt = fmt
    end

    def write_log(request, response, curr_time)
      log = AccessLog.format(@fmt, request, response, curr_time)
      log << "\n"
      @output.write(log)
      @output.flush
      nil
    end
  end

  class AccessLogAdapter
    def initialize(logger, fmt=AccessLog::COMMON_LOG_FORMAT)
      @logger = logger
      @fmt = fmt
    end

    def write_log(request, response, curr_time)
      log = AccessLog.format(@fmt, request, response, curr_time)
      log << "\n"
      @logger.messg(LoggingLevel::LOG_INFO, log, false)
      nil
    end
  end

  class MultiAccessLog
    def initialize
      @log_list = Array.new
    end

    def add(access_log)
      @log_list.push(access_log)
      nil
    end

    def write_log(request, response, curr_time)
      for access_log in @log_list
	access_log.write_log(request, response, curr_time)
      end
      nil
    end
  end

  class SyncAccessLog
    def initialize(access_log)
      @access_log = access_log
      @lock = Mutex.new
    end

    def write_log(request, response, curr_time)
      @lock.synchronize{
	@access_log.write_log(request, response, curr_time)
      }
      nil
    end
  end
end
