# $Id: document.rb,v 1.10 2004/10/02 00:41:41 toki Exp $

require 'time'
require 'delegate'
require 'rucy/error'
require 'rucy/request'
require 'rucy/response'
require 'rucy/loader'

module Rucy
  module HTMLUtil
    def escapeHTML(string)
      escaped_string = string.dup
      escaped_string.gsub!(/&/, '&amp;')
      escaped_string.gsub!(/"/, '&quot;')
      escaped_string.gsub!(/</, '&lt;')
      escaped_string.gsub!(/>/, '&gt;')
      escaped_string
    end
    module_function :escapeHTML
  end

  class Document
    include HTMLUtil

    def self.set_doc_option(option)
      # hook for subclasses.
    end

    def open
      # hook for subclasses.
    end

    def close
      # hook for subclasses.
    end

    def check_traverse(traversed)
      unless (traversed.include? self) then
	traversed[self] = true
	yield
      end
    end
    private :check_traverse

    def each(traversed={})
      check_traverse(traversed) {
	yield(self)
      }
      nil
    end

    def publish(script_name, request, response, logger)
      raise NotImplementedError, 'not defined document.'
    end
  end

  class Filter
    def self.set_filter_option(option)
      # hook for subclasses.
    end

    def init
      # hook for subclasses.
    end

    def final
      # hook for subclasses.
    end

    def filter_open(context, script_name, request, response, logger)
    end

    def filter_head(context, script_name, request, response, logger)
      response.start_body
    end

    def filter_body(context, script_name, request, response, logger, messg_body)
      response.write(messg_body)
    end

    def filter_close(context, script_name, request, response, logger)
    end

    def terminate_filter
      throw(:end_of_filter)
    end
    private :terminate_filter
  end

  class FilterResponse < DelegateClass(Response)
    def initialize(context, filter, script_name, request, response, logger)
      super(response)
      @context = context
      @filter = filter
      @script_name = script_name
      @request = request
      @response = response
      @logger = logger
    end

    def start_body
      @filter.filter_head(@context, @script_name, @request, @response, @logger)
      nil
    end

    def write(messg_body)
      @filter.filter_body(@context, @script_name, @request, @response, @logger, messg_body)
      nil
    end

    def <<(messg_body)
      write(messg_body.to_s)
      self
    end
  end

  class FilterDocument < Document
    def initialize(document, filter)
      @document = document
      @filter = filter
    end

    attr_reader :document
    attr_reader :filter

    def open
      @filter.init
      nil
    end

    def close
      @filter.final
      nil
    end

    def each(traversed={})
      check_traverse(traversed) {
	yield(self)
	@document.each(traversed) do |document|
	  yield(document)
	end
      }
      nil
    end

    def publish(script_name, request, response, logger)
      logger.debug("[#{Time.now.httpdate}] enter filter: #{@filter.class}")
      context = Hash.new
      catch(:end_of_filter) {
	begin
	  @filter.filter_open(context, script_name, request, response, logger)
	  filter_response = FilterResponse.new(context, @filter, script_name, request, response, logger)
	  @document.publish(script_name, request, filter_response, logger)
	ensure
	  @filter.filter_close(context, script_name, request, response, logger)
	end
      }

      nil
    end
  end

  class DocumentFactory
    def initialize
      @option = Hash.new
      @doc_map = Hash.new
      @filter_map = Hash.new
    end

    def add_option(name, value)
      @option[name] = value
      nil
    end

    def setup
      for name, doc in @doc_map
	doc.set_doc_option(@option)
      end
      for name, filter in @filter_map
	filter.set_filter_option(@option)
      end
      nil
    end

    def add_document(doc_class)
      @doc_map[doc_class.doc_name] = doc_class
      nil
    end

    def has_document?(name)
      @doc_map.include? name
    end

    def doc_names
      @doc_map.keys.sort
    end

    def doc_args(name)
      unless (@doc_map.include? name) then
	raise "not found a document: #{name.inspect}"
      end
      @doc_map[name].doc_args
    end

    def doc_build(name, args)
      unless (@doc_map.include? name) then
	raise "not found a document: #{name.inspect}"
      end
      @doc_map[name].new(*args)
    end

    def add_filter(filter_class)
      @filter_map[filter_class.filter_name] = filter_class
      nil
    end

    def has_filter?(name)
      @filter_map.include? name
    end

    def filter_names
      @filter_map.keys.sort
    end

    def filter_args(name)
      unless (@filter_map.include? name) then
	raise "not found a filter: #{name.inspect}"
      end
      @filter_map[name].filter_args
    end

    def filter_build(name, args)
      unless (@filter_map.include? name) then
	raise "not found a filter: #{name.inspect}"
      end
      @filter_map[name].new(*args)
    end
  end

  class DocumentLoader
    def initialize(mod_path)
      @mod_path = mod_path
      @load_errors = Array.new
      @doc_map = nil
      @filter_map = nil
      @factory = nil
    end

    attr_reader :load_errors
    attr_reader :factory

    def load
      @doc_map = Hash.new
      @filter_map = Hash.new
      @factory = DocumentFactory.new
      rucy_consts = Hash[*Rucy.constants.map{|n| [n, true] }.flatten]
      Dir.foreach(@mod_path) do |rb_name|
	path = "#{@mod_path}/#{rb_name}"
	if ((File.file? path) && rb_name =~ /\.rb$/) then
	  loader = SimpleLoader.new
	  begin
	    loader.load(path)
	    for const_name in loader.constants
	      next if (rucy_consts.include? const_name)
	      case (const_name)
	      when /Document$/
		@doc_map[const_name] = loader
		doc_factory = const_name + 'Factory'
		if (loader.const_defined? doc_factory) then
		  @factory.add_document(loader.const_get(doc_factory).instance)
		else
		  @factory.add_document(loader.const_get(const_name))
		end
	      when /Filter$/
		@filter_map[const_name] = loader
		filter_factory = const_name + 'Factory'
		if (loader.const_defined? filter_factory) then
		  @factory.add_filter(loader.const_get(filter_factory).instance)
		else
		  @factory.add_filter(loader.const_get(const_name))
		end
	      end
	    end
	  rescue StandardError, ScriptError
	    @load_errors.push([ path, $! ])
	  end
	end
      end
      nil
    end

    def doc_get(name)
      unless (@doc_map.include? name) then
	raise NameError, "uninitialized document: #{name.inspect}"
      end
      @doc_map[name].const_get(name)
    end

    def filter_get(name)
      unless (@filter_map.include? name) then
	raise NameError, "uninitialized filter: #{name.inspect}"
      end
      @filter_map[name].const_get(name)
    end

    def method_missing(id, *args, &block)
      if (args.empty? && block.nil?) then
	case (id.to_s)
	when /Document$/
	  return doc_get(id.to_s)
	when /Filter$/
	  return filter_get(id.to_s)
	else
	  return super
	end
      else
	return super
      end
    end
  end

  class Page < Document
    def self.doc_name
      'Page'
    end

    def self.doc_args
      [ [ 'content', :text, nil ],
	[ 'content-type', :string, 'text/html' ]
      ]
    end

    def initialize(message, content_type='text/html')
      @message = message
      @content_type = content_type
    end

    def publish(script_name, request, response, logger)
      logger.debug("[#{Time.now.httpdate}] enter document: #{self.class}")
      case (request.method)
      when 'GET', 'HEAD'
	response.status = 200
	response.set_header('Content-Type', @content_type)
	response.set_header('Content-Length', @message.length.to_s)
	response.start_body
	if (request.method != 'HEAD') then
	  response.write(@message.dup)
	end
      else
	http_error = HTTPError.new(405)	# Method Not Allowed
	http_error.set_header('Allow', 'GET, HEAD')
	raise http_error
      end

      nil
    end
  end

  class SubsetDocument < Document
    def initialize(document, path)
      @document = document
      @path = path
    end

    def each(traversed={})
      check_traverse(traversed) {
	yield(self)
	@document.each(traversed) do |document|
	  yield(document)
	end
      }
    end

    def publish(script_name, request, response, logger)
      if (script_name.length < @path.length) then
	raise HTTPError.new(404) # Not Found
      end
      pos = script_name.length - @path.length
      if (script_name[pos, @path.length] != @path) then
	raise HTTPError.new(404) # Not Found
      end
      script_name2 = script_name[0...pos]
      @document.publish(script_name2, request, response, logger)
      nil
    end
  end

  class FolderDocument < Document
    EMPTY_MAP = {}.freeze

    def initialize
      @mount_map = Hash.new
      @alias_map = Hash.new
      @virtual_mount_map = Hash.new
      @virtual_alias_map = Hash.new
    end

    def set_alias(alias_path, orig_path)
      if (alias_path !~ %r"^/") then
	raise "not a path: #{alias_path.inspect}"
      end
      if (orig_path !~ %r"^/") then
	raise "not a path: #{orig_path.inspect}"
      end
      @alias_map[alias_path] = orig_path
      nil
    end

    def set_virtual_alias(host, alias_path, orig_path)
      host += ':80'if (host !~ /:\d+$/)
      if (alias_path !~ %r"^/") then
	raise "not a path: #{alias_path.inspect}"
      end
      if (orig_path !~ %r"^/") then
	raise "not a path: #{orig_path.inspect}"
      end
      unless (@virtual_alias_map.include? host) then
	@virtual_alias_map[host] = Hash.new
      end
      @virtual_alias_map[host][alias_path] = orig_path
      nil
    end

    def _mount(mount_map, document, path, mask=nil)
      if (path == '/') then
	path = ''
      end
      unless (mount_map.include? path) then
	mount_map[path] = {
	  :document => nil,
	  :mask_list => Array.new
	}
      end
      node = mount_map[path]

      if (mask) then
	if (node[:mask_list].find{ |m, d| m == mask }) then
	  raise "duplicated mount at #{path}:#{mask}."
	end
	node[:mask_list].unshift([ mask, document ])
      else
	if (node[:document]) then
	  raise "duplicated mount at #{path}."
	end
	node[:document] = document
      end

      nil
    end
    private :_mount

    def _attach(mount_map, filter, path, mask=nil)
      if (path == '/') then
	path = ''
      end
      if (doc_pair = search(mount_map, path)) then
	document, mount_path = doc_pair
	if (path == mount_path) then
	  node = mount_map[path]
	else
	  len = mount_path.length
	  subpath = path[len..-1]
	  node = {
	    :document => SubsetDocument.new(document, subpath),
	    :mask_list => Array.new
	  }
	  mount_map[path] = node
	end
      else
	raise "not mounted at #{path}."
      end

      if (mask) then
	mask_pair = node[:mask_list].find{ |m, d| m == mask }
	if (mask_pair) then
	  document = mask_pair[1]
	  filter_document = FilterDocument.new(document, filter)
	  mask_pair[1] = filter_document
	else
	  document = node[:document]
	  unless (document) then
	    raise "not mounted at #{path}:#{mask}."
	  end
	  filter_document = FilterDocument.new(document, filter)
	  node[:mask_list].unshift([ mask, filter_document ])
	end
      else
	document = node[:document]
	unless (document) then
	  raise "not mounted at #{path}."
	end
	filter_document = FilterDocument.new(document, filter)
	node[:document] = filter_document
      end

      nil
    end
    private :_attach

    def _umount(mount_map, path, mask=nil)
      unless (mount_map.include? path) then
	raise "not mounted at #{path}."
      end
      node = mount_map[path]

      if (mask) then
	document = nil
	node[:mask_list].delete_if{ |m, d|
	  if (m == mask) then
	    document = d
	    true
	  end
	}
	unless (document) then
	  raise "not mounted at #{path}:#{mask}."
	end
	return document
      else
	unless (node[:document]) then
	  raise "not mounted at #{path}."
	end
	document = node[:document]
	node[:document] = nil
	return document
      end
    end
    private :_umount

    def scan(mount_map)
      for path, node in mount_map
	document = node[:document]
	if (document) then
	  yield(document, path, nil)
	end
	for mask, document2 in node[:mask_list]
	  yield(document2, path, mask)
	end
      end

      nil
    end
    private :scan

    def search(mount_map, path)
      Request.scan(path) do |mount_path, path_info|
	if (node = mount_map[mount_path]) then
	  for mask, document in node[:mask_list]
	    if (mask === path_info) then
	      return document, mount_path
	    end
	  end
	  if (document = node[:document]) then
	    return document, mount_path
	  end
	end
      end

      nil
    end
    private :search

    def find(path)
      if (doc_pair = search(@mount_map, path)) then
	document, mount_path = doc_pair
	if (mount_path.empty?) then
	  mount_path = '/'
	end
	return document, mount_path
      end

      nil
    end

    def virtual_find(host, path)
      host += ':80' if (host !~ /:\d+$/)
      if (@virtual_mount_map.include? host) then
	return search(@virtual_mount_map[host], path)
      end
      nil
    end

    def mount(document, path, mask=nil)
      _mount(@mount_map, document, path, mask)
    end

    def attach(filter, path, mask=nil)
      _attach(@mount_map, filter, path, mask)
    end

    def umount(path, mask=nil)
      _umount(@mount_map, path, mask)
    end

    def virtual_mount(host, document, path, mask=nil)
      host += ':80' if (host !~ /:\d+$/)
      unless (@virtual_mount_map.include? host) then
	@virtual_mount_map[host] = Hash.new
      end
      _mount(@virtual_mount_map[host], document, path, mask)
    end

    def virtual_attach(host, filter, path, mask=nil)
      host += ':80' if (host !~ /:\d+$/)
      unless (@virtual_mount_map.include? host) then
	raise "not mounted virtual host: #{host}"
      end
      _attach(@virtual_mount_map[host], filter, path, mask)
    end

    def virtual_umount(host, path, mask=nil)
      host += ':80' if (host !~ /:\d+$/)
      unless (@virtual_mount_map.include? host) then
	raise "not mounted virtual host: #{host}"
      end
      _umount(@virtual_mount_map[host], path, mask)
    end

    def each(traversed={})
      check_traverse(traversed) {
	yield(self)
	scan(@mount_map) do |document, path, mask|
	  document.each(traversed) do |document2|
	    yield(document2)
	  end
	end
	for host, mount_map in @virtual_mount_map
	  scan(mount_map) do |document, path, mask|
	    document.each(traversed) do |document2|
	      yield(document2)
	    end
	  end
	end
      }
      nil
    end

    def publish(script_name, request, response, logger)
      script_name2, path = request.subpath(script_name)
      host = request.host
      host += ':80' if (host && host !~ /:\d+$/)
      if (@virtual_mount_map[host]) then
	alias_map = @virtual_alias_map[host] || EMPTY_MAP
	if (alias_map.include? path) then
	  path = alias_map[path]
	  request.path = script_name2 + path
	end
	if (doc_pair = search(@virtual_mount_map[host], path)) then
	  document, mount_path = doc_pair
	  document.publish(script_name2 + mount_path, request, response, logger)
	else
	  logger.debug("[#{Time.now.httpdate}] not virtual mounted: http://#{host}#{request.path}")
	  raise HTTPError.new(404) # Not Found
	end
      else
	if (@alias_map.include? path) then
	  path = @alias_map[path]
	  request.path = script_name2 + path
	end
	if (doc_pair = search(@mount_map, path)) then
	  document, mount_path = doc_pair
	  document.publish(script_name2 + mount_path, request, response, logger)
	else
	  logger.debug("[#{Time.now.httpdate}] not mounted: #{request.path}")
	  raise HTTPError.new(404) # Not Found
	end
      end

      nil
    end
  end
end
