
package jp.sourceforge.expression_computer.test;

import jp.sourceforge.expression_computer.CommandList;
import jp.sourceforge.expression_computer.ComputeException;
import jp.sourceforge.expression_computer.Computer;
import jp.sourceforge.expression_computer.Function;
import jp.sourceforge.expression_computer.ParseException;
import junit.framework.TestCase;

public class ComputerTest extends TestCase {

    public ComputerTest(String name) {
        super(name);
    }

    public void testFail_null1() {
        String exp = null;

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (NullPointerException e) {
            assertEquals("expressionがnullです。", e.getMessage());
        }
    }

    public void testFail_null2() {
        CommandList cl = null;

        Computer comp = new Computer();
        try {
            comp.compute(cl);
            fail();
        } catch (NullPointerException e) {
            assertEquals("commandListがnullです。", e.getMessage());
        }
    }

    public void testFail_EmptyStatement() {
        String exp = "";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_IntegerLiteral1() {
        String exp = "1";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_IntegerLiteral2() {
        String exp = "123456789";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(123456789, v, 0);
    }

    public void testNormal_IntegerLiteral3() {
        String exp = "0xfFff";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(65535, v, 0);
    }

    public void testNormal_IntegerLiteral4() {
        String exp = "07777";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(4095, v, 0);
    }

    public void testNormal_IntegerLiteral5() {
        String exp = "0";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testFail_IntegerLiteral1() {
        String exp = "123456789123456789123456789";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (NumberFormatException e) {
        }
    }

    public void testFail_IntegerLiteral2() {
        String exp = "123,456,789";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_FloatingPointLiteral1() {
        String exp = "123.456";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(123.456, v, 0);
    }

    public void testNormal_FloatingPointLiteral2() {
        String exp = "0.123456";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0.123456, v, 0);
    }

    public void testNormal_FloatingPointLiteral3() {
        String exp = ".123456";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0.123456, v, 0);
    }

    public void testNormal_FloatingPointLiteral4() {
        String exp = "0.0";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_FloatingPointLiteral5() {
        String exp = ".0";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testFail_FloatingPointLiteral1() {
        String exp = "123.456.789";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_FunctionExpression1() {
        String exp = "foo()";

        Function func = new Function() {

            public double call(double[] arguments) {
                return 1;
            }

        };

        Computer comp = new Computer();
        comp.addFunction("foo", func);
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_FunctionExpression2() {
        String exp = "foo(1, 2)";

        Function func = new Function() {

            public double call(double[] arguments) {
                return arguments[0] + arguments[1];
            }

        };

        Computer comp = new Computer();
        comp.addFunction("foo", func);
        double v = comp.compute(exp);

        assertEquals(3, v, 0);
    }

    public void testNormal_FunctionExpression3() {
        String exp = "foo(1 + 2)";

        Function func = new Function() {

            public double call(double[] arguments) {
                return arguments[0] * 10;
            }

        };

        Computer comp = new Computer();
        comp.addFunction("foo", func);
        double v = comp.compute(exp);

        assertEquals(30, v, 0);
    }

    public void testNormal_FunctionExpression4() {
        String exp = "foo() + bar()";

        Function fooFunc = new Function() {

            public double call(double[] arguments) {
                return 1;
            }

        };
        Function barFunc = new Function() {

            public double call(double[] arguments) {
                return 2;
            }

        };

        Computer comp = new Computer();
        comp.addFunction("foo", fooFunc);
        comp.addFunction("bar", barFunc);
        double v = comp.compute(exp);

        assertEquals(3, v, 0);
    }

    public void testFail_FunctionExpression1() {
        String exp = "foo()";

        Function func = new Function() {

            public double call(double[] arguments) {
                return 1;
            }

        };

        Computer comp = new Computer();
        comp.addFunction("bar", func);
        try {
            comp.compute(exp);
            fail();
        } catch (ComputeException e) {
            assertEquals("関数{foo}は定義されていません。", e.getMessage());
        }
    }

    public void testFail_FunctionExpression3() {
        String exp = "foo(10, )";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_FunctionExpression4() {
        String exp = "foo(";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_FunctionExpression5() {
        String exp = "foo)";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_FunctionExpression6() {
        String exp = "foo(10 10)";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_Variable1() {
        String exp = "x";

        Computer comp = new Computer();
        comp.setVariable("x", 1);
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_Variable2() {
        String exp = "abc123_";

        Computer comp = new Computer();
        comp.setVariable("abc123_", 1);
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_Variable3() {
        String exp = "_abc";

        Computer comp = new Computer();
        comp.setVariable("_abc", 1);
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_Variable4() {
        String exp = "ほげ";

        Computer comp = new Computer();
        comp.setVariable("ほげ", 1);
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testFail_Variable1() {
        String exp = "123abc";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_ParenthesesExpression1() {
        String exp = "(1 + 2)";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(3, v, 0);
    }

    public void testNormal_ParenthesesExpression2() {
        String exp = "(1 + (2 - 3))";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testFail_ParenthesesExpression1() {
        String exp = "()";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_ParenthesesExpression2() {
        String exp = "(1 + 2";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_ParenthesesExpression3() {
        String exp = "1 + 2)";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_PreIncrementExpression1() {
        String exp = "++x";

        Computer comp = new Computer();
        comp.setVariable("x", 1);
        double v = comp.compute(exp);

        assertEquals(2, v, 0);
        assertEquals(2, comp.getVariable("x"), 0);
    }

    public void testFail_PreIncrementExpression1() {
        String exp = "++1";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_PreIncrementExpression2() {
        String exp = "++++x";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_PreIncrementExpression3() {
        String exp = "++(++x)";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_PostIncrementExpression1() {
        String exp = "x++";

        Computer comp = new Computer();
        comp.setVariable("x", 1);
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
        assertEquals(2, comp.getVariable("x"), 0);
    }

    public void testFail_PostIncrementExpression1() {
        String exp = "1++";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_PostIncrementExpression2() {
        String exp = "x++++";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_PostIncrementExpression3() {
        String exp = "(x++)++";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_PreDecrementExpression1() {
        String exp = "--x";

        Computer comp = new Computer();
        comp.setVariable("x", 1);
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
        assertEquals(0, comp.getVariable("x"), 0);
    }

    public void testFail_PreDecrementExpression1() {
        String exp = "--1";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_PreDecrementExpression2() {
        String exp = "----x";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_PreDecrementExpression3() {
        String exp = "--(--x)";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_PostDecrementExpression1() {
        String exp = "x--";

        Computer comp = new Computer();
        comp.setVariable("x", 1);
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
        assertEquals(0, comp.getVariable("x"), 0);
    }

    public void testFail_PostDecrementExpression1() {
        String exp = "1--";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_PostDecrementExpression2() {
        String exp = "x----";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_PostDecrementExpression3() {
        String exp = "(x--)--";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_UnaryExpressionNotPlusMinus1() {
        String exp = "~1";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(-2, v, 0);
    }

    public void testNormal_UnaryExpressionNotPlusMinus2() {
        String exp = "~x";

        Computer comp = new Computer();
        comp.setVariable("x", 10);
        double v = comp.compute(exp);

        assertEquals(-11, v, 0);
        assertEquals(10, comp.getVariable("x"), 0);
    }

    public void testNormal_UnaryExpressionNotPlusMinus3() {
        String exp = "~(~1)";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_UnaryExpressionNotPlusMinus4() {
        String exp = "!0";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_UnaryExpressionNotPlusMinus5() {
        String exp = "!1";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_UnaryExpressionNotPlusMinus6() {
        String exp = "!0.1";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_UnaryExpressionNotPlusMinus7() {
        String exp = "!(!9)";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testFail_UnaryExpressionNotPlusMinus1() {
        String exp = "~~1";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_UnaryExpressionNotPlusMinus2() {
        String exp = "!!1";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_UnaryExpression1() {
        String exp = "+0";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_UnaryExpression2() {
        String exp = "+999";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(999, v, 0);
    }

    public void testNormal_UnaryExpression3() {
        String exp = "+(+999)";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(999, v, 0);
    }

    public void testNormal_UnaryExpression4() {
        String exp = "+(+x)";

        Computer comp = new Computer();
        comp.setVariable("x", 999);
        double v = comp.compute(exp);

        assertEquals(999, v, 0);
    }

    public void testNormal_UnaryExpression5() {
        String exp = "-0";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_UnaryExpression6() {
        String exp = "-999";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(-999, v, 0);
    }

    public void testNormal_UnaryExpression7() {
        String exp = "-(-999)";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(999, v, 0);
    }

    public void testNormal_UnaryExpression8() {
        String exp = "-(-x)";

        Computer comp = new Computer();
        comp.setVariable("x", 999);
        double v = comp.compute(exp);

        assertEquals(999, v, 0);
    }

    public void testFail_UnaryExpression1() {
        String exp = "+++1";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_UnaryExpression2() {
        String exp = "---1";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_MultiplicativeExpression1() {
        String exp = "2 * 3";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(6, v, 0);
    }

    public void testNormal_MultiplicativeExpression2() {
        String exp = "4 / 2";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(2, v, 0);
    }

    public void testNormal_MultiplicativeExpression3() {
        String exp = "10 % 3";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_MultiplicativeExpression4() {
        String exp = "2 * 6 / 3 % 3";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_MultiplicativeExpression5() {
        String exp = "1 / 0";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertTrue(Double.isInfinite(v));
    }

    public void testNormal_MultiplicativeExpression6() {
        String exp = "0 / 0";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertTrue(Double.isNaN(v));
    }

    public void testFail_MultiplicativeExpression1() {
        String exp = "2 * * 3";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_MultiplicativeExpression2() {
        String exp = "2 / / 3";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_MultiplicativeExpression3() {
        String exp = "2 % % 3";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_MultiplicativeExpression4() {
        String exp = "2 *";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_AdditiveExpression1() {
        String exp = "1 + 2";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(3, v, 0);
    }

    public void testNormal_AdditiveExpression2() {
        String exp = "1 - 2";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(-1, v, 0);
    }

    public void testNormal_AdditiveExpression3() {
        String exp = "1 - 2 + 3";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(2, v, 0);
    }

    public void testFail_AdditiveExpression1() {
        String exp = "1 + + + 1";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_AdditiveExpression2() {
        String exp = "1 - - - 1";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_AdditiveExpression3() {
        String exp = "1 +";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_ShiftExpression1() {
        String exp = "0 << 1";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_ShiftExpression2() {
        String exp = "123 << 3";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(984, v, 0);
    }

    public void testNormal_ShiftExpression3() {
        String exp = "0 >> 1";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_ShiftExpression4() {
        String exp = "123 >> 1";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(61, v, 0);
    }

    public void testNormal_ShiftExpression5() {
        String exp = "-123 >> 1";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(-62, v, 0);
    }

    public void testNormal_ShiftExpression6() {
        String exp = "0 >>> 1";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_ShiftExpression7() {
        String exp = "123 >>> 1";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(61, v, 0);
    }

    public void testNormal_ShiftExpression8() {
        String exp = "-123 >>> 1";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(9.223372036854776E18, v, 0);
    }

    public void testNormal_ShiftExpression9() {
        String exp = "123.456 << 3.456";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(984, v, 0);
    }

    public void testNormal_ShiftExpression10() {
        String exp = "1 << 2 << 3";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(32, v, 0);
    }

    public void testFail_ShiftExpression1() {
        String exp = "1 << << 1";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_ShiftExpression2() {
        String exp = "1 >> >> 1";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_ShiftExpression3() {
        String exp = "1 >>> >>> 1";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_ShiftExpression4() {
        String exp = "1 <<<< 1";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_RelationalExpression1() {
        String exp = "123 > 122";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_RelationalExpression2() {
        String exp = "123.1 > 123.2";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_RelationalExpression3() {
        String exp = "123.1 > 123.1";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_RelationalExpression4() {
        String exp = "122 < 123";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_RelationalExpression5() {
        String exp = "123.2 < 123.1";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_RelationalExpression6() {
        String exp = "123.2 < 123.2";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_RelationalExpression7() {
        String exp = "123 >= 122";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_RelationalExpression8() {
        String exp = "123.1 >= 123.2";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_RelationalExpression9() {
        String exp = "123.1 >= 123.1";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_RelationalExpression10() {
        String exp = "122 <= 123";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_RelationalExpression11() {
        String exp = "123.2 <= 123.1";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_RelationalExpression12() {
        String exp = "123.1 <= 123.1";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_RelationalExpression13() {
        String exp = "0 > 0 > 0";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testFail_RelationalExpression1() {
        String exp = "1 > > 2";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_RelationalExpression2() {
        String exp = "1 < < 2";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_RelationalExpression3() {
        String exp = "1 >= >= 2";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_RelationalExpression4() {
        String exp = "1 <= <= 2";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_EqualityExpression1() {
        String exp = "123.456 == 123.456";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_EqualityExpression2() {
        String exp = "123.456 == 123.4567";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_EqualityExpression3() {
        String exp = "123.456 != 123.456";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_EqualityExpression4() {
        String exp = "123.456 != 123.4567";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_EqualityExpression5() {
        String exp = "NaN == NaN";

        Computer comp = new Computer();
        comp.setVariable("NaN", Double.NaN);
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_EqualityExpression6() {
        String exp = "NaN != NaN";

        Computer comp = new Computer();
        comp.setVariable("NaN", Double.NaN);
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_EqualityExpression7() {
        String exp = "0 == 0 == 0";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testFail_EqualityExpression1() {
        String exp = "1 == == 2";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_EqualityExpression2() {
        String exp = "1 != != 2";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_AndExpression1() {
        String exp = "123 & 456";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(72, v, 0);
    }

    public void testNormal_AndExpression2() {
        String exp = "123.456 & 456.789";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(72, v, 0);
    }

    public void testNormal_AndExpression3() {
        String exp = "123 & 456 & 789";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testFail_AndExpression1() {
        String exp = "123 & & 456";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_ExclusiveOrExpression1() {
        String exp = "123 ^ 456";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(435, v, 0);
    }

    public void testNormal_ExclusiveOrExpression2() {
        String exp = "123.456 ^ 456.789";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(435, v, 0);
    }

    public void testNormal_ExclusiveOrExpression3() {
        String exp = "123 ^ 456 ^ 789";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(678, v, 0);
    }

    public void testFail_ExclusiveOrExpression1() {
        String exp = "123 ^ ^ 456";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_InclusiveOrExpression1() {
        String exp = "123 | 456";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(507, v, 0);
    }

    public void testNormal_InclusiveOrExpression2() {
        String exp = "123.456 | 456.789";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(507, v, 0);
    }

    public void testNormal_InclusiveOrExpression3() {
        String exp = "123 | 456 | 789";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1023, v, 0);
    }

    public void testFail_InclusiveOrExpression1() {
        String exp = "123 | | 456";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_ConditionalAndExpression1() {
        String exp = "0 && 0";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_ConditionalAndExpression2() {
        String exp = "0 && 1";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_ConditionalAndExpression3() {
        String exp = "0 && 123.456";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_ConditionalAndExpression4() {
        String exp = "1 && 0";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_ConditionalAndExpression5() {
        String exp = "123.456 && 0";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_ConditionalAndExpression6() {
        String exp = "1 && 1";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_ConditionalAndExpression7() {
        String exp = "123.456 && 123.456";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testFail_ConditionalAndExpression1() {
        String exp = "0 && && 0";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_ConditionalOrExpression1() {
        String exp = "0 || 0";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(0, v, 0);
    }

    public void testNormal_ConditionalOrExpression2() {
        String exp = "0 || 1";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_ConditionalOrExpression3() {
        String exp = "0 || 123.456";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_ConditionalOrExpression4() {
        String exp = "1 || 0";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_ConditionalOrExpression5() {
        String exp = "123.456 || 0";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_ConditionalOrExpression6() {
        String exp = "1 || 1";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testNormal_ConditionalOrExpression7() {
        String exp = "123.456 || 123.456";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
    }

    public void testFail_ConditionalOrExpression1() {
        String exp = "0 || || 0";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_ConditionalExpression1() {
        String exp = "0 ? 123 : 456";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(456, v, 0);
    }

    public void testNormal_ConditionalExpression2() {
        String exp = "1 ? 123 : 456";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(123, v, 0);
    }

    public void testNormal_ConditionalExpression3() {
        String exp = "789 ? 123 : 456";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(123, v, 0);
    }

    public void testNormal_ConditionalExpression4() {
        String exp = "0 ? 1 : 2 ? 3 : 4";

        Computer comp = new Computer();
        double v = comp.compute(exp);

        assertEquals(3, v, 0);
    }

    public void testFail_ConditionalExpression2() {
        String exp = "0 ? 1";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_ConditionalExpression3() {
        String exp = "1 : 2";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testFail_ConditionalExpression4() {
        String exp = "0 ? 1 ? 2";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

    public void testNormal_AssignmentExpression1() {
        String exp = "x = 123";

        Computer comp = new Computer();
        comp.setVariable("x", 1);
        double v = comp.compute(exp);

        assertEquals(123, v, 0);
        assertEquals(123, comp.getVariable("x"), 0);
    }

    public void testNormal_AssignmentExpression2() {
        String exp = "x += 1";

        Computer comp = new Computer();
        comp.setVariable("x", 123);
        double v = comp.compute(exp);

        assertEquals(124, v, 0);
        assertEquals(124, comp.getVariable("x"), 0);
    }

    public void testNormal_AssignmentExpression3() {
        String exp = "x -= 1";

        Computer comp = new Computer();
        comp.setVariable("x", 123);
        double v = comp.compute(exp);

        assertEquals(122, v, 0);
        assertEquals(122, comp.getVariable("x"), 0);
    }

    public void testNormal_AssignmentExpression4() {
        String exp = "x *= 2";

        Computer comp = new Computer();
        comp.setVariable("x", 123);
        double v = comp.compute(exp);

        assertEquals(246, v, 0);
        assertEquals(246, comp.getVariable("x"), 0);
    }

    public void testNormal_AssignmentExpression5() {
        String exp = "x /= 3";

        Computer comp = new Computer();
        comp.setVariable("x", 123);
        double v = comp.compute(exp);

        assertEquals(41, v, 0);
        assertEquals(41, comp.getVariable("x"), 0);
    }

    public void testNormal_AssignmentExpression6() {
        String exp = "x %= 4";

        Computer comp = new Computer();
        comp.setVariable("x", 123);
        double v = comp.compute(exp);

        assertEquals(3, v, 0);
        assertEquals(3, comp.getVariable("x"), 0);
    }

    public void testNormal_AssignmentExpression7() {
        String exp = "x &= 1";

        Computer comp = new Computer();
        comp.setVariable("x", 123);
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
        assertEquals(1, comp.getVariable("x"), 0);
    }

    public void testNormal_AssignmentExpression8() {
        String exp = "x ^= 1";

        Computer comp = new Computer();
        comp.setVariable("x", 123);
        double v = comp.compute(exp);

        assertEquals(122, v, 0);
        assertEquals(122, comp.getVariable("x"), 0);
    }

    public void testNormal_AssignmentExpression9() {
        String exp = "x |= 1";

        Computer comp = new Computer();
        comp.setVariable("x", 123);
        double v = comp.compute(exp);

        assertEquals(123, v, 0);
        assertEquals(123, comp.getVariable("x"), 0);
    }

    public void testNormal_AssignmentExpression10() {
        String exp = "x <<= 1";

        Computer comp = new Computer();
        comp.setVariable("x", 123);
        double v = comp.compute(exp);

        assertEquals(246, v, 0);
        assertEquals(246, comp.getVariable("x"), 0);
    }

    public void testNormal_AssignmentExpression11() {
        String exp = "x >>= 1";

        Computer comp = new Computer();
        comp.setVariable("x", 123);
        double v = comp.compute(exp);

        assertEquals(61, v, 0);
        assertEquals(61, comp.getVariable("x"), 0);
    }

    public void testNormal_AssignmentExpression12() {
        String exp = "x >>>= 1";

        Computer comp = new Computer();
        comp.setVariable("x", 123);
        double v = comp.compute(exp);

        assertEquals(61, v, 0);
        assertEquals(61, comp.getVariable("x"), 0);
    }

    public void testNormal_AssignmentExpression13() {
        String exp = "x = y = 1";

        Computer comp = new Computer();
        comp.setVariable("x", 123);
        comp.setVariable("y", 456);
        double v = comp.compute(exp);

        assertEquals(1, v, 0);
        assertEquals(1, comp.getVariable("x"), 0);
        assertEquals(1, comp.getVariable("y"), 0);
    }

    public void testFail_AssignmentExpression1() {
        String exp = "x = = 1";

        Computer comp = new Computer();
        try {
            comp.compute(exp);
            fail();
        } catch (ParseException e) {
        }
    }

}
