/*
 * Copyright (C) 2006 uguu@users.sourceforge.jp, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.expression_computer.util;

import java.util.regex.Pattern;

/**
 * <p>
 * 文字列の形式を検証する機能を持ちます。
 * </p>
 * 
 * @author uguu@users.sourceforge.jp
 */
public final class Validator {

    private Validator() {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * 文字列の形式が小数点値の形式かどうかを検証します。
     * </p>
     * 
     * @param text
     *            検証する文字列。
     * @return 文字列が小数点値の形式の場合はtrue、そうでない場合はfalse。
     */
    public static boolean isFloatingPointLiteral(String text) {
        return Pattern.matches("[0-9]*\\.[0-9]+", text);
    }

    /**
     * <p>
     * 文字列の形式が整数値の形式かどうかを検証します。
     * </p>
     * 
     * @param text
     *            検証する文字列。
     * @return 文字列が整数値の形式の場合はtrue、そうでない場合はfalse。
     */
    public static boolean isIntegerLiteral(String text) {
        return Pattern.matches("[1-9][0-9]*|0[xX][0-9a-fA-F]+|0[0-7]+|0", text);
    }

    /**
     * <p>
     * 文字列の形式が識別子の形式かどうかを検証します。
     * </p>
     * 
     * @param text
     *            検証する文字列。
     * @return 文字列が識別子の形式の場合はtrue、そうでない場合はfalse。
     */
    public static boolean isIdentifier(String text) {
        return Pattern.matches("[\\u0024\\u0041-\\u005a\\u005f\\u0061-\\u007a\\u00c0-\\u00d6\\u00d8-\\u00f6\\u00f8-\\u00ff\\u0100-\\u1fff\\u3040-\\u318f\\u3300-\\u337f\\u3400-\\u3d2d\\u4e00-\\u9fff\\uf900-\\ufaff]([\\u0024\\u0041-\\u005a\\u005f\\u0061-\\u007a\\u00c0-\\u00d6\\u00d8-\\u00f6\\u00f8-\\u00ff\\u0100-\\u1fff\\u3040-\\u318f\\u3300-\\u337f\\u3400-\\u3d2d\\u4e00-\\u9fff\\uf900-\\ufaff]|[\\u0030-\\u0039\\u0660-\\u0669\\u06f0-\\u06f9\\u0966-\\u096f\\u09e6-\\u09ef\\u0a66-\\u0a6f\\u0ae6-\\u0aef\\u0b66-\\u0b6f\\u0be7-\\u0bef\\u0c66-\\u0c6f\\u0ce6-\\u0cef\\u0d66-\\u0d6f\\u0e50-\\u0e59\\u0ed0-\\u0ed9\\u1040-\\u1049])*", text);
    }
}
