<?php
/* SVN FILE: $Id: Store.php 616 2008-07-20 03:28:17Z bb_yujiro $ */
/**
 * OpenID store
 *
 * PHP versions 5
 *
 *      hitSuji : Social Network Service <http://rakuto.net/rktSNS/>
 *      Copyright (c) 2008 Yujiro Takahashi
 *
 * Licensed under The MIT License
 * Redistributions of files must retain the above copyright notice.
 *
 * @filesource
 * @package         hitSuji
 * @copyright       Copyright (c) 2008 Yujiro Takahashi
 * @link            http://rakuto.net/
 * @author          Yujiro Takahashi <yujiro@rakuto.net>
 * @version         $Revision: 616 $
 * @modifiedby      $LastChangedBy: bb_yujiro $
 * @lastmodified    $Date: 2008-07-20 12:28:17 +0900 (日, 20 7 2008) $
 * @license         http://opensource.org/licenses/mit-license.php The MIT License
 */

/**
 * Require base class for creating a new interface.
 */
require_once LIB_DIR.'PEAR/Auth/OpenID.php';
require_once LIB_DIR.'PEAR/Auth/OpenID/Interface.php';
require_once LIB_DIR.'PEAR/Auth/OpenID/HMACSHA1.php';
require_once LIB_DIR.'PEAR/Auth/OpenID/Nonce.php';

/**
 * Store
 *
 * @category        Authentication
 * @package         hitSuji
 * @copyright       Copyright (c) 2008 Yujiro Takahashi
 * @link            http://rakuto.net/
 * @author          Yujiro Takahashi <yujiro@rakuto.net>
 * @version         $Revision: 616 $
 * @modifiedby      $LastChangedBy: bb_yujiro $
 * @lastmodified    $Date: 2008-07-20 12:28:17 +0900 (日, 20 7 2008) $
 * @license         http://opensource.org/licenses/mit-license.php The MIT License
 */
class Auth_OpenID_Store extends Auth_OpenID_OpenIDStore
{
    /**
     * Association
     * @const string
     */
    const ASSOCIATIONS_TABLE = 'oid_associations';

    /**
     * NONCES
     * @const string
     */
    const NONCES_TABLE = 'oid_nonces';

    /**
     * DBオブジェクト
     * @var object
     */
    private $_objdb;

    /**
     * コンストラクタ
     *
     * @access public
     * @return void
     */
    public function __construct()
    {
        $this->_objdb = RKT_DB::getInstance();

        $this->max_nonce_age = 21600; // 6 * 60 * 60
    }

    /**
     * テーブルを削除する
     *
     * @access  public
     * @return  void
     */
    public function reset()
    {
        $this->objdb->exec('DELETE FROM '.DB_PREFIX.self::ASSOCIATIONS_TABLE);
        $this->objdb->exec('DELETE FROM '.DB_PREFIX.self::NONCES_TABLE);
    }

    /**
     * テーブルの作成：未サポート
     * hitSujiでは、管理者ページから作成
     *
     * @access  public
     * @return  boolean
     */
    public function createTables()
    {
        return true;
    }

    /**
     * 認証結果を取得する
     *
     * @access  public
     * @param string $phrase
     * @param array $selected
     * @return  boolean
     */
    public function storeAssociation($server_url, $association)
    {
        /* ID設定 */
        $id = sha1($server_url.$association->handle);

        $result = $this->_getAssoc($id);

        if (empty($result)){
            $sql = 'INSERT INTO '.DB_PREFIX.self::ASSOCIATIONS_TABLE.' '.
                   '( id, server_url, handle, secret, issued, lifetime, assoc_type) VALUES '.
                   '(:id,:server_url,:handle,:secret,:issued,:lifetime,:assoc_type)';
        } else {
            $sql = 'UPDATE '.DB_PREFIX.self::ASSOCIATIONS_TABLE.
                   ' SET server_url = :server_url, handle = :handle, secret = :secret, '.
                        'issued = :issued, lifetime = :lifetime, assoc_type = :assoc_type '.
                   ' WHERE id = :id';
        }
        $stmt = $this->_objdb->prepare($sql);

        $stmt->bindParam(':id',         $id,                      PDO::PARAM_STR);
        $stmt->bindParam(':server_url', $server_url,              PDO::PARAM_STR);
        $stmt->bindParam(':handle',     $association->handle,     PDO::PARAM_STR);
        $stmt->bindParam(':secret',     $association->secret,     PDO::PARAM_LOB);
        $stmt->bindParam(':issued',     $association->issued,     PDO::PARAM_STR);
        $stmt->bindParam(':lifetime',   $association->lifetime,   PDO::PARAM_INT);
        $stmt->bindParam(':assoc_type', $association->assoc_type, PDO::PARAM_STR);

        return $stmt->execute();
    }

    /**
     * Associationの削除
     *
     * @access  public
     * @param string $phrase
     * @param array $selected
     * @return  boolean
     */
    public function removeAssociation($server_url, $handle)
    {
        /* ID設定 */
        $id = sha1($server_url.$handle);

        if ($this->_getAssoc($id) == null) {
            return false;
        }

        $sql = 'DELETE FROM '.DB_PREFIX.self::ASSOCIATIONS_TABLE.
              ' WHERE id = :id';
        $stmt = $this->_objdb->prepare($sql);
        $stmt->bindParam(':id', $id, PDO::PARAM_STR);
        return $stmt->execute();
    }

    /**
     * 認証結果を取得する
     *
     * @access  public
     * @param string $phrase
     * @param array $selected
     * @return  boolean
     */
    public function getAssociation($server_url, $handle = null)
    {
        if ($handle !== null) {
            /* ID設定 */
            $id = sha1($server_url.$handle);
            $assoc = $this->_getAssoc($id);

            $assocs = array();
            if ($assoc) {
                $assocs[] = $assoc;
            }
        } else {
            $assocs = $this->_getAssocs($server_url);
        }

        if (!$assocs || (count($assocs) == 0)) {
            return null;
        } else {
            $associations = array();

            foreach ($assocs as $assoc_row) {
                $assoc = new Auth_OpenID_Association($assoc_row['handle'],
                                                     $assoc_row['secret'],
                                                     $assoc_row['issued'],
                                                     $assoc_row['lifetime'],
                                                     $assoc_row['assoc_type']);
                if ($assoc->getExpiresIn() == 0) {
                    $this->removeAssociation($server_url, $assoc->handle);
                } else {
                    $associations[] = array($assoc->issued, $assoc);
                }
            }

            if ($associations) {
                $issued = array();
                $assocs = array();
                foreach ($associations as $key => $assoc) {
                    $issued[$key] = $assoc[0];
                    $assocs[$key] = $assoc[1];
                }

                array_multisort($issued, SORT_DESC, $assocs, SORT_DESC,
                                $associations);

                // return the most recently issued one.
                list($issued, $assoc) = $associations[0];
                return $assoc;
            } else {
                return null;
            }
        }
    }

    /**
     * Nonceの使用
     *
     * @access  public
     * @param string $server_url
     * @param integer $timestamp
     * @param string $salt
     * @return  boolean
     */
    public function useNonce($server_url, $timestamp, $salt)
    {
        global $Auth_OpenID_SKEW;

        if ( abs($timestamp - time()) > $Auth_OpenID_SKEW ) {
            return False;
        }

        $sql = 'INSERT INTO '.DB_PREFIX.self::NONCES_TABLE.' '.
               '( server_url, timestamp, salt) VALUES '.
               '(:server_url,:timestamp,:salt)';
        $stmt = $this->_objdb->prepare($sql);

        $stmt->bindParam(':server_url', $server_url, PDO::PARAM_STR);
        $stmt->bindParam(':timestamp',  $timestamp,  PDO::PARAM_INT);
        $stmt->bindParam(':salt',       $salt,       PDO::PARAM_STR);

        return $stmt->execute();
    }

    /**
     * Nonceリストの削除
     *
     * @access  public
     * @return  integer
     */
    public function cleanupNonces()
    {
        global $Auth_OpenID_SKEW;
        $timestamp = time() - $Auth_OpenID_SKEW;

        $sql = 'DELETE FROM '.DB_PREFIX.self::NONCES_TABLE.
              ' WHERE timestamp < :timestamp';
        $stmt = $this->_objdb->prepare($sql); 
        $stmt->bindParam(':timestamp', $timestamp, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->rowCount();
    }

    /**
     * Associationリストの削除
     *
     * @access  public
     * @return  integer
     */
    public function cleanupAssociations()
    {
        $sql = 'DELETE FROM '.DB_PREFIX.self::ASSOCIATIONS_TABLE.
              ' WHERE issued + lifetime < :timestamp';
        $stmt = $this->_objdb->prepare($sql);

        $timestamp = time();
        $stmt->bindParam(':timestamp', $timestamp, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->rowCount();

    }

    /**
     * Associationデータを1行取得する
     *
     * @access  private
     * @param string $id
     * @return  array
     */
    private function _getAssoc($id)
    {
        $sql = 'SELECT handle, secret, issued, lifetime, assoc_type '.
               'FROM '.DB_PREFIX.self::ASSOCIATIONS_TABLE.' '.
               'WHERE id = :id';
        $stmt = $this->_objdb->prepare($sql);
        $stmt->bindParam(':id', $id, PDO::PARAM_STR);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $stmt->closeCursor();
        
        if (empty($result['handle'])){
            return null;
        }
        return $result;
    }

    /**
     * Associationデータ一覧を取得する
     *
     * @access  private
     * @param string $server_url
     * @return  array
     */
    private function _getAssocs($server_url)
    {
        $sql = 'SELECT handle, secret, issued, lifetime, assoc_type '.
               'FROM '.DB_PREFIX.self::ASSOCIATIONS_TABLE.' '.
               'WHERE server_url = :server_url';
        $stmt = $this->_objdb->prepare($sql);
        $stmt->bindParam(':server_url', $server_url, PDO::PARAM_STR);
        $stmt->execute();
        $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (empty($result)){
            return array();
        }
        return $result;
    }
} // Auth_OpenID_Store
?>