<?php
/**
 * SimpleBlog Flavor系画面用基底クラス
 *
 * SimpleBlog Flavor系画面で使用する基底クラス
 *
 * @package   simpleblog
 * @author    Risoluto Developers
 * @license   http://opensource.org/licenses/bsd-license.php new BSD license
 * @copyright (C) 2008-2011 Risoluto Developers / All Rights Reserved.
 */

  /**
   * Smartyクラス
   */
  require_once( 'Smarty.class.php' );

  /**
   * セッションクラス
   */
  require_once( RISOLUTO_FUNC . 'risoluto_session.php' );
  /**
   * コンフィグ操作クラス
   */
  require_once( RISOLUTO_FUNC . 'risoluto_conf.php' );
  /**
   * ユーティリティファンクションクラス
   */
  require_once( RISOLUTO_FUNC . 'risoluto_util.php' );
  /**
   * ログ出力クラス
   */
  require_once( RISOLUTO_FUNC . 'risoluto_log.php' );

  abstract class simpleblog_base
  {
    //------------------------------------------------------//
    // クラス変数定義
    //------------------------------------------------------//
    /**
     * Smartyインスタンス格納用変数
     * @access protected
     * @var    object
     */
    protected $smarty;
    /**
     * セッションクラスインスタンスを保持する変数
     * @access protected
     * @var    object
     */
    protected $obj_sess;
    /**
     * ユーティリティクラスインスタンスを保持する変数
     * @access protected
     * @var    object
     */
    protected $obj_util;
    /**
     * コンフィグクラスインスタンスを保持する変数
     * @access protected
     * @var    object
     */
    protected $obj_adminconf;
    /**
     * ログクラスインスタンスを保持する変数
     * @access private
     * @var    object
     */
    private $obj_logs;
    /**
     * cage（URL用）を保持する変数
     * @access protected
     * @var    string
     */
    protected $mycage;

    //------------------------------------------------------//
    // 抽象クラスメソッド定義
    //------------------------------------------------------//
    abstract function model();
    abstract function view();

    //------------------------------------------------------//
    // クラスメソッド定義
    //------------------------------------------------------//
    /**
     * コンストラクタメソッド
     *
     * コントローラのコンストラクタメソッド
     *
     * @param     void なし
     * @return    void なし
     */
    public function __construct()
    {
    } // end of function:__construct()

    /**
     * クローンメソッド
     *
     * コントローラのクローンメソッド
     *
     * @param     void なし
     * @return    void なし
     */
    public function __clone()
    {
    } // end of function:__clone()

    /**
     * 初期化メソッド（イニシャライザ）
     *
     * 当該クラスの処理実施に先立ち初期化処理を行う
     *
     * @param     void なし
     * @return    boolean ファンクション実行結果（ true：正常終了 / false: 異常終了 ）
     */
    public function init()
    {
      // セッションクラスインスタンスの生成
      $this->obj_sess      = RisolutoSession::singleton();
      // コンフィグクラスインスタンスの生成
      $this->obj_adminconf = new RisolutoConf();
      $this->obj_adminconf->parse( RISOLUTO_CONF . 'risoluto_admin.ini' );
      // ユーティリティ
      $this->obj_util      = RisolutoUtils::singleton();
      // ログ
      $this->obj_logs      = RisolutoLog::singleton();

      // Smarty関連定義
      $this->smarty                = new Smarty;
      $this->smarty->template_dir  = RISOLUTO_USERLAND . $this->obj_sess->sessLoad( 'currentcage' );
      $this->smarty->config_dir    = RISOLUTO_USERLAND . $this->obj_sess->sessLoad( 'currentcage' );
      $this->smarty->compile_dir   = RISOLUTO_CACHE;
      $this->smarty->cache_dir     = RISOLUTO_CACHE;
      $this->smarty->caching       = false;
      $this->smarty->debugging     = false;
      $this->smarty->force_compile = true;
      $this->smarty->compile_check = true;

      // 自分のcage名をクラス変数に入れておく
      $this->mycage = str_replace( DIRECTORY_SEPARATOR, '.', $this->obj_sess->sessLoad( 'currentcage' ) );

      return true;
    } // end of function:initializer()

    /**
     * エラー処理メソッド（エラーハンドラ）
     *
     * 当該クラス内でエラーが発生したときの処理を行う
     *
     * @param     void なし
     * @return    boolean ファンクション実行結果（ true：正常終了 / false: 異常終了 ）
     */
    public function errHandler()
    {
      // 管理系エラー画面に飛ばす
      $this->obj_util->redirectTo( $this->obj_adminconf->get( 'COMMON', 'error_act'  )
                                 , $this->obj_adminconf->get( 'COMMON', 'error_cage' ) );
      return true;
    } // end of function:errHandler()

    /**
     * 後処理メソッド（クリーナ）
     *
     * 当該クラス内のすべての処理が終了した後に必ず実行される処理を行う
     *
     * @param     void なし
     * @return    boolean ファンクション実行結果（ true：正常終了 / false: 異常終了 ）
     */
    public function clean() {
      // クラス変数を解放
      unset( $this->smarty        );
      unset( $this->obj_sess      );
      unset( $this->obj_util      );
      unset( $this->obj_logs      );
      unset( $this->obj_adminconf );
      unset( $this->mycage );
      return true;
    } // end of function:clean()

    /**
     * トラックバックPing送信メソッド
     *
     * 引数で指定された情報を元にトラックバックPingを送信する
     *
     * @param     $tb_url   string トラックバックPing送信先URL
     * @param     $own_url  string トラックバックPing送信先に通知する自身のURL
     * @param     $blogname string トラックバックPing送信先に通知するブログのタイトル
     * @param     $title    string トラックバックPing送信先に通知する記事のタイトル
     * @param     $content  string トラックバックPing送信先に通知する記事の内容（先頭から200文字までが使われます）
     * @return    boolean ファンクション実行結果（ true：正常終了 / false: 異常終了 ）
     */
    public function sendTb( $tb_url, $own_url, $blogname, $title, $content )
    {

      // トラックバックURLを分解
      $tb_url = html_entity_decode( $tb_url, ENT_QUOTES, 'UTF-8' );
      if ( ( $tb_url = parse_url( $tb_url ) ) == false )
      {
         return false;
      } // end of if

      // 分解したトラックバックURLにおいて、
      // ポートの情報がない場合はデフォルトで「80」をセットする
      if ( !isset( $tb_url[ 'port' ] ) or empty( $tb_url[ 'port' ] ) )
      {
        $tb_url[ 'port' ] = 80;
      } // end of if

      // 分解したトラックバックURLにおいて、
      // クエリの情報がある場合は先頭に「?」を付与する
      if ( isset( $tb_url[ 'query' ] ) and !empty( $tb_url[ 'query' ] ) )
      {
        $tb_url[ 'query' ] = '?' . $tb_url[ 'query' ];
      } // end of if
      else
      {
        $tb_url[ 'query' ] = '';
      } // end of else

      // 自身のURLを分解
      $tmp_own = html_entity_decode( $own_url,  ENT_QUOTES, 'UTF-8' );
      if ( ( $tmp_own = parse_url( $tmp_own ) ) == false )
      {
         return false;
      } // end of if

      // 分解した自身のURLにおいて、
      // クエリの情報がある場合は先頭に「?」を付与する
      if ( isset( $tmp_own[ 'query' ] ) and !empty( $tmp_own[ 'query' ] ) )
      {
        $tmp_own[ 'query' ] = '?' . $tmp_own[ 'query' ];
      } // end of if
      else
      {
        $tmp_own[ 'query' ] = '';
      } // end of else

      // 自身のURLを再合成
      $tmp_own_cat = 'http://' . $tmp_own[ 'host' ] . $tmp_own[ 'path' ] . htmlentities( $tmp_own[ 'query' ], ENT_QUOTES, 'UTF-8' );



      // 引数で指定された値を送信に備えてまとめる（ついでにURLエンコードもかける）
      $tb_dat = "url=" . urlencode( $tmp_own_cat )
              . "&blog_name=" . urlencode( trim( $blogname ) )
              . "&title=" . urlencode(  mb_strimwidth( trim( $title ), 0, 40, '……', 'UTF-8' ) )
              . "&excerpt=" . urlencode( htmlentities( mb_strimwidth( trim( strip_tags( $content ) ), 0, 200, '……', 'UTF-8' ), ENT_QUOTES, 'UTF-8' ) );

      // 送信するトラックバックpingの構築
      $send_tbping  = "POST " . $tb_url[ 'path' ] . $tb_url[ 'query' ] . " HTTP/1.0\r\n";
      $send_tbping .= "Host: " . $tb_url[ 'host' ] . "\r\n";
      $send_tbping .= "Content-Type: application/x-www-form-urlencoded\r\n";
      $send_tbping .= "Content-Length: " . strlen( $tb_dat ) . "\r\n";
      $send_tbping .= "\r\n";
      $send_tbping .= $tb_dat."\r\n";

      // ソケットをオープンし、トラックバックpingを送信する
      if ( ( $tb_fp = fsockopen( $tb_url[ 'host' ], $tb_url[ 'port' ] ) ) )
      {
        // トラックバックpingの送信
        fwrite( $tb_fp, $send_tbping );

        // レスポンスを取得
        $tb_resp = '';
        while( !feof( $tb_fp ) )
        {
          $tb_resp .= fread( $tb_fp, 4096 );
        } // end of while

        // ソケットをクローズ
        fclose( $tb_fp );

        // レスポンス中に「<error>0</error>」の文字列がある場合は成功したと見なす
        if ( preg_match( '/<error>0<\/error>/', $tb_resp ) )
        {
          return true;
        } // end of if
        // 失敗した場合は、ログを出力する
        else
        {
          $this->obj_logs->logging( 'error', "[simpleblog_base::sendTb]Response Error : " . print_r( $tb_resp, true ) );
          $this->obj_logs->logging( 'error', "[simpleblog_base::sendTb]HTTP HEADER    : " . print_r( $send_tbping, true ) );
          return false;
        } // end of else
      } // end of if
      // ソケットのオープンに失敗した場合、すぐに戻る
      else
      {
        $this->obj_logs->logging( 'error', "[simpleblog_base::sendTb]Socket Open Error : " . print_r( $tb_resp, true ) );
        $this->obj_logs->logging( 'error', "[simpleblog_base::sendTb]HTTP HEADER       : " . print_r( $send_tbping, true ) );
        return false;
      } // end of else

    } // end of function: sendTb

  } // end of class:simpleblog_base
?>
