/*
 * Copyright 2009-2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.misc;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.MathContext;
import java.math.RoundingMode;
import java.util.HashSet;
import java.util.Set;

/**
 * 
 *
 *
 * @author MORIGUCHI, Yuichiro 2010
 */
public final class Rational implements Comparable<Rational> {

	//
	private BigInteger numer;
	private BigInteger denom;

	//
	private Rational(BigInteger num, BigInteger den) {
		numer = num;
		denom = den;
	}

	/**
	 * 
	 */
	public static final Rational ZERO =
		new Rational(BigInteger.ZERO, BigInteger.ONE);

	/**
	 * 
	 */
	public static final Rational ONE =
		new Rational(BigInteger.ONE, BigInteger.ONE);

	/**
	 * 
	 * @param num
	 * @param den
	 * @return
	 */
	public static final Rational valueOf(
			BigInteger num, BigInteger den) {
		BigInteger n = num;
		BigInteger d = den;
		int nsig = num.signum();
		int dsig = den.signum();

		if(dsig == 0) {
			throw new ArithmeticException("denominator is zero");
		} else if(nsig == 0) {
			return ZERO;
		} else if(dsig < 0) {
			n = n.negate();
			d = d.negate();
		}

		BigInteger gcd = n.gcd(d);
		n = n.divide(gcd);
		d = d.divide(gcd);

		return new Rational(n, d);
	}

	/**
	 * 
	 * @param num
	 * @param den
	 * @return
	 */
	public static final Rational valueOf(long num, long den) {
		return valueOf(BigInteger.valueOf(num),
				BigInteger.valueOf(den));
	}

	/**
	 * 
	 * @param num
	 * @param den
	 * @return
	 */
	public static final Rational valueOf(long num) {
		return valueOf(num, 1);
	}

	/**
	 * 
	 * @param v
	 * @return
	 */
	public static Rational valueOf(BigDecimal v) {
		BigInteger n = v.unscaledValue();

		if(v.scale() > 0) {
			BigInteger d = BigInteger.TEN.pow(v.scale());

			return valueOf(n, d);
		} else if(v.scale() < 0) {
			BigInteger d = BigInteger.TEN.pow(-v.scale());

			return valueOf(n.multiply(d), BigInteger.ONE);
		} else {
			return valueOf(n, BigInteger.ONE);
		}
	}

	/**
	 * 
	 * @param val
	 * @return
	 */
	public static Rational valueOf(double val) {
		if(Double.isInfinite(val) || Double.isNaN(val)) {
			throw new ArithmeticException(
					"Infinity and NaN is not supported");
		} else {
			BigDecimal v = BigDecimal.valueOf(val);
			return valueOf(v);
		}
	}

	/**
	 * 
	 * @param b
	 * @return
	 */
	public static Rational valueOf(BigInteger b) {
		return new Rational(b, BigInteger.ONE);
	}

	/**
	 * 
	 * @param r
	 * @return
	 */
	public BigDecimal toBigDecimal() {
		BigDecimal n = new BigDecimal(numer);
		BigDecimal d = new BigDecimal(denom);

		return n.divide(d);
	}

	/**
	 * 
	 * @param r
	 * @param mc
	 * @return
	 */
	public BigDecimal toBigDecimal(MathContext mc) {
		BigDecimal n = new BigDecimal(numer);
		BigDecimal d = new BigDecimal(denom);

		return n.divide(d, mc);
	}

	/**
	 * 
	 * @param r
	 * @param scale
	 * @return
	 */
	public BigDecimal toBigDecimal(int scale) {
		BigDecimal n = new BigDecimal(numer);
		BigDecimal d = new BigDecimal(denom);

		return n.divide(d, scale, RoundingMode.HALF_UP);
	}

	/**
	 * 
	 * @param r1
	 * @param r2
	 * @return
	 */
	public static Rational rationalize(Rational r1, Rational r2) {
		Rational e, x, y;
		BigInteger xn, xd, yn, yd;
		BigInteger a, b, c, d;

		e = (r2.compareTo(ZERO) < 0) ? r2.negate() : r2;
		x = r1.add(e);
		y = r1.subtract(e);
		if(x.equals(y)) {
			return x;
		}
		xn = x.numer;  xd = x.denom;
		yn = y.numer;  yd = y.denom;

		a = d = BigInteger.ONE;
		b = c = BigInteger.ZERO;
		while(true) {
			BigInteger[] qx, qy;
			BigInteger xq;

			qx = xn.divideAndRemainder(xd);
			qy = yn.divideAndRemainder(yd);
			xq = qx[0];
			if(qx[1].signum() == 0) {
				return valueOf(
						a.multiply(xq).add(b),
						c.multiply(xq).add(d));
			} else if(xq.compareTo(qy[0]) < 0) {
				return valueOf(
						a.multiply(xq.add(BigInteger.ONE)).add(b),
						c.multiply(xq.add(BigInteger.ONE)).add(d));
			} else if(xq.compareTo(qy[0]) > 0) {
				throw new RuntimeException("implementation error");
			} else {
				BigInteger aa = a, cc = c;

				a = a.multiply(xq).add(b);  b = aa;
				c = c.multiply(xq).add(d);  d = cc;
				xn = yd;  yn = xd;
				xd = qy[1];  yd = qx[1];
			}
		}
	}

	/**
	 * 
	 * @return
	 */
	public BigInteger getNumerator() {
		return numer;
	}

	/**
	 * 
	 * @return
	 */
	public BigInteger getDenominator() {
		return denom;
	}

	/**
	 * 
	 * @return
	 */
	public BigInteger getIntegerPart() {
		return numer.divide(denom);
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lang.algebra.Addable#add(java.lang.Object)
	 */
	public Rational add(Rational n) {
		BigInteger nd = denom.multiply(n.denom);
		BigInteger nn = numer.multiply(n.denom).add(
				n.numer.multiply(denom));

		return Rational.valueOf(nn, nd);
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lang.algebra.Dividable#divide(java.lang.Object)
	 */
	public Rational divide(Rational n) {
		BigInteger nd = denom.multiply(n.numer);
		BigInteger nn = numer.multiply(n.denom);

		return Rational.valueOf(nn, nd);
	}

	/**
	 * 
	 * @param n
	 * @return
	 */
	public boolean isEqualTo(Rational n) {
		return (numer.equals(n.numer) && denom.equals(n.denom));
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lang.algebra.Multipliable#multiply(java.lang.Object)
	 */
	public Rational multiply(Rational n) {
		BigInteger nd = denom.multiply(n.denom);
		BigInteger nn = numer.multiply(n.numer);

		return Rational.valueOf(nn, nd);
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lang.algebra.Subtractable#subtract(java.lang.Object)
	 */
	public Rational subtract(Rational n) {
		BigInteger nd = denom.multiply(n.denom);
		BigInteger nn = numer.multiply(n.denom).subtract(
				n.numer.multiply(denom));

		return Rational.valueOf(nn, nd);
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lang.algebra.Negatable#negate()
	 */
	public Rational negate() {
		return new Rational(numer.negate(), denom);
	}

	/**
	 * 
	 * @return
	 */
	public int signum() {
		return numer.signum();
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lang.algebra.UnitaryRingElement#isUnit()
	 */
	public boolean isUnit() {
		return numer.equals(BigInteger.ONE) && denom.equals(BigInteger.ONE);
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lang.algebra.RingElement#isZero()
	 */
	public boolean isZero() {
		return isEqualTo(ZERO);
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lang.algebra.Addable#multiply(int)
	 */
	public Rational multiply(int n) {
		BigInteger nn = numer.multiply(BigInteger.valueOf(n));

		return Rational.valueOf(nn, denom);
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lang.algebra.Multipliable#power(int)
	 */
	public Rational power(int n) {
		BigInteger nd = denom.pow(n);
		BigInteger nn = numer.pow(n);

		return Rational.valueOf(nn, nd);
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lang.algebra.Calculatable#invert()
	 */
	public Rational invert() {
		return Rational.valueOf(denom, numer);
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lang.number.NumericalRingElement#doubleValue()
	 */
	public double doubleValue() {
		return numer.doubleValue() / denom.doubleValue();
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lang.number.NumericalRingElement#floatValue()
	 */
	public float floatValue() {
		return numer.floatValue() / denom.floatValue();
	}

	/* (non-Javadoc)
	 * @see net.morilib.lang.number.NumericalRingElement#castInt()
	 */
	public int castInt() {
		return castBigInteger().intValue();
	}

	/* (non-Javadoc)
	 * @see net.morilib.lang.number.NumericalRingElement#castLong()
	 */
	public long castLong() {
		return castBigInteger().longValue();
	}

	/* (non-Javadoc)
	 * @see net.morilib.lang.number.NumericalRingElement#castBigInteger()
	 */
	public BigInteger castBigInteger() {
		return numer.divide(denom);
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lang.number.NumericalRingElement#getBigIntegerFloor()
	 */
	public BigInteger getBigIntegerFloor() {
		BigInteger[] r0 = numer.divideAndRemainder(denom);

		if(r0[1].equals(BigInteger.ZERO)) {
			return r0[0];
		} else if((numer.signum() > 0) ^ (denom.signum() > 0)) {
			return r0[0];
		} else {
			return r0[0].subtract(BigInteger.ONE);
		}
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lang.number.NumericalRingElement#intFloor()
	 */
	public int intFloor() {
		return getBigIntegerFloor().intValue();
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lang.number.NumericalRingElement#longFloor()
	 */
	public long longFloor() {
		return getBigIntegerFloor().longValue();
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lang.number.NumericalRingElement#getBigIntegerCeil()
	 */
	public BigInteger getBigIntegerCeil() {
		BigInteger[] r0 = numer.divideAndRemainder(denom);

		if(r0[1].equals(BigInteger.ZERO)) {
			return r0[0];
		} else if((numer.signum() > 0) ^ (denom.signum() > 0)) {
			return r0[0].add(BigInteger.ONE);
		} else {
			return r0[0];
		}
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lang.number.NumericalRingElement#intCeil()
	 */
	public int intCeil() {
		return getBigIntegerCeil().intValue();
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lang.number.NumericalRingElement#longCeil()
	 */
	public long longCeil() {
		return getBigIntegerCeil().longValue();
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lang.number.NumericalRingElement#isInteger()
	 */
	public boolean isInteger() {
		return denom.equals(BigInteger.ONE);
	}

	/* (non-Javadoc)
	 * @see net.morilib.lang.number.NumericalRingElement#getRational()
	 */
	public Rational getRational() {
		return this;
	}

	/**
	 * 
	 * @param digit
	 * @return
	 */
	public int getScale(int digit) {
		BigInteger n = numer.abs(), d = n.remainder(denom), x;
		Set<BigInteger> c = new HashSet<BigInteger>();
		int r;

		c.add(d);
		x = BigInteger.valueOf(digit);
		for(r = 0; r < 1000; r++) {
			d = d.multiply(x).remainder(denom);
			if(d.signum() == 0) {
				return r;
			} else if(c.contains(d)) {
				return Integer.MAX_VALUE;
			} else {
				c.add(d);
			}
		}
		return r;
	}

	/**
	 * 
	 * @return
	 */
	public int getScale() {
		return getScale(10);
	}

	/* (non-Javadoc)
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	public int compareTo(Rational o) {
		BigInteger c = numer.multiply(o.denom);
		BigInteger d = o.numer.multiply(denom);

		return c.compareTo(d);
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(Object x) {
		if(x instanceof Rational) {
			Rational n = (Rational)x;

			return (numer.equals(n.numer) && denom.equals(n.denom));
		}
		return false;
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	public int hashCode() {
		int l = 17;

		l = 37 * l + numer.hashCode();
		l = 37 * l + denom.hashCode();
		return l;
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		if(numer.signum() == 0) {
			return "0";
		} else if(denom.equals(BigInteger.ONE)) {
			return numer.toString();
		} else {
			return numer.toString() + "/" + denom.toString();
		}
	}

}
