unit UFrmNsmMainDebug;
(* NSM SYSTEM CEBhEifobOj *)

interface

uses
  Windows, Messages, SysUtils, Classes, APIUtils, MinApp, APIWindow,
  UFrmNsmMain, UNsmCom, UNsmPlugin, UNsmSystem, UNsmConsts;

type
  TFrmNsmMainDebug = class(TFrmNsmMain)
  private
    FViewMode: Integer;
    FLog: TStringList;
    FUserLog: TStringList;
    FFont: HFONT;
    FColmunCount: Integer;
    FColmuns: array[0..16] of String;
    FColmunWidth: array[0..16] of Integer;

    procedure AddColumn(ACaption: String; AWidth: Integer);
    procedure DrawCell(X, Y: Integer; Text: String);
    function GetCellRect(X, Y: Integer): TRect;
    function GetScrollBarMax(ItemCount: Integer): Integer;

    procedure MakeMenu;
    procedure UpdateList;

    procedure UpdateServiceList;
    procedure UpdateEventList;
    procedure UpdateModuleList;
    procedure UpdateConnectionList;
    procedure UpdateSessionList;
    procedure UpdateLog;
    procedure UpdateUserLog;

    procedure NsmComLog(LogStr: String; User: Boolean);
    procedure NsmComServiceListChange(Sender: TObject);
    procedure NsmComEventListChange(Sender: TObject);
    procedure NsmSystemConnectionChange(Sender: TObject);
    procedure NsmSystemSessionChange(Sender: TObject);
    procedure ON_WM_ERASEBKGND(var Msg: TWMEraseBkgnd); message WM_ERASEBKGND;
    procedure ON_WM_VSCROLL(var Msg: TWMVScroll); message WM_VSCROLL;
    procedure ON_WM_COMMAND(var Msg: TWMCommand); message WM_COMMAND;
  protected
    procedure Paint(var Msg: TWmPaint); override;
  public
    constructor Create(hParent: HWND); override;
    destructor Destroy; override;
  end;

var
  FrmNsmMainDebug: TFrmNsmMainDebug;

implementation

const
  CellHeight = 16;

// -----------------------------------------------------------------------------

constructor TFrmNsmMainDebug.Create(hParent: HWND);
begin
  FLog := TStringList.Create;
  FUserLog := TStringList.Create;
  with NsmCom do
  begin
    OnLog := NsmComLog;
    OnServiceListChange := NsmComServiceListChange;
    OnEventListChange := NsmComEventListChange;
  end;
  inherited;
  with NsmSystem do
  begin
    OnConnectionChange := NsmSystemConnectionChange;
    OnSessionChange := NsmSystemSessionChange;
  end;

  FFont := CreateFont(
      13,                           // tHg̍
      0,                            // tHg̉H
      0,                            // ̕px
      0,                            // x[XC̊px
      FW_NORMAL,                    // tHg̑
      0,                            // Α̂ǂ
      0,                            // ̃^Cv
      0,                            // ł
      SHIFTJIS_CHARSET,             // tHg̕Zbg
      OUT_DEFAULT_PRECIS,           // o͐x
      CLIP_DEFAULT_PRECIS,          // NbsO
      ANTIALIASED_QUALITY,          // o͕i
      VARIABLE_PITCH or FF_SCRIPT,  // tHg̃sb`ƃt@~
      'MS UI Gothic'                // tHg̎
  );

  Icon := LoadIcon(hInstance, 'MAINICON');
  Caption := NMM_SYSTEM + '/Debug';
  FViewMode := 0;
  Color := clrWhite;
  MakeMenu;
  Width := 640;
  Height := 480;
  Center;
  ShowScrollBar(Handle, SB_VERT, True);
end;

destructor TFrmNsmMainDebug.Destroy;
begin
  with NsmCom do
  begin
    OnLog := nil;
    OnServiceListChange := nil;
    OnEventListChange := nil;
  end;
  with NsmSystem do
  begin
    OnConnectionChange := nil;
    OnSessionChange := nil;
  end;
  FLog.Free;
  FUserLog.Free;
  DeleteObject(FFont);
  inherited;
end;

procedure TFrmNsmMainDebug.Paint(var Msg: TWmPaint);
var
  ps: TPaintStruct;
begin
  BeginPaint(Handle, ps);
  UpdateList;
  EndPaint(Handle, ps);
end;

procedure TFrmNsmMainDebug.ON_WM_ERASEBKGND(var Msg: TWMEraseBkgnd);
var
  R: TRect;
  BH: HBRUSH;
begin
  GetClientRect(Handle, R);
  R.Bottom := CellHeight;
  BH := CreateSolidBrush(clrBtnFace);
  FillRect(Msg.DC, R, BH);
  DeleteObject(BH);

  GetClientRect(Handle, R);
  R.Top := CellHeight;
  BH := CreateSolidBrush(Self.Color);
  FillRect(Msg.DC, R, BH);
  DeleteObject(BH);
  
  Msg.Result := 1;
end;

procedure TFrmNsmMainDebug.ON_WM_COMMAND(var Msg: TWMCommand);
begin
  inherited;
  if Msg.NotifyCode = 0 then // Menu=0, Accel=1, NotifyCode = Control
  begin
    FViewMode := Msg.ItemID;
    SetScrollPos(Handle, SB_VERT, 0, False);
    SetScrollRange(Handle, SB_VERT, 0, 0, False);
    InvalidateRect(Handle, nil, True);
  end;
end;

procedure TFrmNsmMainDebug.ON_WM_VSCROLL(var Msg: TWMVScroll);
var
  nPos: Integer;
begin
  inherited;
  nPos := GetScrollPos(Handle, SB_VERT);
  case Msg.ScrollCode of
    SB_LINEDOWN:      SetScrollPos(Handle, SB_VERT, nPos + 2, True);
    SB_LINEUP:        SetScrollPos(Handle, SB_VERT, nPos - 2, True);
    SB_PAGEDOWN:      SetScrollPos(Handle, SB_VERT, nPos + 20, True);
    SB_PAGEUP:        SetScrollPos(Handle, SB_VERT, nPos - 20, True);
    SB_THUMBPOSITION,
    SB_THUMBTRACK:    SetScrollPos(Handle, SB_VERT, Msg.Pos, True);
  end;
  InvalidateRect(Handle, nil, True);
end;

procedure TFrmNsmMainDebug.UpdateList;
begin
  inherited;
  case FViewMode of
    0: UpdateModuleList;
    1: UpdateServiceList;
    2: UpdateEventList;
    3: UpdateConnectionList;
    4: UpdateSessionList;
    5: UpdateLog;
    6: UpdateUserLog;
  end;
end;

procedure TFrmNsmMainDebug.MakeMenu;
var
  hM: HMENU;
begin
  hM := CreateMenu;
  AppendMenu(hM, MF_STRING, 0, '&Modules');
  AppendMenu(hM, MF_STRING, 1, '&Services');
  AppendMenu(hM, MF_STRING, 2, '&Events');
  AppendMenu(hM, MF_STRING, 3, '&Connections');
  AppendMenu(hM, MF_STRING, 4, '&Sessions');
  AppendMenu(hM, MF_STRING, 5, '&Log');
  AppendMenu(hM, MF_STRING, 6, '&Debug');
  SetMenu(Handle, hM);
end;

// -----------------------------------------------------------------------------

procedure TFrmNsmMainDebug.AddColumn(ACaption: String; AWidth: Integer);
begin
  FColmuns[FColmunCount] := ACaption;
  FColmunWidth[FColmunCount] := AWidth;
  DrawCell(FColmunCount, 0, ACaption);
  Inc(FColmunCount);
end;

procedure TFrmNsmMainDebug.DrawCell(X, Y: Integer; Text: String);
var
  DCHandle: HDC;
  BrushHandle: HBRUSH;
  R: TRect;
  OldFont: HFONT;
begin
  R := GetCellRect(X, Y);
  if ((Y = 0) or (R.Bottom > CellHeight)) and (R.Top < ClientHeight) then
  begin
    DCHandle := GetDC(Handle);
    if Y = 0 then
    begin
      if X > 0 then
        Dec(R.Left);
      SetBkMode(DCHandle, TRANSPARENT);
      BrushHandle := CreateSolidBrush(clrBtnShadow);
      FrameRect(DCHandle, R, BrushHandle);
      DeleteObject(BrushHandle);
    end;

    OldFont := HFONT(SelectObject(DCHandle, FFont));

    InflateRect(R, -1, -1);
    DrawText(DCHandle, PChar(Text), -1, R,
        DT_LEFT or DT_EXPANDTABS or DT_SINGLELINE or DT_VCENTER);

    SelectObject(DCHandle, OldFont);
    ReleaseDC(Handle, DCHandle);
  end;
end;

function TFrmNsmMainDebug.GetCellRect(X, Y: Integer): TRect;
var
  I, T: Integer;
begin
  Result := Rect(0, 0, 0, 0);
  for I := 0 to X - 1 do
    Inc(Result.Left, FColmunWidth[I]);
  Result.Right := Result.Left + FColmunWidth[X];
  if (Y > 0) then
  begin
    T := GetScrollPos(Handle, SB_VERT);
    Result.Top := CellHeight * (Y - T);
  end else
    Result.Top := CellHeight * Y;
  Result.Bottom := Result.Top + CellHeight;
end;

function TFrmNsmMainDebug.GetScrollBarMax(ItemCount: Integer): Integer;
begin
  Result := ItemCount - (ClientHeight div CellHeight) + 1;
  if Result < 0 then
    Result := 0;
end;

procedure TFrmNsmMainDebug.UpdateServiceList;
var
  I: Integer;
begin
  FColmunCount := 0;
  with NsmCom.ServiceList.LockList do
    try
      SetScrollRange(Handle, SB_VERT, 0, GetScrollBarMax(Count), False);
      AddColumn('ServiceName',  350);
      AddColumn('ProcAddr',  100);
      for I := 0 to Count - 1 do
      begin
        DrawCell(0, I + 1, Service[I].Name);
        DrawCell(1, I + 1, IntToHex(Cardinal(Addr(Service[I].ServiceProc)), 8));
      end;
    finally
      NsmCom.ServiceList.UnlockList;
    end;
end;

procedure TFrmNsmMainDebug.UpdateEventList;
var
  I: Integer;
begin
  FColmunCount := 0;
  with NsmCom.EventList.LockList do
    try
      SetScrollRange(Handle, SB_VERT, 0, GetScrollBarMax(Count), False);
      AddColumn('EventName',  350);
      AddColumn('HookCount',  100);
      for I := 0 to Count - 1 do
      begin
        DrawCell(0, I + 1, Event[I].Name);
        DrawCell(1, I + 1, IntToStr(Event[I].HookProcs.Count));
      end;
    finally
      NsmCom.EventList.UnlockList;
    end;
end;

procedure TFrmNsmMainDebug.UpdateModuleList;
var
  I: Integer;
begin
  FColmunCount := 0;
  SetScrollRange(Handle, SB_VERT, 0, GetScrollBarMax(NsmSystem.Plugins.Count + 1), False);
  AddColumn('FileName',  140);
  AddColumn('ModuleName',  200);
  AddColumn('API Ver',  60);

  DrawCell(0, 1, ExtractFileName(Application.ExeName));
  DrawCell(1, 1, NMM_SYSTEM);
  DrawCell(2, 1, NSM_API_VERSION);

  for I := 0 to NsmSystem.Plugins.Count - 1 do
  begin
    DrawCell(0, I + 2, ExtractFileName(NsmSystem.Plugins[I].FileName));
    DrawCell(1, I + 2, NsmSystem.Plugins[I].PluginInfo.ModuleName);
    DrawCell(2, I + 2, NsmSystem.Plugins[I].PluginInfo.ApiVersion);
  end;
end;

procedure TFrmNsmMainDebug.UpdateConnectionList;
var
  I: Integer;
begin
  FColmunCount := 0;
  with NsmSystem.Connections.LockList do
    try
      SetScrollRange(Handle, SB_VERT, 0, GetScrollBarMax(Count), False);
      AddColumn('Handle',  100);
      AddColumn('Protocol',  100);
      AddColumn('Caption',  200);
      AddColumn('Status',  60);
      for I := 0 to Count - 1 do
      begin
        DrawCell(0, I + 1, IntToHex(Cardinal(Connection[I]), 8));
        DrawCell(1, I + 1, Connection[I].Protocol);
        DrawCell(2, I + 1, Connection[I].Caption);
        DrawCell(3, I + 1, IntToStr(Connection[I].Status));
      end;
    finally
      NsmSystem.Connections.UnlockList;
    end;
end;

procedure TFrmNsmMainDebug.UpdateSessionList;
var
  I: Integer;
begin
  FColmunCount := 0;
  with NsmSystem.Sessions.LockList do
    try
      SetScrollRange(Handle, SB_VERT, 0, GetScrollBarMax(Count), False);
      AddColumn('Handle',  100);
      AddColumn('Protocol',  100);
      AddColumn('Caption',  200);
      AddColumn('Status',  60);
      for I := 0 to Count - 1 do
      begin
        DrawCell(0, I + 1, IntToHex(Cardinal(Session[I]), 8));
        DrawCell(1, I + 1, Session[I].Protocol);
        DrawCell(2, I + 1, Session[I].Caption);
        DrawCell(3, I + 1, IntToStr(Session[I].Status));
      end;
    finally
      NsmSystem.Sessions.UnlockList;
    end;
end;

procedure TFrmNsmMainDebug.UpdateLog;
var
  I: Integer;
begin
  SetScrollRange(Handle, SB_VERT, 0, GetScrollBarMax(FLog.Count), False);
  FColmunCount := 0;
  AddColumn('LogStr',  ClientWidth);
  for I := 0 to FLog.Count - 1 do
  begin
    DrawCell(0, I + 1, FLog[I]);
  end;
end;

procedure TFrmNsmMainDebug.UpdateUserLog;
var
  I: Integer;
begin
  SetScrollRange(Handle, SB_VERT, 0, GetScrollBarMax(FUserLog.Count), False);
  FColmunCount := 0;
  AddColumn('LogStr',  ClientWidth);
  for I := 0 to FUserLog.Count - 1 do
  begin
    DrawCell(0, I + 1, FUserLog[I]);
  end;
end;

// -----------------------------------------------------------------------------

procedure TFrmNsmMainDebug.NsmComServiceListChange(Sender: TObject);
begin
  if FViewMode = 1 then
    InvalidateRect(Handle, nil, True);
end;

procedure TFrmNsmMainDebug.NsmComEventListChange(Sender: TObject);
begin
  if FViewMode = 2 then
    InvalidateRect(Handle, nil, True);
end;

procedure TFrmNsmMainDebug.NsmSystemConnectionChange(Sender: TObject);
begin
  if FViewMode = 3 then
    InvalidateRect(Handle, nil, True);
end;

procedure TFrmNsmMainDebug.NsmSystemSessionChange(Sender: TObject);
begin
  if FViewMode = 4 then
    InvalidateRect(Handle, nil, True);
end;

procedure TFrmNsmMainDebug.NsmComLog(LogStr: String; User: Boolean);
begin
  if not User then
  begin
    FLog.Add(TimeToStr(Now) + ' ' + LogStr);
    if FViewMode = 5 then
      InvalidateRect(Handle, nil, True);
  end else
  begin
    FUserLog.Add(LogStr);
    if FViewMode = 6 then
      InvalidateRect(Handle, nil, True);
  end;
end;

end.
