unit UWideGraphics;

interface

uses
  Windows, Graphics, UWideUtils;

procedure WideTextOut(Canvas: TCanvas; X, Y: Integer; const Text: WideString);
procedure WideTextRect(Canvas: TCanvas; Rect: TRect; X, Y: Integer; const Text: WideString);
procedure WideDrawText(Canvas: TCanvas; const Text: WideString; var Rect: TRect; Format: Cardinal);
function WideTextExtent(Canvas: TCanvas; const Text: WideString): TSize;
function WideTextWidth(Canvas: TCanvas; const Text: WideString): Integer;
function WideTextHeight(Canvas: TCanvas; const Text: WideString): Integer;

implementation

procedure WideTextOut(Canvas: TCanvas; X, Y: Integer; const Text: WideString);
begin
  ExtTextOutW(Canvas.Handle, X, Y, Canvas.TextFlags, nil,
      PWideChar(Text), Length(Text), nil);
  Canvas.MoveTo(X + WideTextWidth(Canvas, Text), Y);
end;

procedure WideTextRect(Canvas: TCanvas; Rect: TRect; X, Y: Integer; const Text: WideString);
var
  Flags: Integer;
begin
  Flags := ETO_CLIPPED or Canvas.TextFlags;
  ExtTextOutW(Canvas.Handle, X, Y, Flags, @Rect, PWideChar(Text),
      Length(Text), nil);
end;

procedure DrawMultiLine(Canvas: TCanvas; Text: PWideChar; var Rect: TRect; Format: Cardinal);
var
  Head, Tail : PWideChar;
  Flags, Y, LineHeight: Integer;
  CalcRect: Boolean;
begin
  Flags := ETO_CLIPPED or Canvas.TextFlags;
  Head := Text;
  Y := Rect.Top;
  LineHeight := WideTextHeight(Canvas, Text);
  CalcRect := (Format and DT_CALCRECT) <> 0;

  while (Head^ <> WideNull) and (Rect.Top < Rect.Bottom) do
  begin
    Tail := Head;
    while not (Tail^ in [WideNull, WideCR, WideLF]) and (Tail^ <> WideLS) do
      Inc(Tail);

    if not CalcRect then
      ExtTextOutW(Canvas.Handle, Rect.Left, Y, Flags, @Rect,
          Head, Tail - Head, nil);
    if (Tail^ = WideCR) or (Tail^ = WideLS) then
      Inc(Tail);
    if Tail^ = WideLF then
      Inc(Tail);
    Inc(Y, LineHeight);
    Head:=Tail;
  end;
  if CalcRect then
  begin
    Rect.Bottom := Y;
  end;
end;

procedure DrawSingleLine(Canvas: TCanvas; Text: PWideChar; var Rect: TRect; Format: Cardinal);
var
  Flags, X, Y, RectWidth, RectHeight, EllipsisWidth: Integer;
  Extent: TSize;
  DisplayText: WideString;
begin
  Flags := Canvas.TextFlags;
  DisplayText := Text;
  RectWidth := (Rect.Right - Rect.Left);
  RectHeight := (Rect.Bottom - Rect.Top);
  Extent := WideTextExtent(Canvas, DisplayText);

  if (Format and DT_CALCRECT) <> 0 then
  begin
    Rect.Right := Rect.Left + Extent.cx;
    Rect.Bottom := Rect.Top + Extent.cy;
    Exit;
  end;

  if (Format and DT_NOCLIP) <> 0 then
    Flags := (not ETO_CLIPPED) and Flags
  else
    Flags := ETO_CLIPPED or Flags;

  if ((Format and DT_END_ELLIPSIS) <> 0) and (Extent.cx > RectWidth) then
  begin
    EllipsisWidth := WideTextWidth(Canvas, '...');
    while (Extent.cx + EllipsisWidth > RectWidth) and
          (Length(DisplayText) > 1) do
    begin
      SetLength(DisplayText, Length(DisplayText) - 1);
      Extent := WideTextExtent(Canvas, DisplayText);
    end;
    DisplayText := DisplayText + '...';
  end;

  if (Format and DT_CENTER) <> 0 then
    X := Rect.Left + (RectWidth - Extent.cx) div 2
  else if (Format and DT_RIGHT) <> 0 then
    X := Rect.Right - Extent.cx
  else
    X := Rect.Left;
  if (Format and DT_VCENTER) <> 0 then
    Y := Rect.Top + (RectHeight - Extent.cy) div 2
  else if (Format and DT_BOTTOM) <> 0 then
    Y := Rect.Bottom - Extent.cy
  else
    Y := Rect.Top;
  ExtTextOutW(Canvas.Handle, X, Y, Flags, @Rect, PWideChar(DisplayText),
      Length(DisplayText), nil);
end;

procedure WideDrawText(Canvas: TCanvas; const Text: WideString; var Rect: TRect; Format: Cardinal);
begin
  if (Format and DT_WORDBREAK) <> 0 then
    DrawMultiLine(Canvas, PWideChar(Text), Rect, Format)
  else
    DrawSingleLine(Canvas, PWideChar(Text), Rect, Format);
end;

function WideTextExtent(Canvas: TCanvas; const Text: WideString): TSize;
begin
  Result.cx := 0;
  Result.cy := 0;
  Windows.GetTextExtentPoint32W(Canvas.Handle, PWideChar(Text),
      Length(Text), Result);
end;

function WideTextWidth(Canvas: TCanvas; const Text: WideString): Integer;
begin
  Result := WideTextExtent(Canvas, Text).cX;
end;

function WideTextHeight(Canvas: TCanvas; const Text: WideString): Integer;
begin
  Result := WideTextExtent(Canvas, Text).cY;
end;

end.
