unit UFrmNotify;

interface

uses
  Windows, Messages, SysUtils, Classes, Controls, Forms, UFrmBase, ActnList,
  ExtCtrls, StdCtrls, Graphics, USkinResource, ExSpeedButton, ExPanel,
  Menus, UStrResource, ResizeArea, UNsmTypes, USimpleUICore, UConfig;

type
  TNotifyItems = class;

  TNotifyItem = class(TObject)
  private
    FOwner: TNotifyItems;
    FText: String;
    FLinkTo: String;
    FBitmap: TBitmap;
    FTimeToLive: Integer;
    FOnClick: TNotifyClickCallback;
    FData: Integer;
    procedure SetText(Value: String);
    procedure SetBitmap(Value: TBitmap);
    procedure DoClick;
  public
    constructor Create(AOwner: TNotifyItems);
    destructor Destroy; override;
    function DecTimeToLive(N: Integer = 1): Integer;
    property Text: String read FText write SetText;
    property LinkTo: String read FLinkTo write FLinkTo;
    property Bitmap: TBitmap read FBitmap write SetBitmap;
    property TimeToLive: Integer read FTimeToLive write FTimeToLive;
    property OnClick: TNotifyClickCallback read FOnClick write FOnClick;
    property Data: Integer read FData write FData;
  end;

  TNotifyItems = class(TObject)
  private
    FItems: TList;
    FOnChange: TNotifyEvent;
    FUpdating: Integer;
    function GetItem(Index: Integer): TNotifyItem;
    function GetCount: Integer;
    procedure DoChange;
  public
    constructor Create;
    destructor Destroy; override;
    function Add(AText: String; ATimtToLive: Integer; ABitmap: TBitmap): TNotifyItem;
    procedure Delete(Idx: Integer);
    procedure Remove(Item: TNotifyItem);
    procedure Clear;
    function IndexOf(Item: TNotifyItem): Integer;
    procedure BeginUpdate;
    procedure EndUpdate;

    property Items[Index: Integer]: TNotifyItem read GetItem; default;
    property Count: Integer read GetCount;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
  end;

  TFrmNotify = class(TFrmBase)
    Timer: TTimer;
    procedure FormCreate(Sender: TObject);
    procedure pbNotifyPaint(Sender: TObject);
    procedure pbNotifyMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure TimerTimer(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
  private
    FNotifyList: TNotifyItems;
    FNotifySpaceColor: TColor;
    FDummyHandle: hWnd;
    function CalcRect(Index: Integer): TRect;
    procedure NotifyListChange(Sender: TObject);
    function GetItemAt(X, Y: Integer): TNotifyItem;
  protected
    procedure UpdateSkinObjects; override;
    procedure CreateParams(var Params: TCreateParams); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure ApplyConfig; override;
    property  NotifyList: TNotifyItems read FNotifyList;
    property NotifySpaceColor: TColor read FNotifySpaceColor write FNotifySpaceColor;
  end;

var
  FrmNotify: TFrmNotify;

implementation

{$R *.dfm}

// -----------------------------------------------------------------------------

constructor TNotifyItem.Create(AOwner: TNotifyItems);
begin
  FOwner := AOwner;
  FTimeToLive := 10;
  FBitmap := TBitmap.Create;
  FBitmap.TransparentMode := tmAuto;
  FBitmap.Transparent := True;
end;

destructor TNotifyItem.Destroy;
begin
  FBitmap.Free;
end;

procedure TNotifyItem.SetText(Value: String);
begin
  FText := Value;
  FOwner.DoChange;
end;

function TNotifyItem.DecTimeToLive(N: Integer = 1): Integer;
begin
  Dec(FTimeToLive, N);
  Result := FTimeToLive;
end;

procedure TNotifyItem.SetBitmap(Value: TBitmap);
begin
  FBitmap.Assign(Value);
  FOwner.DoChange;
end;

procedure TNotifyItem.DoClick;
begin
  if Length(FLinkTo) > 0 then
    SimpleUICore.OpenURL(FLinkTo);
  if Assigned(FOnClick) then
    FOnClick(FData);
end;

// -----------------------------------------------------------------------------

constructor TNotifyItems.Create;
begin
  FItems := TList.Create;
  FUpdating := 0;
end;

destructor TNotifyItems.Destroy;
begin
  FOnChange := nil;
  Clear;
  FItems.Free;
end;

function TNotifyItems.GetItem(Index: Integer): TNotifyItem;
begin
  Result := TNotifyItem(FItems[Index]);
end;

function TNotifyItems.GetCount: Integer;
begin
  Result := FItems.Count;
end;

procedure TNotifyItems.DoChange;
begin
  if (FUpdating = 0) and Assigned(FOnChange) then
    FOnChange(Self);
end;

// ǉ
function TNotifyItems.Add(AText: String; ATimtToLive: Integer; ABitmap: TBitmap): TNotifyItem;
begin
  BeginUpdate;
  try
    Result := TNotifyItem.Create(Self);
    FItems.Add(Result);
    Result.Text := AText;
    Result.TimeToLive := ATimtToLive;
    Result.Bitmap.Assign(ABitmap);
  finally
    EndUpdate;
  end;
end;

// 폜
procedure TNotifyItems.Delete(Idx: Integer);
begin
  TNotifyItem(FItems[Idx]).Free;
  FItems.Delete(Idx);
  DoChange;
end;

procedure TNotifyItems.Remove(Item: TNotifyItem);
var
  Idx: Integer;
begin
  Idx := IndexOf(Item);
  if Idx > -1 then
    Delete(Idx);
end;

// S
procedure TNotifyItems.Clear;
begin
  BeginUpdate;
  try
    while FItems.Count > 0 do
      Delete(0)
  finally
    EndUpdate;
  end;
  DoChange;
end;

procedure TNotifyItems.BeginUpdate;
begin
  Inc(FUpdating);
end;

procedure TNotifyItems.EndUpdate;
begin
  Dec(FUpdating);
  DoChange;
end;

function TNotifyItems.IndexOf(Item: TNotifyItem): Integer;
var
  I: Integer;
begin
  Result := -1;
  for I := 0 to FItems.Count - 1 do
    if FItems[I] = Item then
    begin
      Result := I;
      Break;
    end;
end;
// -----------------------------------------------------------------------------

constructor TFrmNotify.Create(AOwner: TComponent);
begin
  FDummyHandle := Classes.AllocateHWnd(nil);
  inherited;
end;

destructor TFrmNotify.Destroy;
begin
  inherited;
  Classes.DeallocateHWnd(FDummyHandle);
end;

procedure TFrmNotify.CreateParams(var Params: TCreateParams);
begin
  inherited;
  // _~[eɂă^XNo[B
  // FrmMain eɂ FrmMain ŏ FrmNotify ȂǕssB
  Params.WndParent := FDummyHandle;
end;

procedure TFrmNotify.ApplyConfig;
begin
  inherited;
  if Config.NotifyTransparency > 0 then
  begin
    AlphaBlend := True;
    AlphaBlendValue := (100 - Config.NotifyTransparency) * 255 div 100;
  end else
    AlphaBlend := False;
end;

procedure TFrmNotify.UpdateSkinObjects;
begin
  inherited;
  FNotifySpaceColor := WindowStyle.BackgroundColor;
//  pbNotify.Invalidate;
end;

function TFrmNotify.CalcRect(Index: Integer): TRect;
var
  I, H: Integer;
  R: TRect;
begin
  if (Index < 0) or (Index >= FNotifyList.Count) then
  begin
    Result := Rect(0, 0, 0, 0);
    Exit;
  end;

  H := 0;
  with pnlBack.Canvas do
  begin
    Font.Assign(pnlBack.Font);
    for I := 0 to Index do
    begin
      if I > 0 then
        Inc(H, 4);
      R := Rect(10, H + 10, pnlBack.ClientWidth, H + 10);
      Result := Rect(0, H, pnlBack.ClientWidth, H);
      if not FNotifyList[I].Bitmap.Empty then
        R.Left := FNotifyList[I].Bitmap.Width + 4;
      Inc(H, 20);
      Inc(Result.Bottom, 20);
      DrawText(Handle, PChar(FNotifyList[I].Text), -1, R, DT_CALCRECT or DT_WORDBREAK);
      if not FNotifyList[I].Bitmap.Empty and
         (R.Bottom - R.Top < FNotifyList[I].Bitmap.Height) then
      begin
        Inc(H, FNotifyList[I].Bitmap.Height);
        Inc(Result.Bottom, FNotifyList[I].Bitmap.Height);
      end else
      begin
        Inc(H, R.Bottom - R.Top);
        Inc(Result.Bottom, R.Bottom - R.Top);
      end;
    end;
  end;
end;

procedure TFrmNotify.NotifyListChange(Sender: TObject);
var
  RDesktop, RTaskBar : TRect;
  HTaskBar: HWND;
begin
  if FNotifyList.Count = 0 then
  begin
    Close;
    Exit;
  end;

  pnlBack.Invalidate;
  ClientHeight := pnlBack.Top + CalcRect(FNotifyList.Count - 1).Bottom + 5;

  HTaskBar := FindWindowEx(0, 0, 'Shell_TrayWnd', nil);
  GetWindowRect(HTaskBar, RTaskBar);
  SystemParametersInfo(SPI_GETWORKAREA, 0, @RDesktop, 0);

  if (RTaskBar.Top > 0) and (RTaskBar.Bottom > Screen.Height) then
  begin
    Left := RDesktop.Right - Self.Width - 10;
    Top := RDesktop.Bottom - Self.Height;
  end else
  if (RTaskBar.Top < 0) and (RTaskBar.Bottom < Screen.Height) then
  begin
    Left := RDesktop.Right - Self.Width - 10;
    Top := RDesktop.Top;
  end else
  if (RTaskBar.Top < 0) and (RTaskBar.Right < Screen.Width) then
  begin
    Left := RDesktop.Left + 10;
    Top := RDesktop.Bottom - Self.Height;
  end else
  begin
    Left := RDesktop.Right - Self.Width - 10;
    Top := RDesktop.Bottom - Self.Height;
  end;

  if not Visible then
  begin
    // őOʂɕ\邪ANeBuȂ
    SetWindowPos(Self.Handle, HWND_TOPMOST, 0, 0, 0, 0, SWP_SHOWWINDOW or
        SWP_NOMOVE or SWP_NOSIZE or SWP_NOACTIVATE);
    Visible := True;
  end;
end;

function TFrmNotify.GetItemAt(X, Y: Integer): TNotifyItem;
var
  I: Integer;
  R: TRect;
begin
  Result := nil;
  for I := 0 to FNotifyList.Count - 1 do
  begin
    R := CalcRect(I);
    if (Y > R.Top) and (Y < R.Bottom) then
      Result := FNotifyList[I];
  end;
end;

procedure TFrmNotify.FormCreate(Sender: TObject);
begin
  inherited;
  WindowKind := wkNotify;
  Caption := RS_FORM_NOTIFYTITLE;
  FNotifySpaceColor := WindowStyle.BackgroundColor;
  Sizeable := False;
  WindowFit := False;

  FNotifyList := TNotifyItems.Create;
  FNotifyList.OnChange := NotifyListChange;
end;

procedure TFrmNotify.FormDestroy(Sender: TObject);
begin
  inherited;
  FNotifyList.Free;
  FrmNotify := nil;
end;

procedure TFrmNotify.pbNotifyPaint(Sender: TObject);
var
  I, H: Integer;
  R: TRect;
begin
  inherited;
  H := 0;
  with pnlBack.Canvas do
  begin
    Font.Assign(pnlBack.Font);
    for I := 0 to FNotifyList.Count - 1 do
    begin
      if I > 0 then
      begin
        Brush.Color := FNotifySpaceColor;
        R := pnlBack.ClientRect;
        pnlBack.Style.AdjustClientRect(R);
        R.Top := H;
        R.Bottom := H + 4;
        FillRect(R);
        Inc(H, 4);
      end;
      R := Rect(10, H + 10, pnlBack.ClientWidth - 2, H + 10);
      if not FNotifyList[I].Bitmap.Empty then
      begin
        R.Left := FNotifyList[I].Bitmap.Width + 4;
        Draw(2, R.Top, FNotifyList[I].Bitmap);
      end;
      Inc(H, 20);
      Brush.Color := pnlBack.Color;
      SetBkMode(Handle, Windows.TRANSPARENT);
      DrawText(Handle, PChar(FNotifyList[I].Text), -1, R, DT_CALCRECT or DT_WORDBREAK);
      if not FNotifyList[I].Bitmap.Empty and
         ((R.Bottom - R.Top) < FNotifyList[I].Bitmap.Height) then
      begin
        R.Top := R.Top + (FNotifyList[I].Bitmap.Height - (R.Bottom - R.Top)) div 2;
        R.Bottom := R.Top + FNotifyList[I].Bitmap.Height;
        DrawText(Handle, PChar(FNotifyList[I].Text), -1, R, DT_WORDBREAK);
        Inc(H, FNotifyList[I].Bitmap.Height);
      end else
      begin
        DrawText(Handle, PChar(FNotifyList[I].Text), -1, R, DT_WORDBREAK);
        Inc(H, R.Bottom - R.Top);
      end;
    end;
  end;
end;

procedure TFrmNotify.pbNotifyMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  Item: TNotifyItem;
begin
  inherited;
  Item := GetItemAt(X, Y);
  if Assigned(Item) then
  begin
    Item.DoClick;
    FNotifyList.Remove(Item);
  end;
end;

procedure TFrmNotify.TimerTimer(Sender: TObject);
var
  I: Integer;
begin
  inherited;
  for I := FNotifyList.Count - 1 downto 0 do
  begin
    if FNotifyList[I].DecTimeToLive <= 0 then
      FNotifyList.Delete(I);
  end;
end;

procedure TFrmNotify.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  inherited;
  Action := caFree;
end;

end.
