unit UFrmBaseContainer;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, UFrmBase, Menus, ExtCtrls, ActnList, ExPanel, ExSpeedButton,
  StdCtrls, ExTabSet, UConfig, (*ExLabel,*) ResizeArea, USkinStyles, USkinResource,
  ComCtrls;

const
  DefaultTabSetWidth = 23;

type
  TTabSetStyle = record
    Style: TSkinStyle;
    ActiveItemStyle: TSkinStyle;
    InactiveItemStyle: TSkinStyle;
  end;

  TFrmBaseContainer = class(TFrmBase)
    TabSet: TExTabSet;
    pnlDock: TExPanel;
    pmTab: TPopupMenu;
    mnuTabClose: TMenuItem;
    mnuTabFloat: TMenuItem;
    actTabFloat: TAction;
    actTabClose: TAction;
    actTabNext: TAction;
    mnuTabNext: TMenuItem;
    mnuTabLine01: TMenuItem;
    actTabPrev: TAction;
    mnuTabPrev: TMenuItem;
    mnuTabLine02: TMenuItem;
    actTabPosTop: TAction;
    actTabPosBottom: TAction;
    actTabPosLeft: TAction;
    actTabPosRight: TAction;
    mnuTabPosBottom: TMenuItem;
    mnuTabPosLeft: TMenuItem;
    mnuTabPosRight: TMenuItem;
    mnuTabPosTop: TMenuItem;
    procedure TabSetChange(Sender: TObject);
    procedure TabSetDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure TabSetDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure TabSetMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure actTabFloatExecute(Sender: TObject);
    procedure actTabCloseExecute(Sender: TObject);
    procedure TabSetDblClick(Sender: TObject);
    procedure actTabNextExecute(Sender: TObject);
    procedure TabSetInfoTip(Sender: TObject; Tab: TExTabItem;
      var InfoTip: String);
    procedure actTabPrevExecute(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure actTabPosTopExecute(Sender: TObject);
    procedure actTabPosBottomExecute(Sender: TObject);
    procedure actTabPosLeftExecute(Sender: TObject);
    procedure actTabPosRightExecute(Sender: TObject);
    procedure pmTabPopup(Sender: TObject);
  private
    FTabSetStyle: TTabSetStyle;
    FCaption: WideString;
    procedure LoadTabSetStyle(TabPosition: TTabPosition);
    procedure ApplyTabSetStyle;
    procedure SyncMenuItem(MenuItem: TMenuItem);
  protected
    procedure UpdateSkinObjects; override;
    procedure SetCaption(Value: WideString); override;
    function GetActiveDockedForm: TFrmBase;
    procedure SetActiveDockedForm(Value: TFrmBase); virtual;
    procedure SyncMenuItems(const AItems: array of TMenuItem);
  public
    procedure UpdateActions; override;
    function IsShortCut(var Message: TWMKey): Boolean; override;
    procedure ApplyConfig; override;
    procedure LoadSkinResource; override;
    procedure UpdateCaption; virtual;
    procedure DockWindow(Form: TFrmBase); virtual;
    procedure FloatWindow(Form: TFrmBase); virtual;
    procedure RemoveTab(Form: TFrmBase);
    procedure UpdateTabCaption(Form: TFrmBase);
    function ContainsDockWindow(Form: TFrmBase): Boolean;
    procedure FlashTab(Form: TFrmBase; Invert: Boolean);
    procedure SetTabSetPosition(TabPosition: TTabPosition);
    property ActiveDockedForm: TFrmBase read GetActiveDockedForm write SetActiveDockedForm;
  end;

implementation

{$R *.dfm}

procedure TFrmBaseContainer.UpdateActions;
begin
  inherited;
  actTabFloat.Enabled := (TabSet.Items.Count > 1);
  actTabClose.Enabled := (TabSet.TabIndex > -1);
end;

function TFrmBaseContainer.IsShortCut(var Message: TWMKey): Boolean;
var
  Form: TFrmBase;
begin
  Result := inherited IsShortCut(message);
  if not Result then
  begin
    Form := ActiveDockedForm;
    if Assigned(Form) then
      Result := ActiveDockedForm.IsShortCut(Message);
  end;
end;

procedure TFrmBaseContainer.ApplyConfig;
begin
  inherited;
  TabSet.MaxCaptionLength := Config.MaxTabLength;
  // DoubleBuffered
  TabSet.DoubleBuffered := Config.DoubleBuffered;
  pnlDock.DoubleBuffered := Config.DoubleBuffered;
end;

procedure TFrmBaseContainer.LoadSkinResource;
begin
  inherited;
  LoadTabSetStyle(TabSet.TabPosition);
end;

procedure TFrmBaseContainer.UpdateSkinObjects;
begin
  inherited;
  ApplyTabSetStyle;
end;

procedure TFrmBaseContainer.LoadTabSetStyle(TabPosition: TTabPosition);
var
  ClassName: String;
begin
  case TabPosition of
  tpTop     : ClassName := 'Top';
  tpBottom  : ClassName := 'Bottom';
  tpLeft    : ClassName := 'Left';
  tpRight   : ClassName := 'Right';
  end;
  FTabSetStyle.Style :=
      SkinResources.GetWindowElementStyle(WindowKind, ssActive, seTabSet, ClassName);
  FTabSetStyle.ActiveItemStyle :=
      SkinResources.GetWindowElementStyle(WindowKind, ssActive, seTabButton, ClassName, ssActive);
  FTabSetStyle.InactiveItemStyle :=
      SkinResources.GetWindowElementStyle(WindowKind, ssActive, seTabButton, ClassName, ssInactive);
end;

procedure TFrmBaseContainer.ApplyTabSetStyle;
begin
  TabSet.Style := FTabSetStyle.Style;
  TabSet.ActiveItemStyle := FTabSetStyle.ActiveItemStyle;
  TabSet.InactiveItemStyle := FTabSetStyle.InactiveItemStyle;
end;

procedure TFrmBaseContainer.SetCaption(Value: WideString);
begin
  FCaption := Value;
  UpdateCaption;
end;

procedure TFrmBaseContainer.UpdateCaption;
var
  ACaption, TabCaption: WideString;
begin
  if TabSet.TabIndex > -1 then
  begin
    TabCaption := TFrmBase(TabSet.Items[TabSet.TabIndex].Data).Caption;
    ACaption := WideFormat('%s - %s', [TabCaption, FCaption])
  end else
    ACaption := FCaption;
  inherited SetCaption(ACaption);
end;

procedure TFrmBaseContainer.DockWindow(Form: TFrmBase);
begin
  TabSet.Items.Add(Form.Caption, Form);
  with Form do
  begin
    Hide;
    ManualDock(pnlDock, nil, alClient);
    Align := alClient;
  end;
  if TabSet.TabIndex = -1 then
    SetActiveDockedForm(Form);
  UpdateCaption;
  TabSet.Visible := (TabSet.Items.Count > 1);
end;

procedure TFrmBaseContainer.FloatWindow(Form: TFrmBase);
begin
  RemoveTab(Form);
  Form.Align := alNone;
  Form.ManualFloat(Rect(Left + 30, Top + 30, Left + 30 + Width,
      Top + 30 + Height));
end;

procedure TFrmBaseContainer.RemoveTab(Form: TFrmBase);
var
  Idx: Integer;
begin
  Idx := TabSet.Items.IndexOfData(Form);
  if Idx > -1 then
  begin
    TabSet.Items.Delete(Idx);
    if (TabSet.TabIndex = -1) and (TabSet.Items.Count > 0) then
      TabSet.TabIndex := 0;
    TabSetChange(TabSet);
  end;
  UpdateCaption;
  TabSet.Visible := (TabSet.Items.Count > 1);
end;

procedure TFrmBaseContainer.UpdateTabCaption(Form: TFrmBase);
var
  Idx: Integer;
begin
  Idx := TabSet.Items.IndexOfData(Form);
  if Idx > -1 then
  begin
    TabSet.Items[Idx].Caption := Form.Caption;
  end;
end;

function TFrmBaseContainer.ContainsDockWindow(Form: TFrmBase): Boolean;
begin
  Result := (TabSet.Items.IndexOfData(Form) > - 1);
end;

function TFrmBaseContainer.GetActiveDockedForm: TFrmBase;
begin
  Result := nil;
  if TabSet.TabIndex > - 1 then
    Result := TFrmBase(TabSet.Items[TabSet.TabIndex].Data);
end;

procedure TFrmBaseContainer.SetActiveDockedForm(Value: TFrmBase);
var
  I, Idx: Integer;
  Form: TCustomForm;
begin
  Idx := TabSet.Items.IndexOfData(Value);
  if Idx > -1 then
  begin
    TabSet.TabIndex := Idx;
    Form := TCustomForm(TabSet.Items[TabSet.TabIndex].Data);
    Form.Align := alNone;
    Form.Show;
    Form.Align := alClient;
    if Form is TFrmBase then
      TFrmBase(Form).StopBlink;
  end;
  for I := 0 to TabSet.Items.Count - 1 do
    if I <> TabSet.TabIndex then
      TCustomForm(TabSet.Items[I].Data).Hide;
end;

procedure TFrmBaseContainer.FlashTab(Form: TFrmBase; Invert: Boolean);
var
  Idx: Integer;
begin
  Idx := TabSet.Items.IndexOfData(Form);
  if Idx > -1 then
  begin
    if Invert then
      TabSet.Items[Idx].Emphasis := not TabSet.Items[Idx].Emphasis
    else
      TabSet.Items[Idx].Emphasis := False;
  end;
end;

// DockedForm ɓ̃gbvx MenuItem ꍇɎqړ
procedure TFrmBaseContainer.SyncMenuItems(const AItems: array of TMenuItem);
var
  I: Integer;
begin
  for I := 0 to Length(AItems) - 1 do
    SyncMenuItem(AItems[I]);
end;

procedure TFrmBaseContainer.SyncMenuItem(MenuItem: TMenuItem);
  procedure CopyMenuItem(Src, Dest: TMenuItem);
  begin
    if not Assigned(Src.Action) then
    begin
      Dest.Caption := Src.Caption;
      Dest.ImageIndex := Src.ImageIndex;
      Dest.Tag := Src.Tag;
      Dest.OnClick := Src.OnClick;
      Dest.Enabled := Src.Enabled;
    end else
      Dest.Action := Src.Action;
  end;
  procedure CopyChildMenuItems(Src, Dest: TMenuItem);
  var
    I: Integer;
  begin
    Dest.Clear;
    for I := 0 to Src.Count - 1 do
    begin
      Dest.Add(TMenuItem.Create(Self));
      CopyMenuItem(Src[I], Dest[I]);
      if Src[I].Count > 0 then
        CopyChildMenuItems(Src[I], Dest[I]);
    end;
  end;
var
  Form: TFrmBase;
  FindComp: TComponent;
begin
  Form := ActiveDockedForm;
  if Assigned(Form) then
  begin
    FindComp := Form.FindComponent(MenuItem.Name);
    if Assigned(FindComp) and (FindComp is TMenuItem) then
    begin
      MenuItem.Enabled := True;
      CopyMenuItem(TMenuItem(FindComp), MenuItem);
      CopyChildMenuItems(TMenuItem(FindComp), MenuItem)
    end else
    begin
      MenuItem.Clear;
      MenuItem.Enabled := False;
    end;
  end else
  begin
    MenuItem.Clear;
    MenuItem.Enabled := False;
  end;
end;

procedure TFrmBaseContainer.SetTabSetPosition(TabPosition: TTabPosition);
begin
  LoadTabSetStyle(TabPosition);
  TabSet.TabPosition := TabPosition;
  case TabPosition of
  tpTop     : TabSet.Align := alTop;
  tpBottom  : TabSet.Align := alBottom;
  tpLeft    : TabSet.Align := alLeft;
  tpRight   : TabSet.Align := alRight;
  end;
  case TabPosition of
  tpTop, tpBottom : TabSet.Height := DefaultTabSetWidth;
  tpLeft, tpRight : TabSet.Width := DefaultTabSetWidth;
  end;
  ApplyTabSetStyle;
end;

procedure TFrmBaseContainer.TabSetChange(Sender: TObject);
begin
  inherited;
  if TabSet.TabIndex > -1 then
    SetActiveDockedForm(TFrmBase(TabSet.Items[TabSet.TabIndex].Data))
  else
    SetActiveDockedForm(nil);
end;

procedure TFrmBaseContainer.TabSetDragDrop(Sender, Source: TObject; X,
  Y: Integer);
var
  Dest: Integer;
begin
  inherited;
  if (Source = TabSet) and (TabSet.TabIndex > -1) then
  begin
    Dest := TabSet.ItemAtPos(Point(X, Y));
    if Dest > -1 then
      with TabSet do
      begin
        Items.BeginUpdate;
        Items.Move(TabIndex, Dest);
        TabIndex := Dest;
        Items.EndUpdate;
      end;
  end;
end;

procedure TFrmBaseContainer.TabSetDragOver(Sender, Source: TObject; X,
  Y: Integer; State: TDragState; var Accept: Boolean);
begin
  inherited;
  Accept := (Source = TabSet) and (TabSet.TabIndex > -1) and
            (TabSet.ItemAtPos(Point(X, Y)) > -1);
end;

procedure TFrmBaseContainer.TabSetMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
  P: TPoint;
  TabIdx: Integer;
begin
  inherited;
  TabIdx := TabSet.ItemAtPos(Point(X, Y));
  if (Button = mbLeft) and (TabIdx > -1) then
    TabSet.BeginDrag(False, 10)
  else if (Button = mbRight) then
  begin
    P := TabSet.ClientToScreen(Point(X, Y));
    pmTab.Popup(P.X, P.Y);
  end else if (Button = mbMiddle) and (TabIdx > -1) then
    actTabClose.Execute;
end;

procedure TFrmBaseContainer.TabSetDblClick(Sender: TObject);
begin
  inherited;
  actTabFloat.Execute;
end;

procedure TFrmBaseContainer.actTabFloatExecute(Sender: TObject);
var
  Form: TFrmBase;
begin
  inherited;
  Form := ActiveDockedForm;
  if Assigned(Form) then
    FloatWindow(Form);
end;

procedure TFrmBaseContainer.actTabCloseExecute(Sender: TObject);
var
  Form: TFrmBase;
begin
  inherited;
  Form := ActiveDockedForm;
  if Assigned(Form) then
    Form.Close;
end;

procedure TFrmBaseContainer.actTabNextExecute(Sender: TObject);
var
  Idx: Integer;
begin
  inherited;
  Idx := TabSet.TabIndex + 1;
  if Idx > TabSet.Items.Count - 1 then
    Idx := 0;
  if Idx < TabSet.Items.Count then
    TabSet.TabIndex := Idx;
end;

procedure TFrmBaseContainer.actTabPrevExecute(Sender: TObject);
var
  Idx: Integer;
begin
  inherited;
  Idx := TabSet.TabIndex - 1;
  if Idx < 0 then
    Idx := TabSet.Items.Count - 1;
  if Idx < TabSet.Items.Count then
    TabSet.TabIndex := Idx;
end;

procedure TFrmBaseContainer.TabSetInfoTip(Sender: TObject; Tab: TExTabItem;
  var InfoTip: String);
begin
  inherited;
  if Tab.Caption <> Tab.ShortCaption then
    InfoTip := Tab.Caption;
end;

procedure TFrmBaseContainer.FormCreate(Sender: TObject);
begin
  inherited;
  SetActiveDockedForm(nil);
end;

procedure TFrmBaseContainer.actTabPosTopExecute(Sender: TObject);
begin
  inherited;
  SetTabSetPosition(tpTop);
end;

procedure TFrmBaseContainer.actTabPosBottomExecute(Sender: TObject);
begin
  inherited;
  SetTabSetPosition(tpBottom);
end;

procedure TFrmBaseContainer.actTabPosLeftExecute(Sender: TObject);
begin
  inherited;
  SetTabSetPosition(tpLeft);
end;

procedure TFrmBaseContainer.actTabPosRightExecute(Sender: TObject);
begin
  inherited;
  SetTabSetPosition(tpRight);
end;

procedure TFrmBaseContainer.pmTabPopup(Sender: TObject);
var
  P: TPoint;
  TabIdx: Integer;
begin
  inherited;
  P := TabSet.ScreenToClient(pmTab.PopupPoint);
  TabIdx := TabSet.ItemAtPos(P);
  mnuTabFloat.Visible := (TabIdx > -1);
  mnuTabClose.Visible := (TabIdx > -1);
  mnuTabLine01.Visible := (TabIdx > -1);
end;

end.
