unit ResizeArea;

interface

uses
  Windows, Messages, Forms, SysUtils, Classes, Controls, Graphics, USkinStyles;

type
  TResizeArea = class(TCustomControl)
  private
    FStyle: TSkinStyle;
    FMouseDowned: Boolean;
    FStartPos: TPoint;
    FStartSize: TSize;
    FOnResizing: TNotifyEvent;
    FOnStartResize: TNotifyEvent;
    FOnEndResize: TNotifyEvent;

    procedure SetStyle(Value: TSkinStyle);
    function GetParentForm: TCustomForm;
  protected
    procedure DoResizing; virtual;
    procedure DoStartResize; virtual;
    procedure DoEndResize; virtual;
    procedure Paint; override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
  public
    constructor Create(AOwnwe: TComponent); override;
    destructor Destroy; override;
    property Style: TSkinStyle read FStyle write SetStyle;
  published
    property Anchors;
    property Cursor;
    property Color;
    property ParentColor;
    property Visible;
    property Enabled;

    property OnResizing: TNotifyEvent read FOnResizing write FOnResizing;
    property OnStartResize: TNotifyEvent read FOnStartResize write FOnStartResize;
    property OnEndResize: TNotifyEvent read FOnEndResize write FOnEndResize;

    property OnMouseDown;
		property OnMouseMove;
		property OnMouseUp;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('NSM2', [TResizeArea]);
end;

constructor TResizeArea.Create(AOwnwe: TComponent);
begin
  inherited;
  FStyle := TSkinStyle.Create(nil);
  with FStyle do
  begin
    FontColor := Font.Color;
    BackgroundColor := Color;
  end;
  FMouseDowned := False;
  Anchors := [akRight, akBottom];
  Cursor := crSizeNWSE;

  Width := 20;
  Height := 20;
  if Assigned(Parent) then
  begin
    Left := Parent.ClientWidth - Width;
    Top := Parent.ClientHeight - Height;
  end;
end;

destructor TResizeArea.Destroy;
begin
  FStyle.Free;
  inherited;
end;

procedure TResizeArea.DoResizing;
begin
  if Assigned(FOnResizing) then
    FOnResizing(Self);
end;

procedure TResizeArea.DoStartResize;
begin
  if Assigned(FOnStartResize) then
    FOnStartResize(Self);
end;

procedure TResizeArea.DoEndResize;
begin
  if Assigned(FOnEndResize) then
    FOnEndResize(Self);
end;

procedure TResizeArea.MouseMove(Shift: TShiftState; X, Y: Integer);
var
  P: TPoint;
  Form: TCustomForm;
begin
  if FMouseDowned then
  begin
    P := ClientToScreen(Point(X, Y));
    Form := GetParentForm;
    if Assigned(Form) then
    begin
      Form.SetBounds(Form.Left, Form.Top,
          FStartSize.cx + (P.X - FStartPos.X),
          FStartSize.cy + (P.Y - FStartPos.Y));
      Form.Repaint;
    end;
    DoResizing;
  end;
end;

procedure TResizeArea.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
  Form: TCustomForm;
begin
  if Button = mbLeft then
  begin
    FMouseDowned := True;
    FStartPos := ClientToScreen(Point(X, Y));
    Form := GetParentForm;
    if Assigned(Form) then
    begin
      FStartSize.cx := Form.Width;
      FStartSize.cy := Form.Height;
    end;
    DoStartResize;
  end;
end;

procedure TResizeArea.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  if FMouseDowned then
  begin
    FMouseDowned := False;
    DoEndResize;
  end;
end;

procedure TResizeArea.Paint;
const
  RA_FONTNAME = 'Marlett';
  RA_CHAR1 = 'p';
  RA_CHAR2 = 'o';
var
  Extent: TSize;
  R: TRect;
begin
  FStyle.Draw(Canvas, ClientRect);
  if FStyle.BackgroundImage.Empty then
  begin
    R := ClientRect;
    FStyle.AdjustClientRect(R);
    with Canvas do
    begin
      Font.Name := RA_FONTNAME;
      Font.Charset := SYMBOL_CHARSET;
      Font.Size := 12;
      Extent := TextExtent(RA_CHAR1);

      Brush.Style := bsClear;
      Font.Color := clBtnShadow;
      TextOut(R.Right - Extent.cx, R.Bottom - Extent.cy, RA_CHAR1);
      Font.Color := clBtnHighlight;
      TextOut(R.Right - Extent.cx, R.Bottom - Extent.cy, RA_CHAR2);
    end;
  end;
end;


function TResizeArea.GetParentForm: TCustomForm;
begin
  Result := TCustomForm(Forms.GetParentForm(Self));
end;

procedure TResizeArea.SetStyle(Value: TSkinStyle);
begin
  if Assigned(Value) then
  begin
    FStyle.Assign(Value);
    Invalidate;
  end;
end;

end.
