unit ExTaskTray;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Menus, ShellAPI;

const
  WM_NOTIFYTASKTRAY = WM_USER + 201;

type
  TExTaskTray = class(TComponent)
  private
    FTip: String;
    FIcon: TIcon;
    FPopupMenu: TPopupMenu;
    FIconData: TNotifyIconData;
    FActive: Boolean;
    FWindowHandle: HWND;
    FOnDblClick: TNotifyEvent;
    FOnMouseMove: TMouseMoveEvent;
    FOnMouseDown: TMouseEvent;
    FOnMouseUp: TMouseEvent;
  protected
    procedure SetTip(Value: String); virtual;
    procedure SetIcon(Value: TIcon); virtual;
    procedure SetPopupMenu(Value: TPopupMenu); virtual;
    procedure SetActive(Value: Boolean); virtual;
    procedure IconChange(Sender: TObject);
    procedure InitIcon; virtual;
    procedure FinishIcon; virtual;
    procedure DoDblClick; virtual;
    procedure DoMouseDown(Button : TMouseButton; ShiftState : TShiftState;X, Y : Integer); virtual;
    procedure DoMouseMove(X, Y : Integer); virtual;
    procedure DoMouseUp(Button : TMouseButton; ShiftState : TShiftState;X, Y : Integer); virtual;
    procedure WndProc(var Msg : TMessage);
  public
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
  protected
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  published
    property PopupMenu: TPopupMenu read FPopupMenu write SetPopupMenu;
    property Tip: String read FTip write SetTip;
    property Icon: TIcon read FIcon write SetIcon;
    property Active: Boolean read FActive write SetActive Default False;
    property OnDblClick : TNotifyEvent read FOnDblClick write FOnDblClick;
    property OnMouseMove : TMouseMoveEvent read FOnMouseMove write FOnMouseMove;
    property OnMouseDown : TMouseEvent read  FOnMouseDown write FOnMouseDown;
    property OnMouseUp : TMouseEvent read FOnMouseUp write FOnMouseUp;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('NSM2', [TExTaskTray]);
end;

constructor TExTaskTray.Create(AOwner: TComponent);
begin
  inherited;
  FWindowHandle := Classes.AllocateHWnd(WndProc);
  FIcon := TIcon.Create;
  FIcon.OnChange := IconChange;
  FActive := False;
end;

destructor TExTaskTray.Destroy;
begin
 FinishIcon;
 FIcon.Free;
 Classes.DeallocateHWnd(FWindowHandle);
 inherited;
end;

procedure TExTaskTray.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited;
  if (AComponent = FPopupMenu) and (Operation = opRemove) then
    FPopupMenu := nil;
end;

procedure TExTaskTray.IconChange(Sender: TObject);
begin
  if FActive then
    InitIcon;
end;

procedure TExTaskTray.SetTip(Value: String);
begin
  if Value <> FTip then
  begin
    FTip := Value;
    if FActive then
      InitIcon;
  end;
end;

procedure TExTaskTray.SetIcon(Value: TIcon);
begin
  if Value <> nil then
  begin
    FIcon.Assign(Value);
    if FActive then
      InitIcon;
  end;
end;

procedure TExTaskTray.SetActive(Value: Boolean);
begin
  if csDesigning in ComponentState then Exit;
  if Value <> FActive then
  begin
    if Value then
      InitIcon
    else
      FinishIcon;
  end;
end;

procedure TExTaskTray.SetPopupMenu(Value: TPopupMenu);
begin
  if Value <> FPopupMenu then
    FPopupMenu := Value;
  if FPopupMenu <> nil then
    FPopupMenu.FreeNotification(Self);
end;

procedure TExTaskTray.InitIcon;
var
  R: Boolean;
begin
  with FIconData do
  begin
    cbSize := SizeOf(TNotifyIconData);
    Wnd := FWindowHandle;
    uCallBackMessage := WM_NOTIFYTASKTRAY;
    uId := 1;
    if FTip <> '' then
      uFlags := NIF_MESSAGE or NIF_ICON or NIF_TIP
    else
      uFlags := NIF_MESSAGE or NIF_ICON;
    hIcon := FIcon.Handle;
    StrPLCopy(szTip, FTip, SizeOf(szTip) - 1);
  end;

//  SetForegroundWindow(FWindowHandle);
  if FActive then
    R := Shell_NotifyIcon(NIM_MODIFY, @FIconData)
  else
    R := Shell_NotifyIcon(NIM_ADD, @FIconData);

  if not R then
    FinishIcon
  else
    FActive := True;
end;

procedure TExTaskTray.FinishIcon;
begin
  if FActive then
    Shell_NotifyIcon(NIM_DELETE, @FIconData);
  FActive := False;
end;


procedure TExTaskTray.DoDblClick;
begin
  if Assigned(FOnDblClick) then
    FOnDblClick(Self);
end;

procedure TExTaskTray.DoMouseDown(Button: TMouseButton; ShiftState : TShiftState;X, Y : Integer);
begin
  if Assigned(Owner) and (Owner is TWinControl) Then
    SetForegroundWindow(TWinControl(Owner).Handle);

  if Assigned(FOnMouseDown) then
    FOnMouseDown(Self, Button, ShiftState, X, Y);

  if (Button = mbRight) then
    if Assigned(FPopupMenu) then
      FPopupMenu.Popup(X, Y);
end;

procedure TExTaskTray.DoMouseMove(X, Y: Integer);
begin
  if Assigned(FOnMouseMove) then
    FOnMouseMove(Self, [], X, Y);
end;

procedure TExTaskTray.DoMouseUp(Button: TMouseButton; ShiftState : TShiftState;X, Y : Integer);
begin
  if Assigned(FOnMouseUp) then
    FOnMouseUp(Self, Button, ShiftState, X, Y);
end;

procedure TExTaskTray.WndProc(var Msg: TMessage);
var
  Pos : TPoint;
  ShiftState : TShiftState;
begin
  case Msg.Msg of
  WM_NOTIFYTASKTRAY:
  begin
    GetCursorPos(Pos);
    ShiftState := [];
    Case Msg.lParam Of
      WM_LBUTTONDBLCLK :
        DoDblClick;

      WM_RBUTTONDBLCLK :
        DoDblClick;

      WM_MOUSEMOVE :
        DoMouseMove(Pos.X, Pos.Y);

      WM_LBUTTONDOWN :
      begin
        ShiftState := ShiftState + [ssLeft];
        DoMouseDown(mbLeft, ShiftState, Pos.X, Pos.Y);
      end;

      WM_LBUTTONUP :
      begin
        ShiftState := ShiftState + [ssLeft];
        DoMouseUp(mbLeft, ShiftState, Pos.X, Pos.Y);
      end;

      WM_RBUTTONDOWN :
      begin
        ShiftState := ShiftState + [ssRight];
        DoMouseDown(mbRight, ShiftState, Pos.X, Pos.Y);
      end;

      WM_RBUTTONUP :
      begin
        ShiftState := ShiftState + [ssRight];
        DoMouseUp(mbRight, ShiftState, Pos.X, Pos.Y);
      end;

      WM_MBUTTONDOWN :
      begin
        ShiftState := ShiftState + [ssMiddle];
        DoMouseDown(mbMiddle, ShiftState, Pos.X, Pos.Y);
      end;

      WM_MBUTTONUP :
      begin
        ShiftState := ShiftState + [ssMiddle];
        DoMouseUp(mbMiddle, ShiftState, Pos.X, Pos.Y);
      end;
    end;
  end else
    Msg.Result := DefWindowProc(FWindowHandle, Msg.Msg, Msg.wParam, Msg.lParam);
  end;
end;


end.
