unit ExScrollBar;

interface

uses
  Windows, Messages, SysUtils, Classes, Controls, Graphics, ExtCtrls,
  USkinStyles;

type
  TXSButtonType = (btUp, btDown);
  TXSScrollMode = (smNone, smLineUp, smLineDown, smPageUp, smPageDown, smTrack);
  TExScrollBar = class(TCustomControl)
  private
    FTimer: TTimer;
    FTimerDelay: Integer;
    FTrackPos: TPoint;
    FOnChange: TNotifyEvent;
    FThumbSize: Integer;
    FMin: Integer;
    FMax: Integer;
    FPosition: Integer;
    FSmallChange: Integer;
    FLargeChange: Integer;
    FScrollMode: TXSScrollMode;
    FStyle: TSkinStyle;
    FThumbUpStyle: TSkinStyle;
    FThumbDownStyle: TSkinStyle;
    FUpButtonUpStyle: TSkinStyle;
    FUpButtonDownStyle: TSkinStyle;
    FDownButtonUpStyle: TSkinStyle;
    FDownButtonDownStyle: TSkinStyle;
    procedure BtnTimerProc(Sender: TObject);
    procedure SetThumbSize(Value: Integer);
    procedure SetMin(Value: Integer);
    procedure SetMax(Value: Integer);
    procedure SetPosition(Value: Integer);
    procedure SetStyle(Value: TSkinStyle);
    procedure SetThumbUpStyle(Value: TSkinStyle);
    procedure SetThumbDownStyle(Value: TSkinStyle);
    procedure SetUpButtonUpStyle(Value: TSkinStyle);
    procedure SetUpButtonDownStyle(Value: TSkinStyle);
    procedure SetDownButtonUpStyle(Value: TSkinStyle);
    procedure SetDownButtonDownStyle(Value: TSkinStyle);
    procedure DoChange;
    procedure EraseBackground;
    function GetThumbRect: TRect;
    function GetButtonRect(ButtonType: TXSButtonType): TRect;
    function GetPageRect(ButtonType: TXSButtonType): TRect;
    procedure DrawButton(ButtonType: TXSButtonType);
    procedure DrawThumb;
    procedure ON_WM_ERASEBKGND(var Msg: TWMEraseBkgnd); message WM_ERASEBKGND;
  protected
    procedure Paint; override;
    procedure Resize; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property ThumbSize: Integer read FThumbSize write SetThumbSize default 0;
    property Min: Integer read FMin write SetMin default 0;
    property Max: Integer read FMax write SetMax default 100;
    property Position: Integer read FPosition write SetPosition default 0;
    property Style: TSkinStyle read FStyle write SetStyle;
    property ThumbUpStyle: TSkinStyle read FThumbUpStyle write SetThumbUpStyle;
    property ThumbDownStyle: TSkinStyle read FThumbDownStyle write SetThumbDownStyle;
    property UpButtonUpStyle: TSkinStyle read FUpButtonUpStyle write SetUpButtonUpStyle;
    property UpButtonDownStyle: TSkinStyle read FUpButtonDownStyle write SetUpButtonDownStyle;
    property DownButtonUpStyle: TSkinStyle read FDownButtonUpStyle write SetDownButtonUpStyle;
    property DownButtonDownStyle: TSkinStyle read FDownButtonDownStyle write SetDownButtonDownStyle;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;

    property Align;
    property Anchors;
    property Constraints;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;

    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick;
    property OnContextPopup;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDock;
    property OnStartDrag;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('NSM2', [TExScrollBar]);
end;

const
  ButtonSize = 16;
  MinThumbSize = 16;

// -----------------------------------------------------------------------------

constructor TExScrollBar.Create(AOwner: TComponent);
begin
  inherited;
  FStyle := TSkinStyle.Create(nil);
  FThumbUpStyle := TSkinStyle.Create(nil);
  FThumbDownStyle := TSkinStyle.Create(nil);
  FUpButtonUpStyle := TSkinStyle.Create(nil);
  FUpButtonDownStyle := TSkinStyle.Create(nil);
  FDownButtonUpStyle := TSkinStyle.Create(nil);
  FDownButtonDownStyle := TSkinStyle.Create(nil);

  with FStyle do
  begin
    FontColor := clBtnText;
    FontSize := 10;
    BackgroundColor := clBtnFace;
    BorderStyle := sbInset;
  end;
  with FThumbUpStyle do
  begin
    FontColor := clBtnText;
    FontSize := 10;
    BackgroundColor := clBtnFace;
    BorderStyle := sbOutset;
  end;
  with FThumbDownStyle do
  begin
    FontColor := clBtnText;
    FontSize := 10;
    BackgroundColor := clBtnFace;
    BorderStyle := sbNone;
  end;
  FUpButtonUpStyle.Assign(FThumbUpStyle);
  FUpButtonDownStyle.Assign(FThumbDownStyle);
  FDownButtonUpStyle.Assign(FThumbUpStyle);
  FDownButtonDownStyle.Assign(FThumbDownStyle);

  FTimer := TTimer.Create(Self);
  FTimer.Enabled := False;
  FTimer.Interval := 40;
  FTimer.OnTimer := BtnTimerProc;
  FTimerDelay := 10;
  Width := 16;
  Height := 100;
  FThumbSize := 0;
  FPosition := 0;
  FMax := 100;
  FMin := 0;
  FSmallChange := 1;
  FLargeChange := 10;
  FScrollMode := smNone;
  DoubleBuffered := True;
end;

destructor TExScrollBar.Destroy;
begin
  FStyle.Free;
  FThumbUpStyle.Free;
  FThumbDownStyle.Free;
  FUpButtonUpStyle.Free;
  FUpButtonDownStyle.Free;
  FDownButtonUpStyle.Free;
  FDownButtonDownStyle.Free;
  FOnChange := nil;
  FTimer.Free;
  inherited;
end;

// OnChange CxggK
procedure TExScrollBar.DoChange;
begin
  if Assigned(FOnChange) then
    FOnChange(Self);
end;

procedure TExScrollBar.SetThumbSize(Value: Integer);
begin
  if Value < 0 then
    Value := 0
  else if Value > Abs(FMax - FMin) - 1 then
    Value := (FMax - FMin) - 1;

  if FThumbSize <> Value then
  begin
    FThumbSize := Value;
    Invalidate;
  end;
end;

procedure TExScrollBar.SetMin(Value: Integer);
begin
  if FMin <> Value then
  begin
    FMin := Value;
    if FPosition < FMin then
      SetPosition(FMin);
    Invalidate;
  end;
end;

procedure TExScrollBar.SetMax(Value: Integer);
begin
  if FMax <> Value then
  begin
    FMax := Value;
    if FPosition > FMax then
      SetPosition(FMax);
    Invalidate;
  end;
end;

procedure TExScrollBar.SetPosition(Value: Integer);
begin
  if Value < FMin then
    Value := FMin
  else if Value > FMax - FThumbSize then
    Value := FMax - FThumbSize;
  if FPosition <> Value then
  begin
    FPosition := Value;
    Invalidate;
    DoChange;
  end;
end;

procedure TExScrollBar.SetStyle(Value: TSkinStyle);
begin
  if Assigned(Value) then
  begin
    FStyle.Assign(Value);
    Invalidate;
  end;
end;

procedure TExScrollBar.SetThumbUpStyle(Value: TSkinStyle);
begin
  if Assigned(Value) then
  begin
    FThumbUpStyle.Assign(Value);
    Invalidate;
  end;
end;

procedure TExScrollBar.SetThumbDownStyle(Value: TSkinStyle);
begin
  if Assigned(Value) then
  begin
    FThumbDownStyle.Assign(Value);
    Invalidate;
  end;
end;

procedure TExScrollBar.SetUpButtonUpStyle(Value: TSkinStyle);
begin
  if Assigned(Value) then
  begin
    FUpButtonUpStyle.Assign(Value);
    Invalidate;
  end;
end;

procedure TExScrollBar.SetUpButtonDownStyle(Value: TSkinStyle);
begin
  if Assigned(Value) then
  begin
    FUpButtonDownStyle.Assign(Value);
    Invalidate;
  end;
end;

procedure TExScrollBar.SetDownButtonUpStyle(Value: TSkinStyle);
begin
  if Assigned(Value) then
  begin
    FDownButtonUpStyle.Assign(Value);
    Invalidate;
  end;
end;

procedure TExScrollBar.SetDownButtonDownStyle(Value: TSkinStyle);
begin
  if Assigned(Value) then
  begin
    FDownButtonDownStyle.Assign(Value);
    Invalidate;
  end;
end;

function PointInRect(R: TRect; P: TPoint): Boolean;
begin
  Result := (R.Left <= P.X) and (R.Right > P.X) and (R.Top <= P.Y) and
    (R.Bottom > P.Y);
end;

procedure TExScrollBar.BtnTimerProc(Sender: TObject);
var
  P: TPoint;
begin
  if FTimerDelay <= 0 then
  begin
    if FScrollMode = smLineUp then
      SetPosition(FPosition - FSmallChange)
    else if FScrollMode = smLineDown then
      SetPosition(FPosition + FSmallChange)
    else if FScrollMode = smPageUp then
    begin
      GetCursorPos(P);
      ScreenToClient(P);
      if GetThumbRect.Top < P.Y then
        SetPosition(FPosition - FLargeChange);
    end
    else if FScrollMode = smPageDown then
    begin
      GetCursorPos(P);
      ScreenToClient(P);
      if GetThumbRect.Bottom > P.Y then
        SetPosition(FPosition + FLargeChange);
    end;
  end else
    Dec(FTimerDelay);  
end;

procedure TExScrollBar.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  inherited;
  if Button <> mbLeft then Exit;
  if PointInRect(GetButtonRect(btUp), Point(X, Y)) then
  begin
    FScrollMode := smLineUp;
    FTimer.Enabled := True;
    FTimerDelay := 10;
    SetPosition(FPosition - FSmallChange);
    Invalidate;
  end else
  if PointInRect(GetButtonRect(btDown), Point(X, Y)) then
  begin
    FScrollMode := smLineDown;
    FTimer.Enabled := True;
    FTimerDelay := 10;
    SetPosition(FPosition + FSmallChange);
    Invalidate;
  end else
  if PointInRect(GetThumbRect, Point(X, Y)) then
  begin
    FScrollMode := smTrack;
    FTrackPos := Point(X, GetThumbRect.Top - Y);
  end else
  if PointInRect(GetPageRect(btUp), Point(X, Y)) then
  begin
    FScrollMode := smPageUp;
    FTimer.Enabled := True;
    FTimerDelay := 10;
    SetPosition(FPosition - FLargeChange);
    Invalidate;
  end else
  if PointInRect(GetPageRect(btDown), Point(X, Y)) then
  begin
    FScrollMode := smPageDown;
    FTimer.Enabled := True;
    FTimerDelay := 10;
    SetPosition(FPosition + FLargeChange);
    Invalidate;
  end else
    FScrollMode := smNone;
end;

procedure TExScrollBar.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  inherited;
  case FScrollMode of
  smLineDown, smLineUp, smPageDown, smPageUp:
    begin
      FTimer.Enabled:= False;
      Invalidate;
    end;
  end;
  FScrollMode := smNone;
end;

procedure TExScrollBar.MouseMove(Shift: TShiftState; X, Y: Integer);
var
  P: Integer;
begin
  inherited;
  if FScrollMode = smTrack then
  begin
    P := Trunc(((Y + FTrackPos.Y) - ButtonSize) / (ClientHeight - ButtonSize *
      2) * Abs(FMax - FMin));
    SetPosition(P);
  end;
end;

procedure TExScrollBar.Resize;
begin
  Invalidate;
  inherited;
end;


procedure TExScrollBar.ON_WM_ERASEBKGND(var Msg:TWMEraseBkgnd);
begin
  Msg.Result := 1;
end;

procedure TExScrollBar.Paint;
begin
  EraseBackground;
  DrawButton(btUp);
  DrawButton(btDown);
  DrawThumb;
end;

procedure TExScrollBar.EraseBackground;
var
  R: TRect;
begin
  with Canvas do
  begin
    R := ClipRect;
    FStyle.Draw(Canvas, R);
  end;
end;

procedure TExScrollBar.DrawButton(ButtonType: TXSButtonType);
var
  R: TRect;
  AStyle: TSkinStyle;
begin
  with Canvas do
  begin
    Font.Name := 'Marlett';
    Font.Charset := SYMBOL_CHARSET;
    if ButtonType = btUp then
    begin
      R := GetButtonRect(ButtonType);
      if FScrollMode = smLineUp then
        AStyle := FUpButtonDownStyle
      else
        AStyle := FUpButtonUpStyle;
      AStyle.Draw(Canvas, R);

      Font.Size := AStyle.FontSize;
      Font.Color := AStyle.FontColor;
      SetBkMode(Handle, TRANSPARENT);

      R := GetButtonRect(ButtonType);
      R.TopLeft := Point(R.Left + Ord(FScrollMode = smLineUp),
                         R.Top + Ord(FScrollMode = smLineUp));
      if AStyle.BackgroundImage.Empty then
        DrawText(Handle, '5', -1, R, DT_VCENTER or DT_CENTER);
    end else
    begin
      R := GetButtonRect(ButtonType);
      if FScrollMode = smLineDown then
        AStyle := FDownButtonDownStyle
      else
        AStyle := FDownButtonUpStyle;
      AStyle.Draw(Canvas, R);

      Font.Size := AStyle.FontSize;
      Font.Color := AStyle.FontColor;
      SetBkMode(Handle, TRANSPARENT);
      
      R := GetButtonRect(ButtonType);
      R.TopLeft := Point(R.Left + Ord(FScrollMode = smLineDown),
                         R.Top + Ord(FScrollMode = smLineDown));
      if AStyle.BackgroundImage.Empty then
        DrawText(Handle, '6', -1, R, DT_VCENTER or DT_CENTER);
    end;
  end;
end;

function TExScrollBar.GetButtonRect(ButtonType: TXSButtonType): TRect;
begin
  if ButtonType = btUp then
  begin
    Result := Rect(0, 0, ClientWidth, ButtonSize);
    FStyle.AdjustClientRect(Result);
    Result.Bottom := ButtonSize;
  end else
  begin
    Result := Rect(0, ClientHeight - ButtonSize, ClientWidth, ClientHeight);
    FStyle.AdjustClientRect(Result);
    Result.Top := ClientHeight - ButtonSize;
  end;
end;

function TExScrollBar.GetPageRect(ButtonType: TXSButtonType): TRect;
var
  BtnRect, ThumbRect: TRect;
begin
  ThumbRect := GetThumbRect;
  BtnRect := GetButtonRect(ButtonType);
  if ButtonType = btUp then
    Result := Rect(ThumbRect.Left, BtnRect.Bottom, ThumbRect.Right, ThumbRect.Top)
  else
    Result := Rect(ThumbRect.Left, ThumbRect.Bottom, ThumbRect.Right, BtnRect.Top);
end;

function TExScrollBar.GetThumbRect: TRect;
var
  H, T: Integer;
begin
  H := Trunc((ClientHeight - ButtonSize * 2) * FThumbSize / Abs(FMax - FMin));
  if H < MinThumbSize then
    H := MinThumbSize;
  T := ButtonSize + Trunc((ClientHeight - ButtonSize * 2) *
        FPosition / Abs(FMax - FMin));
  if T < ButtonSize then
    T := ButtonSize
  else if T > ClientHeight - ButtonSize - H then
    T := ClientHeight - ButtonSize - H;
  Result := Rect(0, T, ClientWidth, T + H);
  FStyle.AdjustClientRect(Result);
  Result.Top := T;
  Result.Bottom := T + H;
end;

procedure TExScrollBar.DrawThumb;
begin
  FThumbUpStyle.Draw(Canvas, GetThumbRect);
end;

end.
