unit ExPanel;
(* tbgX^Cpl *)

interface

uses
  Windows, Messages, SysUtils, Classes, Controls, Graphics, USkinStyles;

type
  TExPanel = class(TCustomControl)
  private
    FStyle: TSkinStyle;
    FTransparent: Boolean;
    FOnPaint: TNotifyEvent;
    procedure SetColor(Value: TColor);
    procedure SetBorderColor(Value: TColor);
    procedure SetBorderStyle(Value: TSkinBorderStyle);
    procedure SetBorderWidth(Value: Integer);
    procedure SetPadding(Value: Integer);
    function GetColor: TColor;
    function GetBorderColor: TColor;
    function GetBorderStyle: TSkinBorderStyle;
    function GetBorderWidth: Integer;
    function GetPadding: Integer;
    procedure SetStyle(Value: TSkinStyle);
    procedure SetTransparent(Value: Boolean);
  protected
    function FindParentExPanel: TExPanel;
    procedure UpdateStyle;
    procedure DoPaint;
    procedure Paint; override;
    procedure AdjustClientRect(var Rect: TRect); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure DrawChildBackground(Child: TControl; ACanvas: TCanvas; R: TRect);
    property Style: TSkinStyle read FStyle write SetStyle;
    property Canvas;
  published
    property Color: TColor read GetColor write SetColor default clWindow;
    property BorderColor: TColor read GetBorderColor write SetBorderColor default clBlack;
    property BorderStyle: TSkinBorderStyle read GetBorderStyle write SetBorderStyle default sbSolid;
    property BorderWidth: Integer read GetBorderWidth write SetBorderWidth default 1;
    property Padding: Integer read GetPadding write SetPadding default 0;
    property Transparent: Boolean read FTransparent write SetTransparent default False;
    property Align;
    property Anchors;
    property Hint;
    property Ctl3D;
    property DockSite;
    property Font;
    property ParentColor;
    property ParentFont;
    property ParentCtl3D;
    property ShowHint;
    property ParentShowHint;
    property Visible;
    property TabStop;
    property TabOrder;

    property OnPaint: TNotifyEvent read FOnPaint write FOnPaint;
    property OnClick;
    property OnDblClick;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnResize;
    property OnDockDrop;
		property OnDockOver;
		property OnEnter;
		property OnExit;
		property OnGetSiteInfo;
		property OnKeyDown;
		property OnKeyPress;
		property OnKeyUp;
		property OnUnDock;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('NSM2', [TExPanel]);
end;

constructor TExPanel.Create(AOwner: TComponent);
begin
  inherited;
  ControlStyle := ControlStyle + [csAcceptsControls];
  FStyle := TSkinStyle.Create(nil);
  inherited Color := clWindow;  
  with FStyle do
  begin
    BackgroundColor := inherited Color;
    BorderColor := clBlack;
    BorderStyle := sbSolid;
    BorderWidth := 1;
  end;

  Transparent := False;
end;

destructor TExPanel.Destroy;
begin
  FStyle.Free;
  inherited;
end;

procedure TExPanel.DoPaint;
begin
  if Assigned(FOnPaint) then
    FOnPaint(Self);
end;

procedure TExPanel.Paint;
var
  ParentPanel: TExPanel;
begin
  if FTransparent then
    ParentPanel := FindParentExPanel
  else
    ParentPanel := nil;
  if Assigned(ParentPanel) then
  begin
    ParentPanel.DrawChildBackground(Self, Canvas, ClientRect);
    FStyle.DrawBorder(Canvas, ClientRect);
  end else
    FStyle.Draw(Canvas, ClientRect);
  DoPaint;
end;

function TExPanel.FindParentExPanel: TExPanel;
var
  Control: TWinControl;
begin
  Result := nil;
  Control := Parent;
  while Assigned(Control) do
  begin
    if Control is TExPanel then
    begin
      Result := TExPanel(Control);
      Break;
    end;
    Control := Control.Parent;
  end;
end;

procedure TExPanel.AdjustClientRect(var Rect: TRect);
begin
  FStyle.AdjustClientRect(Rect);
end;

procedure TExPanel.UpdateStyle;
var
  I: Integer;
begin
  // qplĕ`
  for I := 0 to ControlCount - 1 do
    if (Controls[I] is TExPanel) and TExPanel(Controls[I]).Transparent then
      TExPanel(Controls[I]).UpdateStyle;
  Invalidate;
end;

procedure TExPanel.DrawChildBackground(Child: TControl; ACanvas: TCanvas; R: TRect);
var
  ParentPanel: TExPanel;
  ParentTopLeft, ChildTopLeft: TPoint;
  OfsX, OfsY: Integer;
  ParentRect: TRect;
begin
  if FTransparent then
    ParentPanel := FindParentExPanel
  else
    ParentPanel := nil;
  if Assigned(ParentPanel) then
    ParentPanel.DrawChildBackground(Child, ACanvas, R)
  else
  begin
    ParentTopLeft := ClientToScreen(Point(0, 0));
    ChildTopLeft := Child.ClientToScreen(Point(0, 0));
    OfsX := (ChildTopLeft.X - ParentTopLeft.X);
    OfsY := (ChildTopLeft.Y - ParentTopLeft.Y);
    ParentRect := FStyle.GetBackgroundRect(ClientRect);    
    OffsetRect(ParentRect, - OfsX, - OfsY);
    FStyle.FillBackGround(ACanvas, ParentRect);
  end;
end;

procedure TExPanel.SetStyle(Value: TSkinStyle);
begin
  if Assigned(Value) then
  begin
    FStyle.Assign(Value);
    inherited Color := FStyle.BackgroundColor;
    Invalidate;
    Realign;
    UpdateStyle;
  end;
end;

procedure TExPanel.SetColor(Value: TColor);
begin
  if FStyle.BackgroundColor <> Value then
  begin
    FStyle.BackgroundColor := Value;
    inherited Color := Value;
    Invalidate;
  end;
end;

procedure TExPanel.SetBorderColor(Value: TColor);
begin
  if FStyle.BorderColor <> Value then
  begin
    FStyle.BorderColor := Value;
    Invalidate;
  end;
end;

procedure TExPanel.SetBorderStyle(Value: TSkinBorderStyle);
begin
  if FStyle.BorderStyle <> Value then
  begin
    FStyle.BorderStyle := Value;
    Invalidate;
    Realign;
  end;
end;

procedure TExPanel.SetBorderWidth(Value: Integer);
begin
  if FStyle.BorderWidth <> Value then
  begin
    FStyle.BorderWidth := Value;
    Invalidate;
    Realign;
  end;
end;

procedure TExPanel.SetPadding(Value: Integer);
begin
  FStyle.PaddingLeft := Value;
  FStyle.PaddingRight := Value;
  FStyle.PaddingTop := Value;
  FStyle.PaddingBottom := Value;
  Invalidate;
  Realign;
end;

function TExPanel.GetColor: TColor;
begin
  Result := FStyle.BackgroundColor;
end;

function TExPanel.GetBorderColor: TColor;
begin
  Result := FStyle.BorderColor;
end;

function TExPanel.GetBorderStyle: TSkinBorderStyle;
begin
  Result := FStyle.BorderStyle;
end;

function TExPanel.GetBorderWidth: Integer;
begin
  Result := FStyle.BorderWidth;
end;

function TExPanel.GetPadding: Integer;
begin
  Result := FStyle.PaddingLeft;
end;

procedure TExPanel.SetTransparent(Value: Boolean);
begin
  if FTransparent <> Value then
  begin
    FTransparent := Value;
    if FTransparent then
    begin
//      FStyle.ClearBitmap;
      ParentColor := True;
    end;
    Invalidate;
  end;
end;

end.
