unit UNsmUIPluginMainEx;

interface

uses
  Windows, SysUtils, UNsmConsts, UNsmTypes, UNsmUIPluginMain, IniFiles, Classes,
  UNsmInfo;

type
  TNsmUIPluginMainEx = class;
  TNsmUIConnection = class;
  TNsmUISession = class;
  TNsmUIConnections = class;
  TNsmUISessions = class;
  TNsmUIMembers = class;
  TNsmUIGroups = class;

  TNsmUIObject = class(TObject)
  private
    FOwnerPlugin: TNsmUIPluginMainEx;
  public
    property OwnerPlugin: TNsmUIPluginMainEx read FOwnerPlugin;
  end;

  TNsmUIMember = class(TNsmUIObject)
  private
    FOwner: TNsmUIMembers;
    FAccount: String;
    FData: Pointer;
    function GetName: WideString;
    function GetStatusStr: WideString;
    function GetStatus: Integer;
    function GetBusyness: Integer;
    function GetGroupID: Integer;
  public
    constructor Create(AOwner: TNsmUIMembers; AAccount: String);
    property Owner: TNsmUIMembers read FOwner;
    function GetMemberInfoStr(InfoKey: Integer): WideString;
    function GetMemberInfoInt(InfoKey: Integer): Integer;
    function IsBlocked: Boolean;

    property Account: String read FAccount;
    property Name: WideString read GetName;
    property StatusStr: WideString read GetStatusStr;
    property Status: Integer read GetStatus;
    property Busyness: Integer read GetBusyness;
    property GroupID: Integer read GetGroupID;
    property Data: Pointer read FData;
  end;

  TSortKey = (skAccount, skName, skStatus, skBusyness);
  TNsmUIMembers = class(TNsmUIObject)
  private
    FMembers: THashedStringList;
    FOwner: TNsmUIObject;
    FListKind: Integer;
    function GetCount: Integer;
    function GetMember(Idx: Integer): TNsmUIMember;
  public
    constructor Create(AOwner: TNsmUIObject; AListKind: Integer = NMLK_FORWARDLIST);
    destructor Destroy; override;
    procedure SyncMembers;
    function Add(Account: String): TNsmUIMember;
    procedure Delete(Idx: Integer);
    procedure Remove(Account: String);
    procedure Clear;
    function IndexOfAccount(Account: String): Integer;
    function Contains(Account: String): Boolean;
    procedure Sort(Key1, Key2, Key3: TSortKey);

    property Owner: TNsmUIObject read FOwner;
    property ListKind: Integer read FListKind;
    property Count: Integer read GetCount;
    property Member[Idx: Integer]: TNsmUIMember read GetMember; default;
  end;

  TNsmUIGroup = class(TNsmUIObject)
  private
    FOwner: TNsmUIGroups;
    FId: Integer;
    FData: Pointer;
    function GetName: WideString;
    function GetExpanded: Boolean;
    procedure SetExpanded(Value: Boolean);
  public
    constructor Create(AOwner: TNsmUIGroups; AId: Integer);
    property Owner: TNsmUIGroups read FOwner;

    property Id: Integer read FId;
    property Name: WideString read GetName;
    property Expanded: Boolean read GetExpanded write SetExpanded;
    property Data: Pointer read FData;    
  end;

  TNsmUIGroups = class(TNsmUIObject)
  private
    FGroups: TList;
    FOwner: TNsmUIObject;
    function GetCount: Integer;
    function GetGroup(Idx: Integer): TNsmUIGroup;
  public
    constructor Create(AOwner: TNsmUIObject);
    destructor Destroy; override;
    procedure SyncGroups;
    function Add(Id: Integer): TNsmUIGroup;
    procedure Delete(Idx: Integer);
    procedure Remove(Id: Integer);
    procedure Clear;
    function IndexOfId(Id: Integer): Integer;
    procedure Sort;

    property Owner: TNsmUIObject read FOwner;
    property Count: Integer read GetCount;
    property Group[Idx: Integer]: TNsmUIGroup read GetGroup; default;
  end;

  // RlNṼbp[NX
  TNsmUIConnection = class(TNsmUIObject)
  private
    FOwner: TNsmUIConnections;
    FHandle: HNsmConnection;
    FMembers: TNsmUIMembers;
    FReverseMembers: TNsmUIMembers;
    FAllowMembers: TNsmUIMembers;
    FBlockMembers: TNsmUIMembers;
    FGroups: TNsmUIGroups;
    FData: Pointer;
    function GetProtocol: String;
    function GetCaption: WideString;
    function GetStatus: Integer;
    function GetUserAccount: String;
    function GetUserName: WideString;
    function GetUserStatusStr: WideString;
    function GetUserStatus: Integer;
    function GetUserBusyness: Integer;
  public
    constructor Create(AOwner: TNsmUIConnections);
    destructor Destroy; override;
    function ChangeUserStatus(AStatus: Integer): Integer;
    function ChangeUserName(AName: WideString): Integer;
    function AddMember(Account: String; ListKind: Integer): Integer;
    function RemoveMember(Account: String; ListKind: Integer): Integer;
    function AddGroup(GroupName: WideString): Integer;
    function RemoveGroup(GroupID: Integer): Integer;
    function ChangeGroupName(GroupID: Integer; NewName: WideString): Integer;
    function ChangeMemberGroup(Account: String; NewGroupId: Integer): Integer;
    function LogOut: Integer;
    function OpenSession(ToAccount: String): Integer;

    property Handle: HNsmConnection read FHandle write FHandle;
    property Owner: TNsmUIConnections read FOwner;
    property Members: TNsmUIMembers read FMembers;
    property ReverseMembers: TNsmUIMembers read FReverseMembers;
    property AllowMembers: TNsmUIMembers read FAllowMembers;
    property BlockMembers: TNsmUIMembers read FBlockMembers;
    property Groups: TNsmUIGroups read FGroups;
    property Data: Pointer read FData;
    property Protocol: String read GetProtocol;
    property Caption: WideString read GetCaption;
    property Status: Integer read GetStatus;
    property UserAccount: String read GetUserAccount;
    property UserName: WideString read GetUserName;
    property UserStatusStr: WideString read GetUserStatusStr;
    property UserStatus: Integer read GetUserStatus;
    property UserBusyness: Integer read GetUserBusyness;
  end;

  // RlNVXg
  TNsmUIConnections = class(TNsmUIObject)
  private
    FOwner: TNsmUIPluginMainEx;
    FConnections: TList;
    function GetConnection(Idx: Integer): TNsmUIConnection;
    function GetCount: Integer;
  public
    constructor Create(AOwner: TNsmUIPluginMainEx);
    destructor Destroy; override;
    property Owner: TNsmUIPluginMainEx read FOwner;
    property Connection[Idx: Integer]: TNsmUIConnection read GetConnection; default;
    property Count: Integer read GetCount;
    function Add(Handle: HNsmConnection): TNsmUIConnection;
    procedure Remove(Handle: HNsmConnection);
    procedure Clear;
    function IndexOfHandle(Handle: HNsmConnection): Integer;
    function Find(Handle: HNsmConnection): TNsmUIConnection;
  end;

  // ZbV
  TNsmUISession = class(TNsmUIObject)
  private
    FOwner: TNsmUISessions;
    FHandle: HNsmSession;
    FMembers: TNsmUIMembers;
    function GetCaption: WideString;
    function GetStatus: Integer;
    function GetConnection: TNsmUIConnection;
    function GetProtocol: String;
  public
    constructor Create(AOwner: TNsmUISessions);
    destructor Destroy; override;
    function InviteMember(Account: String): Integer;
    function SendMessage(MsgInfo: TMessageInfo): Integer;
    function Close: Integer;

    property Owner: TNsmUISessions read FOwner;
    property Handle: HNsmSession read FHandle write FHandle;
    property Members: TNsmUIMembers read FMembers;
    property Caption: WideString read GetCaption;
    property Status: Integer read GetStatus;
    property Connection: TNsmUIConnection read GetConnection;
    property Protocol: String read GetProtocol;
  end;

  TNsmUISessions = class(TNsmUIObject)
  private
    FOwner: TNsmUIPluginMainEx;
    FSessions: TList;
    function GetSession(Idx: Integer): TNsmUISession;
    function GetCount: Integer;
  public
    constructor Create(AOwner: TNsmUIPluginMainEx);
    destructor Destroy; override;
    property Owner: TNsmUIPluginMainEx read FOwner;
    property Session[Idx: Integer]: TNsmUISession read GetSession; default;
    property Count: Integer read GetCount;
    function Add(SHandle: HNsmSession): TNsmUISession;
    procedure Remove(SHandle: HNsmSession);
    procedure Clear;
    function IndexOfHandle(SHandle: HNsmSession): Integer;
    function Find(SHandle: HNsmSession): TNsmUISession;
  end;

  TNsmUIPluginMainEx = class(TNsmUIPluginMain)
  private
    FConnections: TNsmUIConnections;
    FSessions: TNsmUISessions;
    FGetInfo: TNsmGetInfo;
    FSetInfo: TNsmSetInfo;
  protected
    procedure DoInitialize; override;
    procedure DoTerminate; override;

    procedure DoCreateConnection(AHandle: HNsmConnection; lParam: Cardinal); override;
    procedure DoDeleteConnection(AHandle: HNsmConnection; lParam: Cardinal); override;
    procedure DoAddMember(AHandle: HNsmConnection; AMInfo: TAddMemberInfo); override;
    procedure DoRemoveMember(AHandle: HNsmConnection; RMInfo: TRemoveMemberInfo); override;
    procedure DoAddGroup(AHandle: HNsmConnection; GroupID: Integer); override;
    procedure DoRemoveGroup(AHandle: HNsmConnection; GroupID: Integer); override;
    procedure DoCreateSession(AHandle: HNsmSession; lParam: Cardinal); override;
    procedure DoDeleteSession(AHandle: HNsmSession; lParam: Cardinal); override;
    procedure DoSessionAddMember(AHandle: HNsmSession; Account: String); override;
    procedure DoSessionRemoveMember(AHandle: HNsmSession; Account: String); override;
  public
    constructor Create; override;
    destructor Destroy; override;

    function SystemExit: Integer;
    function LogIn(Protocol, Account, Password, Caption: String; Status: Integer): Integer;

    property Connections: TNsmUIConnections read FConnections;
    property Sessions: TNsmUISessions read FSessions;
    property GetInfo: TNsmGetInfo read FGetInfo;
    property SetInfo: TNsmSetInfo read FSetInfo;
  end;

implementation


// -----------------------------------------------------------------------------
function NsmStatusToStr(Status: Integer): WideString;
begin
  Result := '';
  case Status of
  NMST_OFFLINE:     Result := 'ItC';
  NMST_ONLINE:      Result := 'IC';
  NMST_BUSY:        Result := '荞ݒ';
  NMST_IDLE:        Result := 'ACh';
  NMST_BERIGHTBACK: Result := 'ꎞސȒ';
  NMST_AWAY:        Result := 'ސȒ';
  NMST_ONTHEPHONE:  Result := 'db';
  NMST_OUTTOLUNCH:  Result := 'x';
  NMST_HIDDEN:      Result := 's';
  end;
end;

// TNsmUIMember ----------------------------------------------------------------

constructor TNsmUIMember.Create(AOwner: TNsmUIMembers; AAccount: String);
begin
  FOwnerPlugin := AOwner.OwnerPlugin;
  FOwner := AOwner;
  FAccount := AAccount;
  FData := nil;
end;

// o[𓾂
function TNsmUIMember.GetMemberInfoStr(InfoKey: Integer): WideString;
begin
  if FOwner.Owner is TNsmUIConnection then
    Result := FOwnerPlugin.GetInfo.GetMemberInfo
        (TNsmUIConnection(FOwner.Owner).Handle, FAccount, InfoKey, FOwner.ListKind)
  else if FOwner.Owner is TNsmUISession then
    Result := FOwnerPlugin.GetInfo.GetSessionMemberInfo
        (TNsmUISession(FOwner.Owner).Handle, FAccount,InfoKey);
end;

function TNsmUIMember.GetMemberInfoInt(InfoKey: Integer): Integer;
begin
  Result := 0;
  if FOwner.Owner is TNsmUIConnection then
    Result := FOwnerPlugin.GetInfo.GetMemberInfoInt
        (TNsmUIConnection(FOwner.Owner).Handle, FAccount, InfoKey, FOwner.ListKind)
  else if FOwner.Owner is TNsmUISession then
    Result := FOwnerPlugin.GetInfo.GetSessionMemberInfoInt
        (TNsmUISession(FOwner.Owner).Handle, FAccount,InfoKey);
end;

function TNsmUIMember.GetName: WideString;
begin
  Result := GetMemberInfoStr(NMMI_NAME);
end;

function TNsmUIMember.GetStatusStr: WideString;
var
  Status: Integer;
begin
  Status := GetStatus;
  if Status = NMST_CUSTOM then
    Result := GetMemberInfoStr(NMMI_STATUSSTRING)
  else
    Result := NsmStatusToStr(Status);
end;

function TNsmUIMember.GetStatus: Integer;
begin
  Result := GetMemberInfoInt(NMMI_STATUS);
end;

function TNsmUIMember.GetBusyness: Integer;
begin
  Result := GetMemberInfoInt(NMMI_BUSYNESS);
end;

function TNsmUIMember.GetGroupID: Integer;
begin
  Result := GetMemberInfoInt(NMMI_GROUPID);
end;

function TNsmUIMember.IsBlocked: Boolean;
begin
  if FOwner.Owner is TNsmUIConnection then
    Result := (TNsmUIConnection(FOwner.Owner).BlockMembers.IndexOfAccount(FAccount) > -1)
  else
    Result := False;
end;

// TNsmUIMembers ---------------------------------------------------------------

constructor TNsmUIMembers.Create(AOwner: TNsmUIObject; AListKind: Integer = NMLK_FORWARDLIST);
begin
  FOwnerPlugin := AOwner.OwnerPlugin;
  FOwner := AOwner;
  FListKind := AListKind;
  FMembers := THashedStringList.Create;
end;

destructor TNsmUIMembers.Destroy;
begin
  inherited;
  Clear;
  FMembers.Free;
end;

function TNsmUIMembers.GetCount: Integer;
begin
  Result := FMembers.Count;
end;

function TNsmUIMembers.GetMember(Idx: Integer): TNsmUIMember;
begin
  Result := TNsmUIMember(FMembers.Objects[Idx]);
end;

function TNsmUIMembers.Add(Account: String): TNsmUIMember;
begin
  Result := TNsmUIMember.Create(Self, Account);
  FMembers.AddObject(Account, Result);
end;

procedure TNsmUIMembers.Delete(Idx: Integer);
begin
  TNsmUIMember(FMembers.Objects[Idx]).Free;
  FMembers.Delete(Idx);
end;

procedure TNsmUIMembers.Remove(Account: String);
var
  Idx: Integer;
begin
  Idx := IndexOfAccount(Account);
  if Idx > -1 then
    Delete(Idx);
end;

procedure TNsmUIMembers.Clear;
begin
  while FMembers.Count > 0 do
    Delete(0);
end;

function TNsmUIMembers.IndexOfAccount(Account: String): Integer;
begin
  Result := FMembers.IndexOf(Account);
end;

function TNsmUIMembers.Contains(Account: String): Boolean;
begin
  Result := IndexOfAccount(Account) > -1;
end;

var
  _Key1, _Key2, _Key3: TSortKey;
function CompareProc(List: TStringList; Index1, Index2: Integer): Integer;
  function CompareByAccount(Item1, Item2: TNsmUIMember): Integer;
  begin
    Result := CompareText(Item1.Account, Item2.Account);
  end;
  function CompareByName(Item1, Item2: TNsmUIMember): Integer;
  begin
    Result := CompareText(Item1.Name, Item2.Name);
  end;
  function CompareByStatus(Item1, Item2: TNsmUIMember): Integer;
  begin
    Result := Ord(Item1.Status) - Ord(Item2.Status);
  end;
  function CompareByBusyness(Item1, Item2: TNsmUIMember): Integer;
  begin
    Result := Item1.Busyness - Item2.Busyness;
  end;
  function KeyCompare(Key: TSortKey): Integer;
  var
    Item1, Item2: TNsmUIMember;
  begin
    Item1 := TNsmUIMember(List.Objects[Index1]);
    Item2 := TNsmUIMember(List.Objects[Index2]);
    case Key of
    skAccount   : Result := CompareByAccount(Item1, Item2);
    skName      : Result := CompareByName(Item1, Item2);
    skStatus    : Result := CompareByStatus(Item1, Item2);
    skBusyness  : Result := CompareByBusyness(Item1, Item2);
    else
      Result := 0;
    end;
  end;
begin
  Result := KeyCompare(_Key1);
  if Result = 0 then
  begin
    Result := KeyCompare(_Key2);
    if Result = 0 then
      Result := KeyCompare(_Key3);
  end;
end;

procedure TNsmUIMembers.Sort(Key1, Key2, Key3: TSortKey);
begin
  _Key1 := Key1;
  _Key2 := Key2;
  _Key3 := Key3;
  FMembers.CustomSort(CompareProc);
end;

function EnumMember(lpAccount: LPTSTR; nData: Integer): Integer; StdCall;
begin
  TNsmUIMembers(nData).Add(lpAccount);
  Result := 0;
end;

procedure TNsmUIMembers.SyncMembers;

  procedure SyncSessionMembers;
  var
    EMINFO: TEnumMemberInfo;
  begin
    with EMINFO do
    begin
      cbSize := SizeOf(TEnumMemberInfo);
      nFlags := 0;
      lpCallBackProc := EnumMember;
      nData := Integer(Self);
    end;

    Clear;
    FOwnerPlugin.CallService(NMS_SYSTEM_SESSION_MEMBERS_ENUM,
        TNsmUISession(FOwner).Handle, Cardinal(@EMINFO));
  end;

  procedure SyncConnectionMembers;
  var
    EMINFO: TEnumMemberInfo;
  begin
    with EMINFO do
    begin
      cbSize := SizeOf(TEnumMemberInfo);
      nListKind := FListKind;
      nFlags := 0;
      lpCallBackProc := EnumMember;
      nData := Integer(Self);
    end;

    Clear;
    FOwnerPlugin.CallService(NMS_SYSTEM_CONNECTION_MEMBERS_ENUM,
        TNsmUIConnection(FOwner).Handle, Cardinal(@EMINFO));
  end;
begin
  if FOwner is TNsmUIConnection then
    SyncConnectionMembers
  else if FOwner is TNsmUISession then
    SyncSessionMembers;
end;

// TNsmUIGroup -----------------------------------------------------------------

constructor TNsmUIGroup.Create(AOwner: TNsmUIGroups; AId: Integer);
begin
  FOwnerPlugin := AOwner.OwnerPlugin;
  FOwner := AOwner;
  FId := AId;
  FData := nil;
end;

function TNsmUIGroup.GetName: WideString;
begin
  Result := FOwnerPlugin.GetInfo.GetGroupInfo(
    TNsmUIConnection(FOwner.FOwner).Handle, FId, NMGI_NAME);
end;

function TNsmUIGroup.GetExpanded: Boolean;
begin
  Result := FOwnerPLugin.GetInfo.GetGroupInfoBool(
    TNsmUIConnection(FOwner.FOwner).Handle, FId, NMGI_EXPANDED);
end;

procedure TNsmUIGroup.SetExpanded(Value: Boolean);
begin
  FOwnerPlugin.SetInfo.SetGroupInfo(
    TNsmUIConnection(FOwner.FOwner).Handle, FId, NMGI_EXPANDED, Value,
    NMIF_NOCHANGEEVENT);
end;

// TNsmUIGroups ----------------------------------------------------------------

constructor TNsmUIGroups.Create(AOwner: TNsmUIObject);
begin
  FOwnerPlugin := AOwner.OwnerPlugin;
  FOwner := AOwner;
  FGroups := TList.Create;
end;

destructor TNsmUIGroups.Destroy;
begin
  inherited;
  Clear;
  FGroups.Free;
end;

function TNsmUIGroups.GetCount: Integer;
begin
  Result := FGroups.Count;
end;

function TNsmUIGroups.GetGroup(Idx: Integer): TNsmUIGroup;
begin
  Result := TNsmUIGroup(FGroups[Idx]);
end;

function TNsmUIGroups.Add(Id: Integer): TNsmUIGroup;
begin
  Result := TNsmUIGroup.Create(Self, Id);
  FGroups.Add(Result);
end;

procedure TNsmUIGroups.Delete(Idx: Integer);
begin
  TNsmUIGroup(FGroups[Idx]).Free;
  FGroups.Delete(Idx);
end;

procedure TNsmUIGroups.Remove(Id: Integer);
var
  Idx: Integer;
begin
  Idx := IndexOfId(Id);
  if Idx > -1 then
    Delete(Idx);
end;

procedure TNsmUIGroups.Clear;
begin
  while FGroups.Count > 0 do
    Delete(0);
end;

function TNsmUIGroups.IndexOfId(Id: Integer): Integer;
var
  I: Integer;
begin
  Result := -1;
  for I := 0 to FGroups.Count - 1 do
    if TNsmUIGroup(FGroups[I]).Id = Id then
    begin
      Result := I;
      Break;
    end;
end;

function CompareGroupProc(Item1, Item2: Pointer): Integer;
begin
  if (TNsmUIGroup(Item1).Id = 0) and (TNsmUIGroup(Item2).Id <> 0) then
    Result := 1
  else if (TNsmUIGroup(Item1).Id <> 0) and (TNsmUIGroup(Item2).Id = 0) then
    Result := -1
  else
    Result := CompareStr(TNsmUIGroup(Item1).Name, TNsmUIGroup(Item2).Name);
  if Result = 0 then
    Result := TNsmUIGroup(Item1).Id - TNsmUIGroup(Item2).Id;
end;

procedure TNsmUIGroups.Sort;
begin
  FGroups.Sort(CompareGroupProc);
end;

function EnumGroup(nId, nData: Integer): Integer; StdCall;
begin
  TNsmUIGroups(nData).Add(nId);
  Result := 0;
end;

procedure TNsmUIGroups.SyncGroups;
var
  EGINFO: TEnumGroupInfo;
begin
  with EGINFO do
  begin
    cbSize := SizeOf(TEnumGroupInfo);
    nFlags := 0;
    lpCallBackProc := EnumGroup;
    nData := Integer(Self);
  end;

  Clear;
  FOwnerPlugin.CallService(NMS_SYSTEM_CONNECTION_GROUPS_ENUM,
      TNsmUIConnection(FOwner).Handle, Cardinal(@EGINFO));
end;

// TNsmConnection --------------------------------------------------------------

constructor TNsmUIConnection.Create(AOwner: TNsmUIConnections);
begin
  FOwnerPlugin := AOwner.OwnerPlugin;
  FOwner := AOwner;
  FMembers := TNsmUIMembers.Create(Self, NMLK_FORWARDLIST);
  FReverseMembers := TNsmUIMembers.Create(Self, NMLK_REVERSELIST);
  FAllowMembers := TNsmUIMembers.Create(Self, NMLK_ALLOWLIST);
  FBlockMembers := TNsmUIMembers.Create(Self, NMLK_BLOCKLIST);
  FGroups := TNsmUIGroups.Create(Self);
  FData := nil;
end;

destructor TNsmUIConnection.Destroy;
begin
  FMembers.Free;
  FReverseMembers.Free;
  FAllowMembers.Free;
  FBlockMembers.Free;
  FGroups.Free;
end;

function TNsmUIConnection.GetProtocol: String;
begin
  Result := FOwnerPlugin.GetInfo.GetConnectionInfo(FHandle, NMCI_PROTOCOL);
end;

function TNsmUIConnection.GetCaption: WideString;
begin
  Result := FOwnerPlugin.GetInfo.GetConnectionInfo(FHandle, NMCI_CAPTION);
end;

function TNsmUIConnection.GetStatus: Integer;
begin
  Result := FOwnerPlugin.GetInfo.GetConnectionInfoInt(FHandle, NMCI_STATUS);
end;

function TNsmUIConnection.GetUserAccount: String;
begin
  Result := FOwnerPlugin.GetInfo.GetConnectionInfo(FHandle, NMCI_USER_ACCOUNT);
end;

function TNsmUIConnection.GetUserName: WideString;
begin
  Result := FOwnerPlugin.GetInfo.GetConnectionInfo(FHandle, NMCI_USER_NAME);
end;

function TNsmUIConnection.GetUserStatusStr: WideString;
var
  Status: Integer;
begin
  Status := GetUserStatus;
  if Status = NMST_CUSTOM then
    Result := FOwnerPlugin.GetInfo.GetConnectionInfo(FHandle, NMCI_USER_STATUSSTRING)
  else
    Result := NsmStatusToStr(Status);
end;

function TNsmUIConnection.GetUserStatus: Integer;
begin
  Result := FOwnerPlugin.GetInfo.GetConnectionInfoInt(FHandle, NMCI_USER_STATUS);
end;

function TNsmUIConnection.GetUserBusyness: Integer;
begin
  Result := FOwnerPlugin.GetInfo.GetConnectionInfoInt(FHandle, NMCI_USER_BUSYNESS);
end;

function TNsmUIConnection.ChangeUserStatus(AStatus: Integer): Integer;
var
  StatusInfo: TUserStatusInfo;
begin
  with StatusInfo do
  begin
    cbSize      := SizeOf(TUserStatusInfo);
    nStatus     := AStatus;
    lpStatusStr := nil;
    nBusyness   := 0;
  end;
  Result := FOwnerPlugin.CallService(
    Format(NMS_PROTOCOL_CONNECTION_CHANGEUSERSTATUS, [GetProtocol]),
    FHandle, Cardinal(@StatusInfo));
end;

function TNsmUIConnection.ChangeUserName(AName: WideString): Integer;
var
  NameInfo: TUserNameInfo;
begin
  with NameInfo do
  begin
    cbSize := SizeOf(TUserNameInfo);
    lpName := PWideChar(AName);
  end;
  Result := FOwnerPlugin.CallService(
    Format(NMS_PROTOCOL_CONNECTION_CHANGEUSERNAME, [GetProtocol]),
    FHandle, Cardinal(@NameInfo));
end;

function TNsmUIConnection.AddMember(Account: String; ListKind: Integer): Integer;
var
  AMInfo: TAddMemberInfo;
begin
  with AMInfo do
  begin
    cbSize := SizeOf(TAddMemberInfo);
    lpAccount := PChar(Account);
    nListKind := ListKind;
  end;
  Result := FOwnerPlugin.CallService(
    Format(NMS_PROTOCOL_CONNECTION_ADDMEMBER, [GetProtocol]),
    FHandle, Cardinal(@AMInfo));
end;

function TNsmUIConnection.RemoveMember(Account: String; ListKind: Integer): Integer;
var
  RMInfo: TRemoveMemberInfo;
begin
  with RMInfo do
  begin
    cbSize := SizeOf(TRemoveMemberInfo);
    lpAccount := PChar(Account);
    nListKind := ListKind;
  end;
  Result := FOwnerPlugin.CallService(
    Format(NMS_PROTOCOL_CONNECTION_REMOVEMEMBER, [GetProtocol]),
    FHandle, Cardinal(@RMInfo));
end;

function TNsmUIConnection.AddGroup(GroupName: WideString): Integer;
begin
  Result := FOwnerPlugin.CallService(
    Format(NMS_PROTOCOL_CONNECTION_ADDGROUP, [GetProtocol]),
    FHandle, Cardinal(PWideChar(GroupName)));
end;

function TNsmUIConnection.RemoveGroup(GroupId: Integer): Integer;
begin
  Result := FOwnerPlugin.CallService(
    Format(NMS_PROTOCOL_CONNECTION_REMOVEGROUP, [GetProtocol]),
    FHandle, GroupId);
end;

function TNsmUIConnection.ChangeGroupName(GroupID: Integer; NewName: WideString): Integer;
var
  GNInfo: TGroupNameInfo;
begin
  with GNInfo do
  begin
    cbSize := SizeOf(TGroupNameInfo);
    nGroupId := GroupId;
    lpName := PWideChar(NewName);
  end;
  Result := FOwnerPlugin.CallService(
    Format(NMS_PROTOCOL_CONNECTION_CHANGEGROUPNAME, [GetProtocol]),
    FHandle, Cardinal(@GNInfo));
end;

function TNsmUIConnection.ChangeMemberGroup(Account: String; NewGroupID: Integer): Integer;
var
  MGInfo: TMemberGroupInfo;
begin
  with MGInfo do
  begin
    cbSize := SizeOf(TMemberGroupInfo);
    lpAccount := PChar(Account);
    nGroupId := NewGroupID;
  end;
  Result := FOwnerPlugin.CallService(
    Format(NMS_PROTOCOL_CONNECTION_CHANGEMEMBERGROUP, [GetProtocol]),
    FHandle, Cardinal(@MGInfo));
end;

function TNsmUIConnection.LogOut: Integer;
begin
  Result := FOwnerPlugin.CallService(
    Format(NMS_PROTOCOL_CONNECTION_DISCONNECT, [GetProtocol]), FHandle, 0);
end;

function TNsmUIConnection.OpenSession(ToAccount: String): Integer;
var
  OSI: TOpenSessionInfo;
begin
  with OSI do
  begin
    cbSize := SizeOf(TOpenSessionInfo);
    lpToAccount := PChar(ToAccount);
  end;
  Result := FOwnerPlugin.CallService(
    Format(NMS_PROTOCOL_SESSION_OPEN, [GetProtocol]), FHandle, Cardinal(@OSI));
end;

// TNsmUIConnections -----------------------------------------------------------

constructor TNsmUIConnections.Create(AOwner: TNsmUIPluginMainEx);
begin
  FOwnerPlugin := AOwner;
  FOwner := AOwner;
  FConnections := TList.Create;
end;

destructor TNsmUIConnections.Destroy;
begin
  Clear;
  FConnections.Free;
  inherited;
end;

function TNsmUIConnections.Add(Handle: HNsmConnection): TNsmUIConnection;
begin
  Result := TNsmUIConnection.Create(Self);
  Result.Handle := Handle;
  FConnections.Add(Result);
end;

procedure TNsmUIConnections.Remove(Handle: HNsmConnection);
var
  I: Integer;
begin
  for I := 0 to FConnections.Count - 1 do
    if TNsmUIConnection(FConnections[I]).Handle = Handle then
    begin
      TNsmUIConnection(FConnections[I]).Free;
      FConnections.Delete(I);
      Break;
    end;
end;

procedure TNsmUIConnections.Clear;
begin
  while FConnections.Count > 0 do
  begin
    TNsmUIConnection(FConnections[0]).Free;
    FConnections.Delete(0);
  end;
end;

function TNsmUIConnections.GetConnection(Idx: Integer): TNsmUIConnection;
begin
  Result := TNsmUIConnection(FConnections[Idx]);
end;

function TNsmUIConnections.GetCount: Integer;
begin
  Result := FConnections.Count;
end;

function TNsmUIConnections.IndexOfHandle(Handle: HNsmConnection): Integer;
var
  I: Integer;
begin
  Result := -1;
  for I := 0 to FConnections.Count - 1 do
    if TNsmUIConnection(FConnections[I]).Handle = Handle then
    begin
      Result := I;
      Break;
    end;
end;

function TNsmUIConnections.Find(Handle: HNsmConnection): TNsmUIConnection;
var
  Idx: Integer;
begin
  Idx := IndexOfHandle(Handle);
  if Idx > -1 then
    Result := FConnections[Idx]
  else
    Result := nil;
end;

// TNsmUISession ---------------------------------------------------------------

constructor TNsmUISession.Create(AOwner: TNsmUISessions);
begin
  FOwnerPlugin := AOwner.OwnerPlugin;
  FOwner := AOwner;
  FMembers := TNsmUIMembers.Create(Self);
end;

destructor TNsmUISession.Destroy;
begin
  FMembers.Free;
  inherited;
end;

function TNsmUISession.GetCaption: WideString;
begin
  Result := FOwnerPlugin.GetInfo.GetSessionInfo(FHandle, NMSI_CAPTION);
end;

function TNsmUISession.GetStatus: Integer;
begin
  Result := FOwnerPlugin.GetInfo.GetSessionInfoInt(FHandle, NMSI_STATUS);
end;

function TNsmUISession.GetConnection: TNsmUIConnection;
var
  Handle: HNsmConnection;
begin
  Result := nil;
  Handle := FOwnerPlugin.GetInfo.GetSessionInfoInt(FHandle, NMSI_CONNECTION);
  if Handle > 0 then
    Result := FOwnerPlugin.Connections.Find(Handle);
end;

function TNsmUISession.GetProtocol: String;
begin
  Result := FOwnerPlugin.GetInfo.GetSessionInfo(FHandle, NMSI_PROTOCOL);
end;

function TNsmUISession.InviteMember(Account: String): Integer;
begin
  Result := FOwnerPlugin.CallService(
      Format(NMS_PROTOCOL_SESSION_INVITEMEMBER, [GetProtocol]),
      FHandle, Cardinal(PChar(Account)));
end;

function TNsmUISession.SendMessage(MsgInfo: TMessageInfo): Integer;
begin
  Result := FOwnerPlugin.CallService(
      NMS_SYSTEM_SESSION_SENDMESSAGE, FHandle, Cardinal(@MsgInfo));
end;

function TNsmUISession.Close: Integer;
begin
  Result := FOwnerPlugin.CallService(
      Format(NMS_PROTOCOL_SESSION_CLOSE, [GetProtocol]), FHandle, 0);
end;

// TNsmUISessions ---------------------------------------------------------------


constructor TNsmUISessions.Create(AOwner: TNsmUIPluginMainEx);
begin
  FOwnerPlugin := AOwner;
  FOwner := AOwner;
  FSessions := TList.Create;
end;

destructor TNsmUISessions.Destroy;
begin
  Clear;
  FSessions.Free;
  inherited;
end;

function TNsmUISessions.Add(SHandle: HNsmSession): TNsmUISession;
begin
  Result := TNsmUISession.Create(Self);
  Result.Handle := SHandle;
  FSessions.Add(Result);
end;

procedure TNsmUISessions.Remove(SHandle: HNsmSession);
var
  I: Integer;
begin
  for I := 0 to FSessions.Count - 1 do
    if TNsmUISession(FSessions[I]).Handle = SHandle then
    begin
      TNsmUISession(FSessions[I]).Free;
      FSessions.Delete(I);
      Break;
    end;
end;

procedure TNsmUISessions.Clear;
begin
  while FSessions.Count > 0 do
  begin
    TNsmUISession(FSessions[0]).Free;
    FSessions.Delete(0);
  end;
end;

function TNsmUISessions.GetSession(Idx: Integer): TNsmUISession;
begin
  Result := TNsmUISession(FSessions[Idx]);
end;

function TNsmUISessions.GetCount: Integer;
begin
  Result := FSessions.Count;
end;

function TNsmUISessions.IndexOfHandle(SHandle: HNsmSession): Integer;
var
  I: Integer;
begin
  Result := -1;
  for I := 0 to FSessions.Count - 1 do
    if TNsmUISession(FSessions[I]).Handle = SHandle then
    begin
      Result := I;
      Break;
    end;
end;

function TNsmUISessions.Find(SHandle: HNsmSession): TNsmUISession;
var
  Idx: Integer;
begin
  Idx := IndexOfHandle(SHandle);
  if Idx > -1 then
    Result := FSessions[Idx]
  else
    Result := nil;
end;

// TNsmUIPluginMainEx ------------------------------------------------------------

constructor TNsmUIPluginMainEx.Create;
begin
  inherited;
  FConnections := TNsmUIConnections.Create(Self);
  FSessions := TNsmUISessions.Create(Self);
end;

destructor TNsmUIPluginMainEx.Destroy;
begin
  FConnections.Free;
  FSessions.Free;
  inherited;
end;

function TNsmUIPluginMainEx.SystemExit: Integer;
begin
  Result := CallService(NMS_SYSTEM_EXIT, 0, 0);
end;

function TNsmUIPluginMainEx.LogIn(Protocol, Account, Password, Caption: String; Status: Integer): Integer;
var
  LInfo: TLogInInfo;
begin
  with LInfo do
  begin
    cbSize := SizeOf(TLogInInfo);
    lpAccount := PChar(Account);
    lpPassword := PChar(Password);
    nStatus := Status;
    lpCaption := PWideChar(WideString(Caption));
  end;
  Result := CallService(
      Format(NMS_PROTOCOL_CONNECTION_CONNECT, [Protocol]),
      Cardinal(@LInfo), 0);
end;

procedure TNsmUIPluginMainEx.DoInitialize;
begin
  inherited;
  FGetInfo := TNsmGetInfo.Create(InitInfo);
  FSetInfo := TNsmSetInfo.Create(InitInfo);
end;

procedure TNsmUIPluginMainEx.DoTerminate;
begin
  inherited;
  FGetInfo.Free;
  FSetInfo.Free;
end;

procedure TNsmUIPluginMainEx.DoCreateConnection(AHandle: HNsmConnection; lParam: Cardinal);
begin
  inherited;
  FConnections.Add(AHandle);
end;

procedure TNsmUIPluginMainEx.DoDeleteConnection(AHandle: HNsmConnection; lParam: Cardinal);
begin
  inherited;
  FConnections.Remove(AHandle);
end;

procedure TNsmUIPluginMainEx.DoAddMember(AHandle: HNsmConnection; AMInfo: TAddMemberInfo);
var
  Idx: Integer;
  List: TNsmUIMembers;
begin
  inherited;
  Idx := FConnections.IndexOfHandle(AHandle);
  if Idx > -1 then
  begin
    case AMInfo.nListKind of
      NMLK_FORWARDLIST : List := FConnections[Idx].Members;
      NMLK_REVERSELIST : List := FConnections[Idx].ReverseMembers;
      NMLK_ALLOWLIST   : List := FConnections[Idx].AllowMembers;
      NMLK_BLOCKLIST   : List := FConnections[Idx].BlockMembers;
    else
      List := FConnections[Idx].Members;
    end;
    List.Add(AMInfo.lpAccount);
  end;
end;

procedure TNsmUIPluginMainEx.DoRemoveMember(AHandle: HNsmConnection; RMInfo: TRemoveMemberInfo);
var
  Idx: Integer;
  List: TNsmUIMembers;
begin
  inherited;
  Idx := FConnections.IndexOfHandle(AHandle);
  if Idx > -1 then
  begin
    case RMInfo.nListKind of
      NMLK_FORWARDLIST : List := FConnections[Idx].Members;
      NMLK_REVERSELIST : List := FConnections[Idx].ReverseMembers;
      NMLK_ALLOWLIST   : List := FConnections[Idx].AllowMembers;
      NMLK_BLOCKLIST   : List := FConnections[Idx].BlockMembers;
    else
      List := FConnections[Idx].Members;
    end;
    List.Remove(RMInfo.lpAccount);
  end;
end;

procedure TNsmUIPluginMainEx.DoAddGroup(AHandle: HNsmConnection; GroupID: Integer);
var
  Idx: Integer;
begin
  inherited;
  Idx := FConnections.IndexOfHandle(AHandle);
  if Idx > -1 then
    FConnections[Idx].Groups.Add(GroupID);
end;

procedure TNsmUIPluginMainEx.DoRemoveGroup(AHandle: HNsmConnection; GroupID: Integer);
var
  Idx: Integer;
begin
  inherited;
  Idx := FConnections.IndexOfHandle(AHandle);
  if Idx > -1 then
    FConnections[Idx].Groups.Remove(GroupID);
end;

procedure TNsmUIPluginMainEx.DoCreateSession(AHandle: HNsmSession; lParam: Cardinal);
begin
  inherited;
  FSessions.Add(AHandle);
end;

procedure TNsmUIPluginMainEx.DoDeleteSession(AHandle: HNsmSession; lParam: Cardinal);
begin
  inherited;
  FSessions.Remove(AHandle);
end;

procedure TNsmUIPluginMainEx.DoSessionAddMember(AHandle: HNsmSession; Account: String);
var
  Idx: Integer;
begin
  inherited;
  Idx := FSessions.IndexOfHandle(AHandle);
  if Idx > -1 then
    FSessions[Idx].Members.Add(Account);
end;

procedure TNsmUIPluginMainEx.DoSessionRemoveMember(AHandle: HNsmSession; Account: String);
var
  Idx: Integer;
begin
  inherited;
  Idx := FSessions.IndexOfHandle(AHandle);
  if Idx > -1 then
    FSessions[Idx].Members.Remove(Account);
end;

end.
 