/*************************************************************************************************
 * Administration Utility for RBBS
 *                                                      Copyright (C) 2003-2004 Mikio Hirabayashi
 * This file is part of RBBS, a personal full-text search system.
 * RBBS is free software; you can redistribute it and/or modify it under the terms of the GNU
 * General Public License as published by the Free Software Foundation; either version 2 of the
 * License, or any later version.
 * RBBS is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with RBBS;
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA.
 *************************************************************************************************/


#include "rbbscommon.h"

#define ENCODING    "UTF-8"              /* encoding of the page */

enum {                                   /* enumeration for the format of output data */
  FTXML,                                 /* original XML */
  FTTEXT,                                /* plain text */
  FTHTML,                                /* human readable HTML */
  FTATOM                                 /* Atom feed */
};


/* global variables */
const char *g_progname = NULL;           /* program name */
const char *g_rbbsuri = NULL;            /* base URI of the RBBS system */
const char *g_artprfx = NULL;            /* prefix of articles in HTML format */
const char *g_dumplang = NULL;           /* language of the dumped data */
const char *g_dumptitle = NULL;          /* title of the dump data */
const char *g_dumpauthor = NULL;         /* author of the dump data */
const char *g_dumpsummary = NULL;        /* summary of the dump data */


/* function prototypes */
int main(int argc, char **argv);
void setglobal(void);
void usage(void);
int runimport(int argc, char **argv);
int runexport(int argc, char **argv);
int rundelete(int argc, char **argv);
int runlist(int argc, char **argv);
int rundump(int argc, char **argv);
int runoptimize(int argc, char **argv);
int runinit(int argc, char **argv);
int procimport(const char *name, const char *file, const char *id);
int procexport(const char *name, const char *id, int ftype);
int procdelete(const char *name, const char *id);
int proclist(const char *name, int sort);
int procdump(const char *name, int ftype);
int procoptimize(const char *name);
int procinit(const char *name);
char *readfiledata(const char *file);
void xprintf(const char *format, ...);


/* main routine */
int main(int argc, char **argv){
  int rv;
  cbstdiobin();
  g_progname = argv[0];
  setglobal();
  if(argc < 2) usage();
  rv = 0;
  if(!strcmp(argv[1], "import")){
    rv = runimport(argc, argv);
  } else if(!strcmp(argv[1], "export")){
    rv = runexport(argc, argv);
  } else if(!strcmp(argv[1], "delete")){
    rv = rundelete(argc, argv);
  } else if(!strcmp(argv[1], "list")){
    rv = runlist(argc, argv);
  } else if(!strcmp(argv[1], "dump")){
    rv = rundump(argc, argv);
  } else if(!strcmp(argv[1], "optimize")){
    rv = runoptimize(argc, argv);
  } else if(!strcmp(argv[1], "init")){
    rv = runinit(argc, argv);
  } else {
    usage();
  }
  return rv;
}


/* set global variables */
void setglobal(void){
  const char *tmp;
  g_rbbsuri = "http://localhost/rbbs/rbbs.cgi";
  g_artprfx = "http://localhost/rbbs/rbbs.cgi?id=";
  g_dumplang = "en";
  g_dumptitle = "RBBS Articles";
  g_dumpauthor = "anonymous";
  g_dumpsummary = "This page includes all articles of the Raving Bulletin Board System.";
  if((tmp = getenv("RBBSURI")) != NULL) g_rbbsuri = tmp;
  if((tmp = getenv("RBBSARTPRFX")) != NULL) g_artprfx = tmp;
  if((tmp = getenv("RBBSDUMPLANG")) != NULL) g_dumplang = tmp;
  if((tmp = getenv("RBBSDUMPTITLE")) != NULL) g_dumptitle = tmp;
  if((tmp = getenv("RBBSDUMPAUTHOR")) != NULL) g_dumpauthor = tmp;
  if((tmp = getenv("RBBSDUMPSUMMARY")) != NULL) g_dumpsummary = tmp;
}


/* print the usage and exit */
void usage(void){
  fprintf(stderr, "%s: administration utility for RBBS\n", g_progname);
  fprintf(stderr, "\n");
  fprintf(stderr, "usage:\n");
  fprintf(stderr, "  %s import [-mod id] name [file]\n", g_progname);
  fprintf(stderr, "  %s export [-ft] [-fh] [-fa] name key\n", g_progname);
  fprintf(stderr, "  %s delete name key\n", g_progname);
  fprintf(stderr, "  %s list [-scd] [-sma] [-smd] name\n", g_progname);
  fprintf(stderr, "  %s dump [-ft] [-fh] [-fa] name\n", g_progname);
  fprintf(stderr, "  %s optimize name\n", g_progname);
  fprintf(stderr, "  %s init name\n", g_progname);
  fprintf(stderr, "\n");
  exit(1);
}


/* parse arguments of import command */
int runimport(int argc, char **argv){
  char *name, *file, *id;
  int i, rv;
  name = NULL;
  file = NULL;
  id = NULL;
  for(i = 2; i < argc; i++){
    if(!name && argv[i][0] == '-'){
      if(!strcmp(argv[i], "-mod")){
        if(++i >= argc) usage();
        id = argv[i];
      } else {
        usage();
      }
    } else if(!name){
      name = argv[i];
    } else if(!file){
      file = argv[i];
    } else {
      usage();
    }
  }
  if(!name) usage();
  rv = procimport(name, file, id);
  return rv;
}


/* parse arguments of import command */
int runexport(int argc, char **argv){
  char *name, *id;
  int i, ftype, rv;
  name = NULL;
  id = NULL;
  ftype = FTXML;
  for(i = 2; i < argc; i++){
    if(!name && argv[i][0] == '-'){
      if(!strcmp(argv[i], "-ft")){
        ftype = FTTEXT;
      } else if(!strcmp(argv[i], "-fh")){
        ftype = FTHTML;
      } else if(!strcmp(argv[i], "-fa")){
        ftype = FTATOM;
      } else {
        usage();
      }
    } else if(!name){
      name = argv[i];
    } else if(!id){
      id = argv[i];
    } else {
      usage();
    }
  }
  if(!name || !id) usage();
  rv = procexport(name, id, ftype);
  return rv;
}


/* parse arguments of delete command */
int rundelete(int argc, char **argv){
  char *name, *id;
  int i, rv;
  name = NULL;
  id = NULL;
  for(i = 2; i < argc; i++){
    if(!name && argv[i][0] == '-'){
      usage();
    } else if(!name){
      name = argv[i];
    } else if(!id){
      id = argv[i];
    } else {
      usage();
    }
  }
  if(!name || !id) usage();
  rv = procdelete(name, id);
  return rv;
}


/* parse arguments of list command */
int runlist(int argc, char **argv){
  char *name;
  int i, sort, rv;
  name = NULL;
  sort = SORTCDATEA;
  for(i = 2; i < argc; i++){
    if(!name && argv[i][0] == '-'){
      if(!strcmp(argv[i], "-scd")){
        sort = SORTCDATED;
      } else if(!strcmp(argv[i], "-sma")){
        sort = SORTMDATEA;
      } else if(!strcmp(argv[i], "-smd")){
        sort = SORTMDATED;
      } else {
        usage();
      }
    } else if(!name){
      name = argv[i];
    } else {
      usage();
    }
  }
  if(!name) usage();
  rv = proclist(name, sort);
  return rv;
}


/* parse arguments of dump command */
int rundump(int argc, char **argv){
  char *name;
  int i, ftype, rv;
  name = NULL;
  ftype = FTXML;
  for(i = 2; i < argc; i++){
    if(!name && argv[i][0] == '-'){
      if(!strcmp(argv[i], "-ft")){
        ftype = FTTEXT;
      } else if(!strcmp(argv[i], "-fh")){
        ftype = FTHTML;
      } else if(!strcmp(argv[i], "-fa")){
        ftype = FTATOM;
      } else {
        usage();
      }
    } else if(!name){
      name = argv[i];
    } else {
      usage();
    }
  }
  if(!name) usage();
  rv = procdump(name, ftype);
  return rv;
}


/* parse arguments of optimize command */
int runoptimize(int argc, char **argv){
  char *name;
  int i, rv;
  name = NULL;
  for(i = 2; i < argc; i++){
    if(!name && argv[i][0] == '-'){
      usage();
    } else if(!name){
      name = argv[i];
    } else {
      usage();
    }
  }
  if(!name) usage();
  rv = procoptimize(name);
  return rv;
}


/* parse arguments of init command */
int runinit(int argc, char **argv){
  char *name;
  int i, rv;
  name = NULL;
  for(i = 2; i < argc; i++){
    if(!name && argv[i][0] == '-'){
      usage();
    } else if(!name){
      name = argv[i];
    } else {
      usage();
    }
  }
  if(!name) usage();
  rv = procinit(name);
  return rv;
}


/* import an article */
int procimport(const char *name, const char *file, const char *id){
  RBBS *rbbs;
  ARTICLE *art;
  char *xml;
  int rv, isdir;
  rv = 0;
  xml = NULL;
  rbbs = NULL;
  if(cbfilestat(name, &isdir, NULL, NULL) && !isdir){
    fprintf(stderr, "%s: %s: not a database\n", g_progname, name);
    rv = 1;
  } else if(!(xml = readfiledata(file))){
    fprintf(stderr, "%s: %s: cannot read\n", g_progname, file ? file : "(stdin)");
    rv = 1;
  } else if(!(rbbs = rbbsopen(name, TRUE))){
    fprintf(stderr, "%s: %s: cannot open\n", g_progname, name);
    rv = 1;
  } else {
    art = makearticle(xml);
    if(!articleisok(art)){
      fprintf(stderr, "%s: %s\n", g_progname, artemsg(art));
      rv = 1;
    } else {
      if(!postarticle(rbbs, art, id)){
        fprintf(stderr, "%s: %s\n", g_progname, rbbsemsg(rbbs));
        rv = 1;
      } else {
        printf("%s: %s: imported as \"%s\"\n",
               name, file ? file : "(stdin)", id ? id : artid(art));
      }
    }
    freearticle(art);
  }
  if(rbbs && !rbbsclose(rbbs)){
    fprintf(stderr, "%s: %s: some errors occurred around the database\n", g_progname, name);
    rv = 1;
  }
  if(xml) free(xml);
  return rv;
}


/* export an article */
int procexport(const char *name, const char *id, int ftype){
  RBBS *rbbs;
  ARTICLE *art;
  char *xml;
  int rv;
  rv = 0;
  rbbs = NULL;
  art = NULL;
  if(!(rbbs = rbbsopen(name, FALSE))){
    fprintf(stderr, "%s: \"%s\" could not open\n", g_progname, name);
    rv = 1;
  } else if(!(art = getarticle(rbbs, id))){
    fprintf(stderr, "%s: %s\n", g_progname, rbbsemsg(rbbs));
    rv = 1;
  } else {
    switch(ftype){
    case FTTEXT:
      xml = arttotext(art);
      printf("%s", xml);
      free(xml);
      break;
    case FTHTML:
      xml = arttohtml(art, 0, 0);
      printf("%s", xml);
      free(xml);
      break;
    case FTATOM:
      xml = arttoatom(art, g_artprfx, 0, 0);
      printf("%s", xml);
      free(xml);
      break;
    default:
      xml = arttoxml(art);
      printf("%s", xml);
      free(xml);
      break;
    }
  }
  if(art) freearticle(art);
  if(rbbs && !rbbsclose(rbbs)){
    fprintf(stderr, "%s: %s: some errors occurred around the database\n", g_progname, name);
    rv = 1;
  }
  return rv;
}


/* delete an article */
int procdelete(const char *name, const char *id){
  RBBS *rbbs;
  ARTICLE *art;
  int rv;
  rv = 0;
  rbbs = NULL;
  art = NULL;
  if(!(rbbs = rbbsopen(name, TRUE))){
    fprintf(stderr, "%s: \"%s\" could not open\n", g_progname, name);
    rv = 1;
  } else if(!deletearticle(rbbs, id)){
    fprintf(stderr, "%s: %s\n", g_progname, rbbsemsg(rbbs));
    rv = 1;
  } else {
    printf("%s: %s: deleted\n", g_progname, id);
  }
  if(rbbs && !rbbsclose(rbbs)){
    fprintf(stderr, "%s: %s: some errors occurred around the database\n", g_progname, name);
    rv = 1;
  }
  return rv;
}


/* list ID numbers of articles */
int proclist(const char *name, int sort){
  RBBS *rbbs;
  char *id;
  int rv;
  rv = 0;
  rbbs = NULL;
  if(!(rbbs = rbbsopen(name, FALSE))){
    fprintf(stderr, "%s: \"%s\" could not open\n", g_progname, name);
    rv = 1;
  } else {
    setsortorder(rbbs, sort);
    while((id = getnextid(rbbs)) != NULL){
      printf("%s\n", id);
      free(id);
    }
  }
  if(rbbs && !rbbsclose(rbbs)){
    fprintf(stderr, "%s: %s: some errors occurred around the database\n", g_progname, name);
    rv = 1;
  }
  return rv;
}


/* dump all articles */
int procdump(const char *name, int ftype){
  RBBS *rbbs;
  ARTICLE *art;
  char *tmp, *id, *xml;
  int rv;
  rv = 0;
  rbbs = NULL;
  if(!(rbbs = rbbsopen(name, FALSE))){
    fprintf(stderr, "%s: \"%s\" could not open\n", g_progname, name);
    rv = 1;
  } else {
    setsortorder(rbbs, SORTCDATEA);
    switch(ftype){
    case FTTEXT:
      printf("%s\n", g_dumplang);
      printf("%s\n", g_dumptitle);
      printf("%s\n", g_dumpauthor);
      printf("%s\n", g_dumpsummary);
      break;
    case FTHTML:
      xprintf("<?xml version=\"1.0\" encoding=\"%@\"?>\n", ENCODING);
      xprintf("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\""
              " \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">\n");
      xprintf("<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"%@\" lang=\"%@\">\n",
              g_dumplang, g_dumplang);
      xprintf("<head>\n");
      xprintf("<title>%@</title>\n", g_dumptitle);
      xprintf("<meta http-equiv=\"Content-Type\" content=\"text/html; charset=%@\" />\n",
              ENCODING);
      xprintf("<meta name=\"author\" content=\"%@\" />\n", g_dumpauthor);
      xprintf("<meta name=\"generator\" content=\"RBBS %@\" />\n", RBBS_VERSION);
      xprintf("<link rel=\"contents\" href=\"%@\" />\n", g_rbbsuri);
      xprintf("</head>\n");
      xprintf("<body>\n");
      xprintf("<h1>%@</h1>\n", g_dumptitle);
      xprintf("<p>%@</p>\n", g_dumpsummary);
      xprintf("<hr />\n");
      break;
    case FTATOM:
      xprintf("<?xml version=\"1.0\" encoding=\"%@\"?>\n", ENCODING);
      xprintf("<feed version=\"0.3\" xmlns=\"http://purl.org/atom/ns#\" xml:lang=\"%@\">\n",
              g_dumplang);
      xprintf("<link rel=\"alternate\" type=\"text/html\" href=\"%@\"/>\n", g_rbbsuri);
      xprintf("<title>%@</title>\n", g_dumptitle);
      xprintf("<author>\n");
      xprintf("<name>%@</name>\n", g_dumpauthor);
      xprintf("</author>\n");
      tmp = dateforwww(rbbsmdate(rbbs), FALSE);
      xprintf("<modified>%@</modified>\n", tmp);
      free(tmp);
      xprintf("<tagline>%@</tagline>\n", g_dumpsummary);
      xprintf("<generator name=\"RBBS\">%@?version=%?</generator>\n", RBBS_URI, RBBS_VERSION);
      break;
    default:
      xprintf("<?xml version=\"1.0\" encoding=\"%@\"?>\n", ENCODING);
      xprintf("<!DOCTYPE rbbs SYSTEM \"rbbs.dtd\">\n");
      xprintf("<rbbs version=\"%@\" cdate=\"%@\" mdate=\"%@\">\n",
              RBBS_VERSION, rbbscdate(rbbs), rbbsmdate(rbbs));
      break;
    }
    while((id = getnextid(rbbs)) != NULL){
      if(!(art = getarticle(rbbs, id))){
        fprintf(stderr, "%s: %s\n", g_progname, rbbsemsg(rbbs));
        rv = 1;
      }
      switch(ftype){
      case FTTEXT:
        printf("-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-\n");
        xml = arttotext(art);
        printf("%s\n", xml);
        free(xml);
        break;
      case FTHTML:
        xml = arttohtml(art, 0, 0);
        printf("%s", xml);
        free(xml);
        xprintf("<hr />\n");
        break;
      case FTATOM:
        xml = arttoatom(art, g_artprfx, 0, 0);
        printf("%s", xml);
        free(xml);
        break;
      default:
        xml = arttoxml(art);
        printf("%s", xml);
        free(xml);
        break;
      }
      if(art) freearticle(art);
      free(id);
    }
    switch(ftype){
    case FTTEXT:
      printf("-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-\n");
      break;
    case FTHTML:
      xprintf("<div class=\"rbbsversion\">Powered by RBBS %@</div>", RBBS_VERSION);
      xprintf("</body>\n");
      xprintf("</html>\n");
      break;
    case FTATOM:
      xprintf("</feed>\n");
      break;
    default:
      xprintf("</rbbs>\n");
      break;
    }
  }
  if(rbbs && !rbbsclose(rbbs)){
    fprintf(stderr, "%s: %s: some errors occurred around the database\n", g_progname, name);
    rv = 1;
  }
  return rv;
}


/* optimize a database */
int procoptimize(const char *name){
  RBBS *rbbs;
  int rv;
  rv = 0;
  rbbs = NULL;
  if(!(rbbs = rbbsopen(name, TRUE))){
    fprintf(stderr, "%s: \"%s\" could not open\n", g_progname, name);
    rv = 1;
  } else if(!rbbsoptimize(rbbs)){
    fprintf(stderr, "%s: %s\n", g_progname, rbbsemsg(rbbs));
    rv = 1;
  } else {
    printf("%s: %s: optimized\n", g_progname, name);
  }
  if(rbbs && !rbbsclose(rbbs)){
    fprintf(stderr, "%s: %s: some errors occurred around the database\n", g_progname, name);
    rv = 1;
  }
  return rv;
}


/* remove a database */
int procinit(const char *name){
  RBBS *rbbs;
  int rv;
  rv = 0;
  rbbs = NULL;
  rbbsremove(name);
  if(!(rbbs = rbbsopen(name, TRUE))){
    fprintf(stderr, "%s: \"%s\" could not open\n", g_progname, name);
    rv = 1;
  } else {
    printf("%s: %s: initialized\n", g_progname, name);
  }
  if(rbbs && !rbbsclose(rbbs)){
    fprintf(stderr, "%s: %s: some errors occurred around the database\n", g_progname, name);
    rv = 1;
  }
  return rv;
}


/* read whole data of a file */
char *readfiledata(const char *file){
  CBDATUM *buf;
  char cc;
  int c;
  if(file) return cbreadfile(file, NULL);
  buf = cbdatumopen("", 0);
  while((c = getchar()) != EOF){
    cc = (char)c;
    cbdatumcat(buf, &cc, 1);
  }
  return cbdatumtomalloc(buf, NULL);
}


/* XML-oriented printf */
void xprintf(const char *format, ...){
  va_list ap;
  char *tmp;
  unsigned char c;
  va_start(ap, format);
  while(*format != '\0'){
    if(*format == '%'){
      format++;
      switch(*format){
      case 's':
        tmp = va_arg(ap, char *);
        if(!tmp) tmp = "(null)";
        printf("%s", tmp);
        break;
      case 'd':
        printf("%d", va_arg(ap, int));
        break;
      case '@':
        tmp = va_arg(ap, char *);
        if(!tmp) tmp = "(null)";
        while(*tmp){
          switch(*tmp){
          case '&': printf("&amp;"); break;
          case '<': printf("&lt;"); break;
          case '>': printf("&gt;"); break;
          case '"': printf("&quot;"); break;
          default:
            if(!((*tmp >= 0 && *tmp <= 0x8) || (*tmp >= 0x0e && *tmp <= 0x1f))) putchar(*tmp);
            break;
          }
          tmp++;
        }
        break;
      case '?':
        tmp = va_arg(ap, char *);
        if(!tmp) tmp = "(null)";
        while(*tmp){
          c = *(unsigned char *)tmp;
          if((c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z') ||
             (c >= '0' && c <= '9') || (c != '\0' && strchr("_-.", c))){
            putchar(c);
          } else {
            printf("%%%02X", c);
          }
          tmp++;
        }
        break;
      case '%':
        putchar('%');
        break;
      }
    } else {
      putchar(*format);
    }
    format++;
  }
  va_end(ap);
}



/* END OF FILE */
