/*************************************************************************************************
 * Interface of common features
 *                                                      Copyright (C) 2003-2004 Mikio Hirabayashi
 * This file is part of RBBS, a personal full-text search system.
 * RBBS is free software; you can redistribute it and/or modify it under the terms of the GNU
 * General Public License as published by the Free Software Foundation; either version 2 of the
 * License, or any later version.
 * RBBS is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with RBBS;
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA.
 *************************************************************************************************/


#ifndef _RBBSCOMMON_H                    /* duplication check */
#define _RBBSCOMMON_H



/*************************************************************************************************
 * headers
 *************************************************************************************************/


#include <depot.h>
#include <cabin.h>
#include <villa.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <stdarg.h>
#include <limits.h>
#include <time.h>
#include <signal.h>

#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>



/*************************************************************************************************
 * macros
 *************************************************************************************************/


#ifndef RBBS_VERSION                     /* version and URI */
#define RBBS_VERSION   "0.0.0"
#endif
#define RBBS_URI       "http://rbbs.sourceforge.net/"

#undef TRUE                              /* boolean true */
#define TRUE           1
#undef FALSE                             /* boolean false */
#define FALSE          0

#define ARTDBNAME      "artdb"           /* name of the database for articles */
#define CDIDXNAME      "cdidx"           /* name of the index for modification date */
#define MDIDXNAME      "mdidx"           /* name of the index for creation date */
#define CDATEKEY       "cdate"           /* key of the creation date */
#define MDATEKEY       "mdate"           /* key of the modification date */
#define PATHBUFSIZ     2048              /* size of a path buffer */
#define NUMBUFSIZ      32                /* size of a number buffer */
#define HASHNUM        7                 /* number of hash classes */
#define LONESTR        "lone"            /* string for lone articles */
#define LONECHAR       'l'               /* character for lone articles */
#define SINKSTR        "sink"            /* string for sinking responses */
#define SINKCHAR       's'               /* character for sinking responses */



/*************************************************************************************************
 * types and constants
 *************************************************************************************************/


typedef struct {                         /* type of structure for a database handle */
  DEPOT *artdb;                          /* datebase for articles */
  VILLA *cdidx;                          /* database for the creation date */
  VILLA *mdidx;                          /* database for the modification date */
  VILLA *idx;                            /* temporary index */
  int (*next)(VILLA *);                  /* temporary cursor function */
  char *cdate;                           /* creation date */
  char *mdate;                           /* modification date */
  char *errmsg;                          /* error message */
} RBBS;

typedef struct {                         /* type of structure for an article */
  char *id;                              /* identification */
  char *flags;                           /* flags for multi purpose */
  char *language;                        /* language */
  char *category;                        /* category */
  char *subject;                         /* subject */
  char *author;                          /* author */
  char *cdate;                           /* creation date */
  char *mdate;                           /* modification date */
  CBLIST *body;                          /* list of blocks in the body */
  CBLIST *tail;                          /* list of responses in the body */
  char *errmsg;                          /* error message */
} ARTICLE;

enum {                                   /* enumeration for sorting orders */
  SORTCDATEA,                            /* ascending order of the creation date */
  SORTCDATED,                            /* descending order of the creation date */
  SORTMDATEA,                            /* ascending order of the modification date */
  SORTMDATED                             /* descending order of the modification date */
};



/*************************************************************************************************
 * functions
 *************************************************************************************************/


/* Get a handle connected to the databases.
   `name' specities the path of a directory containing database files.
   `writer' specifies the handle is a writer or not.
   The return value is a handle connected to the databases or `NULL' if an error occures. */
RBBS *rbbsopen(const char *name, int writer);


/* Release a database handle.
   `rbbs' specifies a database handle.
   The return value is true if successful, else it is false. */
int rbbsclose(RBBS *rbbs);


/* Remove a database directory.
   `name' specities the path of a directory containing database files.
   The return value is true if successful, else it is false. */
int rbbsremove(const char *name);


/* Get the error message of a database.
   `rbbs' specifies a database handle.
   The return value is the error message of the database. */
const char *rbbsemsg(RBBS *rbbs);


/* Get the creation date of a database.
   `rbbs' specifies a database handle.
   The return value is the string of the creation date of the database. */
const char *rbbscdate(RBBS *rbbs);


/* Get the modification date of a database.
   `rbbs' specifies a database handle.
   The return value is the string of the modification date of the database. */
const char *rbbsmdate(RBBS *rbbs);


/* Get the number of articles in a database.
   `rbbs' specifies a database handle.
   The return value is the number of articles in the database. */
int rbbsanum(RBBS *rbbs);


/* Optimize a database.
   `rbbs' specifies a database handle.
   The return value is true if successful, else it is false. */
int rbbsoptimize(RBBS *rbbs);


/* Set the sorting order of a database.
   `rbbs' specifies a database handle.
   `sort' specifies the sorting order. */
void setsortorder(RBBS* rbbs, int sort);


/* Get the ID string of the next article.
   `rbbs' specifies a database handle.
   The return value is the string of the ID string of the next article.  If there is no next
   article, it is `NULL'.  */
char *getnextid(RBBS *rbbs);


/* Retrieve article from a database.
   `rbbs' specifies a database handle.
   `id' specifies the string of the ID of the exisging article.
   The return value is a handle of an article object or `NULL' if error. */
ARTICLE *getarticle(RBBS *rbbs, const char *id);


/* Post a new article and store it into a database.
   `rbbs' specifies a database handle.
   `art' specifies a handle of an article object.
   `id' specifies the string of the ID of the exisging article.  If it is `NULL', the article is
   treated as a new article.
   The return value is true if successful, else it is false. */
int postarticle(RBBS *rbbs, ARTICLE *art, const char *id);


/* Delete an article existing in a database.
   `rbbs' specifies a database handle.
   `id' specifies the string of the ID of the exisging article.
   The return value is true if successful, else it is false. */
int deletearticle(RBBS *rbbs, const char *id);


/* Make a object of article from an XML string.
   `xml' specifies a string of XML.
   The return value is a handle of an article object. */
ARTICLE *makearticle(const char *xml);


/* Release resources of an article object.
   `art' specifies a handle of an article object. */
void freearticle(ARTICLE *art);


/* Check the article is OK.
   `art' specifies a handle of an article object.
   The return value is true if the article is OK, else it is false. */
int articleisok(ARTICLE *art);


/* Get the error message of an article.
   `art' specifies a handle of an article object.
   The return value is the error message of the article. */
const char *artemsg(ARTICLE *art);


/* Get the ID string of an article.
   `art' specifies a handle of an article object.
   The return value is the ID string of the article. */
const char *artid(ARTICLE *art);


/* Get the flags of an article.
   `art' specifies a handle of an article object.
   The return value is the flagst of the article. */
const char *artflags(ARTICLE *art);


/* Get the language of an article.
   `art' specifies a handle of an article object.
   The return value is the language of the article. */
const char *artlanguage(ARTICLE *art);


/* Get the category of an article.
   `art' specifies a handle of an article object.
   The return value is the category of the article. */
const char *artcategory(ARTICLE *art);


/* Get the subject of an article.
   `art' specifies a handle of an article object.
   The return value is the subject of the article. */
const char *artsubject(ARTICLE *art);


/* Get the author of an article.
   `art' specifies a handle of an article object.
   The return value is the author of the article. */
const char *artauthor(ARTICLE *art);


/* Get the creation date of an article.
   `art' specifies a handle of an article object.
   The return value is the creation date of the article. */
const char *artcdate(ARTICLE *art);


/* Get the modification date of an article.
   `art' specifies a handle of an article object.
   The return value is the modification date of the article. */
const char *artmdate(ARTICLE *art);


/* Get the number of responses of an article.
   `art' specifies a handle of an article object.
   The return value is the number of responses of the article. */
int artresnum(ARTICLE *art);


/* Add a response to an article.
   `art' specifies a handle of an article object.
   `date' specifies the string of a date. */
void artaddres(ARTICLE *art, const char *flags, const char *author, const char *cdate,
               const char *text);


/* Set the creation date of an article.
   `art' specifies a handle of an article object.
   `date' specifies the creation date the response. */
void artsetcdate(ARTICLE *art, const char *date);


/* Set the modification date of an article.
   `art' specifies a handle of an article object.
   `date' specifies the modification date the response. */
void artsetmdate(ARTICLE *art, const char *date);


/* Serialize an article into a XML string.
   `art' specifies a handle of an article object.
   The return value is a string of XML. */
char *arttoxml(ARTICLE *art);


/* Serialize an article into a string of plain text.
   `art' specifies a handle of an article object.
   The return value is a string of plain text. */
char *arttotext(ARTICLE *art);


/* Serialize an article into a string of HTML.
   `art' specifies a handle of an article object.
   `bnum' specifies the number of shown blocks in the body.  If it is zero, all responses are
   shown.  If it is negative, the absolute value is calculated and the lower blocks are shown.
   `rnum' specifies the number of shown responses.  If it is zero, all responses are shown.
   If it is negative, the absolute value is calculated and the lower responses are shown.
   The return value is a string of HTML. */
char *arttohtml(ARTICLE *art, int bnum, int rnum);


/* Serialize an article into a string of Atom.
   `art' specifies a handle of an article object.
   `prefix' specifies the prefix of the URI of the generator of the alternative HTML content.
   `bnum' specifies the number of shown blocks in the body.  If it is zero, all responses are
   shown.  If it is negative, the absolute value is calculated and the lower blocks are shown.
   `rnum' specifies the number of shown responses.  If it is zero, all responses are shown.
   If it is negative, the absolute value is calculated and the lower responses are shown.
   The return value is a string of Atom. */
char *arttoatom(ARTICLE *art, const char *prefix, int bnum, int rnum);


/* Create the string for a current date in internal form.
   `date' specifies the string of a date in internal form.
   The return value is the string of the current date in internal form. */
char *datecurrent(void);


/* Create the string for a date in W3DTF.
   `date' specifies the string of a date in internal form.
   `local' specifies whether the expression is for the local timezone or not.
   The return value is the string of the date in W3DTF (YYYY-MM-DDThh:mm:ddTZD). */
char *dateforwww(const char *date, int local);


/* Create the string for a date in RFC1123.
   `date' specifies the string of a date in internal form.
   `local' specifies whether the expression is for the local timezone or not.
   The return value is the string of the date in RFC1123 (Wdy, DD-Mon-YYYY hh:mm:dd TZD). */
char *dateforhttp(const char *date, int local);


/* Create the string for a date which is human readable.
   `date' specifies the string of a date in internal form.
   The return value is the string of the date for human readable (YYYY/MM/DD hh:mm). */
char *dateformen(const char *date);


/* Normalize a string.
   `str' specifies a string.
   `nolf' specifies whether to convert a line feed to a space.
   `trim' specifies whether to cut the heading spaces and tailing spaces.
   The return value is a converted string. */
char *strnormalize(const char *str, int nolf, int trim);


/* Perform formatted conversion and write the result into a buffer.
   `buf' specifies a datum handle for a result buffer.
   `format' specifies a format string.  The format is as with `printf'.  Meta characters `c', `s',
   `d', `@'(XML escaped string), `?'(URL escaped string), `%' are available. */
void xsprintf(CBDATUM *buf, const char *format, ...);



#endif                                   /* duplication check */


/* END OF FILE */
