/*************************************************************************************************
 * Raving Bulletin Board System
 *                                                      Copyright (C) 2003-2004 Mikio Hirabayashi
 * This file is part of RBBS, a personal full-text search system.
 * RBBS is free software; you can redistribute it and/or modify it under the terms of the GNU
 * General Public License as published by the Free Software Foundation; either version 2 of the
 * License, or any later version.
 * RBBS is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with RBBS;
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA.
 *************************************************************************************************/


#include "rbbscommon.h"

#define CONFFILE    "rbbs.conf"          /* name of the configuration file */
#define ENCODING    "UTF-8"              /* encoding of the page */
#define PETITBNUM   31                   /* bucket number of a petit map */
#define RDATAMAX    1048576              /* max size of data to read */
#define PAGENAME    "rbbs.html"          /* saving filename of the page */
#define FEEDNAME    "rbbs.xml"           /* saving filename of the feed */
#define FEEDCDATE   "cdate"              /* ID for feed ordered by creation date */
#define FEEDMDATE   "mdate"              /* ID for feed ordered by modification date */
#define DUMPNAME    "rbbsdump.xml"       /* saving filename of the dump */
#define DUMPDBCMD   "dumpdb"             /* command name to dump all articles */
#define DELEDBCMD   "deledb"             /* command name to delete the database */
#define SINKPREFIX  SINKSTR ":"          /* prefix for sinking responses */

enum {                                   /* enumeration for actions */
  ACTVIEW,                               /* show articles */
  ACTLIST,                               /* show list of articles */
  ACTSEARCH,                             /* show the search form and its result */
  ACTARTFORM,                            /* show the form to post an article */
  ACTARTCNFM,                            /* confirm an article */
  ACTPOSTART,                            /* post an article */
  ACTDELEART,                            /* delete an article */
  ACTPOSTRES,                            /* post a response */
  ACTDUMPDB,                             /* dump all articles */
  ACTDELEDB,                             /* delete the database */
  ACTHELP                                /* show the help page */
};

enum {                                   /* enumeration for focusing article */
  FCSNORMAL,                             /* show the body and a part of responses */
  FCSDETAIL,                             /* show the body and all responses */
  FCSSIMPLE                              /* show a part of responses */
};


/* global variables for configurations */
const char *g_systemuri = NULL;          /* URI of this system */
const char *g_dbname = NULL;             /* name of the database */
const char *g_helpfile = NULL;           /* path of the help file */
const char *g_stylefile = NULL;          /* path of the style file */
const char *g_pagelang = NULL;           /* language of the page */
const char *g_pagetitle = NULL;          /* title of the page */
const char *g_pageauthor = NULL;         /* author of the page */
const char *g_pagesummary = NULL;        /* summary of the page */
const char *g_homepageuri = NULL;        /* URI of the home page */
const char *g_adminpass = NULL;          /* password for administration */
const char *g_writepass = NULL;          /* password for writing */
const char *g_artflags = NULL;           /* default flags for a new article */
const char *g_resflags = NULL;           /* default flags for a new response */
const char *g_languages = NULL;          /* list of supported languages */
const char *g_categories = NULL;         /* list of supported categories */
int g_sort = 0;                          /* default sorting order of articles */
int g_listnum = 0;                       /* number of shown items in the subject list */
int g_shownum = 0;                       /* number of shown articles in the page */
int g_feednum = 0;                       /* number of shown articles in the feed */
int g_blknum = 0;                        /* number of shown blocks for an article */
int g_resnum = 0;                        /* number of shown responses for an article */
const char *g_homelabel = NULL;          /* label to go to the home page */
const char *g_helplabel = NULL;          /* label to show the help page */
const char *g_postlabel = NULL;          /* label to post a new article */
const char *g_feedlabel = NULL;          /* label to get the feed the page or an article */
const char *g_reloadlabel = NULL;        /* label to reload the page */
const char *g_searchlabel = NULL;        /* label to search articles */
const char *g_listlabel = NULL;          /* label to show list of all articles */
const char *g_prevlabel = NULL;          /* label to go to the previous page */
const char *g_nextlabel = NULL;          /* label to go to the next page */
const char *g_editlabel = NULL;          /* label to edit the existing */
const char *g_focuslabel = NULL;         /* label to focus the page or an article */
const char *g_passwordlabel = NULL;      /* label for the password */
const char *g_upstamplabel = NULL;       /* label for the update the time stamp */
const char *g_idlabel = NULL;            /* label for the ID string of an article */
const char *g_sourcelabel = NULL;        /* label for the raw XML of an article */
const char *g_languagelabel = NULL;      /* label for the language of an article */
const char *g_categorylabel = NULL;      /* label for the category of an article */
const char *g_subjectlabel = NULL;       /* label for the subject of an article */
const char *g_authorlabel = NULL;        /* label for the author of an article */
const char *g_cdatelabel = NULL;         /* label for creation date */
const char *g_mdatelabel = NULL;         /* label for modification date */
const char *g_textlabel = NULL;          /* label for the body text of an article */
const char *g_ftslabel = NULL;           /* label for full-text search */
const char *g_orderlabel = NULL;         /* label for showing order */
const char *g_descordlabel = NULL;       /* label for descending order */
const char *g_submitlabel = NULL;        /* label for general submit button */
const char *g_resetlabel = NULL;         /* label for general reset button */
const char *g_backlabel = NULL;          /* label for general back button */
const char *g_anykindlabel = NULL;       /* label for selection of any kind */


/* global variables for parameters and cookies */
int p_action = ACTVIEW;                  /* number of the action */
int p_page = 0;                          /* number of skipping pages */
int p_focus = FCSNORMAL;                 /* number of the focusing mode */
int p_sort = -1;                         /* sorting order of articles */
int p_upstamp = FALSE;                   /* whether to update the modification time */
const char *p_id = NULL;                 /* ID of an article */
const char *p_password = NULL;           /* password for administration or writing */
const char *p_language = NULL;           /* language of an article */
const char *p_category = NULL;           /* category of an article */
const char *p_subject = NULL;            /* subject of an article */
const char *p_author = NULL;             /* author of an article */
const char *p_text = NULL;               /* body text of an article */
const char *p_feed = NULL;               /* ID of an article for Atom feed */
const char *c_author = NULL;             /* author in the cookie */


/* other global variables */
RBBS *g_rbbs = NULL;                     /* database handler */
const char *g_curdate = NULL;            /* current date */
int g_rmpost = FALSE;                     /* whether the request method is POST */
int g_cannext = FALSE;                   /* whether the next page can be shown */
int g_tabidx = 0;                        /* counter for tab indexes */


/* function prototypes */
int main(int argc, char **argv);
const char *skiplabel(const char *str);
CBMAP *getparameters(void);
CBMAP *getcookies(void);
void xprintf(const char *format, ...);
void senderror(const char *msg);
void sendnotfound(void);
void senddump(void);
void sendfeed(void);
void sendhtml(void);
int canadmin(void);
int canwrite(void);
int authorize(int admin);
void shownote(const char *msg);
void showpagenavi(void);
void showcaption(void);
int matchingart(ARTICLE *art);
void showlist(void);
void showsearchform(void);
void showarticles(void);
void showartwithctl(ARTICLE *art);
void showartform(void);
void showartconfirm(void);
void showartformback(void);
void asiscat(CBDATUM *buf, const char *text);
void mypostarticle(void);
void mydeletearticle(void);
int mypostresponse(void);
void deletedb(void);
void showhelp(void);


/* main routine */
int main(int argc, char **argv){
  CBLIST *lines;
  CBMAP *params, *cooks;
  const char *tmp;
  char *buf;
  int i;
  /* set configurations */
  signal(SIGINT, SIG_IGN);
  signal(SIGQUIT, SIG_IGN);
  signal(SIGPIPE, SIG_IGN);
  signal(SIGTERM, SIG_IGN);
  signal(SIGTSTP, SIG_IGN);
  cbstdiobin();
  if(!(lines = cbreadlines(CONFFILE))) senderror("the configuration file is missing.");
  cbglobalgc(lines, (void (*)(void *))cblistclose);
  for(i = 0; i < cblistnum(lines); i++){
    tmp = cblistval(lines, i, NULL);
    if(cbstrfwimatch(tmp, "systemuri:")){
      g_systemuri = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "dbname:")){
      g_dbname = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "helpfile:")){
      g_helpfile = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "stylefile:")){
      g_stylefile = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "pagelang:")){
      g_pagelang = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "pagetitle:")){
      g_pagetitle = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "pageauthor:")){
      g_pageauthor = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "pagesummary:")){
      g_pagesummary = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "homepageuri:")){
      g_homepageuri = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "adminpass:")){
      g_adminpass = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "writepass:")){
      g_writepass = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "artflags:")){
      g_artflags = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "resflags:")){
      g_resflags = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "languages:")){
      g_languages = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "categories:")){
      g_categories = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "sort:")){
      tmp = skiplabel(tmp);
      if(!strcmp(tmp, "cda")){
        g_sort = SORTCDATEA;
      } else if(!strcmp(tmp, "cdd")){
        g_sort = SORTCDATED;
      } else if(!strcmp(tmp, "mda")){
        g_sort = SORTMDATEA;
      } else if(!strcmp(tmp, "mdd")){
        g_sort = SORTMDATED;
      }
    } else if(cbstrfwimatch(tmp, "listnum:")){
      g_listnum = atoi(skiplabel(tmp));
    } else if(cbstrfwimatch(tmp, "shownum:")){
      g_shownum = atoi(skiplabel(tmp));
    } else if(cbstrfwimatch(tmp, "feednum:")){
      g_feednum = atoi(skiplabel(tmp));
    } else if(cbstrfwimatch(tmp, "blknum:")){
      g_blknum = atoi(skiplabel(tmp));
    } else if(cbstrfwimatch(tmp, "resnum:")){
      g_resnum = atoi(skiplabel(tmp));
    } else if(cbstrfwimatch(tmp, "homelabel:")){
      g_homelabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "helplabel:")){
      g_helplabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "postlabel:")){
      g_postlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "feedlabel:")){
      g_feedlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "reloadlabel:")){
      g_reloadlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "searchlabel:")){
      g_searchlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "listlabel:")){
      g_listlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "prevlabel:")){
      g_prevlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "nextlabel:")){
      g_nextlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "editlabel:")){
      g_editlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "focuslabel:")){
      g_focuslabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "passwordlabel:")){
      g_passwordlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "upstamplabel:")){
      g_upstamplabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "idlabel:")){
      g_idlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "sourcelabel:")){
      g_sourcelabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "languagelabel:")){
      g_languagelabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "categorylabel:")){
      g_categorylabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "subjectlabel:")){
      g_subjectlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "authorlabel:")){
      g_authorlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "cdatelabel:")){
      g_cdatelabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "mdatelabel:")){
      g_mdatelabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "textlabel:")){
      g_textlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "ftslabel:")){
      g_ftslabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "orderlabel:")){
      g_orderlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "descordlabel:")){
      g_descordlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "submitlabel:")){
      g_submitlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "resetlabel:")){
      g_resetlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "backlabel:")){
      g_backlabel = skiplabel(tmp);
    } else if(cbstrfwimatch(tmp, "anykindlabel:")){
      g_anykindlabel = skiplabel(tmp);
    }
  }
  if(!g_systemuri) senderror("systemuri is undefined.");
  if(!g_dbname) senderror("dbname is undefined.");
  if(!g_helpfile) senderror("helpfile is undefined.");
  if(!g_stylefile) senderror("stylefile is undefined.");
  if(!g_pagelang) senderror("pagelang is undefined.");
  if(!g_pagetitle) senderror("pagetitle is undefined.");
  if(!g_pageauthor) senderror("pageauthor is undefined.");
  if(!g_pagesummary) senderror("pagesummary is undefined.");
  if(!g_homepageuri) senderror("homepageuri is undefined.");
  if(!g_adminpass) senderror("adminpass is undefined.");
  if(!g_writepass) senderror("writepass is undefined.");
  if(!g_artflags) senderror("artflags is undefined.");
  if(!g_resflags) senderror("resflags is undefined.");
  if(!g_languages) senderror("languages is undefined.");
  if(!g_categories) senderror("categories is undefined.");
  if(g_sort < SORTCDATEA || g_sort >  SORTMDATED) g_sort = SORTCDATEA;
  if(g_listnum < 0) g_listnum = 0;
  if(g_shownum < 0) g_shownum = 0;
  if(g_feednum < 0) g_feednum = 0;
  if(!g_homelabel) senderror("homelabel is undefined.");
  if(!g_helplabel) senderror("helplabel is undefined.");
  if(!g_postlabel) senderror("postlabel is undefined.");
  if(!g_feedlabel) senderror("feedlabel is undefined.");
  if(!g_reloadlabel) senderror("reloadlabel is undefined.");
  if(!g_searchlabel) senderror("searchlabel is undefined.");
  if(!g_listlabel) senderror("listlabel is undefined.");
  if(!g_prevlabel) senderror("prevlabel is undefined.");
  if(!g_nextlabel) senderror("nextlabel is undefined.");
  if(!g_editlabel) senderror("editlabel is undefined.");
  if(!g_focuslabel) senderror("focuslabel is undefined.");
  if(!g_passwordlabel) senderror("passwordlabel is undefined");
  if(!g_upstamplabel) senderror("upstamplabel is undefined");
  if(!g_idlabel) senderror("idlabel is undefined");
  if(!g_sourcelabel) senderror("sourcelabel is undefined");
  if(!g_languagelabel) senderror("languagelabel is undefined");
  if(!g_categorylabel) senderror("categorylabel is undefined");
  if(!g_subjectlabel) senderror("subjectlabel is undefined");
  if(!g_authorlabel) senderror("authorlabel is undefined");
  if(!g_cdatelabel) senderror("cdatelabel is undefined");
  if(!g_mdatelabel) senderror("mdatelabel is undefined");
  if(!g_textlabel) senderror("textlabel is undefined");
  if(!g_ftslabel) senderror("ftslabel is undefined");
  if(!g_orderlabel) senderror("orderlabel is undefined");
  if(!g_descordlabel) senderror("descordlabel is undefined");
  if(!g_submitlabel) senderror("submitlabel is undefined");
  if(!g_resetlabel) senderror("resetlabel is undefined");
  if(!g_backlabel) senderror("backlabel is undefined");
  if(!g_anykindlabel) senderror("anykindlabel is undefined");
  /* read parameters */
  params = getparameters();
  cbglobalgc(params, (void (*)(void *))cbmapclose);
  if((tmp = cbmapget(params, "action", -1, NULL)) != NULL){
    if(!strcmp(tmp, DUMPDBCMD)){
      p_action = ACTDUMPDB;
    } else if(!strcmp(tmp, DELEDBCMD)){
      p_action = ACTDELEDB;
    } else {
      p_action = atoi(tmp);
    }
  }
  if((tmp = cbmapget(params, "page", -1, NULL)) != NULL) p_page = atoi(tmp);
  if((tmp = cbmapget(params, "focus", -1, NULL)) != NULL) p_focus = atoi(tmp);
  if((tmp = cbmapget(params, "sort", -1, NULL)) != NULL) p_sort = atoi(tmp);
  if((tmp = cbmapget(params, "upstamp", -1, NULL)) != NULL) p_upstamp = atoi(tmp);
  if((tmp = cbmapget(params, "id", -1, NULL)) != NULL) p_id = tmp;
  if((tmp = cbmapget(params, "password", -1, NULL)) != NULL) p_password = tmp;
  if((tmp = cbmapget(params, "language", -1, NULL)) != NULL) p_language = tmp;
  if((tmp = cbmapget(params, "category", -1, NULL)) != NULL) p_category = tmp;
  if((tmp = cbmapget(params, "subject", -1, NULL)) != NULL) p_subject = tmp;
  if((tmp = cbmapget(params, "author", -1, NULL)) != NULL) p_author = tmp;
  if((tmp = cbmapget(params, "text", -1, NULL)) != NULL) p_text = tmp;
  if((tmp = cbmapget(params, "feed", -1, NULL)) != NULL) p_feed = tmp;
  if(p_action < ACTVIEW || p_action > ACTHELP) p_action = ACTVIEW;
  if(p_page < 0) p_page = 0;
  if(p_focus < FCSNORMAL || p_focus > FCSSIMPLE) p_focus = FCSNORMAL;
  if(p_sort < SORTCDATEA || p_sort >  SORTMDATED) p_sort = g_sort;
  if(!p_id) p_id = "";
  if(!p_password) p_password = "";
  if(!p_language) p_language = "";
  if(!p_category) p_category = "";
  if(!p_subject) p_subject = "";
  if(!p_author) p_author = "";
  if(!p_text) p_text = "";
  if(!p_feed) p_feed = "";
  /* read cookies */
  cooks = getcookies();
  cbglobalgc(cooks, (void (*)(void *))cbmapclose);
  if((tmp = cbmapget(cooks, "author", -1, NULL)) != NULL) c_author = tmp;
  if(!c_author) c_author = "";
  /* open the database */
  switch(p_action){
  case ACTVIEW:
  case ACTLIST:
  case ACTSEARCH:
  case ACTARTFORM:
  case ACTARTCNFM:
    g_rbbs = rbbsopen(g_dbname, FALSE);
    break;
  case ACTPOSTART:
  case ACTDELEART:
    g_rbbs = rbbsopen(g_dbname, g_rmpost);
    break;
  case ACTPOSTRES:
    g_rbbs = rbbsopen(g_dbname, g_rmpost && p_author[0] != '\0' && p_text != '\0');
    break;
  case ACTDUMPDB:
    g_rbbs = rbbsopen(g_dbname, FALSE);
    break;
  default:
    break;
  }
  if(g_rbbs) cbglobalgc(g_rbbs, (void (*)(void *))rbbsclose);
  /* set the current date */
  buf = datecurrent();
  cbglobalgc(buf, free);
  g_curdate = buf;
  /* show page frame */
  if(p_action == ACTDUMPDB){
    senddump();
  } else if(p_feed[0] != '\0'){
    sendfeed();
  } else {
    sendhtml();
  }
  return 0;
}


/* skip the label of a line */
const char *skiplabel(const char *str){
  if(!(str = strchr(str, ':'))) return str;
  str++;
  while(*str != '\0' && (*str == ' ' || *str == '\t')){
    str++;
  }
  return str;
}


/* get a map of the CGI parameters */
CBMAP *getparameters(void){
  CBMAP *map;
  CBLIST *pairs;
  char *rbuf, *buf, *key, *val, *dkey, *dval;
  const char *tmp;
  int i, len, c;
  map = cbmapopenex(PETITBNUM);
  rbuf = NULL;
  buf = NULL;
  if((tmp = getenv("CONTENT_LENGTH")) != NULL && (len = atoi(tmp)) > 0 && len <= RDATAMAX){
    rbuf = cbmalloc(len + 1);
    for(i = 0; i < len && (c = getchar()) != EOF; i++){
      rbuf[i] = c;
    }
    rbuf[i] = '\0';
    if(i == len) buf = rbuf;
    g_rmpost = TRUE;
  } else {
    buf = getenv("QUERY_STRING");
  }
  if(buf != NULL){
    buf = cbmemdup(buf, -1);
    pairs = cbsplit(buf, -1, "&");
    for(i = 0; i < cblistnum(pairs); i++){
      key = cbmemdup(cblistval(pairs, i, NULL), -1);
      if((val = strchr(key, '=')) != NULL){
        *(val++) = '\0';
        dkey = cburldecode(key, NULL);
        dval = cburldecode(val, NULL);
        cbmapput(map, dkey, -1, dval, -1, FALSE);
        free(dval);
        free(dkey);
      }
      free(key);
    }
    cblistclose(pairs);
    free(buf);
  }
  free(rbuf);
  return map;
}


/* get a map of the CGI cookies */
CBMAP *getcookies(void){
  CBMAP *map;
  CBLIST *pairs;
  char *key, *val, *dkey, *dval;
  const char *buf;
  int i;
  map = cbmapopenex(PETITBNUM);
  if(!(buf = getenv("HTTP_COOKIE"))) return map;
  pairs = cbsplit(buf, -1, "; ");
  for(i = 0; i < cblistnum(pairs); i++){
    key = cbmemdup(cblistval(pairs, i, NULL), -1);
    if((val = strchr(key, '=')) != NULL){
      *(val++) = '\0';
      dkey = cburldecode(key, NULL);
      dval = cburldecode(val, NULL);
      cbmapput(map, dkey, -1, dval, -1, FALSE);
      free(dval);
      free(dkey);
    }
    free(key);
  }
  cblistclose(pairs);
  return map;
}


/* XML-oriented printf */
void xprintf(const char *format, ...){
  va_list ap;
  char *tmp;
  unsigned char c;
  va_start(ap, format);
  while(*format != '\0'){
    if(*format == '%'){
      format++;
      switch(*format){
      case 's':
        tmp = va_arg(ap, char *);
        if(!tmp) tmp = "(null)";
        printf("%s", tmp);
        break;
      case 'd':
        printf("%d", va_arg(ap, int));
        break;
      case '@':
        tmp = va_arg(ap, char *);
        if(!tmp) tmp = "(null)";
        while(*tmp){
          switch(*tmp){
          case '&': printf("&amp;"); break;
          case '<': printf("&lt;"); break;
          case '>': printf("&gt;"); break;
          case '"': printf("&quot;"); break;
          default:
            if(!((*tmp >= 0 && *tmp <= 0x8) || (*tmp >= 0x0e && *tmp <= 0x1f))) putchar(*tmp);
            break;
          }
          tmp++;
        }
        break;
      case '?':
        tmp = va_arg(ap, char *);
        if(!tmp) tmp = "(null)";
        while(*tmp){
          c = *(unsigned char *)tmp;
          if((c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z') ||
             (c >= '0' && c <= '9') || (c != '\0' && strchr("_-.", c))){
            putchar(c);
          } else {
            printf("%%%02X", c);
          }
          tmp++;
        }
        break;
      case '%':
        putchar('%');
        break;
      }
    } else {
      putchar(*format);
    }
    format++;
  }
  va_end(ap);
}


/* show the error page and exit */
void senderror(const char *msg){
  printf("Status: 500 Internal Server Error\r\n");
  printf("Content-Type: text/plain; charset=%s\r\n", ENCODING);
  printf("\r\n");
  printf("Error: %s\n", msg);
  exit(1);
}


/* show the error page of not found */
void sendnotfound(void){
  printf("Status: 404 Not Found\r\n");
  printf("Content-Type: text/plain; charset=%s\r\n", ENCODING);
  printf("\r\n");
  printf("The target of the requested URI was not found\n");
}


/* send dump data of all articles */
void senddump(void){
  ARTICLE *art;
  char *tmp, *id, *xml;
  if(!g_rbbs){
    sendnotfound();
    return;
  }
  printf("Cache-Control: no-cache, must-revalidate\r\n");
  printf("Pragma: no-cache\r\n");
  printf("Content-Disposition: inline; filename=%s\r\n", DUMPNAME);
  tmp = dateforhttp(rbbsmdate(g_rbbs), FALSE);
  printf("Last-Modified: %s\r\n", tmp);
  free(tmp);
  printf("Content-Type: application/xml\r\n");
  printf("\r\n");

  xprintf("<?xml version=\"1.0\" encoding=\"%@\"?>\n", ENCODING);
  xprintf("<!DOCTYPE rbbs SYSTEM \"rbbs.dtd\">\n");
  xprintf("<rbbs version=\"%@\" cdate=\"%@\" mdate=\"%@\">\n",
          RBBS_VERSION, rbbscdate(g_rbbs), rbbsmdate(g_rbbs));
  setsortorder(g_rbbs, SORTCDATEA);
  while((id = getnextid(g_rbbs)) != NULL){
    if((art = getarticle(g_rbbs, id)) != NULL){
      xml = arttoxml(art);
      printf("%s", xml);
      free(xml);
      freearticle(art);
    }
    free(id);
  }
  xprintf("</rbbs>\n");
}


/* send Atom feed */
void sendfeed(void){
  ARTICLE *art;
  char *tmp, *prefix, *xml;
  int snum;
  if(!g_rbbs){
    sendnotfound();
    return;
  }
  art = NULL;
  if(!strcmp(p_feed, FEEDCDATE)){
    setsortorder(g_rbbs, SORTCDATED);
  } else if(!strcmp(p_feed, FEEDMDATE)){
    setsortorder(g_rbbs, SORTMDATED);
  } else if(!(art = getarticle(g_rbbs, p_feed))){
    sendnotfound();
    return;
  }
  printf("Cache-Control: no-cache, must-revalidate\r\n");
  printf("Pragma: no-cache\r\n");
  if(!strcmp(p_feed, FEEDCDATE) || !strcmp(p_feed, FEEDMDATE)){
    printf("Content-Disposition: inline; filename=%s\r\n", FEEDNAME);
  } else {
    printf("Content-Disposition: inline; filename=%s.xml\r\n", p_feed);
  }
  tmp = dateforhttp(rbbsmdate(g_rbbs), FALSE);
  printf("Last-Modified: %s\r\n", tmp);
  free(tmp);
  printf("Content-Type: application/xml\r\n");
  printf("\r\n");
  xprintf("<?xml version=\"1.0\" encoding=\"%@\"?>\n", ENCODING);
  xprintf("<feed version=\"0.3\" xmlns=\"http://purl.org/atom/ns#\" xml:lang=\"%@\">\n",
          g_pagelang);
  prefix = cbsprintf("%s?id=", g_systemuri);
  if(art){
    xprintf("<link rel=\"alternate\" type=\"text/html\" href=\"%@?id=%@\"/>\n",
            g_systemuri, artid(art));
    xprintf("<title>%@</title>\n", artsubject(art));
    xprintf("<author>\n");
    xprintf("<name>%@</name>\n", artauthor(art));
    xprintf("</author>\n");
    tmp = dateforwww(artmdate(art), FALSE);
    xprintf("<modified>%@</modified>\n", tmp);
    free(tmp);
    xprintf("<generator name=\"RBBS\">%@?version=%?</generator>\n", RBBS_URI, RBBS_VERSION);
    xml = arttoatom(art, prefix, g_blknum, g_resnum);
    printf("%s", xml);
    free(xml);
    freearticle(art);
  } else {
    xprintf("<link rel=\"alternate\" type=\"text/html\" href=\"%@\"/>\n", g_systemuri);
    xprintf("<title>%@</title>\n", g_pagetitle);
    xprintf("<author>\n");
    xprintf("<name>%@</name>\n", g_pageauthor);
    xprintf("</author>\n");
    tmp = dateforwww(rbbsmdate(g_rbbs), FALSE);
    xprintf("<modified>%@</modified>\n", tmp);
    free(tmp);
    xprintf("<tagline>%@</tagline>\n", g_pagesummary);
    xprintf("<generator name=\"RBBS\">%@?version=%?</generator>\n", RBBS_URI, RBBS_VERSION);
    snum = 0;
    while(snum < g_feednum && (tmp = getnextid(g_rbbs)) != NULL){
      if((art = getarticle(g_rbbs, tmp)) != NULL){
        xml = arttoatom(art, prefix, g_blknum, g_resnum);
        printf("%s", xml);
        free(xml);
        freearticle(art);
        snum++;
      }
      free(tmp);
    }
  }
  free(prefix);
  xprintf("</feed>\n");
}


/* send HTML page */
void sendhtml(void){
  char *tmp;
  printf("Cache-Control: no-cache, must-revalidate\r\n");
  printf("Pragma: no-cache\r\n");
  if(p_id[0] != '\0'){
    printf("Content-Disposition: inline; filename=%s.html\r\n", p_id);
  } else {
    printf("Content-Disposition: inline; filename=%s\r\n", PAGENAME);
  }
  if(g_rbbs && (p_action == ACTVIEW || p_action == ACTLIST || p_action == ACTSEARCH)){
    tmp = dateforhttp(rbbsmdate(g_rbbs), FALSE);
    printf("Last-Modified: %s\r\n", tmp);
    free(tmp);
  }
  if((p_action == ACTPOSTART || p_action == ACTPOSTRES) && p_author[0] != '\0'){
    tmp = cburlencode(p_author, -1);
    printf("Set-Cookie: author=%s; expires=Thu, 1-Jan-2037 00:00:00 GMT\r\n", tmp);
    free(tmp);
  }
  printf("Content-Type: text/html; charset=%s\r\n", ENCODING);
  printf("\r\n");
  xprintf("<?xml version=\"1.0\" encoding=\"%@\"?>\n", ENCODING);
  xprintf("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\""
          " \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">\n");
  xprintf("<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"%@\" lang=\"%@\">\n",
          g_pagelang, g_pagelang);
  xprintf("<head>\n");
  xprintf("<title>%@</title>\n", g_pagetitle);
  xprintf("<meta http-equiv=\"Content-Type\" content=\"text/html; charset=%@\" />\n",
          ENCODING);
  xprintf("<meta http-equiv=\"Content-Style-Type\" content=\"text/css\" />\n");
  xprintf("<meta name=\"author\" content=\"%@\" />\n", g_pageauthor);
  xprintf("<meta name=\"generator\" content=\"RBBS %@\" />\n", RBBS_VERSION);
  xprintf("<link rel=\"contents\" href=\"%@\" />\n", g_systemuri);
  if((tmp = cbreadfile(g_stylefile, NULL)) != NULL){
    xprintf("<style type=\"text/css\">\n");
    xprintf("%s", tmp);
    xprintf("</style>\n");
    free(tmp);
  }
  xprintf("</head>\n");
  xprintf("<body>\n");
  if((p_action == ACTVIEW || p_action == ACTSEARCH) && p_id[0] == '\0'){
    showcaption();
    showlist();
  } else if(p_action == ACTHELP){
    showcaption();
  }
  showpagenavi();
  switch(p_action){
  case ACTVIEW:
    showarticles();
    break;
  case ACTLIST:
    showlist();
    break;
  case ACTSEARCH:
    showsearchform();
    showarticles();
    break;
  case ACTARTFORM:
    if(authorize(p_id[0] != '\0')) showartform();
    break;
  case ACTARTCNFM:
    if(authorize(p_id[0] != '\0')) showartconfirm();
    break;
  case ACTPOSTART:
    if(authorize(p_id[0] != '\0')) mypostarticle();
    break;
  case ACTDELEART:
    if(authorize(p_id[0] != '\0')) mydeletearticle();
    break;
  case ACTPOSTRES:
    if(mypostresponse()) showarticles();
    break;
  case ACTDELEDB:
    if(authorize(TRUE)) deletedb();
    break;
  case ACTHELP:
    showhelp();
    break;
  default:
    break;
  }
  showpagenavi();
  if((p_action == ACTVIEW || p_action == ACTSEARCH) && p_id[0] == '\0')
    xprintf("<div class=\"sysversion\">Powered by RBBS %@.</div>\n", RBBS_VERSION);
  xprintf("</body>\n");
  xprintf("</html>\n");
}


/* check whether the user can administrate the system */
int canadmin(void){
  if(g_adminpass[0] == '\0') return TRUE;
  if(!strcmp(p_password, g_adminpass)) return TRUE;
  return FALSE;
}


/* check whether the user can write articles */
int canwrite(void){
  if(canadmin()) return TRUE;
  if(g_writepass[0] == '\0') return TRUE;
  if(!strcmp(p_password, g_writepass)) return TRUE;
  return FALSE;
}


/* check the password and show the form to input a password */
int authorize(int admin){
  if(admin){
    if(canadmin()) return TRUE;
  } else {
    if(canwrite()) return TRUE;
  }
  if(p_password[0] != '\0'){
    shownote("The password did not match.  Try again.");
  } else if(admin){
    shownote("The password of the administrator is needed.");
  } else {
    shownote("The password for writers is needed.");
  }
  xprintf("<form method=\"post\" action=\"%@\">\n", g_systemuri);
  xprintf("<div class=\"comform\">\n");
  xprintf("<table summary=\"form table\">\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\">%@:</td>\n", g_passwordlabel);
  xprintf("<td class=\"tbody\">\n");
  xprintf("<input type=\"password\" name=\"password\" value=\"\" size=\"16\""
          " class=\"password\" tabindex=\"%d\" />\n", ++g_tabidx);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\"></td>\n");
  xprintf("<td class=\"tbody\"><input type=\"submit\" value=\"%@\" class=\"submit\""
          " tabindex=\"%d\" /></td>\n", g_submitlabel, ++g_tabidx);
  xprintf("</tr>\n");
  xprintf("</table>\n");
  xprintf("</div>\n");
  xprintf("<div class=\"hidden\">\n");
  xprintf("<input type=\"hidden\" name=\"action\" value=\"%d\" />\n", p_action);
  xprintf("<input type=\"hidden\" name=\"id\" value=\"%@\" />\n", p_id);
  xprintf("</div>\n");
  xprintf("</form>\n");
  return FALSE;
}


/* show a note line */
void shownote(const char *msg){
  xprintf("<p class=\"note\">%@</p>\n", msg);
}


/* show the page navigation */
void showpagenavi(void){
  CBDATUM *buf;
  const char *fstr, *cond;
  buf = cbdatumopen("", 0);
  if(p_action == ACTSEARCH){
    xsprintf(buf, "&amp;action=%d", p_action);
    xsprintf(buf, "&amp;language=%@", p_language);
    xsprintf(buf, "&amp;category=%@", p_category);
    xsprintf(buf, "&amp;subject=%@", p_subject);
    xsprintf(buf, "&amp;author=%@", p_author);
    xsprintf(buf, "&amp;text=%@", p_text);
    xsprintf(buf, "&amp;sort=%d", p_sort);
    cond = cbdatumptr(buf);
  } else {
    cond = "";
  }
  xprintf("<div class=\"pagenavi\">\n");
  if(g_homepageuri[0] != '\0')
    xprintf("<a href=\"%@\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
            g_homepageuri, ++g_tabidx, g_homelabel);
  xprintf("<a href=\"%@?action=%d\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
          g_systemuri, ACTHELP, ++g_tabidx, g_helplabel);
  xprintf("<a href=\"%@?action=%d\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
          g_systemuri, ACTARTFORM, ++g_tabidx, g_postlabel);
  fstr = FEEDCDATE;
  if(p_sort == SORTMDATEA || p_sort == SORTMDATED) fstr = FEEDMDATE;
  xprintf("<a href=\"%@?feed=%@\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
          g_systemuri, fstr, ++g_tabidx, g_feedlabel);
  xprintf("<a href=\"%@\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
          g_systemuri, ++g_tabidx, g_reloadlabel);
  xprintf("<a href=\"%@?action=%d\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
          g_systemuri, ACTSEARCH, ++g_tabidx, g_searchlabel);
  xprintf("<a href=\"%@?action=%d\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
          g_systemuri, ACTLIST, ++g_tabidx, g_listlabel);
  if(p_page > 0){
    xprintf("<a href=\"%@?page=%d%s\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
            g_systemuri, p_page - 1, cond, ++g_tabidx, g_prevlabel);
  } else {
    xprintf("<span class=\"navivoid\">%@</span>\n", g_prevlabel);
  }
  if(g_cannext && p_action != ACTLIST){
    xprintf("<a href=\"%@?page=%d%s\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
            g_systemuri, p_page + 1, cond, ++g_tabidx, g_nextlabel);
  } else {
    xprintf("<span class=\"navivoid\">%@</span>\n", g_nextlabel);
  }
  xprintf("</div>\n");
  cbdatumclose(buf);
}


/* show the caption of the page */
void showcaption(void){
  xprintf("<div class=\"pagecaption\">\n");
  xprintf("<h1>%@</h1>\n", g_pagetitle);
  xprintf("<p>%@</p>\n", g_pagesummary);
  xprintf("</div>\n");
}


/* check whether an article matches the narrowing condition. */
int matchingart(ARTICLE *art){
  CBLIST *words;
  const char *word;
  char *text;
  int i, hit;
  if(p_language[0] != '\0' && strcmp(artlanguage(art), p_language)) return FALSE;
  if(p_category[0] != '\0' && strcmp(artcategory(art), p_category)) return FALSE;
  if(p_subject[0] != '\0' && !strstr(artsubject(art), p_subject)) return FALSE;
  if(p_author[0] != '\0' && !strstr(artauthor(art), p_author)) return FALSE;
  if(p_text[0] != '\0'){
    text = arttotext(art);
    words = cbsplit(p_text, -1, " \t\n");
    hit = TRUE;
    for(i = 0; i < cblistnum(words); i++){
      word = cblistval(words, i, NULL);
      if(word[0] == '\0') continue;
      if(!strstr(text, word)){
        hit = FALSE;
        break;
      }
    }
    cblistclose(words);
    free(text);
    if(!hit) return FALSE;
  }
  return TRUE;
}


/* show titles of articles */
void showlist(void){
  ARTICLE *art;
  char *id;
  int i, skip, snum;
  if(g_listnum < 1 || !g_rbbs || rbbsanum(g_rbbs) < 1) return;
  setsortorder(g_rbbs, p_sort);
  snum = 0;
  skip = 0;
  xprintf("<div class=\"artlist\">\n");
  for(i = 0; (p_action == ACTLIST || snum < g_listnum) && (id = getnextid(g_rbbs)) != NULL; i++){
    if(p_action != ACTSEARCH){
      if(skip++ < p_page * g_shownum){
        free(id);
        continue;
      }
    }
    if((art = getarticle(g_rbbs, id)) != NULL){
      if(matchingart(art) && skip++ >= p_page * g_shownum){
        xprintf("<span class=\"artitem\">\n");
        xprintf("<span class=\"artseq\">%d</span>\n", i + 1);
        xprintf("<a href=\"%@?id=%@\">%@</a>\n", g_systemuri, artid(art), artsubject(art));
        xprintf("<span class=\"resnum\">%d</span>\n", artresnum(art));
        xprintf("</span>\n");
        snum++;
      }
      freearticle(art);
    }
    free(id);
  }
  if(snum < 1) xprintf("<span class=\"note\">There is no corresponding article.</span>\n");
  xprintf("</div>\n");
  if(snum > g_shownum) g_cannext = TRUE;
}


/* show the search form */
void showsearchform(void){
  CBLIST *list;
  const char *tmp;
  int i;
  xprintf("<form method=\"get\" action=\"%@\">\n", g_systemuri);
  xprintf("<div class=\"comform\">\n");
  xprintf("<table summary=\"form table\">\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\">%@:</td>\n", g_languagelabel);
  xprintf("<td class=\"tbody\">\n");
  xprintf("<label><input type=\"radio\" name=\"language\" value=\"\""
          " class=\"radio\" tabindex=\"%d\"%s />%@</label>\n",
          ++g_tabidx, p_language[0] == '\0' ? " checked=\"checked\"" : "", g_anykindlabel);
  list = cbsplit(g_languages, -1, ",");
  for(i = 0; i < cblistnum(list); i++){
    tmp = cblistval(list, i, NULL);
    if(tmp[0] == '\0') continue;
    xprintf("<label><input type=\"radio\" name=\"language\" value=\"%@\""
            " class=\"radio\" tabindex=\"%d\"%s />%@</label>\n",
            tmp, ++g_tabidx, !strcmp(tmp, p_language) ? " checked=\"checked\"" : "", tmp);

  }
  cblistclose(list);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\">%@:</td>\n", g_categorylabel);
  xprintf("<td class=\"tbody\">\n");
  xprintf("<label><input type=\"radio\" name=\"category\" value=\"\""
          " class=\"radio\" tabindex=\"%d\"%s />%@</label>\n",
          ++g_tabidx, p_language[0] == '\0' ? " checked=\"checked\"" : "", g_anykindlabel);
  list = cbsplit(g_categories, -1, ",");
  for(i = 0; i < cblistnum(list); i++){
    tmp = cblistval(list, i, NULL);
    if(tmp[0] == '\0') continue;
    xprintf("<label><input type=\"radio\" name=\"category\" value=\"%@\""
            " class=\"radio\" tabindex=\"%d\"%s />%@</label>\n",
            tmp, ++g_tabidx, !strcmp(tmp, p_category) ? " checked=\"checked\"" : "", tmp);
  }
  cblistclose(list);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\">%@:</td>\n", g_subjectlabel);
  xprintf("<td class=\"tbody\">\n");
  xprintf("<input type=\"text\" name=\"subject\" value=\"%@\" size=\"48\""
          " class=\"text\" tabindex=\"%d\" />\n", p_subject, ++g_tabidx);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\">%@:</td>\n", g_authorlabel);
  xprintf("<td class=\"tbody\">\n");
  xprintf("<input type=\"text\" name=\"author\" value=\"%@\" size=\"16\""
          " class=\"text\" tabindex=\"%d\" />\n", p_author, ++g_tabidx);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\">%@:</td>\n", g_ftslabel);
  xprintf("<td class=\"tbody\">\n");
  xprintf("<input type=\"text\" name=\"text\" value=\"%@\" size=\"48\""
          " class=\"text\" tabindex=\"%d\" />\n", p_text, ++g_tabidx);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\">%@:</td>\n", g_orderlabel);
  xprintf("<td class=\"tbody\">\n");
  xprintf("<label><input type=\"radio\" name=\"sort\" value=\"%d\""
          " class=\"radio\" tabindex=\"%d\"%s />%@</label>\n",
          SORTCDATEA, ++g_tabidx, p_sort == SORTCDATEA ? " checked=\"checked\"" : "",
          g_cdatelabel);
  xprintf("<label><input type=\"radio\" name=\"sort\" value=\"%d\""
          " class=\"radio\" tabindex=\"%d\"%s />%@ %@</label>\n",
          SORTCDATED, ++g_tabidx, p_sort == SORTCDATED ? " checked=\"checked\"" : "",
          g_cdatelabel, g_descordlabel);
  xprintf("<label><input type=\"radio\" name=\"sort\" value=\"%d\""
          " class=\"radio\" tabindex=\"%d\"%s />%@</label>\n",
          SORTMDATEA, ++g_tabidx, p_sort == SORTMDATEA ? " checked=\"checked\"" : "",
          g_mdatelabel);
  xprintf("<label><input type=\"radio\" name=\"sort\" value=\"%d\""
          " class=\"radio\" tabindex=\"%d\"%s />%@ %@</label>\n",
          SORTMDATED, ++g_tabidx, p_sort == SORTMDATED ? " checked=\"checked\"" : "",
          g_mdatelabel, g_descordlabel);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\"></td>\n");
  xprintf("<td class=\"tbody\">");
  xprintf("<input type=\"submit\" value=\"%@\" class=\"submit\" tabindex=\"%d\" />",
          g_submitlabel, ++g_tabidx);
  xprintf("<input type=\"reset\" value=\"%@\" class=\"reset\" tabindex=\"%d\" />",
          g_resetlabel, ++g_tabidx);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  xprintf("</table>\n");
  xprintf("</div>\n");
  xprintf("<div class=\"hidden\">\n");
  xprintf("<input type=\"hidden\" name=\"action\" value=\"%d\" />\n", ACTSEARCH);
  xprintf("</div>\n");
  xprintf("</form>\n");
}


/* show articles */
void showarticles(void){
  ARTICLE *art;
  char *id;
  int skip, snum;
  snum = 0;
  if(g_rbbs && p_id[0] != '\0'){
    if((art = getarticle(g_rbbs, p_id)) != NULL){
      showartwithctl(art);
      snum++;
      freearticle(art);
    }
  } else if(g_rbbs){
    setsortorder(g_rbbs, p_sort);
    skip = 0;
    while(snum < g_shownum && (id = getnextid(g_rbbs)) != NULL){
      if(p_action != ACTSEARCH){
        if(skip++ < p_page * g_shownum){
          free(id);
          continue;
        }
      }
      if((art = getarticle(g_rbbs, id)) != NULL){
        if(matchingart(art) && skip++ >= p_page * g_shownum){
          showartwithctl(art);
          snum++;
        }
        freearticle(art);
      }
      free(id);
    }
  }
  if(snum < 1) shownote("There is no corresponding article.");
}


/* show one article with the navigation control */
void showartwithctl(ARTICLE *art){
  char *html;
  xprintf("<div class=\"artctl\">\n");
  xprintf("<div class=\"artnavi\">\n");
  xprintf("<a href=\"%@?action=%d&amp;id=%@\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
          g_systemuri, ACTARTFORM, artid(art), ++g_tabidx, g_editlabel);
  xprintf("<a href=\"%@?feed=%@\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
          g_systemuri, artid(art), ++g_tabidx, g_feedlabel);
  xprintf("<a href=\"%@?id=%@&amp;focus=%d\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
          g_systemuri, artid(art), FCSDETAIL, ++g_tabidx, g_focuslabel);
  xprintf("</div>\n");
  html = arttohtml(art, p_focus != FCSSIMPLE ? 0 : g_blknum, p_focus == FCSDETAIL ? 0 : g_resnum);
  xprintf("%s", html);
  free(html);
  if(!strchr(artflags(art), LONECHAR) && p_id[0] != '\0'){
    xprintf("<form method=\"post\" action=\"%@#%@T\">\n", g_systemuri, artid(art));
    xprintf("<div class=\"resform\">\n");
    xprintf("<span class=\"formitem\">\n");
    xprintf("<input type=\"text\" name=\"author\" value=\"%@\" size=\"8\""
            " class=\"text\" tabindex=\"%d\" />\n",
            p_author[0] != '\0' ? p_author : c_author, ++g_tabidx);
    xprintf("</span>\n");
    xprintf("<span class=\"formitem\">\n");
    xprintf("<input type=\"text\" name=\"text\" value=\"\" size=\"80\""
            " class=\"text\" tabindex=\"%d\" />\n", ++g_tabidx);
    xprintf("</span>\n");
    xprintf("<span class=\"formitem\">\n");
    xprintf("<input type=\"submit\" value=\"%@\" class=\"submit\""
            " tabindex=\"%d\" />\n", g_submitlabel, ++g_tabidx);
    xprintf("</span>\n");
    xprintf("</div>\n");
    xprintf("<div class=\"hidden\">\n");
    xprintf("<input type=\"hidden\" name=\"action\" value=\"%d\" />\n", ACTPOSTRES);
    xprintf("<input type=\"hidden\" name=\"id\" value=\"%@\" />\n", artid(art));
    xprintf("<input type=\"hidden\" name=\"focus\" value=\"%d\" />\n", FCSSIMPLE);
    xprintf("</div>\n");
    xprintf("</form>\n");
  }
  xprintf("</div>\n");
}


/* show the form to post an article */
void showartform(void){
  ARTICLE *art;
  CBLIST *list;
  const char *tmp;
  char *xml;
  int i, fst;
  xprintf("<form method=\"post\" action=\"%@\">\n", g_systemuri);
  xprintf("<div class=\"comform\">\n");
  xprintf("<table summary=\"form table\">\n");
  if(p_id[0] != '\0'){
    xml = NULL;
    if(p_text[0] != '\0'){
      xml = cbbasedecode(p_text, NULL);
    } else {
      if(g_rbbs && (art = getarticle(g_rbbs, p_id)) != NULL){
        xml = arttoxml(art);
        freearticle(art);
      }
    }
    if(!xml) xml = cbmemdup("", 0);
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\">%@:</td>\n", g_idlabel);
    xprintf("<td class=\"tbody\">%@</td>\n", p_id);
    xprintf("</tr>\n");
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\">%@:</td>\n", g_sourcelabel);
    xprintf("<td class=\"tbody\">\n");
    xprintf("<textarea name=\"text\" cols=\"80\" rows=\"20\" tabindex=\"%d\">%@</textarea>\n",
            ++g_tabidx, xml);
    xprintf("</td>\n");
    xprintf("</tr>\n");
    free(xml);
  } else {
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\">%@:</td>\n", g_languagelabel);
    xprintf("<td class=\"tbody\">\n");
    list = cbsplit(g_languages, -1, ",");
    fst = p_language[0] == '\0';
    for(i = 0; i < cblistnum(list); i++){
      tmp = cblistval(list, i, NULL);
      if(tmp[0] == '\0') continue;
      xprintf("<label><input type=\"radio\" name=\"language\" value=\"%@\""
              " class=\"radio\" tabindex=\"%d\"%s />%@</label>\n",
              tmp, ++g_tabidx, fst || !strcmp(tmp, p_language) ? " checked=\"checked\"" : "", tmp);
      fst = FALSE;
    }
    cblistclose(list);
    xprintf("</td>\n");
    xprintf("</tr>\n");
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\">%@:</td>\n", g_categorylabel);
    xprintf("<td class=\"tbody\">\n");
    list = cbsplit(g_categories, -1, ",");
    fst = p_category[0] == '\0';
    for(i = 0; i < cblistnum(list); i++){
      tmp = cblistval(list, i, NULL);
      if(tmp[0] == '\0') continue;
      xprintf("<label><input type=\"radio\" name=\"category\" value=\"%@\""
              " class=\"radio\" tabindex=\"%d\"%s />%@</label>\n",
              tmp, ++g_tabidx, fst || !strcmp(tmp, p_category) ? " checked=\"checked\"" : "", tmp);
      fst = FALSE;
    }
    cblistclose(list);
    xprintf("</td>\n");
    xprintf("</tr>\n");
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\">%@:</td>\n", g_subjectlabel);
    xprintf("<td class=\"tbody\">\n");
    xprintf("<input type=\"text\" name=\"subject\" value=\"%@\" size=\"48\""
            " class=\"text\" tabindex=\"%d\" />\n", p_subject, ++g_tabidx);
    xprintf("</td>\n");
    xprintf("</tr>\n");
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\">%@:</td>\n", g_authorlabel);
    xprintf("<td class=\"tbody\">\n");
    xprintf("<input type=\"text\" name=\"author\" value=\"%@\" size=\"16\""
            " class=\"text\" tabindex=\"%d\" />\n", p_author, ++g_tabidx);
    xprintf("</td>\n");
    xprintf("</tr>\n");
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\">%@:</td>\n", g_textlabel);
    xprintf("<td class=\"tbody\">\n");
    xml = cbbasedecode(p_text, NULL);
    xprintf("<textarea name=\"text\" cols=\"80\" rows=\"20\" tabindex=\"%d\">%@</textarea>\n",
            ++g_tabidx, xml);
    free(xml);
    xprintf("</td>\n");
    xprintf("</tr>\n");
  }
  xprintf("<tr>\n");
  xprintf("<td class=\"tlabel\"></td>\n");
  xprintf("<td class=\"tbody\">");
  xprintf("<input type=\"submit\" value=\"%@\" class=\"submit\" tabindex=\"%d\" />",
          g_submitlabel, ++g_tabidx);
  xprintf("<input type=\"reset\" value=\"%@\" class=\"reset\" tabindex=\"%d\" />",
          g_resetlabel, ++g_tabidx);
  xprintf("</td>\n");
  xprintf("</tr>\n");
  xprintf("</table>\n");
  xprintf("</div>\n");
  xprintf("<div class=\"hidden\">\n");
  xprintf("<input type=\"hidden\" name=\"action\" value=\"%d\" />\n", ACTARTCNFM);
  xprintf("<input type=\"hidden\" name=\"id\" value=\"%@\" />\n", p_id);
  xprintf("<input type=\"hidden\" name=\"password\" value=\"%@\" />\n", p_password);
  xprintf("</div>\n");
  xprintf("</form>\n");
}


/* show the confirmation to post an article */
void showartconfirm(void){
  ARTICLE *art;
  CBDATUM *buf;
  char *xml, *tmp;
  buf = cbdatumopen("", 0);
  if(p_id[0] != '\0'){
    xsprintf(buf, "%s", p_text);
  } else {
    xsprintf(buf, "<article>\n");
    xsprintf(buf, "<head>\n");
    xsprintf(buf, "<flags>%@</flags>\n", g_artflags);
    xsprintf(buf, "<language>%@</language>\n", p_language);
    xsprintf(buf, "<category>%@</category>\n", p_category);
    xsprintf(buf, "<subject>%@</subject>\n", p_subject);
    xsprintf(buf, "<author>%@</author>\n", p_author);
    xsprintf(buf, "<cdate>%@</cdate>\n", g_curdate);
    xsprintf(buf, "<mdate>%@</mdate>\n", g_curdate);
    xsprintf(buf, "</head>\n");
    xsprintf(buf, "<body>\n");
    xml = strnormalize(p_text, FALSE, TRUE);
    if(xml[0] != '<'){
      asiscat(buf, xml);
    } else {
      xsprintf(buf, "%s", xml);
    }
    free(xml);
    xsprintf(buf, "</body>\n");
    xsprintf(buf, "<tail>\n");
    xsprintf(buf, "</tail>\n");
    xsprintf(buf, "</article>\n");
  }
  art = makearticle(cbdatumptr(buf));
  cbdatumclose(buf);
  if(p_id[0] != '\0' && p_text[0] == '\0'){
    shownote("The article is being deleted.");
    xprintf("<form method=\"post\" action=\"%@\">\n", g_systemuri);
    xprintf("<div class=\"comform\">\n");
    xprintf("<table summary=\"form table\">\n");
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\">%@:</td>\n", g_idlabel);
    xprintf("<td class=\"tbody\">%@</td>\n", p_id);
    xprintf("</tr>\n");
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\"></td>\n");
    xprintf("<td class=\"tbody\"><input type=\"submit\" value=\"%@\" class=\"submit\""
            " tabindex=\"%d\" /></td>\n", g_submitlabel, ++g_tabidx);
    xprintf("</tr>\n");
    xprintf("</table>\n");
    xprintf("</div>\n");
    xprintf("<div class=\"hidden\">\n");
    xprintf("<input type=\"hidden\" name=\"action\" value=\"%d\" />\n", ACTDELEART);
    xprintf("<input type=\"hidden\" name=\"id\" value=\"%@\" />\n", p_id);
    xprintf("<input type=\"hidden\" name=\"password\" value=\"%@\" />\n", p_password);
    xprintf("</div>\n");
    xprintf("</form>\n");
  } else if(!articleisok(art)){
    shownote(artemsg(art));
  } else {
    shownote("Confirm the content then submit it.");
    xml = arttoxml(art);
    xprintf("<div class=\"artctl\">\n");
    tmp = arttohtml(art, 0, 0);
    xprintf("%s", tmp);
    free(tmp);
    xprintf("</div>\n");
    xprintf("<form method=\"post\" action=\"%@\">\n", g_systemuri);
    xprintf("<div class=\"comform\">\n");
    xprintf("<table summary=\"form table\">\n");
    if(p_id[0] != '\0'){
      xprintf("<tr>\n");
      xprintf("<td class=\"tlabel\">%@:</td>\n", g_idlabel);
      xprintf("<td class=\"tbody\">%@</td>\n", p_id);
      xprintf("</tr>\n");
    }
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\">%@:</td>\n", g_sourcelabel);
    xprintf("<td class=\"tbody\">\n");
    xprintf("<pre class=\"source\">%@</pre>", xml);
    xprintf("</td>\n");
    xprintf("</tr>\n");
    if(p_id[0] != '\0'){
      xprintf("<tr>\n");
      xprintf("<td class=\"tlabel\"></td>\n");
      xprintf("<td class=\"tbody\">\n");
      xprintf("<label><input type=\"checkbox\" name=\"upstamp\" value=\"%d\""
              " class=\"checkbox\" tabindex=\"%d\" checked=\"checked\" />%@</label>\n",
              TRUE, ++g_tabidx, g_upstamplabel);
      xprintf("</td>\n");
      xprintf("</tr>\n");
    }
    xprintf("<tr>\n");
    xprintf("<td class=\"tlabel\"></td>\n");
    xprintf("<td class=\"tbody\"><input type=\"submit\" value=\"%@\" class=\"submit\""
            " tabindex=\"%d\" /></td>\n", g_submitlabel, ++g_tabidx);
    xprintf("</tr>\n");
    xprintf("</table>\n");
    xprintf("</div>\n");
    xprintf("<div class=\"hidden\">\n");
    xprintf("<input type=\"hidden\" name=\"action\" value=\"%d\" />\n", ACTPOSTART);
    xprintf("<input type=\"hidden\" name=\"id\" value=\"%@\" />\n", p_id);
    xprintf("<input type=\"hidden\" name=\"password\" value=\"%@\" />\n", p_password);
    tmp = cbbaseencode(xml, -1);
    xprintf("<input type=\"hidden\" name=\"text\" value=\"%@\" />\n", tmp);
    free(tmp);
    xprintf("</div>\n");
    xprintf("</form>\n");
    free(xml);
  }
  freearticle(art);
  showartformback();
}


/* show a button to go back to the article form */
void showartformback(void){
  char *tmp;
  xprintf("<form method=\"post\" action=\"%@\">\n", g_systemuri);
  xprintf("<div class=\"todo\">\n");
  xprintf("<input type=\"submit\" value=\"%@\" class=\"submit\""
          " tabindex=\"%d\" />\n", g_backlabel, ++g_tabidx);
  xprintf("</div>\n");
  xprintf("<div class=\"hidden\">\n");
  xprintf("<input type=\"hidden\" name=\"action\" value=\"%d\" />\n", ACTARTFORM);
  xprintf("<input type=\"hidden\" name=\"id\" value=\"%@\" />\n", p_id);
  xprintf("<input type=\"hidden\" name=\"password\" value=\"%@\" />\n", p_password);
  xprintf("<input type=\"hidden\" name=\"language\" value=\"%@\" />\n", p_language);
  xprintf("<input type=\"hidden\" name=\"category\" value=\"%@\" />\n", p_category);
  xprintf("<input type=\"hidden\" name=\"subject\" value=\"%@\" />\n", p_subject);
  xprintf("<input type=\"hidden\" name=\"author\" value=\"%@\" />\n", p_author);
  tmp = cbbaseencode(p_text, -1);
  xprintf("<input type=\"hidden\" name=\"text\" value=\"%@\" />\n", tmp);
  free(tmp);
  xprintf("</div>\n");
  xprintf("</form>\n");
}


/* concatenate an as-is text to an XML buffer */
void asiscat(CBDATUM *buf, const char *text){
  const char *prefs[] = {
    "http://", "https://", "ftp://", "gopher://", "ldap://", "file://", NULL
  };
  const char *pref, *rp;
  char *tmp;
  int i, j;
  xsprintf(buf, "<asis>");
  for(i = 0; text[i] != '\0'; i++){
    pref = NULL;
    for(j = 0; prefs[j]; j++){
      if(cbstrfwimatch(text + i, prefs[j])){
        pref = prefs[j];
      }
    }
    if(pref){
      rp = text + i;
      while(*rp != '\0' && ((*rp >= '0' && *rp <= '9') || (*rp >= 'A' && *rp <= 'Z') ||
                            (*rp >= 'a' && *rp <= 'z') || strchr("-_.!~*'();/?:@&=+$,%#", *rp))){
        rp++;
      }
      tmp = cbmemdup(text + i, rp - text - i);
      xsprintf(buf, "<link to=\"%@\">%@</link>", tmp, tmp);
      free(tmp);
      i = rp - text - 1;
    } else {
      switch(text[i]){
      case '&': xsprintf(buf, "&amp;"); break;
      case '<': xsprintf(buf, "&lt;"); break;
      case '>': xsprintf(buf, "&gt;"); break;
      default: xsprintf(buf, "%c", text[i]); break;
      }
    }
  }
  xsprintf(buf, "\n</asis>\n");
}


/* post an article */
void mypostarticle(void){
  ARTICLE *art;
  char *xml;
  if(!g_rbbs){
    shownote("The database is broken or not writable.");
    return;
  }
  xml = cbbasedecode(p_text, NULL);
  art = makearticle(xml);
  if(p_id[0] != '\0'){
    if(p_upstamp) artsetmdate(art, g_curdate);
  } else {
    artsetcdate(art, g_curdate);
    artsetmdate(art, g_curdate);
  }
  if(!articleisok(art)){
    shownote(artemsg(art));
  } else if(!postarticle(g_rbbs, art, p_id[0] != '\0' ? p_id : NULL)){
    shownote(rbbsemsg(g_rbbs));
  } else {
    shownote("Posting the article have been completed.");
    xprintf("<div class=\"todo\">\n");
    xprintf("<a href=\"%@\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
            g_systemuri, ++g_tabidx, g_reloadlabel);
    xprintf("</div>\n");
  }
  freearticle(art);
  free(xml);
}


/* delete an article */
void mydeletearticle(void){
  if(!g_rbbs){
    shownote("The database is broken or not writable.");
    return;
  }
  if(!deletearticle(g_rbbs, p_id)){
    shownote(rbbsemsg(g_rbbs));
  } else {
    shownote("Deleting the article have been completed.");
    xprintf("<div class=\"todo\">\n");
    xprintf("<a href=\"%@\" class=\"navianc\" tabindex=\"%d\">%@</a>\n",
            g_systemuri, ++g_tabidx, g_reloadlabel);
    xprintf("</div>\n");
  }
}


/* post a response */
int mypostresponse(void){
  ARTICLE *art;
  char flags[PATHBUFSIZ], *wp, *author, *ap, *text;
  int err;
  art = NULL;
  err = TRUE;
  if(!g_rbbs || !(art = getarticle(g_rbbs, p_id))){
    shownote("There is no corresponding article.");
    return FALSE;
  }
  author = strnormalize(p_author, TRUE, TRUE);
  text = strnormalize(p_text, TRUE, TRUE);
  if(author[0] == '\0' || text[0] == '\0'){
    free(text);
    free(author);
    freearticle(art);
    return TRUE;
  }
  wp = flags;
  wp += sprintf(wp, "%s", g_resflags);
  ap = author;
  if(cbstrfwimatch(author, SINKPREFIX)){
    if(!strchr(flags, SINKCHAR)) *(wp++) = SINKCHAR;
    ap += strlen(SINKPREFIX);
  }
  *wp = '\0';
  if(!strchr(flags, SINKCHAR)) artsetmdate(art, g_curdate);
  artaddres(art, flags, ap, g_curdate, text);
  err = FALSE;
  if(!postarticle(g_rbbs, art, p_id)){
    shownote(rbbsemsg(g_rbbs));
    err = TRUE;
  }
  free(text);
  free(author);
  freearticle(art);
  return err ? FALSE : TRUE;
}


/* delete the database directory */
void deletedb(void){
  if(rbbsremove(g_dbname)){
    shownote("The database was deleted.");
  } else {
    shownote("The database could not be deleted.");
  }
}


/* show the help file */
void showhelp(void){
  ARTICLE *art;
  char *xml, *html;
  if(!(xml = cbreadfile(g_helpfile, NULL))){
    shownote("The help file was not found.");
    return;
  }
  art = makearticle(xml);
  if(!articleisok(art)){
    shownote(artemsg(art));
    freearticle(art);
    free(xml);
    return;
  }
  xprintf("<div class=\"artctl\">\n");
  html = arttohtml(art, 0, 0);
  xprintf("%s", html);
  free(html);
  xprintf("</div>\n");
  freearticle(art);
  free(xml);
}



/* END OF FILE */
