<?php
/**
 * Research Artisan Lite: Website Access Analyzer
 * Copyright (C) 2009 Research Artisan Project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * @copyright Copyright (C) 2009 Research Artisan Project
 * @license GNU General Public License (see license.txt)
 * @author ossi
 */
abstract class BaseHelper extends RaHelper {

  const RESEARCH_DAY = '1';
  const RESEARCH_MONTH = '2';
  const RESEARCH_TERM = '3';
  const PAGE_COUNT_ALL = 'all';
  const PAGE_COUNT_UNIQUE = 'unq';

  private $_research = null;

  public function __construct(RaRequest $request, RaSession $session, RaMessage $message, RaResult $result=null, $controller, $action) {
    parent::__construct($request, $session, $message, $result, $controller, $action);
    $this->_research = $this->_getResearch();
  }
  
  public function isInstalled() {
    return file_exists(SETTING_INSTALL_COMPLETE_FILE) ? true : false;
  }

  public function isMenuWrite() {
    $controller = $this->controller;
    return $controller == 'research' || $controller == 'admin' ? true : false;
  }

  public function substrMax($str, $max) {
    $lastLength = 10;
    $rtn = $str;
    if (RaUtil::strLen($str) > $max) {
      if (!Track::checkDoubleStr(RaUtil::subStr($str, $max, RaUtil::strLen($str)))) {
        $rtn = RaUtil::subStr($str, 0, $max - $lastLength) . '&#133;'. RaUtil::subStr($str, RaUtil::strLen($str) - $lastLength, $lastLength);
      }
    }
    return $rtn;
  }

  public function getFormatNumber($value) {
    return number_format($value);
  }

  public function getFormatDataSize($value) {
    $len = strlen($value);
    switch ($len) {
      case 4:
      case 5:
      case 6:
        $rtn = round($value/1000, 1). ' KB';
        break;
      case 7:
      case 8:
      case 9:
        $rtn = round($value/(1000*1000), 1). ' MB';
        break;
      case 10:
      case 11:
      case 12:
        $rtn = round($value/(1000*1000*1000), 1). ' GB';
        break;
      default:
        $rtn = $this->getFormatNumber($value). ' bytes';
        break;
    }
    return $rtn;
  }

  public function escapeHtml($value) {
    return htmlentities($value, ENT_QUOTES, RaConfig::CHARSET);
  }

  public function urlEncode($url) {
    return urlencode($url);
  }

  public function getZeroPadding($value, $cnt) {
    return str_pad($value, $cnt+1, '0', STR_PAD_LEFT);
  }

  public function getZeroSuppress($value) {
    return is_numeric($value) ? intval($value) : $value;
  }

  public function getVersion() {
    $version = !is_null($this->session->get('version')) ? $this->session->get('version') : '';
    if ($version == '') $version = File::replaceCrlf(File::readFile(RA_CORE_DIR. 'version.txt'), '');
    return $version != '' ? '<a href="'. RaConfig::RA_URL. 'version/?ver='. $version. '" title="バージョンチェック">'. 'ver.'. $version. '</a>' : '';
  }

  public function getIndexUrl($controller, $action, $querystring=null) {
    $url = './'. Config::INDEX_FILENAME;
    if (trim($controller) != '') $url .= '?controller='. $controller; 
    if (trim($action) != '') $url .= '&amp;action='. $action; 
    if (!is_null($querystring)) $url .= $querystring; 
    return $url;
  }

  public function getJumpUrl($link) {
    return Track::checkUrl($link) ? '../jump.php?link='. $this->urlEncode($link) : '../jump.php';
  }

  public function getMessage() {
    $controller = $this->controller;
    $action = $this->action;
    $code = $this->message->getCode();
    $message = $this->message->getMessage();
    switch ($code) {
      case RaConfig::PHP_NOTSUPPORT_VERSION_ERR_CODE:
        $rtn = 'PHPの設定に問題があります。<br />';
        $rtn .= 'PHPのバージョンがサポート対象外です。<br />';
        $rtn .= $message. '<br />';
        break;
      case RaConfig::PHP_EXTENSION_NOTFOUND_ERR_CODE:
        $rtn = 'PHPの環境設定に問題があります。<br />';
        $rtn .= '次のモジュールがインストールされていません。<br />';
        $rtn .= $message. '<br />';
        break;
      case RaConfig::URL_NOTFOUND_ERR_CODE:
        if (($controller == 'login' && $action == 'login') || ($controller == 'install' && $action == 'step1')) {
          $rtn = null;
        } else {
          $rtn = '指定したURLに誤りがあります。';
        }
        break;
      case RaConfig::SESSION_ERROR_ERR_CODE:
        $rtn = '一定時間操作がありませんでしたので、処理を中断しました。<br />再度ログインしてご利用下さい。';
        break;
      case RaConfig::DATABASE_CONFIG_UNDEFINED_ERR_CODE:
        $rtn = 'データベース接続設定の読込みに失敗しました。<br />';
        $rtn .= '<a href="'. $this->getIndexUrl('', ''). '">こちら</a>よりデータベース接続の再設定を行なってください。';
        break;
      default:
        $rtn = $message; 
        if ($code != '' && $code != RaConfig::DEFAULT_ERR_CODE) $rtn .= ' ('. $code. ')';
        break;
    }
    return $rtn;
  }

  public function getActionTitle($char=false) {
    $rtn = '';
    $action = $this->action;
    if (!is_null($action)) {
      $action = isset(MenuConfig::$convertActions[$action]) ? MenuConfig::$convertActions[$action] : $action;
      if (trim($rtn) == '') {
        foreach (MenuConfig::$titles as $title) {
          if (isset($title[$action])) {
            $rtn = $title[$action];
            break;
          }
        }
      }
    }
    if (trim($rtn) != '' && $char) {
      $rtn .= ' - ';
    }
    return $rtn;
  }

  public function getSelect() {
    $action = $this->action;
    $select = $this->request->get('select');
    $html = '';
    switch ($action) {
      case 'domain_user':
      case 'jpdomain_user':
        $html .= $this->getDomain($select);
        $html .= '&nbsp;'. $this->getDomainEdit($select, false);
        break;
      case 'rank_user':
      case 'pagein_user':
      case 'pageout_user':
      case 'referer_user':
      case 'adsensepagerank_user':
      case 'btnrank_user':
      case 'clickrank_user':
        $html .= Track::checkUrl($select) ? '<a href="'. $this->getJumpUrl($select). '" title="'. $select. '" >'. $this->getTitle($select). '</a>' : $this->escapeHtml($select);
        break;
      case 'time_detail':
        $html .= $this->getZeroSuppress($select, false);
        break;
      default:
        $html .= $this->escapeHtml($select);
        break;
    }

    return $html;
  }

  public function getTitle($url) {
    $titleData = $this->result->get('titleData');
    $rtn = $url;
    if (is_array($titleData)) {
      foreach($titleData as $value) {
        if ($value['url']==$url) {
          $rtn = $value['title'];
          break;
        }
      }
    }
    return $rtn;
  }

  public function getAlias($uid) {
    $aliasData = $this->session->get('aliasData');

    $rtn = $uid;
    foreach($aliasData as $value) {
      if ($value['uid']==$uid) $rtn = $value['name'];
    }
    return $this->escapeHtml($rtn);
  }

  public function getDomain($domain=null) {
    $rtn = '';
    if (is_null($domain)) $domain = $this->result->get('domain');
    if (is_array($domain)) {
      if (count($domain) == 0) return $rtn;
      foreach($domain as $v) {
        $domain = $v;
        break;
      }
    } 
    $rtn = $domain;
    $domainData = $this->session->get('domainData');
    foreach($domainData as $value) {
      if ($value['domain'] == $domain) $rtn = $value['name'];
    }
    return $this->escapeHtml($rtn);
  }

  public function getMenuTermSelect($fromto) {
    $yyyyFrom = !is_null($this->request->get('yyyy_from')) ? $this->request->get('yyyy_from') : $this->session->get('yyyyFrom');
    $mmFrom =  !is_null($this->request->get('mm_from')) ? $this->request->get('mm_from') : $this->session->get('mmFrom');
    $ddFrom = !is_null($this->request->get('dd_from')) ? $this->request->get('dd_from') : $this->session->get('ddFrom');
    $yyyyTo = !is_null($this->request->get('yyyy_to')) ? $this->request->get('yyyy_to') : $this->session->get('yyyyTo');
    $mmTo =  !is_null($this->request->get('mm_to')) ? $this->request->get('mm_to') : $this->session->get('mmTo');
    $ddTo = !is_null($this->request->get('dd_to')) ? $this->request->get('dd_to') : $this->session->get('ddTo');
    $research = $this->_research; 

    $now = time();
    $yyyy = date('Y', $now);
    $selectYyyy = array(
                          $yyyy - 4 => $yyyy -4,
                          $yyyy - 3 => $yyyy -3,
                          $yyyy - 2 => $yyyy -2,
                          $yyyy - 1 => $yyyy -1,
                          $yyyy     => $yyyy
                 );
    $selectMm = array(
                          '01' => '1',
                          '02' => '2',
                          '03' => '3',
                          '04' => '4',
                          '05' => '5',
                          '06' => '6',
                          '07' => '7',
                          '08' => '8',
                          '09' => '9',
                          '10' => '10',
                          '11' => '11',
                          '12' => '12'
                 );
    $selectDd = array(
                          '01' => '1',
                          '02' => '2',
                          '03' => '3',
                          '04' => '4',
                          '05' => '5',
                          '06' => '6',
                          '07' => '7',
                          '08' => '8',
                          '09' => '9',
                          '10' => '10',
                          '11' => '11',
                          '12' => '12',
                          '13' => '13',
                          '14' => '14',
                          '15' => '15',
                          '16' => '16',
                          '17' => '17',
                          '18' => '18',
                          '19' => '19',
                          '20' => '20',
                          '21' => '21',
                          '22' => '22',
                          '23' => '23',
                          '24' => '24',
                          '25' => '25',
                          '26' => '26',
                          '27' => '27',
                          '28' => '28',
                          '29' => '29',
                          '30' => '30',
                          '31' => '31'
                   );
    $html = '<select name="yyyy_' . $fromto . '">';
    foreach ($selectYyyy as $key => $value) {
      if (($fromto == 'from' && $key == $yyyyFrom) || ($fromto == 'to' && $key == $yyyyTo)) {
        $html .= '<option value="' . $key . '" selected="selected">' . $value . '</option>';
      } else {
        $html .= '<option value="' . $key . '">' . $value . '</option>';
      }
    }
    $html .= '</select>';
    $html .= '<select name="mm_' . $fromto . '">';
    foreach ($selectMm as $key => $value) {
      if (($fromto == 'from' && $key == $mmFrom) || ($fromto == 'to' && $key == $mmTo)) {
        $html .= '<option value="' . $key . '" selected="selected">' . $value . '</option>';
      } else {
        $html .= '<option value="' . $key . '">' . $value . '</option>';
      }
    }
    $html .= '</select>';
    $html .= '<select name="dd_' . $fromto . '">';
    foreach ($selectDd as $key => $value) {
      if ($key > '31') break;
      if (($fromto == 'from' && $key == $ddFrom) || ($fromto == 'to' && $key == $ddTo)) {
        $html .= '<option value="' . $key . '" selected="selected">' . $value . '</option>';
      } else {
        $html .= '<option value="' . $key . '">' . $value . '</option>';
      }
    }
    $html .= '</select>';
    return $html;
  }

  public function getDomainEdit($domain, $divTag=true){
    $rtn = '';
    if (trim($domain) != '' && $domain != Config::NO_DATA) {
      $domainName = $this->getDomain($domain);
      if ($divTag) {
        $rtn .= '<div class="domain">';
      } else {
        $rtn .= '<span class="domain">';
      }
      $rtn .= '<a title="'. $domainName. '" href="'. $this->getIndexUrl('admin', 'domainedit', '&amp;select='. $this->urlEncode($domain)). '">ドメイン名設定</a>';
      if ($divTag) {
        $rtn .= '</div>';
      } else {
        $rtn .= '</span>';
      }
    }
    return $rtn;
  }

  public function versionTag() {
    print RaConfig::RA_NAME. ' '. $this->getVersion();
  }

  public function indexUrl($controller, $action, $querystring=null) {
    print $this->getIndexUrl($controller, $action, $querystring);
  }

  public function jumpUrl($link) {
    print $this->getJumpUrl($link);
  }

  public function copyrightTag() {
    $thisYear = date('Y');
    $yearString = $thisYear <= Config::RELEASE_YEAR ? Config::RELEASE_YEAR : Config::RELEASE_YEAR. '-'. $thisYear;
    print 'Copyright &copy; '. $yearString. ' '. RAConfig::RA_PROJECT. ' All Rights Reserved.';
  }

  public function systemDataSize() {
    $siteData = $this->session->get('siteData');
    print $this->getFormatDataSize(SystemUtil::getSystemDataSize(str_replace('-', '', substr($siteData['created_on'], 0, 7))));
  }

  public function messageTag() {
    $code = $this->message->getCode();
    $message = $this->getMessage(); 
    print !is_null($message) ? $message : '';
  }

  public function errorMessageTag() {
    $code = $this->message->getCode();
    $message = $this->getMessage(); 
    print !is_null($message) ? '<div class="fatal">'. $message. '</div>' : '';
  }

  public function confirmMessageTag() {
    $code = $this->message->getCode();
    $message = $this->getMessage(); 
    print !is_null($message) ? '<div class="notice">'. $message. '</div>' : '';
  }

  public function actionTitle($char=false) {
    print $this->getActionTitle($char);
  }

  public function select() {
    print $this->getSelect();
  }

  public function alias() {
    print $this->getAlias($this->getSelect());
  }

  public function domain() {
    print $this->getDomain($this->getSelect());
  }

  public function domainEdit($domain, $divTag=true){
    print $this->getDomainEdit($domain, $divTag);
  }

  public function htmlChecked($value, $compvalue) {
    print $value == $compvalue ? 'checked=\'checked\'' : '';
  }

  public function targetBlank($targetflg) {
    print $targetflg == RaConfig::ON ? ' onclick=\'window.open(this.href);return false;\'' : '';
  }

  public function jsTag($ssl=false) {
    $host = $_SERVER['SERVER_NAME']. str_replace(ADMIN_DIR_NAME. '/'. Config::INDEX_FILENAME, '', $_SERVER['PHP_SELF']);
    $protocol = $ssl ? 'https://' : 'http://';
    $html = '<script type="text/javascript" src="'. $protocol. $host. 'script.php"></script>';
    $html .= '<noscript><p><img src="'. $protocol. $host. 'track.php" alt="" width="1" height="1" /></p></noscript>';
    print $this->escapeHtml($html);
  }

  public function imgTag($ssl=false) {
    $host = $_SERVER['SERVER_NAME']. str_replace(ADMIN_DIR_NAME. '/'. Config::INDEX_FILENAME, '', $_SERVER['PHP_SELF']);
    $protocol = $ssl ? 'https://' : 'http://';
    $html = '<img src="'. $protocol. $host. 'track.php" alt="" width="1" height="1" />';
    print $this->escapeHtml($html);
  }

  public function imgTagPage($ssl=false) {
    $host = $_SERVER['SERVER_NAME']. str_replace(ADMIN_DIR_NAME. '/'. Config::INDEX_FILENAME, '', $_SERVER['PHP_SELF']);
    $protocol = $ssl ? 'https://' : 'http://';
    $html = '<img src="'. $protocol. $host. 'track.php?page=ページ1" alt="" width="1" height="1" />';
    print $this->escapeHtml($html);
  }

  public function phpCode($title=false) {
    $phpTrack = realpath(dirname(SETTING_PATH_FILE). DIRECTORY_SEPARATOR. '..'). DIRECTORY_SEPARATOR. 'phptrack.php';
    $code = '<?php'. "\r\n";
    $code .= '  include \''. $phpTrack. '\';'. "\r\n";
    $code .= $title ? '  _raTrack(タイトルの値);' : '  _raTrack();';
    $code .= "\r\n";
    $code .= '?>';
    print $code;
  }

  public function searchFormTag($label) {
    $controller = $this->controller;
    $action = $this->action;
    $html = '';
    $formValue = $this->escapeHtml($this->request->get('search'));
    if (RaUtil::strLen(trim($formValue)) > 0) $html = '<h3>&quot;' . $formValue . '&quot;で検索した結果</h3>';
    $querystring = '';
    if (!is_null($this->request->get('search'))) $querystring .= '&amp;search=' . $this->urlEncode($this->request->get('search'));
    if (!is_null($this->request->get('select'))) $querystring .= '&amp;select=' . $this->urlEncode($this->request->get('select'));
    $html .= '<form id="search" action="' . $this->getIndexUrl($controller, $action, $querystring) . '" method="post">';
    $html .= '<dl class="searchdl">';
    $html .= '<dt class="searchdt">' . $label . ':</dt>';
    $html .= '<dd class="searchdd"><input type="text" class="text" name="search" maxlength="100" value="' . $formValue . '" /></dd>';
    $html .= '<dd class="searchdd"><input class="button" type="submit" value=" 検索 " /></dd>';
    $html .= '</dl>';
    $html .= '</form>';
    print $html;
  }

  public function radioChecked($name, $value) {
    $checked = '';
    if (!is_null($this->request->get($name)) && $this->request->get($name) == $value) $checked = 'checked="checked"';
    print $checked;
  }

  public function pageTag($pageCount) {
    $controller = $this->controller;
    $action = $this->action;
    $siteData = $this->session->get('siteData');

    if (substr_count($pageCount, ',') > 0) $pageCount = str_replace(',', '', $pageCount);
 
    $querystring = '';
    if (!is_null($this->request->get('search'))) $querystring .= '&amp;search=' . $this->urlEncode($this->request->get('search'));
    if (!is_null($this->request->get('select'))) $querystring .= '&amp;select=' . $this->urlEncode($this->request->get('select'));
    if (!is_null($this->request->get('browser'))) $querystring .= '&amp;browser=' . $this->urlEncode($this->request->get('browser'));
    if (!is_null($this->request->get('engine'))) $querystring .= '&amp;engine=' . $this->urlEncode($this->request->get('engine'));
    $querystring .= '&amp;page=';
    $href = $this->getIndexUrl($controller, $action, $querystring);

    $html = '<div class="page">';
    $page = !is_null($this->request->get('page')) ? $this->request->get('page') : 1;
    $lastpage = ceil($pageCount / $siteData['dispview']);
    if ($lastpage < Config::MAX_PAGE) {
      $maxpage = $lastpage;
      $cnt = 1;
    } else {
      $base = ceil(Config::MAX_PAGE / 2);
      if ($page > $base) {
        $maxpage = Config::MAX_PAGE + ($page - $base);
        if ($maxpage > $lastpage) {
          $maxpage = $lastpage;
        }
        $cnt = 1 + ($page - $base);
        if (($maxpage - $cnt) < Config::MAX_PAGE) {
          while(($maxpage - $cnt+1) < Config::MAX_PAGE) {
            $cnt--;
          }
        }
      } else {
        $maxpage = Config::MAX_PAGE;
        $cnt = 1;
      }
    }
    if ($page > 1 && $page <= $maxpage) {
      $html .= '<a href="'. $href. '1'. '">&#8810;最初へ</a>';
      $html .= '<a href="'. $href. ($page-1). '">&#8810;前の'. $siteData['dispview']. '件</a>';
    }
    for ($i = $cnt; $i <= $maxpage; $i++) {
      if ($i == $page) {
        if ($lastpage > 1) {
          $html .= '<span class="thispage">' . $i . '</span>';
        }
      } else {
        $html .= '<a href="'. $href. $i. '">'. $i. '</a>';
      }
    }
    if ($maxpage > $page) {
      $html .= '<a href="'. $href. ($page+1). '">次の'. $siteData['dispview']. '件&#8811;</a>';
      $html .= '<a href="'. $href. $lastpage. '">最後へ&#8811;</a>';
    }
    $html .= '</div>';
    print $html;
  }

  public function menuCalendarHeaderTag() {
    $yyyyFrom = $this->session->get('yyyyFrom');
    $mmFrom =  $this->session->get('mmFrom');
    $ddFrom = $this->session->get('ddFrom');
    $research = $this->_research; 
    $controller = $this->controller;
    $action = $this->action;

    $convertAction = $this->_getConvertAction($action);
    $convertAction = $this->_getCalendarConvertAction($convertAction);

    $lastMonths = $this->_getLastNextMonths('last', $yyyyFrom, $mmFrom);
    $nextMonths = $this->_getLastNextMonths('next', $yyyyFrom, $mmFrom);
    if ($research == self::RESEARCH_DAY) {
      $lastDdFrom = checkdate($lastMonths['mm'], $ddFrom, $lastMonths['yyyy']) ? $ddFrom : '01';
      $lastDdTo = $lastDdFrom;
      $nextDdFrom = checkdate($nextMonths['mm'], $ddFrom, $nextMonths['yyyy']) ? $ddFrom : '01';
      $nextDdTo = $nextDdFrom;
    } else {
      $lastDdFrom = '01';
      $lastDdTo = Calendar::getLastDay($lastMonths['yyyy'], $lastMonths['mm']);
      $nextDdFrom = '01';
      $nextDdTo = Calendar::getLastDay($nextMonths['yyyy'], $nextMonths['mm']);
    }
    $clsName = $convertAction == 'term' ? 'select' : '';
    $html = '<a class="selectmonth" href="'. $this->getIndexUrl($controller, $convertAction, 
                                                                 '&amp;yyyy_from=' . $lastMonths['yyyy'] .                                                                                  
                                                                 '&amp;mm_from=' . $lastMonths['mm'] .                                                                                  
                                                                 '&amp;dd_from=' . $lastDdFrom .                                                                                  
                                                                 '&amp;yyyy_to=' . $lastMonths['yyyy'] .                                                                                  
                                                                 '&amp;mm_to=' . $lastMonths['mm'] .                                                                                  
                                                                 '&amp;dd_to=' . $lastDdTo) .                                                                                  
                                                                 '" title="前月へ">&#8810;</a>';
    $html .= '<a class="selectmonth'. $clsName. '" href="'.$this->getIndexUrl('research', 'term',
                                                                 '&amp;yyyy_from=' . $yyyyFrom .                                                                                  
                                                                 '&amp;mm_from=' . $mmFrom .                                                                                  
                                                                 '&amp;dd_from=' . '01' .                                                                                  
                                                                 '&amp;yyyy_to=' . $yyyyFrom .                                                                                  
                                                                 '&amp;mm_to=' . $mmFrom .                                                                                  
                                                                 '&amp;dd_to=' . Calendar::getLastDay($yyyyFrom, $mmFrom)) .                                                                                  
                                                                 '" title="' . $this->getZeroSuppress($yyyyFrom). '年'. $this->getZeroSuppress($mmFrom) . '月">' . $this->getZeroSuppress($yyyyFrom). '年'. $this->getZeroSuppress($mmFrom) . '月</a>';
    $html .= '<a class="selectmonth" href="'. $this->getIndexUrl($controller, $convertAction, 
                                                                 '&amp;yyyy_from=' . $nextMonths['yyyy'] .                                                                                  
                                                                 '&amp;mm_from=' . $nextMonths['mm'] .                                                                                  
                                                                 '&amp;dd_from=' . $nextDdFrom .                                                                                  
                                                                 '&amp;yyyy_to=' . $nextMonths['yyyy'] .                                                                                  
                                                                 '&amp;mm_to=' . $nextMonths['mm'] .                                                                                  
                                                                 '&amp;dd_to=' . $nextDdTo) .                                                                                  
                                                                 '" title="次月へ">&#8811;</a>';
    print $html;
  }

  public function menuCalendarTag() {
    $yyyyFrom = $this->session->get('yyyyFrom');
    $mmFrom =  $this->session->get('mmFrom');
    $ddFrom = $this->session->get('ddFrom');
    $yyyyTo = $this->session->get('yyyyTo');
    $mmTo =  $this->session->get('mmTo');
    $ddTo = $this->session->get('ddTo');
    $research = $this->_research; 
    $controller = $this->controller;
    $action = $this->action;

    $convertAction = $this->_getConvertAction($action);
    $convertAction = $this->_getCalendarConvertAction($convertAction);

    $now = time();
    $wday = 0; 
    $mday = 1; 
    $firstDay = getDate(mktime(0,0,0,$mmFrom,1,$yyyyFrom)); 
  
    $html = '<table class="calendar">';
    $html .= '<tr class="calendardateheaders"><th>日</th><th>月</th><th>火</th><th>水</th><th>木</th><th>金</th><th>土</th></tr>';
    $html .= '<tr>'; 
    while ($wday < $firstDay['wday']) { 
      $wday++; 
      $html .= '<td>&nbsp;</td>';
    } 
    while (checkdate($mmFrom, $mday, $yyyyFrom)) { 
      switch ($research) {
        case self::RESEARCH_DAY:
          if ($mday == $ddFrom) {
            $html .=  '<td class="today">';
          } else {
            $html .=  '<td class="' . $this->_getMenuWeekdayCssClass($yyyyFrom, $mmFrom, $mday) . '">';
          }
          break;
        case self::RESEARCH_MONTH:
          $html .=  '<td class="today">';
          break;
        case self::RESEARCH_TERM:
          $termday = $this->getZeroPadding($mday, 1);
          if ($yyyyFrom . $mmFrom . $termday >= $yyyyFrom . $mmFrom . $ddFrom && $yyyyFrom . $mmFrom . $termday <= $yyyyTo . $mmTo . $ddTo ) {
            $html .=  '<td class="today">';
          } else {
            $html .=  '<td class="' . $this->_getMenuWeekdayCssClass($yyyyFrom, $mmFrom, $mday) . '">';
          }
          break;
      }
      $dispday = $mday;
      $mday = $this->getZeroPadding($mday, 1);
      if ($yyyyFrom . $mmFrom == date('Y', $now) . date('m', $now)) {
        if ($mday > date('d', $now)) {
          $html .=  '<div class="days">' . $dispday . '</div></td>'; 
        } else {
          $html .=  '<div class="days">' . '<a href="'. $this->getIndexUrl($controller, $convertAction, 
                                           '&amp;yyyy_from=' . $yyyyFrom .
                                           '&amp;mm_from=' . $mmFrom .
                                           '&amp;dd_from=' . $mday .
                                           '&amp;yyyy_to=' . $yyyyFrom .
                                           '&amp;mm_to=' . $mmFrom .
                                           '&amp;dd_to=' . $mday) .
                                           '" title="' . $dispday . '日">' .  $dispday . '</a></div></td>'; 
        }
      } elseif ($yyyyFrom . $mmFrom < date('Y', $now) . date('m', $now)) {
        $html .=  '<div class="days">' . '<a href="'. $this->getIndexUrl($controller, $convertAction,
                                           '&amp;yyyy_from=' . $yyyyFrom .
                                           '&amp;mm_from=' . $mmFrom .
                                           '&amp;dd_from=' . $mday .
                                           '&amp;yyyy_to=' . $yyyyFrom .
                                           '&amp;mm_to=' . $mmFrom .
                                           '&amp;dd_to=' . $mday) .
                                           '" title="' . $dispday . '日">' .  $dispday . '</a></div></td>'; 
      } else {
        $html .=  '<div class="days">' . $dispday . '</div></td>'; 
      }
      $mday++; $wday++; 
      if (($wday > 6) && (checkdate($mmFrom, $mday, $yyyyFrom))) { 
        $html .=  '</tr><tr>'; 
        $wday = 0; 
      } 
    }
    while ($wday++ < 7) $html .=  '<td>&nbsp;</td>'; 
    $html .= '</tr>'; 
    $html .= '</table>'; 
    print $html;
  }

  public function menuResearchDayMonthTag($dispResearch) {
    $yyyyFrom = $this->session->get('yyyyFrom');
    $mmFrom =  $this->session->get('mmFrom');
    $ddFrom = $this->session->get('ddFrom');
    $research = $this->_research; 
    $controller = $this->controller;
    $action = $this->action;

    $convertAction = $this->_getConvertAction($action);
    $convertAction = $this->_getCalendarConvertAction($convertAction);

    switch($dispResearch) {
      case self::RESEARCH_DAY:
        $title = $this->getZeroSuppress($mmFrom) . '月' . $this->getZeroSuppress($ddFrom) . '日の解析';
        $ddTo = $ddFrom;
        break;
      case self::RESEARCH_MONTH:
        $title = $this->getZeroSuppress($mmFrom) . '月の解析';
        $ddFrom = '01';
        $ddTo = Calendar::getLastDay($yyyyFrom, $mmFrom);
        break;
    } 
    $clsName = $research == $dispResearch ? 'select' : '';
    $html = '<a class="selectresearch' . $clsName . '" href="'. $this->getIndexUrl($controller, $convertAction, 
                                                                 '&amp;yyyy_from=' . $yyyyFrom .
                                                                 '&amp;mm_from=' . $mmFrom .
                                                                 '&amp;dd_from=' . $ddFrom .
                                                                 '&amp;yyyy_to=' . $yyyyFrom .
                                                                 '&amp;mm_to=' . $mmFrom .
                                                                 '&amp;dd_to=' . $ddTo) .
                                                                 '" title="' . $title . '">' . $title . '</a>';
    print $html;
  }


  public function menuTermFormTag() {
    $research = $this->_research; 
    $action = $this->action;
    $controller = $this->controller;
    
    $convertAction = $this->_getConvertAction($action);
    $convertAction = $this->_getCalendarConvertAction($convertAction);

    $clsName = $research == self::RESEARCH_TERM ? 'select' : '';
    $html  = '<div class="selectresearchterm' . $clsName . '">';
    $html .= '<form id="searchterm" action="'. $this->getIndexUrl($controller, $convertAction) . '" method="post">';
    $html .= '<dl class="termdl">';
    $html .= '<dd class="termdd">';
    $html .= $this->getMenuTermSelect('from');
    $html .= '</dd>';
    $html .= '<dd class="termddcenter">&rarr;</dd>';
    $html .= '<dd class="termdd">';
    $html .= $this->getMenuTermSelect('to');
    $html .= '</dd>';
    $html .= '<dd class="termddcenter">&nbsp;</dd>';
    $html .= '<dd class="termdd">';
    $html .= '<input type="submit" value="期間選択" />';
    $html .= '</dd>';
    $html .= '</dl>';
    $html .= '</form>';
    $html .= '</div>';
    print $html;
  }

  public function menuTitlesTag() {
    $html = '';
    $html .= $this->_getMenuTitles(MenuConfig::$titles);
    print $html;
  }
  
  public function menuSelectHeaderTag() {
    $yyyyFrom = $this->session->get('yyyyFrom');
    $mmFrom =  $this->session->get('mmFrom');
    $ddFrom = $this->session->get('ddFrom');
    $yyyyTo = $this->session->get('yyyyTo');
    $mmTo =  $this->session->get('mmTo');
    $ddTo = $this->session->get('ddTo');
    $research = $this->_research; 
    $selectPage = $this->session->get('selectPage'); 
    $selectTitle = $this->session->get('selectTitle'); 
    $selectName = $this->session->get('selectName'); 
    $controller = $this->controller; 
    $action = $this->action; 
    $siteData = $this->session->get('siteData'); 

    $html = '';

    $systemMessage = SystemUtil::getSystemMessage();
    if (is_array($systemMessage)) {
      $html .= '<div class="system_message">';
      foreach ($systemMessage as $message) {
        $html .= '<p>'. $message. '</p>';
      }
      $html .= '</div>';
    }
    $actionTitle = $this->getActionTitle();
    $html .= trim($actionTitle) != '' ? '<h1>'. $actionTitle. '</h1>' : '';
    $html .= '<table class="header_value">';
    $html .= '<tr>';
    $html .= '<td class="label">サイト名</td>';
    $html .= '<td class="label">&nbsp;</td>';
    $html .= '<td class="value"><span class="header_value"><a href="'.$this->getJumpUrl($siteData['url']). '" title="'. $siteData['url']. '">'. $siteData['sitename']. '</a></span></td>';
    $html .= '</tr>';
    if (($controller == 'research' && ($action != 'download' && $action != 'log')) ||
        ($controller == 'admin' && ($action == 'selectpage' || $action == 'selectuid'))) {
      if ($research == self::RESEARCH_DAY) {
        $html .= '<tr>';
        $html .= '<td class="label">解析対象日</td>';
        $html .= '<td class="label">&nbsp;</td>';
        $html .= '<td class="value"><span class="header_value">' . $yyyyFrom . '年' . $this->getZeroSuppress($mmFrom) . '月' . $this->getZeroSuppress($ddFrom) . '日' . '</span></td>';
        $html .= '</tr>';
      } elseif ($research == self::RESEARCH_MONTH) {
        $html .= '<tr>';
        $html .= '<td class="label">解析対象月</td>';
        $html .= '<td class="label">&nbsp;</td>';
        $html .= '<td class="value"><span class="header_value">' . $yyyyFrom . '年' . $this->getZeroSuppress($mmFrom) . '月' . '</span></td>';
        $html .= '</tr>';
      } elseif ($research == self::RESEARCH_TERM) {
        $html .= '<tr>';
        $html .= '<td class="label">解析対象期間</td>';
        $html .= '<td class="label">&nbsp;</td>';
        $html .= '<td class="value"><span class="header_value">' . $yyyyFrom . '年' . $this->getZeroSuppress($mmFrom) . '月' . $this->getZeroSuppress($ddFrom) . '日～' 
                                   . $yyyyTo . '年' . $this->getZeroSuppress($mmTo) . '月' . $this->getZeroSuppress($ddTo) . '日' . '</span></td>';
        $html .= '</tr>';
      }
      if (!is_null($selectTitle)) {
        $html .= '<tr>';
        $html .= '<td class="label">選択ページ</td>';
        $html .= '<td class="label">&nbsp;</td>';
        $html .= '<td class="value"><span class="header_value"><a href="'. $this->getJumpUrl($selectPage). '" title="'. $selectPage. '">' . $this->escapeHtml($selectTitle) . '</a></span></td>';
        $html .= '</tr>';
      }
      if (!is_null($selectName)) {
        $html .= '<tr>';
        $html .= '<td class="label">選択訪問者</td>';
        $html .= '<td class="label">&nbsp;</td>';
        $html .= '<td class="value"><span class="header_value">' . $this->escapeHtml($selectName) . '</span></td>';
        $html .= '</tr>';
      }
    }
    $html .= '</table>';
    print $html;
  }

  public function menuTermSelectTag($fromto) {
    print $this->getMenuTermSelect($fromto);
  }

  private function _getLastNextMonths($lastNext, $yyyy, $mm) {

     switch ($lastNext) {
        case 'last':
          $month = $mm - 1; 
          $year = $yyyy; 
          if (!checkdate($month, 1, $year)) { 
            $month += 12; 
            $year--; 
          } 
          break;   
        default:
          $month = $mm + 1; 
          $year = $yyyy; 
          if (!checkdate($month, 1, $year)) { 
            $year++; 
            $month -= 12; 
          } 
          break;   
      }
      if ($month < 10) $month = '0'.$month; 
      return array('yyyy' => $year, 'mm' => $month);
  }
  
  private function _getMenuWeekdayCssClass($yyyy, $mm, $dd) {
    $clsnm = 'weekday';
    $dates = getdate(mktime(0, 0, 0, $mm, $dd, $yyyy));
    $wday = $dates['wday'];
    if ($wday == 0) $clsnm = 'sun';
    if ($wday == 6) $clsnm = 'sat';
    return $clsnm;
  }

  private function _getResearch() {
    $yyyyFrom = $this->session->get('yyyyFrom');
    $mmFrom =  $this->session->get('mmFrom');
    $ddFrom = $this->session->get('ddFrom');
    $yyyyTo = $this->session->get('yyyyTo');
    $mmTo =  $this->session->get('mmTo');
    $ddTo = $this->session->get('ddTo');

    $research = self::RESEARCH_TERM;
    if ($yyyyFrom. $mmFrom. $ddFrom == $yyyyTo. $mmTo. $ddTo) $research = self::RESEARCH_DAY; 
    if (($yyyyFrom. $mmFrom == $yyyyTo. $mmTo) && ($ddFrom == '01' && $ddTo == Calendar::getLastDay($yyyyFrom, $mmFrom))) $research = self::RESEARCH_MONTH; 

    return $research;
  }

  private function _getConvertAction($action) {
    return isset(MenuConfig::$convertActions[$action]) ? MenuConfig::$convertActions[$action] : $action;
  }

  private function _getCalendarConvertAction($action) {
    return isset(MenuConfig::$calendarConvertActions[$action]) ? MenuConfig::$calendarConvertActions[$action] : $action;
  }

  private function _getMenuTitles($menuTitles) {
    $yyyyFrom = $this->session->get('yyyyFrom');
    $mmFrom =  $this->session->get('mmFrom');
    $action = $this->action;
    $convertAction = $this->_getConvertAction($action);

    $html = '';
    foreach($menuTitles as $titles) {
      $html .= '<div class="menuline">';
      foreach ($titles as $key => $value) {
        if (!isset(MenuConfig::$actionControllers[$key])) continue;
        $controller = MenuConfig::$actionControllers[$key];
        $clsName = isset(MenuConfig::$actionStyleSheets[$key]) ? MenuConfig::$actionStyleSheets[$key] : 'selectresearch';
        $clsName .= $convertAction == $key ? 'select' : '';
        $html .= '<div class="adjust">';
        if (in_array($key, MenuConfig::$monthActions)) {
          $html .= '<a class="'. $clsName . '" href="'. $this->getIndexUrl($controller, $key,
                                                                                        '&amp;yyyy_from='. $yyyyFrom.
                                                                                        '&amp;mm_from='. $mmFrom. 
                                                                                        '&amp;dd_from='. '01'. 
                                                                                        '&amp;yyyy_to='. $yyyyFrom. 
                                                                                        '&amp;mm_to='. $mmFrom. 
                                                                                        '&amp;dd_to='. Calendar::getLastDay($yyyyFrom, $mmFrom)). 
                                                                                        '">' . $value . '</a>';
        } else {
          $html .= '<a class="'. $clsName . '" href="'. $this->getIndexUrl($controller, $key) . '">' . $value . '</a>';
        }
        $html .= '</div>';
      }
      $html .=  '</div>';
    }
    return $html;
  }

}
?>
