/*!
  \example md_scan.c

  \brief MD スキャンサンプル

  MD コマンドを用いて、距離データを指定回数だけ連続して取得する

  \author Satofumi KAMIMURA

  $Id: md_scan.c 1298 2009-09-14 20:46:35Z satofumi $
*/

#include "urg_ctrl.h"
#include "delay.h"
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>


static void urg_exit(urg_t *urg, const char *message)
{
  printf("%s: %s\n", message, urg_error(urg));
  urg_disconnect(urg);

#ifdef MSC
  getchar();
#endif
  exit(1);
}


//! main
int main(int argc, char *argv[])
{
  enum {
    CaptureTimes = 10,          /* データ取得回数 */
  };

  //const char device[] = "COM3"; /* Windows での例 */
  const char device[] = "/dev/ttyACM0"; /* Linux での例 */

  int data_max;
  long* data;
  int timestamp;
  int previous_timestamp;
  int remain_times;
  int scan_msec;
  urg_parameter_t parameter;
  int ret;
  int n;
  int i;

  /* 接続 */
  urg_t urg;
  ret = urg_connect(&urg, device, 115200);
  if (ret < 0) {
    urg_exit(&urg, "urg_connect()");
    exit(1);
  }

  /* 受信バッファの確保 */
  data_max = urg_dataMax(&urg);
  data = (long*)malloc(sizeof(long) * data_max);
  if (data == NULL) {
    fprintf(stderr, "data_max: %d\n", data_max);
    perror("data buffer");
    exit(1);
  }
  urg_parameters(&urg, &parameter);
  scan_msec = urg_scanMsec(&urg);

  /* MD でのデータ要求。取得回数は CaptureTimes 回 */
  /* 100 回以上の連続データを取得するには、取得回数を無限回に設定すること */
  /* urg_setCaptureTimes(&urg, UrgInfinityTimes); */
  assert(CaptureTimes < 100);
  urg_setCaptureTimes(&urg, CaptureTimes);

  /* データ要求 */
  ret = urg_requestData(&urg, URG_MD, URG_FIRST, URG_LAST);
  if (ret < 0) {
    urg_exit(&urg, "urg_requestData()");
  }

  // !!! わかりにくい。作り直すこと

  /* データ取得 */
  for (i = 0; i < CaptureTimes; ++i) {
    /* 受信 */
    n = urg_receiveData(&urg, data, data_max);
    printf("n = %d\n", n);
    if (n < 0) {
      urg_exit(&urg, "urg_receiveData()");
    } else if (n == 0) {
      //delay(scan_msec / 2);
      printf("n == 0\n");
      --i;
      continue;
    }

    /* 前方の距離データを、タイムスタンプ付きで表示する */
    /*
      PC 側の受信が遅延した場合、URG は送信できないデータを破棄するため、
      remain_times は不連続になる場合がある
    */
    previous_timestamp = timestamp;
    timestamp = urg_recentTimestamp(&urg);
    remain_times = urg_remainCaptureTimes(&urg);

    /* urg_minDistance() 以下の距離データは、無効値を表す */
    printf("%d/%d: %ld [mm], %d [msec], (%d)\n",
           remain_times, CaptureTimes, data[parameter.area_front_], timestamp,
           timestamp - previous_timestamp);

    printf("%d, %d\n", i, remain_times);

    if (remain_times <= 0) {
      break;
    }
  }

  urg_disconnect(&urg);
  free(data);

#ifdef MSC
  getchar();
#endif

  return 0;
}
